<?php

/**
 * Command-line utility to convert usual PNG/JPG/BMP file to ZX Spectrum binary sprite
 *
 * @author Epsilon
 * @version 1.1 [2016/02/21]
 * @license MIT
 */

$cwd = getcwd().'/';

$out = false;
$src = false;

$opts = array(
	'b' => false,
);

foreach ($argv as $k => $ar) {
	if ($k == 0) {
		continue;
	}
	if ($k == 1) {
		$out = $ar;
	} else {
		if (preg_match('~^\-[a-z0-9]+$~', $ar)) {
			// Option
			if ($ar == '-b') {
				$opts['b'] = true;	// B/W mode
			}
		} else {
			if ($k == count($argv) - 1) {
				$src = $ar;
			}
		}
	}
}

if ($src !== false) {
	
	$src = $cwd.$src;
	
	if (is_file($src) && file_exists($src)) {
		// Ok
		$a = file_get_contents($src);
	} else {
		echo 'Can\'t read input file "'.$src.'"';
		exit(1);
	}
	
} else {
	echo 'No source file specified';
	exit(1);
}


if ($out !== false) {
	$out = $cwd.$out;
	
	// Try to create output file
	file_put_contents($out, '1');
	if (is_file($out) && (file_exists($out))) {
		// Ok
	} else {
		echo 'Can not create output file "'.$out.'"';
		exit(1);
	}
	
} else {
	echo 'No output file specified';
	exit(1);
}

// Go.

$rgb = array(
	array(0, 0, 0),
	array(0, 0, 128),
	array(128, 0, 0),
	array(128, 0, 128),
	array(0, 128, 0),
	array(0, 128, 128),
	array(128, 128, 0),
	array(128, 128, 128),
	array(0, 0, 0),
	array(0, 0, 255),
	array(255, 0, 0),
	array(255, 0, 255),
	array(0, 255, 0),
	array(0, 255, 255),
	array(255, 255, 0),
	array(255, 255, 255),
);
$ycc = array();
foreach ($rgb as $cl) {
	$ycc[] = rgb2ycc($cl);
}

function rgb2ycc($v) {
	$y = (16 * 256 + 65.738 * $v[0] + 129.057 * $v[1] + 25.064 * $v[2]) / 256;
	$cb = (128 * 256 - 37.945 * $v[0] - 74.494 * $v[1] + 112.439 * $v[2]) / 256;
	$cr = (128 * 256 + 112.439 * $v[0] - 94.154 * $v[1] - 18.285 * $v[2]) / 256;
	
	return array($y, $cb, $cr);
}

function deltaYcc($v1, $v2) {
	//return sqrt(1.4 * pow($v1[0] - $v2[0], 2) + 0.8 * pow($v1[1] - $v2[1], 2) + 0.8 * pow($v1[2] - $v2[2], 2));
	return sqrt(0.5 * pow($v1[0] - $v2[0], 2) + 1.4 * pow($v1[1] - $v2[1], 2) + 1.4 * pow($v1[2] - $v2[2], 2));
}

function closestYcc($i, $y) {
	
	global $ycc;
	
	$ycc_p = ($i == 0) ? array_slice($ycc, 0, 8) : array_slice($ycc, 8, 8);
	
	$min = 10000;
	$best = -1;
	foreach ($ycc_p as $k => $cl) {
		$cdelt = deltaYcc($cl, $y);
		if ($cdelt < $min) {
			$min = $cdelt;
			$best = $k;
		}
	}
	
	return array($best, $min);
}

// Get image info
$img = getimagesize($src);

if ($img && isset($img[0]) && ($img[0] > 0) && isset($img[1]) && ($img[1] > 0)) {
	
    switch ($img[2]) {
		case 1:
			$im = imagecreatefromgif($src);
			break;
		case 2:
			$im = imagecreatefromjpeg($src);
			break;
		case 3:
			$im = imagecreatefrompng($src);
			break;
		case 6:
			$im = imagecreatefrombmp($src);
			break;
		default:
			echo 'Source file image is not supported yet';
			exit(1);
	}
	
	$line_len = ceil($img[0] / 8);
	if (!$opts['b']) {
		$fullrow_len = $line_len * 9;
	} else {
		$fullrow_len = $line_len * 8;
	}
	
	$outbin = array();
	$outbin_len = ceil($img[1] / 8) * $fullrow_len;
	for ($i = 0; $i < $outbin_len; $i ++) {
		$outbin[$i] = 0;
	}
	
	$dst = imagecreatetruecolor($img[0] * 2, $img[1] * 2);
	
	for ($x = 0; $x < $img[0]; $x += 8) {
		for ($y = 0; $y < $img[1]; $y += 8) {
			
			if (!$opts['b']) {
				// Count optimal points color!
				$cstat0 = array();
				$cstat1 = array();
				for ($xx = $x; $xx < min($img[0], $x + 8); $xx ++) {
					for ($yy = $y; $yy < min($img[1], $y + 8); $yy ++) {
						$ct = imagecolorsforindex($im, imagecolorat($im, $xx, $yy));
						$ycc_s = rgb2ycc(array($ct['red'], $ct['green'], $ct['blue']));
						$clos0 = closestYcc(0, $ycc_s);
						if (isset($cstat0[$clos0[0]])) {
							$cstat0[$clos0[0]][0] ++;
							$cstat0[$clos0[0]][1] += $clos0[1];
						} else {
							$cstat0[$clos0[0]] = array(1, $clos0[1]);
						}

						$clos1 = closestYcc(1, $ycc_s);
						if (isset($cstat1[$clos1[0]])) {
							$cstat1[$clos1[0]][0] ++;
							$cstat1[$clos1[0]][1] += $clos1[1];
						} else {
							$cstat1[$clos1[0]] = array(1, $clos1[1]);
						}

						// Put closest color to dst
						$dst_r = $rgb[$clos0[0]][0];
						$dst_g = $rgb[$clos0[0]][1];
						$dst_b = $rgb[$clos0[0]][2];
						imagesetpixel($dst, $xx, $yy, imagecolorallocate($dst, $dst_r, $dst_g, $dst_b));
						$dst_r = $rgb[$clos1[0] + 8][0];
						$dst_g = $rgb[$clos1[0] + 8][1];
						$dst_b = $rgb[$clos1[0] + 8][2];
						imagesetpixel($dst, $xx, $yy + $img[1], imagecolorallocate($dst, $dst_r, $dst_g, $dst_b));
					}
				}

				//print_r($cstat0);
				//print_r($cstat1);
				// Select best stat (actually select proper brightness)
				uksort($cstat0, function($k1, $k2) use ($cstat0) {
					return $cstat0[$k1][0] < $cstat0[$k2][0];
				});
				reset($cstat0);
				$ct = current($cstat0);
				$dt0 = ($ct[0] > 0) ? $ct[1] / $ct[0] : 0;
				if (count($cstat0) > 1) {
					next($cstat0);
					$ct = current($cstat0);
					$dt0 = ($dt0 + (($ct[0] > 0) ? $ct[1] / $ct[0] : 0)) / 2;
				}

				uksort($cstat1, function($k1, $k2) use ($cstat1) {
					return $cstat1[$k1][0] < $cstat1[$k2][0];
				});
				reset($cstat1);
				$ct = current($cstat1);
				$dt1 = ($ct[0] > 0) ? $ct[1] / $ct[0] : 0;
				if (count($cstat1) > 1) {
					next($cstat1);
					$ct = current($cstat1);
					$dt1 = ($dt1 + (($ct[0] > 0) ? $ct[1] / $ct[0] : 0)) / 2;
				}

				$cst = ($dt0 < $dt1) ? $cstat0 : $cstat1;
				$br = ($dt0 < $dt1) ? 0 : 1;
				//$cst = $cstat0;
				//$br = 0;
				reset($cst);

				$paper = key($cst);
				if (count($cst) > 1) {
					next($cst);
					$ink = key($cst);
				} else {
					if ($paper < 4) {
						$ink = 7;
					} else {
						$ink = 0;
					}
				}
				//echo count($cst).'; paper = '.$paper.'; ink = '.$ink."\n";
			} else {
				$br = 1;
				$paper = 0;
				$ink = 7;
			}
			
			// Converting with selected papers and inks
			$paper_ycc = $ycc[$paper + $br * 8];
			$ink_ycc = $ycc[$ink + $br * 8];
			for ($yy = $y; $yy < $y + 8; $yy ++) {
				$byte = 0;
				$shft = 0x80;
				for ($xx = $x; $xx < $x + 8; $xx ++) {
					$ct = imagecolorsforindex($im, imagecolorat($im, $xx, $yy));
					$ycc_s = rgb2ycc(array($ct['red'], $ct['green'], $ct['blue']));
					
					if ((deltaYcc($paper_ycc, $ycc_s) < deltaYcc($ink_ycc, $ycc_s)) || ($xx >= $img[0]) || ($yy >= $img[1])) {
						$bit = 0;
					} else {
						$bit = 1;
						$byte |= $shft;
					}
					
					// Put closest color to dst
					$dst_r = $bit ? $rgb[$ink + $br * 8][0] : $rgb[$paper + $br * 8][0];
					$dst_g = $bit ? $rgb[$ink + $br * 8][1] : $rgb[$paper + $br * 8][1];
					$dst_b = $bit ? $rgb[$ink + $br * 8][2] : $rgb[$paper + $br * 8][2];
					imagesetpixel($dst, $xx + $img[0], $yy, imagecolorallocate($dst, $dst_r, $dst_g, $dst_b));
					
					$shft >>= 1;
				}
				$outbin[floor($y / 8) * $fullrow_len + ($yy - $y) * $line_len + floor($x / 8)] = chr($byte);
			}
			if (!$opts['b']) {
				$outbin[floor($y / 8) * $fullrow_len + 8 * $line_len + floor($x / 8)] = chr($br * 0x40 + $paper * 8 + $ink);
			}
			
			//echo 'x = '.$x.'; y = '.$y."\n";
			//print_r($cstat);
			
		}
		//exit();
	}
	
	imagecopy($dst, $im, $img[0], $img[1], 0, 0, $img[0], $img[1]);
	
	imagepng($dst, $cwd.'out.png', 0);
	
	file_put_contents($out, implode('', $outbin));
	
	echo 'Done.';
	
} else {
	echo 'Source file is not an image';
	exit(1);
}


