' ----------------------------------------------------------------
' ZX BASIC INTERFACE LIBRARY FOR THE BIFROST*2 ENGINE
'
' If you use this interface library, you must load afterwards the
' BIFROST*2 ENGINE and a multicolor tile set. For a detailed sample
' see file "bifrost2dm.bas".
'
' Original version and further information is available at
' ?
' ----------------------------------------------------------------

#ifndef __LIBRARY_BIFROST2__
#define __LIBRARY_BIFROST2__
#pragma push(case_insensitive)
#pragma case_insensitive = true

const BIFROST2STATIC AS UBYTE = 128
const BIFROST2DISABLED AS UBYTE = 255

' ----------------------------------------------------------------
' Install BIFROST*2 ENGINE
' ----------------------------------------------------------------
#define BIFROST2install() \
    asm                   \
        call 65226        \
    end asm

' ----------------------------------------------------------------
' Activate multicolor rendering with BIFROST*2 ENGINE
' ----------------------------------------------------------------
#define BIFROST2start() \
    asm                 \
        call 51625      \
    end asm

' ----------------------------------------------------------------
' Deactivate multicolor rendering with BIFROST*2 ENGINE
' ----------------------------------------------------------------
#define BIFROST2stop() \
    asm                \
        call 51634     \
    end asm

' ----------------------------------------------------------------
' Execute HALT (wait for next frame).
'
' If an interrupt occurs while certain routines (BIFROST2drawTileH,
' BIFROST2showTilePosH, BIFROST2showNextTile, BIFROST2fillTileAttrH)
' are under execution, the entire screen will "glitch".
'
' Routine BIFROST2halt can be used to avoid these problems.
' Immediately after calling it, your program will have some time
' to execute a few other routines without any interruption.
' ----------------------------------------------------------------
#define BIFROST2halt() \
    asm                \
        halt           \
    end asm

' ----------------------------------------------------------------
' Place a multicolor tile index into the tile map. Add value
' BIFROST2STATIC for static tile, otherwise it will be animated
'
' Parameters:
'     px: tile vertical position (0-10)
'     py: tile horizontal position (0-9)
'     tile: tile index (0-255)
' ----------------------------------------------------------------
#define BIFROST2setTile(px, py, tile)   POKE 65281+(px)*10+(py), (tile)

' ----------------------------------------------------------------
' Obtain a multicolor tile index from the tile map
'
' Parameters:
'     px: tile vertical position (0-10)
'     py: tile horizontal position (0-9)
'
' Returns:
'     Tile index currently stored in this position
' ----------------------------------------------------------------
#define BIFROST2getTile(px, py)   PEEK(65281+(px)*10+(py))

' ----------------------------------------------------------------
' Convert multicolor tile index into the equivalent animation group
'
' Parameters:
'     tile: tile index (0-255)
'
' Returns:
'     Animation group for animated tile, otherwise the same tile index
' ----------------------------------------------------------------
function FASTCALL BIFROST2getAnimGroup(tile as UBYTE) AS UBYTE
    asm
        cp 128          ; compare with BIFROSTSTATIC
        ret nc          ; if (static tile) return tile
        srl a
        ld b,a
        ld a,(51708)
        and a
        ld a,b
        ret z           ; if (2 frames per animation) return tile/2
        rra             ; if (4 frames per animation) return tile/4
    end asm
end function

' ----------------------------------------------------------------
' Locate memory address that stores the multicolor attribute of a
' certain screen position inside the multicolor area
'
' Parameters:
'     lin: pixel line (0-192)
'     col: char column (1-20)
'
' Returns:
'     Memory address of the multicolor attribute
' ----------------------------------------------------------------
function FASTCALL BIFROST2findAttrH(lin as UBYTE, col as UBYTE) as UINTEGER
    asm
        ld e,a          ; E=lin
        ld d,102        ; DE=lookup/2+lin
        pop hl          ; RET address
        ex (sp),hl      ; H=col
        inc h           ; H=col+1
        srl h           ; H=INT((col-1)/2)+1
        ex af,af'       ; preserve ((col-1)%2)
        xor a           ; A=0
        ld l,a          ; HL=256*INT((col-1)/2)+256
        ld a,h          ; AL=256*INT((col-1)/2)+256
        rra
        rr l            ; AL=128*INT((col-1)/2)+128
        add a,h         ; AL=384*INT((col-1)/2)+384
        ld h,a          ; HL=384*INT((col-1)/2)+384
        add hl, de      ; HL=384*INT((col-1)/2)+384+lookup/2+lin
        add hl, de      ; HL=384*INT((col-1)/2)+384+lookup+2*lin
        ld e,(hl)
        inc hl
        ld d,(hl)       ; DE=PEEK (384*INT((col-1)/2)+384+lookup+2*lin)
        ex de,hl        ; HL=PEEK (384*INT((col-1)/2)+384+lookup+2*lin)
        ex af,af'
        ret nc
        inc hl          ; HL=PEEK (384*INT((col-1)/2)+384+lookup+2*lin)+((col-1)%2)
    end asm
end function

' ----------------------------------------------------------------
' Reconfigure BIFROST*2 ENGINE to read tile images from another address
'
' Parameters:
'     addr: New tile images address
' ----------------------------------------------------------------
#define BIFROST2resetTileImages(addr)   POKE UINTEGER 51735, (addr)

' ----------------------------------------------------------------
' Reconfigure BIFROST* ENGINE to use 2 frames per animation group
' ----------------------------------------------------------------
sub FASTCALL BIFROST2resetAnim2Frames()
    asm
        halt
        xor a
        ld (51708),a
        ld hl,128
        ld (51710),hl
    end asm
end sub

' ----------------------------------------------------------------
' Reconfigure BIFROST* ENGINE to use 4 frames per animation group
' ----------------------------------------------------------------
sub FASTCALL BIFROST2resetAnim4Frames()
    asm
        halt
        ld a,15
        ld (51708),a
        ld hl,64+(256*7)
        ld (51710),hl
    end asm
end sub

' ----------------------------------------------------------------
' Advanced conversions
' ----------------------------------------------------------------
#define PX2LIN(px)              (((px)+1)<<4)
#define PX2ROW(px)              (((px)<<1)+1)

#define ROW2LIN(row)            (((row)+1)<<3)
#define ROW2PX_UP(row)          ((row)>>1)
#define ROW2PX_DOWN(row)        (((row)-1)>>1)

#define LIN2ROW_UP(lin)         (((lin)>>3)-1)
#define LIN2ROW_DOWN(lin)       (((lin)-1)>>3)
#define LIN2PX_UP(lin)          (((lin)>>4)-1)
#define LIN2PX_DOWN(lin)        (((lin)-1)>>4)

#define PY2COL(py)              (((py)<<1)+1)
#define COL2PY_LEFT(col)        (((col)-1)>>1)
#define COL2PY_RIGHT(col)       ((col)>>1)

' ----------------------------------------------------------------
' Instantly draw a multicolor tile at the specified screen position
'
' Parameters:
'     lin: pixel line (0-192)
'     col: char column (0-20)
'     tile: tile index (0-255)
'
' WARNING: Incorrect use of this routine may cause glitches
' ----------------------------------------------------------------
sub FASTCALL BIFROST2drawTileH(lin as UBYTE, col as UBYTE, tile as UBYTE)
    asm
        ld d,a          ; D=lin
        pop hl          ; RET address
        pop bc          ; B=col
        ld e,b          ; E=col
        ex (sp),hl      ; H=tile
        ld a,h          ; A=tile
        di
        call 51714      ; execute 'draw_tile'
        ei
    end asm
end sub

' ----------------------------------------------------------------
' Instantly show/animate the multicolor tile currently stored in
' the specified tile map position
'
' Parameters:
'     lin: pixel line (16,32,48..176)
'     col: char column (1,3,5..19)
'
' WARNING: Incorrect use of this routine may cause glitches
' ----------------------------------------------------------------
sub FASTCALL BIFROST2showTilePosH(lin as UBYTE, col as UBYTE)
    asm
        ld d,a          ; D=lin
        pop hl          ; RET address
        ex (sp),hl      ; H=col
        ld e,h          ; E=col
        di
        call 51683      ; execute 'show_tile_pos'
        ei
    end asm
end sub

' ----------------------------------------------------------------
' Instantly show/animate the next multicolor tile currently stored
' in the tile map position, according to a pre-established drawing
' order
'
' WARNING: Incorrect use of this routine may cause glitches
' ----------------------------------------------------------------
#define BIFROST2showNextTile() \
    asm                        \
        di                     \
        call 51653             \
        ei                     \
    end asm

' ----------------------------------------------------------------
' Instantly change the attributes in a tile area (16x16 pixels) to
' the specified value (use the same INK and PAPER values to "erase"
' a tile)
'
' Parameters:
'     lin: pixel line (0-192)
'     col: char column (0-20)
'     attr: attribute value (0-255), INK+8*PAPER+64*BRIGHT+128*FLASH
'
' WARNING: Incorrect use of this routine may cause glitches
' ----------------------------------------------------------------
sub FASTCALL BIFROST2fillTileAttrH(lin as UBYTE, col as UBYTE, attr as UBYTE)
    asm
        ld d,a          ; D=lin
        pop hl          ; RET address
        pop bc          ; B=col
        ld e,b          ; E=col
        ex (sp),hl      ; H=attr
        ld c,h          ; C=attr
        di
        call 64829      ; execute 'fill_tile_attr'
        ei
    end asm
end sub

#pragma pop(case_insensitive)
#endif
