/* Program rysujacy lokacje z gry Fairlight

   Problemy z przenosnoscia programu:

1. program zaklada kodowanie znakow ASCII

2. pobieranie z bufor 16-bitowych danych za pomoca rzutu (uint*) jest znacznie
   szybsze niz pobranie osobno 2 bajtow i zlozenie ich w slowo, ale narzuca
   nastepujace ograniczenia, ktorych nie spelnia np. Motorola 68000:
   - uchar ma 8 bitow, uint ma 16 bitow
   - slowo 16-bitowe jest zapisywane w pamieci w kolejnosci LSB, MSB
   - procesor moze pobierac 16-bitowe slowo z nieparzystego adresu
*/

#include <stdio.h>
#include "typedefs.h"
#include "bool.h"
#include "fl_grap2.h"
#include "keyboard.h"

#define OFFSET 0x5b00

#define BUFSIZE (0xBC90 - OFFSET)
uchar bufor[BUFSIZE];
uint loc=1;		/* lokacja */
uint skip=1;		/* pominiete dane bloku grafiki (atrybut koloru) */
uint licznik=0x180;	/* ograniczenie ilosci analizowanych danych */
uint cnt;		/* licznik analizowanych danych */
uint ret_indeks;	/* indeks na ktorym zakonczona zostala analiza */


/* struktury danych dla funkcji display */
struct disp_stru
{
  uint punkt_x3d;	/* odpowiada zmiennej FFDF = IY+5F */
  uint punkt_y3d;	/* odpowiada zmiennej FFE0 = IY+60 */
  uint punkt_z3d;	/* odpowiada zmiennej FFE1 = IY+61 */
  uint x1;		/* wspolrzedna x punktu 1, odpowiada FFE4 = IY+64 */
  uint y1;		/* wspolrzedna y punktu 1, odpowiada FFE5 = IY+65 */
  uint x2;		/* wspolrzedna x punktu 2, odpowiada FFE6 = IY+66 */
  uint y2;		/* wspolrzedna y punktu 2, odpowiada FFE7 = IY+67 */
  uint screen_segm;	/* odpowiada zmiennej FFEA = IY+6A */
  uint mode;		/* odpowiada zmiennej FFEB = IY+6B */
  uint loop_cnt;	/* licznik petli iteracyjnej, odpow. FFEC = IY+6C */
  uint loop_indeks;	/* wskazuje poczatek petli iteracyjnej, odpowiada
			   zmiennej FFEE:FFEF = IY+6E i IY+6F */
};

/* zmienne uzywane przez display, ktore sa przechowywane przez komende E1 */
struct disp_stru disp_var;

/* wartosci inicjujace disp_var */
struct disp_stru disp_init = {
        0,0,0,
	50,50,50,50,
	SCREEN,
	0,
	0, 0	};


/* tablica struktur danych dotyczacych obiektow znajdujacych sie w biezacej
   lokacji, odpowiada tablicy pod adresem BC90 */

struct objstru {
  uint coord_x2d;	/* wspolrzedna X lewego gornego rogu bitmapy */
  uint coord_y2d;	/* wspolrzedna Y lewego gornego rogu bitmapy */
  uint size_x2d;	/* szerokosc bitmapy */
  uint size_y2d;	/* wysokosc bitmapy */
  uint bmp;		/* indeks bitmapy w buforze bufor + OFFSET */
  uint coord_x3d;	/* wspolrzedna X obiektu (3D) */
  uint coord_y3d;	/* wspolrzedna Y obiektu (3D) */
  uint coord_z3d;	/* wspolrzedna Z obiektu (3D) */
  uint size_x3d;	/* rozmiar X obiektu (3D) */
  uint size_y3d;	/* rozmiar Y (wysokosc) obiektu (3D) */
  uint size_z3d;	/* rozmiar Z obiektu (3D) */
  uint attr;		/* atrybut obiektu */
  uint entry0d;
  uint entry0e;
  uint entry0f;
  uint entry10;		/* bit 4 set gdy przedmiot ma ciezar (na bitach 3-0) */
  uint entry11;
  uint entry12;
  uint obj_code;	/* kod obiektu, indeks do tablicy obiektow A924
			   liczony od 1, rycerz ma tu wartosc 0 */
};

#define OBJTABSIZE 64
struct objstru ObjTab[OBJTABSIZE];
uint ObjTabIndex;


/* zmiana wspolrzednych pozycji obiektu, odpowiednik procedury E4F7 */
void obj_pos (
  uint i,			/* indeks w tablicy obiektow ObjTab */
  uint x, uint y, uint z )	/* nowe wspolrzedne X,Y,Z (3D) */
{
  struct objstru *ptr;

  ptr = &ObjTab[i];
/* wspolrzedna X_2D += (delta_X_3D - delta_Z_3D) */
  ptr->coord_x2d = ptr->coord_x2d + x - ptr->coord_x3d - z + ptr->coord_z3d;
/* wspolrzedna Y_2D += (delta_X_3D / 2 + delta_Y_3D + delta_Z_3D / 2) */
  ptr->coord_y2d = ptr->coord_y2d + y - ptr->coord_y3d +
	((x - ptr->coord_x3d + z - ptr->coord_z3d) >> 1);
  ptr->coord_x3d = x;
  ptr->coord_y3d = y;
  ptr->coord_z3d = z;
}


/* umieszczenie obiektu w tablicy ObjTab,
   oczekuje wskaznika do tablicy A924 do bajtu zawierajacego kod obiektu,
   zwraca TRUE gdy operacja sie powiodla (bylo miejsce w ObjTab),
   odpowiednik procedury EB1A */
bool add_object (uchar *ptr)
{
  uchar *src_ptr;
  struct objstru *dest_ptr;
  uint kod;		/* kod obiektu */
  uint x, y, z;
  uint tmp;

  if (ObjTabIndex >= OBJTABSIZE)
    return FALSE;
  kod = (uint) *ptr++;
  dest_ptr = &ObjTab[ObjTabIndex];
  src_ptr = (kod < 0x46) ?
    &bufor[0xB734 - OFFSET + 11*kod] :		/* obiekt typu nie-drzwi */
    &bufor[0xB9AA - OFFSET - 9*0x46 + 9*kod];	/* obiekt typu drzwi */

  dest_ptr->coord_x2d = (uint) *src_ptr++;
  dest_ptr->coord_y2d = (uint) *src_ptr++;
  dest_ptr->size_x2d = (uint) *src_ptr++;
  dest_ptr->size_y2d = (uint) *src_ptr++;
  dest_ptr->bmp = *((uint*) src_ptr)++;		/* nieprzenosne */

  dest_ptr->attr = (kod < 0x46) ? (uint) *ptr++ : 1;				

  x = disp_var.punkt_x3d + (uint) *ptr++;
  y = disp_var.punkt_y3d + (uint) *ptr++;
  z = disp_var.punkt_z3d + (uint) *ptr++;
  dest_ptr->size_x3d = (uint) *src_ptr++;
  dest_ptr->size_y3d = (uint) *src_ptr++;
  dest_ptr->size_z3d = (uint) *src_ptr++;

  if (kod < 0x46)		/* obiekt typu nie-drzwi */
  {
    dest_ptr->entry0d = 0;
    dest_ptr->entry0e = (uint) *src_ptr++;
    dest_ptr->entry0f = 1;
    dest_ptr->entry10 = (uint) *src_ptr++;
    dest_ptr->entry11 = 0;
    dest_ptr->entry12 = 0;
    tmp = dest_ptr->entry0e & 0x0F;
    if (tmp == 2)
      dest_ptr->entry0d = 0x05;
    else if (tmp == 3)
      dest_ptr->entry0d = 0x45;
    else if (tmp>3 && tmp<11)
    {
      dest_ptr->entry0d = 0x05;
      dest_ptr->entry11 = 0x10;
    }
  }
  else				/* obiekt typu drzwi */
  {
    dest_ptr->entry0d = (uint) *ptr++;
    dest_ptr->entry0e = (uint) *ptr++;
    dest_ptr->entry0f = (uint) *ptr++;
    dest_ptr->entry10 = (uint) *ptr++;
    dest_ptr->entry11 = (uint) *ptr++;
    dest_ptr->entry12 = (uint) *ptr++;
  }

  dest_ptr->coord_x3d = 50;
  dest_ptr->coord_y3d = 50 + dest_ptr->size_y3d;
  dest_ptr->coord_z3d = 50;
  obj_pos (ObjTabIndex, x, y, z);
  dest_ptr->obj_code = 0;
  ObjTabIndex++;
  return TRUE;
}


/* umieszczenie obiektow nalezacych do lokacji loc w tablicy ObjTab,
   odpowiednik procedury EACC */
void add_objects (void)
{
  uchar *ptr;
  uint kod_obiektu;

  ptr = &bufor[0xA924 - OFFSET];
  kod_obiektu = 1;
  while ((uint) *ptr != 0xFF)
  {
    if ((uint) *ptr++ == loc)
    {
      if (add_object(ptr) && (uint) *ptr < 0x46)
        ObjTab[ObjTabIndex - 1].obj_code = kod_obiektu;
    }
    ptr += ((uint) *ptr < 0x46) ? 6-1 : 11-1;	/* czy obiekt typu drzwi ? */
    kod_obiektu++;
  }
}


/* procedura zwraca indeks bloku danych dotyczacych rysunku n,
   odpowiada fragmentowi kodu E55B-E580 */
uint picture (
uint n,			/* numer rysunku, n=1,2,... */
uint indeks )		/* indeks bloku danych w buforze */
{
  while (--n != 0)
  {
/* ponizsze wyrazenie jest nieprzenosne, ale kompiluje sie bardzo efektywnie
   za pomoca rozkazu:
	add	si, word ptr bufor[si]
   przenosna wersja tego wyrazenia moglaby wygladac nastepujaco:
	indeks += (uint) bufor[indeks] + ((uint) bufor[indeks+1] << 8);
*/
    indeks += *((uint*) &bufor[indeks]);
  }
  return (indeks);
}


void display_init (void)
{
  disp_var = disp_init;
  cnt = 0;
  ret_indeks = 0;
}


/* procedura rysujaca rysunek, ktorego dane wskazuje indeks,
   indeks musi wskazywac dane graficzne a nie adres nastepnego bloku z listy
   lub atrybut koloru
   odpowiada mniej wiecej procedurze E55B, od adresu E597 */
void display (uint indeks)
{
  uint c, x, y;
  struct disp_stru disp_save;

  disp_save = disp_init;	/* na wszelki wypadek i zeby Splint sie
				   nie czepial */
  while (cnt++ < licznik)
  {
    c = (uint) bufor[indeks++];
    if (c == 0xe5)		/* bajt E5 jest koncem bloku danych */
      break;
    if ((disp_var.mode & 0x80) != 0)
/* grafika rastrowa */
    {
      if (c < 0xe4)
        indeks+=4;
      else if (c > 0xe4)
        ;
    }
    else
/* grafika wektorowa */
    {
      if (c < 0xc0)
/* dana ponizej C0 oznacza pare wspolrzednych Y i X punktu 1 */
      {
        y = c;
        x = (uint) bufor[indeks++];
        switch (disp_var.mode & 0x50)
        {
          case 0x00:
            break;
          case 0x10:
            x += disp_var.x1;
            y += disp_var.y1;
            break;
          case 0x40:
            x ^= 0xff;
            break;
          case 0x50:
            x = disp_var.x1 - x;
            y += disp_var.y1;
            break;
        }
        x &= 0xff;
        if (y >= YSIZE)
           y -= YSIZE;
        if ((disp_var.mode & 0x08) != 0)
        {
          disp_var.x2 += x - disp_var.x1;
          disp_var.y2 += y - disp_var.y1;
          disp_var.x2 &= 0xff;
          if (disp_var.y2 >= YSIZE)
            disp_var.y2 -= YSIZE;
        }
        disp_var.x1 = x;
        disp_var.y1 = y;
        if ((disp_var.mode & 0x20) != 0)
          draw_line (disp_var.x1, YSIZE-1-disp_var.y1,
			disp_var.x2, YSIZE-1-disp_var.y2,
			disp_var.mode, disp_var.screen_segm);
        if ((disp_var.mode & 0x04) != 0)
        {
          disp_var.x2 = disp_var.x1;
          disp_var.y2 = disp_var.y1;
        }
      }

      else if (c >= 0xe6)	/* wypelnianie obszaru wzorem */
        fill_area (disp_var.x1, YSIZE-1-disp_var.y1, c-0xe6);

      else
      {
        switch (c)
        {
          case 0xcf:
            disp_var.x2 = disp_var.x1;
            disp_var.y2 = disp_var.y1;
            break;
          case 0xc0:
            y = (uint) bufor[indeks++];
            x = (uint) bufor[indeks++];
            switch (disp_var.mode & 0x50)
            {
              case 0x00:
                break;
              case 0x10:
                x += disp_var.x2;
                y += disp_var.y2;
                break;
              case 0x40:
                x ^= 0xff;
                break;
              case 0x50:
                x = disp_var.x2 - x;
                y += disp_var.y2;
                break;
            }
            x &= 0xff;
            if (y >= YSIZE)
              y -= YSIZE;
            disp_var.x2 = x;
            disp_var.y2 = y;
            break;
          case 0xd6:		/* petla iteracyjna */
            if (--disp_var.loop_cnt != 0)
              indeks = disp_var.loop_indeks;
            break;
          case 0xd0:		/* zamiana wspolrzednych punktow 1 i 2 */
            x = disp_var.x2;
            y = disp_var.y2;
            disp_var.x2 = disp_var.x1;
            disp_var.y2 = disp_var.y1;
            disp_var.x1 = x;
            disp_var.y1 = y;
            break;
          case 0xd2:
            draw_line (disp_var.x1, YSIZE-1-disp_var.y1,
			disp_var.x2, YSIZE-1-disp_var.y2,
			disp_var.mode, disp_var.screen_segm);
            break;
          case 0xcb:	/* w trybie OR rysowany piksel w kolorze tla */
            disp_var.mode |= 0x01;
            break;
          case 0xcd:	/* tryb XOR */
            disp_var.mode |= 0x02;
            break;
          case 0xc3:	/* przekopiowanie wspolrzednych punktu 1 do
			   wspolrzednych punktu 2 po wykresleniu odcinka */
            disp_var.mode |= 0x04;
            break;
          case 0xc5:
            disp_var.mode |= 0x08;
            break;
          case 0xc1:	/* wspolrzedne punktu sa wzgledne w stosunku do
			   poprzedniej wartosci */
            disp_var.mode |= 0x10;
            break;
          case 0xc7:	/* po pobraniu wspolrzednych nastepnego punktu
			   wykreslic odcinek do poprzedniego punktu */
            disp_var.mode |= 0x20;
            break;
          case 0xc9:	/* zanegowana wspolrzedna X (lustrzane odbicie) */
            disp_var.mode |= 0x40;
            break;
          case 0xcc:	/* tryb OR */
            disp_var.mode &= ~0x01;
            break;
          case 0xce:	/* w trybie OR rysowany piksel czarny */
            disp_var.mode &= ~0x02;
            break;
          case 0xc4:
            disp_var.mode &= ~0x04;
            break;
          case 0xc6:	/* wspolrzedne punktu sa absolutne (nie wzgledne) */
            disp_var.mode &= ~0x08;
            break;
          case 0xc2:
            disp_var.mode &= ~0x10;
            break;
          case 0xc8:
            disp_var.mode &= ~0x20;
            break;
          case 0xca:
            disp_var.mode &= ~0x40;
            break;
          case 0xd5:		/* inicjacja petli iteracyjnej */
            disp_var.loop_cnt = (uint) bufor[indeks++];
            disp_var.loop_indeks = indeks;
            break;
          case 0xe0:		/* wywolanie rysunku o numerze za kodem
				   komendy bez przechowania i odtworzenia
				   wartosci zmiennych */
            display (picture ((uint) bufor[indeks++], 0x758c-OFFSET) + 2);
            disp_var.mode &= ~0x80;
            break;
          case 0xe1:		/* wywolanie rysunku o numerze za kodem
				   komendy z zachowaniem i odtworzeniem
				   wartosci zmiennych */
            disp_save=disp_var;
            display ( picture ((uint) bufor[indeks++], 0x758c-OFFSET) + 2);
            disp_var=disp_save;
            disp_var.mode &= ~0x80;
            break;
          case 0xe2:		/* kopiowanie ekranu do Shadow-Screen */
            copy_screen (SCREEN,SHADOW);
            break;
          case 0xe4:		/* grupa rozkazow prefiksowanych */
            c = (uint) bufor[indeks++];
            switch (c)
            {
              case 0x00: /* przejscie na Shadow-Screen i skasowanie go */
                clear_screen (disp_var.screen_segm = SHADOW);
                break;
              case 0x01: 	/* przejscie na Video-RAM */
                disp_var.screen_segm = SCREEN;
                break;
              case 0x04:	/* kopiowanie obiektow z tablicy A924 do
				   tablicy obiektow nalezacych do biezacej
				   lokacji */
                disp_var.mode |= 0x80;
                if (ObjTabIndex == 0)	/* test zapobiega wielokrotnemu
					   wykonaniu funkcji add_objects */
                  add_objects();
                break;
              case 0x05:	/* j.w. oraz pobranie wartosci wspolrzednych
				   punktu 3D, wzglednych lub bezwzglednych */
                if (ObjTabIndex == 0)	/* test zapobiega wielokrotnemu
					   wykonaniu funkcji add_objects */
                  add_objects();
                if ((disp_var.mode & 0x10) != 0) /* wspolrzedne wzgledne */
                {
                  disp_var.punkt_x3d += (uint) bufor[indeks++];
                  disp_var.punkt_y3d += (uint) bufor[indeks++];
                  disp_var.punkt_z3d += (uint) bufor[indeks++];
                }
                else		/* wspolrzedne bezwzgledne */
                {
                  disp_var.punkt_x3d = (uint) bufor[indeks++];
                  disp_var.punkt_y3d = (uint) bufor[indeks++];
                  disp_var.punkt_z3d = (uint) bufor[indeks++];
                }
                break;
            }
            break;
        }
      }
    }
  }
  if (cnt >= licznik)
    ret_indeks=indeks;
}


/* wyswietlenie znaku 8x8 pikseli, fonty pochodza z kodu gry, kod ASCII
   znaku w c, x0 i y0 sa wspolrzednymi lewego gornego punktu znaku,
   procedura nieprzenosna, zaklada kodowanie znakow ASCII */
void print_char (char c, uint x0, uint y0)
{
  uint d;

  if (c == '.')
    d = 0x25;
  else if (c == '-')
    d = 0x26;
  else if (c >= '0' && c <= '9')
    d = (uint) (c - '0') + 0x1B;
  else if (c >= 'A' && c <= 'Z')
    d = (uint) (c - 'A') + 1;
  else if (c >= 'a' && c <= 'z')
    d = (uint) (c - 'a') + 1;
  else			/* rowniez spacja */
    d = 0;
  image ((uint) &bufor[0xBAD8 - OFFSET + (d<<3)], 0, x0, y0, 1, 8);
}


/* wyswietlenie lancucha *s zakonczonego '\0',
   x0 i y0 sa wspolrzednymi lewego gornego punktu lancucha */
void print_string (char *s, uint x0, uint y0)
{
  while (*s != '\0')
  {
    print_char (*s, x0, y0);
    s++;
    x0 += 8;
  }
}


/* wyswietlenie liczby szesnastkowo */
void hex_print (uint liczba, uint x0, uint y0)
{
  char znaki[6];
  sprintf (znaki, "%04X", liczba);
  print_string (znaki, x0, y0);
}


/* wyswietlenie szesnastkowo fragmentu interpretowanych danych */
void print_data (uint indeks)
{
  char znaki[4];
  uint x;

  for (x=0; x<XSIZE; x+=8)
  {
    print_char (' ', x, YSIZE);
  }
  if (indeks != 0)
  {
    indeks-=4;
    for (x=0; x<8*3*13; x+=8*3)
    {
      sprintf (znaki, "%02X ", (uint) bufor[indeks++]);
      print_string (znaki, x, YSIZE);
    }
    print_char ('.', 11*8, YSIZE);
  }
}


/* test, czy c jest poprawna cyfra szesnastkowa, mozna by zamiast tego
   uzyc isxdigit, ale Splint nie przechodzi przez CTYPE.H
   procedura nieprzenosna, zaklada kodowanie znakow ASCII */
bool ishexdigit (int c)
{
  if (c < (int) '0')
    return FALSE;
  if (c <= (int) '9')
    return TRUE;
  if (c < (int) 'A')
    return FALSE;
  if (c <= (int) 'F')
    return TRUE;
  if (c < (int) 'a')
    return FALSE;
  return (c <= (int) 'f');
}


#define CURSOR '.'	/* kropka */

/* funkcja zwraca wpisana z klawitury dana liczbowa, w x0 i y0 oczekuje
   wspolrzednych na ekranie, gdzie ma wyswietlac wpisywana liczbe */
uint get_hex (uint x0, uint y0)
{
  int key;
  uint x = 0;
  char mybufor[5];

  mybufor[0] = CURSOR;
  mybufor[1] = mybufor[2] = mybufor[3] = ' ';
  mybufor[4] = '\0';
  do {
    print_string (mybufor, x0, y0);
    key = read_key();
    if (ishexdigit(key) && x<4)
    {
      mybufor[x++] = (char) key;
      if (x < 4)
        mybufor[x] = CURSOR;
    }
    if (key == (int) '\b' && x>0)
    {
      if (x < 4)
        mybufor[x] = ' ';
      mybufor[--x] = CURSOR;
    }
  } while (key != (int) '\r');
  mybufor[x] = '\0';
  if (x > 0)
    (void) sscanf (mybufor, "%x", &x);
  return x;
}


int main (void)
{
  FILE *infp, *outfp;
  uint old_video_mode;
  int bajt;
  uint x;

  if ((infp=fopen("FL_CODE.BIN","rb"))==NULL)
  {
    fprintf (stderr,"\nNie moge otworzyc pliku FL_CODE.BIN\n");
    return 0;
  }
  for (x=0; x<BUFSIZE && (bajt=getc(infp))!=EOF; x++ )
    bufor[x] = (uchar) bajt;
  (void) fclose (infp);

  if ((outfp=fopen("FL7A.LOG","wt"))==NULL)
  {
    fprintf (stderr,"\nNie moge utworzyc pliku FL7A.LOG\n");
    return 0;
  }

  old_video_mode = read_video_mode();
  set_video_mode (5);		/* CGA 320x200, grayscale */
  do {
    ObjTabIndex = 0;
    clear_screen (SCREEN);
    x = picture (loc, 0x68b0-OFFSET);
    print_string ("L ", 34*8, 0);
    hex_print (loc, 36*8, 0);
    print_string ("O ", 34*8, 10);
    hex_print (skip, 36*8, 10);
    print_string ("A ", 34*8, 20);
    hex_print (x+OFFSET, 36*8, 20);
    print_string ("C ", 34*8, 30);
    hex_print (licznik, 36*8, 30);
    display_init();
    display (x+2+skip);

    fprintf (outfp, "\n\nLOC = %02X\n", loc);
    for (x=0; x<ObjTabIndex; x++)
    {
      fprintf (outfp,
"%02X %02X %02X %02X  %04X %02X %02X %02X  %02X %02X %02X %02X  %02X %02X %02X %02X  %02X %02X %02X\n",
	ObjTab[x].coord_x2d, ObjTab[x].coord_y2d, ObjTab[x].size_x2d,
	ObjTab[x].size_y2d, ObjTab[x].bmp, ObjTab[x].coord_x3d,
	ObjTab[x].coord_y3d, ObjTab[x].coord_z3d, ObjTab[x].size_x3d,
	ObjTab[x].size_y3d, ObjTab[x].size_z3d, ObjTab[x].attr,
	ObjTab[x].entry0d, ObjTab[x].entry0e, ObjTab[x].entry0f,
	ObjTab[x].entry10, ObjTab[x].entry11, ObjTab[x].entry12,
	ObjTab[x].obj_code);
    }

    print_string ("PUNKT1", 34*8, 60);
    print_string ("X ", 34*8, 70);
    hex_print (disp_var.x1, 36*8, 70);
    print_string ("Y ", 34*8, 80);
    hex_print (disp_var.y1, 36*8, 80);
    print_string ("PUNKT2", 34*8, 100);
    print_string ("X ", 34*8, 110);
    hex_print (disp_var.x2, 36*8, 110);
    print_string ("Y ", 34*8, 120);
    hex_print (disp_var.y2, 36*8, 120);
    print_string ("M ", 34*8, 140);
    hex_print (disp_var.mode, 36*8, 140);
    print_data (ret_indeks);
    switch (bajt = read_key())
    {
      case (int) 's':
      case (int) 'S':
        copy_screen (SHADOW,SCREEN);
        do {
          bajt = read_key();
        } while (bajt != (int) 's' && bajt!= (int) 'S');
        break;
      case (int) 'l':
      case (int) 'L':
        loc = get_hex (36*8,0) & 0xff;
/* lokacje > 0x51 sa podprogramami i nie maja na poczatku atrybutu koloru */
        skip = (loc>0x51) ? (uint) 0 : (uint) 1;
        break;
      case (int) 'o':
      case (int) 'O':
        skip=get_hex(36*8,10) & 0x01;
        break;
      case (int) 'c':
      case (int) 'C':
        licznik=get_hex(36*8,30);
        break;
      case (int) 'u':
      case (int) 'U':
      case 72+256:	/* strzalka w gore */
        if (loc<0xff)
          loc++;
        if (loc>0x51)
          skip=0;
        break;
      case (int) 'd':
      case (int) 'D':
      case 80+256:	/* strzalka w dol */
        if (loc>1)
          loc--;
        if (loc<=0x51)
          skip=1;
        break;
    }
  } while (bajt != (int) 'q' && bajt != (int) 'Q');
  set_video_mode (old_video_mode);
  fclose (outfp);
  return 0;
}
