module SQRUM (
	input wire res_n,
	input wire gclk_n,
	input wire mreq_n,
	input wire iorq_n,
	input wire rd_n,
	input wire wr_n,
	input wire m1_n,
	input wire [15:0] a,
	inout wire [7:0] dt,
	inout wire [7:0] md,
	output reg [9:0] ma,
	output reg ras_n,
	output reg cas_n,
	output wire clk_z, // may be used, same as ras_n
	output wire [3:0] vout, // 0-B, 1-R, 2-G, 3-Y
	output reg syn_n,
	output wire [15:14] rom,
	output wire romcs_n,
	output wire ramwe_n,
	output wire int_n
);

///
///
///// COUNTER -------------------------------------------------
parameter HPixOffset = 4; 	// pixels offset

parameter HLineEnd = 448-1;
parameter HBorderStart = HPixOffset + 256-1; // X resolution
parameter HBlancStart = HPixOffset + 320-1;
parameter HBlancEnd = HPixOffset + 384-1;
parameter HSynStart = HPixOffset + 320-1;
parameter HSynEnd = HPixOffset + 352-1;
parameter VBorderStart = 192-1; // Y Resolution
parameter VBlancStart = 248-1;
parameter VBlancEnd = 256-1;
parameter VSynStart = 248-1;
parameter VSynEnd = 256-1;
parameter VLineEnd = 320-1;


// pixelclock 7MHz generation
reg pixclk = 0;
always @(negedge gclk_n)
	pixclk <= ~pixclk;

// horizontal/vertical counter
reg [8:0] hcnt = 0;
reg [8:0] vcnt = 0;
reg vsyn = 0, hsyn = 0;
reg hblanc = 0, vblanc = 0;
reg vbord = 0, hbord = 0;
reg bord, blanc;
always @(negedge pixclk)
begin
	if(hcnt < HLineEnd)
		hcnt <= hcnt + 1'b1;
	else
	begin	
		hcnt <= 9'd0;
		if(vcnt < VLineEnd)
			vcnt <= vcnt + 9'd1;
		else
			vcnt <= 9'd0;
		vsyn <= (vcnt >= VSynStart && vcnt <= VSynEnd ) ? 1'b1 : 1'b0;
		vbord <= (vcnt >= VBorderStart) ? 1'b1 : 1'b0;
		vblanc <= (vcnt >= VBlancStart && vcnt <= VBlancEnd ) ? 1'b1 : 1'b0;
	end	

	hsyn <= (hcnt >= HSynStart && hcnt <= HSynEnd ) ? 1'b1 : 1'b0;
	hbord <= (hcnt >= HBorderStart || hcnt < HPixOffset) ? 1'b1 : 1'b0;
	hblanc <= (hcnt >= HBlancStart && hcnt <= HBlancEnd ) ? 1'b1 : 1'b0;
	bord <= hbord | vbord;
	blanc <= hblanc | vblanc;
	syn_n <= ~(hsyn ^ vsyn);
end


assign int_n = (vcnt == 0 && hcnt[8:3] == HPixOffset) ? 1'b0 : 1'b1;

// CAS
always @(posedge gclk_n)
	cas_n <= ~hcnt[0];

// /RAS
always @(posedge pixclk)
	ras_n <= hcnt[0];

assign clk_z = ras_n;
/// END COUNTER -----------------------------------------------

///
///
///// PORTS MODULE --------------------------------------------
wire port_wr_req = wr_n | iorq_n;
reg [7:0] port_FE = 0; // border, beeper, tapeout
reg [7:0] p7FFD_data = 0;
reg [7:0] p1FFD_data = 0;

// reading ports (port write by CPU)
always @(negedge res_n or negedge port_wr_req)
	if(!res_n)
	begin
		p7FFD_data <= 8'd0;
		p1FFD_data <= 8'd0;
	end
	else
	begin
		casex({DOS,a[15:0]})
		/*#FE*/		'b0XXXXXXXXXXXXXXX0: port_FE <= dt;
		/*#1FFD*/	'b0000XXX1XXXXXXX01: p1FFD_data <= dt;
		/*#7FFD*/	'b001XXXXXXXXXXXX01: if(!p7FFD_data[5]) p7FFD_data <= dt;
		endcase
	end
	
///// END PORTS MODULE ----------------------------------------


///
///
///// DOS SELECT MODULE ---------------------------------------
reg DOS = 0;
wire fetch_opcode = mreq_n | m1_n;// | rd_n;
always @(negedge fetch_opcode or negedge res_n)
	if(!res_n)
		DOS <= 'b0;
	else
	begin
		if (rom == 2'b01 && a[15:8] == 'h3D)
			DOS <= 'b1;
		else
			if(a[14] | a[15]) DOS <= 'b0;
	end
///// END DOS SELECT MODULE -----------------------------------


///
///
///// MEMORY MODULE -------------------------------------------
wire [19:14] ram = a[14] & a[15] ? {p1FFD_data[7:6],p1FFD_data[4],p7FFD_data[2:0]} : {3'b000,a[14],a[15],a[14]}; // port ram address
wire scr_addr = p7FFD_data[3]; // 128k screen select
wire p0_ram = p1FFD_data[0];
assign rom[15] = p1FFD_data[1] | DOS;
assign rom[14] = p7FFD_data[4];
//assign rom = { 1'b0, 1'b1 };
/// END MEMORY MODULE -----------------------------------------

///
///
///// MULTIPLEXOR ---------------------------------------------
always @(*)
	casex({hcnt[1],hcnt[0],hcnt[2]}) // 1024K multiplexor
		'b001: ma <= {vcnt[7:6], vcnt[5:3], hcnt[7:3]}; 								// column attr
		'b000: ma <= {vcnt[1:0], vcnt[5:3], hcnt[7:3]}; 								// column pixels
		'b011: ma <= {2'b00,scr_addr,1'b0,6'b110011};									// row attr
		'b010: ma <= {2'b00,scr_addr,1'b0,vcnt[7],vcnt[6],vcnt[2], 3'b011};				// row pixels
		'b10x: ma <= a[9:0];															// column Z80 address
		'b11x: ma <= {ram[19], ram[17], ram[15], a[13:10], ram[18], ram[16], ram[14]};	// row Z80 address
	endcase
/// END MULTIPLEXOR -------------------------------------------


///
///
///// VIDEO ---------------------------------------------------
reg [7:0] pixels = 0;
reg [7:0] attr = 15;//0;
reg [3:0] pixshift = 0;
reg [4:0] flash_cnt = 0;

always @(negedge pixclk)
begin
	if(hcnt[2:0] == 'b001)
		pixels <= md;
	else
		pixels <= pixels << 1;

	pixshift <= {pixshift[2:0],pixels[7]};

	if(hcnt[2:0] == 'b101)
		attr <= md;
end

// flash generator
always @(posedge vsyn)
	flash_cnt <= flash_cnt + 1'b1;

wire fpix = pixshift[3] ^ (attr[7] & flash_cnt[4]); // flash
wire [2:0] img = fpix ? attr[2:0] : attr[5:3];

assign vout[2:0] = blanc ? 3'd0 : bord ? port_FE[2:0] : img[2:0];
assign vout[3] = bord ? 1'b0 : |attr[5:0] & attr[6];

/// END VIDEO -------------------------------------------------

///
///
///// BUFFERS & WRITE -----------------------------------------
wire romn = a[14] | a[15];
assign romcs_n = rd_n | mreq_n | romn;
wire ramcsn = (rd_n | mreq_n) | ~romcs_n;

// ram buffer -----------
reg [7:0] wr_buf;
wire wrbuf =  hcnt[0] & hcnt[1] & pixclk;
always @(posedge wrbuf)
	wr_buf <= md;
assign dt = ramcsn ? 8'bzzzzzzzz : wr_buf;

// SIMM write buffer ----
reg ramwen = 1'b1;
assign ramwe_n = ramwen;
wire en_wr = wr_n | mreq_n | ras_n;
always @ (negedge hcnt[1] or negedge en_wr)
	if (!hcnt[1])
		ramwen <= 1'b1;
	else
		ramwen <= (romn | p0_ram) ? 1'b0 : 1'b1;
assign md = ramwen ? 8'bzzzzzzzz : dt;
/// END BUFFERS & WRITE ---------------------------------------


endmodule
