module SQRUM (
	input wire res_n,
	input wire gclk_n,
	input wire mreq_n,
	input wire iorq_n,
	input wire rd_n,
	input wire wr_n,
	input wire m1_n,
	input wire [15:0] a,
	inout wire [7:0] dt,
	inout wire [7:0] md,
	output reg [9:0] ma,
	output reg ras_n,	
	output reg cas_n,
	output reg [3:0] vout, // 0-B, 1-R, 2-G, 3-Y
	output reg syn_n,
	output wire [15:14] rom,
	output wire romcs_n,
	output reg ramwe_n = 1,
	output reg int_n
	// need output: (move to second)
	//	speaker out
	//	tape out
	//	dos
	// removed clk_z, (pin 73)
	// move address a[1] pin from (pin 81) to (pin 2)
	// free pin 4
	// 		total 3 pins, if move rom14-15 - 5 pins
	//
	// move DOS to second MAX7000S and change iorq_n with it
	// move rom[15], rom[14] to second altera
	// move portXFD bit4, port1FFD bit1
	// move portFE bits [3:4]
	// move portFF
	//   need pins a[15:0], dt[7:0], m1_n, iorq_n, wr_n, rd_n
);


///
///
///// COUNTER -------------------------------------------------
parameter HPixOffset = 4; 	// pixels offset

parameter HLineEnd = 447;
parameter HBorderStart = HPixOffset + 256; // X resolution
parameter HBlancStart = HPixOffset + 319;
parameter HBlancEnd = HPixOffset + 395;
parameter HSynStart = HPixOffset + 319;
parameter HSynEnd = HPixOffset + 363;

parameter VBorderStart = 192; // Y Resolution
parameter VBlancStart = 242;
parameter VBlancEnd = 250;
parameter VSynStart = 242;
parameter VSynEnd = 250;
parameter VLineEnd = 311;


// pixelclock 7MHz generation
reg pixclk = 0;
always @(negedge gclk_n)
	pixclk <= ~pixclk;

// horizontal/vertical counter
reg [8:0] hcnt = 0;
reg [8:0] vcnt = 0;
reg vsyn = 0, hsyn = 0;
reg hblanc = 0, vblanc = 0;
reg vbord = 0, hbord = 0;

always @(negedge pixclk)
	hcnt <= (hcnt < HLineEnd) ? hcnt + 1'b1 : 9'd0; // horisontal counter increase

always @(posedge pixclk)
	begin
		hsyn <= (hcnt >= HSynStart && hcnt <= HSynEnd ) ? 1'b1 : 1'b0;
		hbord <= (hcnt >= HBorderStart || hcnt < HPixOffset ) ? 1'b1 : 1'b0; ///
		hblanc <= (hcnt >= HBlancStart && hcnt <= HBlancEnd ) ? 1'b1 : 1'b0;
	end

always @(posedge hsyn)
	vcnt <= ( vcnt < VLineEnd) ? vcnt + 9'd1 : 9'd0; // vertical counter increase		 

always @(negedge hsyn)
	begin
		vsyn <= (vcnt >= VSynStart && vcnt <= VSynEnd ) ? 1'b1 : 1'b0;
		vbord <= (vcnt >= VBorderStart && vcnt <= VLineEnd) ? 1'b1 : 1'b0;
		vblanc <= (vcnt >= VBlancStart && vcnt <= VBlancEnd ) ? 1'b1 : 1'b0;
	end

// /CAS
always @(posedge gclk_n)
	cas_n <= ~hcnt[0];

// /RAS
always @(posedge pixclk)
	ras_n <= hcnt[0];

/// END COUNTER -----------------------------------------------


///
///
///// PORTS MODULE --------------------------------------------

reg short_port = 0; // SHORT PORT ADDRESS active 1
always @(posedge fetch_opcode or negedge res_n)
	if(!res_n)
		short_port <= 1'b0;
	else
		short_port <= (dt[7:4] == 4'b1101 && dt[2:0] == 3'b011) ? 1'b1 : 1'b0;

reg [2:0] port_FE = 0; // border only
reg [5:0] port_7FFD = 0;
reg [4:0] port_DFFD = 0;
reg port_1FFD = 0; // bit 1 only

// reading ports data (port write by CPU)
wire port_wr = iorq_n | wr_n | ~m1_n;
wire block_7FFD = ~port_7FFD[5] | port_DFFD[4] | DOS;
always @(negedge res_n or negedge port_wr)
	if(!res_n)
	begin
		port_7FFD <= 6'd0;
		port_1FFD <= 1'b0;
		port_DFFD <= 5'd0;
	end
	else
	casex({a[15:0],DOS,short_port})
		/*write #FE */
		'bXXXXXXXX_1XX11X10_0X: port_FE <= dt[2:0];

		/*write #1FFD */
		// port used only to change ROM bank 0 //
		'b000XXX1X_XXXXXX0X_00: if(dt == 8'd2) port_1FFD <= 1'b1; // change ROM to SYS
		
		/* write #7FFD */
		// don't block if DOS active for RAM DISK to work
		'b01XXXXXX_XXXXXX0X_X0: if (block_7FFD)  port_7FFD <= dt[5:0];
				
		/* write #DFFD */	
		'b11011111_XXXXXX0X_X0: port_DFFD[4:0] <= dt[4:0];

		/*write #FD */
		'b0XXXXXXX_11111101_X1: port_7FFD[4:0] <= dt[4:0];
	endcase


// Port FF read
//wire port_FF_sel = iorq_n == 0 && bord == 0 && a[3:0] == 4'b1111 && a[7] == 1 && DOS == 0; // active 1
//assign dt = port_FF_sel & res_n ? attr : 8'bzzzzzzzz;
	
///// END PORTS MODULE ----------------------------------------

///// DOS SELECT MODULE --------------------------------------- TEMPORARY!!!!!!!!!!!!!!!!!
reg DOS = 0;
wire fetch_opcode = rd_n | m1_n;// | mreq_n;
always @(negedge fetch_opcode or negedge res_n)
	if(!res_n)
		DOS <= 1'b0;
	else
	begin
		if (port_7FFD[4] && a[15:8] == 'h3D)
			DOS <= 1'b1;
		else
			if(a[14] | a[15]) DOS <= 1'b0;
	end
///// END DOS SELECT MODULE -----------------------------------

///
///
///// RAM PORT MODULE -----------------------------------------
wire [19:14] ram = &a[15:14] ? {port_DFFD[2:0], block_7FFD ? port_7FFD[2:0] : 3'b000} : {3'd0,a[14],a[15:14]}; // port ram address
wire scr_addr = port_7FFD[3]; // 128k screen select
wire ROM14 = (port_1FFD & ~port_7FFD[4]) | DOS;
assign rom =  { ROM14, port_7FFD[4] }; // rom A14, A15
/// END RAM PORT MODULE ---------------------------------------


///
///
///// RAM MULTIPLEXOR -----------------------------------------
always @(*)
	casex({hcnt[1],hcnt[0],hcnt[2]}) // 1024K multiplexor
		'b001: ma = {vcnt[7:6], vcnt[5:3], hcnt[7:3]}; 								// column attr
		'b000: ma = {vcnt[1:0], vcnt[5:3], hcnt[7:3]}; 								// column pixels
		'b011: ma = {2'b00,scr_addr,1'b0,6'b110011};								// row attr
		'b010: ma = {2'b00,scr_addr,1'b0,vcnt[7],vcnt[6],vcnt[2],3'b011};			// row pixels
		'b10x: ma = a[9:0];															// column Z80 address
		'b11x: ma = {ram[19],ram[17],ram[15],a[13:10],ram[18],ram[16],ram[14]};		// row Z80 address
	endcase
/// END RAM MULTIPLEXOR ---------------------------------------


///
///
///// BUFFERS & WRITE -----------------------------------------
wire rom_ram = a[14] | a[15] | port_DFFD[4]; // 0 - rom, 1 - ram

assign romcs_n = rd_n | mreq_n | rom_ram;
wire ramcsn = rd_n | mreq_n | ~romcs_n;

// ram buffer -----------
reg [7:0] wr_buf;
wire wrbuf = hcnt[0] & hcnt[1] & pixclk;
assign dt = ramcsn ? 8'bzzzzzzzz : wr_buf;
always @(posedge wrbuf or negedge res_n)
	if(!res_n)
		wr_buf <= 8'bzzzzzzzz;	// IT IS IMPORTANT!!!!!!
	else
		wr_buf <= md;


// SIMM write buffer ----
wire en_wr = wr_n | mreq_n | ras_n;
always @ (negedge hcnt[1] or negedge en_wr)
	if (!hcnt[1])
		ramwe_n <= 1'b1;
	else
		ramwe_n <= ~rom_ram;

assign md = ramwe_n ? 8'bzzzzzzzz : dt;
/// END BUFFERS & WRITE ---------------------------------------


///
///
///// VIDEO MODULE --------------------------------------------

// FLASH
reg [4:0] flash_cnt = 0; // flash counter
always @(posedge vsyn)
	flash_cnt <= flash_cnt + 1'b1;// flash generator

// video output
reg [11:0] pixels = 0;
reg [7:0] attr = 0;
wire fpix = pixels[11] ^ (attr[7] & flash_cnt[4]);
wire [2:0] pix_attr = fpix ? attr[2:0] : attr[5:3];
reg	bord, blanc;
always @(posedge pixclk)
begin
	// get pixels & attributes from memory
	pixels <= pixels << 1;
	if({hcnt[2:0]} == 'b001) pixels[7:0] <= md;
	if({hcnt[2:0]} == 'b101) attr <= md;

	bord <= hbord | vbord;
	blanc <= hblanc | vblanc;

	int_n <= vcnt == VSynStart + 6 && hblanc == 1 ? 1'b0 : 1'b1;
	syn_n <= ~(hsyn ^ vsyn);
	vout <= blanc ? 4'd0 : bord ? {1'b0, port_FE[2:0]} : {|pix_attr[2:0] & attr[6], pix_attr};
end
///// END VIDEO MODULE ----------------------------------------


endmodule
