; CBIOS for the 8080 Computer. Implements IOBYTE.
;
; Portions by Digital Research

; (c)Freddy 2017 antenn@land.ru
; For non comercial use only


msize	equ	62		;cp/m version memory size in kilobytes
drives	equ	1		;number of installed disk drives
;
;	"bias" is address offset from 3400h for memory systems
;	than 16k (referred to as"b" throughout the text)
;
bias	equ	(msize-20)*1024
ccp	equ	3400h+bias	;base of ccp
bdos	equ	ccp+806h	;base of bdos
bios	equ	ccp+1600h	;base of bios
cdisk	equ	0004h		;current disk number 0=a,... l5=p
iobyte	equ	0003h		;intel i/o byte

	org	bios		;origin of this program

consd	equ	0F0H	; console data
conss	equ	0F1H	; console status
consc	equ	0F1H	; console control
auxd	equ	0F4H	; aux serial data
auxs	equ	0F5H	; aux serial status
auxc	equ	0F5H	; aux serial control

rxd	equ	2	; receive flag mask
txd	equ	1	; transmit flag mask

idedat	EQU	0E8H	; IDE DATA REGISTER (LOW)
			; HIGH IS LATCHED INTO E4H ON READ
			; E4H MUST CONTAIN HIGH DATA ON WRITE
ideerr	EQU	0E9H	; IDE ERROR REGISTER
idesc	EQU	0EAH	; IDE SECTOR COUNT REGISTER
idest	EQU	0EBH	; IDE START SECTOR REGISTER
idecyll	EQU	0ECH	; IDE CYLINDER LOW REGISTER
idecylh	EQU	0EDH	; IDE CYLINDER HIGH REGISTER
idehead	EQU	0EEH	; IDE HEAD REGISTER
idecmd	EQU	0EFH	; IDE COMMAND REGISTER
idestat	EQU	0EFH	; IDE STATUS REGISTER
idehi	EQU	0E4H	; IDE HIGH BYTE REGISTER

blksiz	equ	4096		;CP/M allocation size
hstsiz	equ	512		;host disk sector size
hstsph	equ	63		;sectors per head
hsthpc	equ	1		;heads per cylinder
hstspt	equ	hstsph*hsthpc	;host disk sectors/trk
hstblk	equ	hstsiz/128	;CP/M sects/host buff
cpmspt	equ	hstblk*hstspt	;CP/M sectors/track
secmsk	equ	hstblk-1		;sector mask
;secshf	equ	2		;log2(hstblk)
nsects	equ	($-ccp)/hstsiz	;warm start sector count

;
wrall	equ	0		;write to allocated
wrdir	equ	1		;write to directory
wrual	equ	2		;write to unallocated


;
;	jump vector for individual subroutines
;
	jmp	boot	;cold start
wboote:	jmp	wboot	;warm start
	jmp	const	;console status
	jmp	conin	;console character in
	jmp	conout	;console character out
	jmp	list	;list character out
	jmp	punch	;punch character out
	jmp	reader	;reader character out
	jmp	home	;move head to home position
	jmp	seldsk	;select disk
	jmp	settrk	;set track number
	jmp	setsec	;set sector number
	jmp	setdma	;set dma address
	jmp	read	;read disk
	jmp	write	;write disk
	jmp	listst	;return list status
	jmp	sectran	;sector translate

;drive's parameters block
dpbase:	
	DW 0000h,0000h,0000h,0000h,dirbuf,dpb0,0000h,alv00

	
dpb0:
	DW cpmspt	;SPT - sectors per track
	DB 5	;BSH - block shift factor
	DB 31	;BLM - block mask
	DB 1	;EXM - Extent mask
	DW 2043	; (2047-4) DSM - Storage size (blocks - 1)
	DW 511	;DRM - Number of directory entries - 1
	DB 240	;AL0 - 1 bit set per directory block
	DB 0	;AL1 -            "
	DW 0	;CKS - DIR check vector size (DRM+1)/4 (0=fixed disk)
	DW 1	;OFF - Reserved tracks
;end of fixed tables

;	individual subroutines to perform each function
boot:	;simplest case is to just perform parameter initialization
	mvi	a, 01h		;CRT is the default device
	sta	iobyte		;clear the iobyte
	xra	a		;zero in the accum
	sta	cdisk		;select disk zero
	sta	hstact		;host buffer inactive
	sta	unacnt		;clear unalloc count
	lxi	h,mesg		; Signon message
	call	print
	jmp	gocpm		;initialize and go to cp/m
;
wboot:	;simplest case is to read the disk until all sectors loaded
	lxi	sp, 80h		;use space below buffer for stack
	mvi	c, 0		;select disk 0
	call	seldsk
	call	home		;go to track 00
;
	mvi	b, nsects	;b counts * of sectors to load
	mvi	c, 0		;c has the current track number
	lxi	d, 1		;d has the next sector to read
;	note that we begin by reading track 0, sector 1 since sector 0
;	contains the cold start loader, which is skipped in a warm start
	lxi	h, ccp		;base of cp/m (initial load point)
rdloop	mov	a, e
	sta	hstsec
	mov	a, d
	sta	hsttrk
	mvi	a, 0
	sta	hsttrk+1
	push	b
	push	d
	push	h
	call	readhst
	pop	h
	lxi	d,hstbuf
	lxi	b,512
cploop	ldax	d
	mov	m,a
	inx	d
	inx	h
	dcr	c
	jnz	cploop
	dcr	b
	jnz	cploop
	pop	d
	pop	b
	dcr	b
	jz	gocpm		; Go to CP/M when done
	inr	e
	mvi	a, hstspt
	cmp	e
	jnz	rdloop
	mvi	e, 0
	inr	d
	jp	rdloop
;
;	end of	load operation, set parameters and go to cp/m
gocpm:
	mvi	a, 0c3h		;c3 is a jmp instruction
	sta	0		;for jmp to wboot
	lxi	h, wboote	;wboot entry point
	shld	1		;set address field for jmp at 0
;
	sta	5		;for jmp to bdos
	lxi	h, bdos		;bdos entry point
	shld	6		;address field of Jump at 5 to bdos
;
	lxi	b, 80h		;default dma address is 80h
	call	setdma
;
	;ei			;enable the interrupt system
	lda	cdisk		;get current disk number
	mov	c, a		;send to the ccp
	jmp	ccp		;go to cp/m for further processing
;
;
; I/O handlers
; 

; Console status
const:	call	cons
	ora	a
	rz
	mvi	a, 0FFh
	ret

cons	lda	iobyte
	call	indxit
	dw	ttystat
	dw	crtstat
	dw	rdrstat
	dw	rdrstat

rdrstat	lda	iobyte
	rrc
	call	gotoit
	dw	ttyin
	dw	rdrin
	dw	crtin
	dw	usrin

; Console in
conin	lda	iobyte
	call	indxit
	dw	ttyin
	dw	crtin
	dw	reader
	dw	reader

; Console out
conout	lda	iobyte
	call	indxit
	dw	ttyout
	dw	crtout
	dw	list
	dw	punch

; List out
list	lda	iobyte
	rlc
	rlc
	call	indxit
	dw	ttyout
	dw	crtout
	dw	lptout
	dw	punch

; List status
listst	lda	iobyte
	rlc
	rlc
	call	indxit
	dw	ttystat
	dw	crtstat
	dw	nulstat
	dw	nulstat

; Punch out
punch	lda	iobyte
	rrc
	rrc
	rrc
	call	gotoit
	dw	ttyout
	dw	crtout
	dw	usrout
	dw	usrout

; Reader in
reader	lda	iobyte
	rrc
	call	gotoit
	dw	ttyin
	dw	rdrin
	dw	crtin
	dw	usrin

; Subroutine dispatch
indxit	rlc
gotoit	ani	6
	xthl
	push	d
	mov	e, a
	mvi	d, 0
	dad	d
	mov	a, m
	inx	h
	mov	h, m
	mov	l, a
	pop	d
	xthl
	ret

; Device drivers

; TTY input
ttyin	in	auxs		; Poll for TTY data
	ani	rxd
	jz	ttyin
	in	auxd
	ret
	
; TTY input status
ttystat	in	auxs
	ani	rxd		; Mask recieve data
	rz			; If there's nothing, return
	mvi	a, 0FFh		; Else load acc
	ret
	
; CRT input from keyboard
crtin:	in	conss
	ani	rxd
	jz	crtin
	in	consd
	ani	07Fh
	ret
	
; CRT keyboard status
crtstat	;console status, return 0ffh if character ready, 00h if not
	in	conss
	ani	rxd		; Mask recieve data
	rz			; If there's nothing, return
	mvi	a, 0FFh		; Else load acc
	ret
	

; TTY output
ttyout	in	auxs
	ani	txd
	jz	ttyout
	mov	a, c
	out	auxd
	ret
	
; CRT output
crtout	in	conss
	ani	txd
	jz	crtout
	mov	a, c
	ani	07Fh
	out	consd
	ret

rdrin	mvi	a, 01Ah		; Return EOF for reader device
	ret
	
nulstat	xra	a
usrin
usrout
lptout	ret


;
;
;	i/o drivers for the disk follow
;	for now, we will simply store the parameters away for use
;	in the read and write	subroutines
;
	;home the selected disk
home:	lda	hstwrt	;check for pending write
	ora	a
	jnz	homed
	sta	hstact	;clear host active flag
homed:	ret
;
seldsk: ;select disk
	lxi	h,0		;not exist by default
	mvi	a,drives-1
	cmp	c
	rc	seldsk1		;drive not exist
	
	mov	a,c		;selected disk number
	sta	sekdsk		;seek disk number
	mov	l,a		;disk number to HL
	mvi	h,0
	dad	h		;multiply by 16
	dad	h
	dad	h
	dad	h
	lxi	d,dpbase		;base of parm block
	dad	d		;hl=.dpb(curdsk)
	ret
	;
settrk:	;set track given by register bc
	mov	h,b
	mov	l,c
	shld	sektrk
	ret
;
setsec:
	;set sector given by register c 
	mov	a,c
	sta	seksec		;sector to seek
	ret

setdma:
	;set dma address given by BC
	mov	h,b
	mov	l,c
	shld	dmaadr
	ret
;
sectran:
	;translate sector number BC
	mov	h,b
	mov	l,c
	ret
;
;*****************************************************
;*                                                   *
;*	The READ entry point takes the place of      *
;*	the previous BIOS defintion for READ.        *
;*                                                   *
;*****************************************************
read:
	;read the selected CP/M sector
	xra	a
	sta	unacnt
	mvi	a,1
	sta	readop		;read operation
	sta	rsflag		;must read data
	mvi	a,wrual
	sta	wrtype		;treat as unalloc
	jmp	rwoper		;to perform the read
;
;*****************************************************
;*                                                   *
;*	The WRITE entry point takes the place of     *
;*	the previous BIOS defintion for WRITE.       *
;*                                                   *
;*****************************************************
write:
	;write the selected CP/M sector
	xra	a		;0 to accumulator
	sta	readop		;not a read operation
	mov	a,c		;write type in c
	sta	wrtype
	cpi	wrual		;write unallocated?
	jnz	chkuna		;check for unalloc
;
;	write to unallocated, set parameters
	mvi	a,blksiz/128	;next unalloc recs
	sta	unacnt
	lda	sekdsk		;disk to seek
	sta	unadsk		;unadsk = sekdsk
	lhld	sektrk
	shld	unatrk		;unatrk = sectrk
	lda	seksec
	sta	unasec		;unasec = seksec
;
chkuna:
	;check for write to unallocated sector
	lda	unacnt		;any unalloc remain?
	ora	a
	jz	alloc		;skip if not
;
;	more unallocated records remain
	dcr	a		;unacnt = unacnt-1
	sta	unacnt
	lda	sekdsk		;same disk?
	lxi	h,unadsk
	cmp	m		;sekdsk = unadsk?
	jnz	alloc		;skip if not
;
;	disks are the same
	lxi	h,unatrk
	call	sektrkcmp	;sektrk = unatrk?
	jnz	alloc		;skip if not
;
;	tracks are the same
	lda	seksec		;same sector?
	lxi	h,unasec
	cmp	m		;seksec = unasec?
	jnz	alloc		;skip if not
;
;	match, move to next sector for future ref
	inr	m		;unasec = unasec+1
	mov	a,m		;end of track?
	cpi	cpmspt		;count CP/M sectors
	jc	noovf		;skip if no overflow
;
;	overflow to next track
	mvi	m,0		;unasec = 0
	lhld	unatrk
	inx	h
	shld	unatrk		;unatrk = unatrk+1
;
noovf:
	;match found, mark as unnecessary read
	xra	a		;0 to accumulator
	sta	rsflag		;rsflag = 0
	jmp	rwoper		;to perform the write
;
alloc:
	;not an unallocated record, requires pre-read
	xra	a		;0 to accum
	sta	unacnt		;unacnt = 0
	inr	a		;1 to accum
	sta	rsflag		;rsflag = 1
;
;*****************************************************
;*                                                   *
;*	Common code for READ and WRITE follows       *
;*                                                   *
;*****************************************************
rwoper:
	;enter here to perform the read/write
	xra	a		;zero to accum
	sta	erflag		;no errors (yet)
	lda	seksec		;compute host sector

	ora	a		;carry = 0
	rar			;shift right
	ora	a		;carry = 0
	rar			;shift right
	
	sta	sekhst		;host sector to seek
;
;	active host sector?
	lxi	h,hstact	;host active flag
	mov	a,m
	mvi	m,1		;always becomes 1
	ora	a		;was it already?
	jz	filhst		;fill host if not
;
;	host buffer active, same as seek buffer?
	lda	sekdsk
	lxi	h,hstdsk	;same disk?
	cmp	m		;sekdsk = hstdsk?
	jnz	nomatch
;
;	same disk, same track?
	lxi	h,hsttrk
	call	sektrkcmp	;sektrk = hsttrk?
	jnz	nomatch
;
;	same disk, same track, same buffer?
	lda	sekhst
	lxi	h,hstsec	;sekhst = hstsec?
	cmp	m
	jz	match		;skip if match
;
nomatch:
	;proper disk, but not correct sector
	lda	hstwrt		;host written?
	ora	a
	cnz	writehst	;clear host buff
;
filhst:
	;may have to fill the host buffer
	lda	sekdsk
	sta	hstdsk
	lhld	sektrk
	shld	hsttrk
	lda	sekhst
	sta	hstsec
	lda	rsflag		;need to read?
	ora	a
	cnz	readhst		;yes, if 1
	xra	a		;0 to accum
	sta	hstwrt		;no pending write
;
match:
	;copy data to or from buffer
	lda	seksec		;mask buffer number
	ani	secmsk		;least signif bits
	mov	l,a		;ready to shift
	mvi	h,0		;double count
	dad	h		;shift left 7
	dad	h
	dad	h
	dad	h
	dad	h
	dad	h
	dad	h
;	hl has relative host buffer address
	lxi	d,hstbuf
	dad	d		;hl = host address
	xchg			;now in DE
	lhld	dmaadr		;get/put CP/M data
	mvi	c,128		;length of move
	lda	readop		;which way?
	ora	a
	jnz	rwmove		;skip if read
;
;	write operation, mark and switch direction
	mvi	a,1
	sta	hstwrt		;hstwrt = 1
	xchg			;source/dest swap
;
rwmove:
	;C initially 128, DE is source, HL is dest
	ldax	d		;source character
	inx	d
	mov	m,a		;to dest
	inx	h
	dcr	c		;loop 128 times
	jnz	rwmove
;
;	data has been moved to/from host buffer
	lda	wrtype		;write type
	cpi	wrdir		;to directory?
	lda	erflag		;in case of errors
	rnz			;no further processing
;
;	clear host buffer for directory write
	ora	a		;errors?
	rnz			;skip if so
	xra	a		;0 to accum
	sta	hstwrt		;buffer written
	call	writehst
	lda	erflag
	ret
;
;*****************************************************
;*                                                   *
;*	Utility subroutine for 16-bit compare        *
;*                                                   *
;*****************************************************
sektrkcmp:
	;HL = .unatrk or .hsttrk, compare with sektrk
	xchg
	lxi	h,sektrk
	ldax	d		;low byte compare
	cmp	m		;same?
	rnz			;return if not
;	low bytes equal, test high 1s
	inx	d
	inx	h
	ldax	d
	cmp	m	;sets flags
	ret
;
;*****************************************************
;*                                                   *
;*	WRITEHST performs the physical write to      *
;*	the host disk, READHST reads the physical    *
;*	disk.					     *
;*                                                   *
;*****************************************************
writehst:
	;hstdsk = host disk #, hsttrk = host track #,
	;hstsec = host sect #. write "hstsiz" bytes
	;from hstbuf and return error flag in erflag.
	;return erflag non-zero if error
hdwrl	in	idestat	; get status of drive
	xri	040h		; invert ready bit
	ani	0c0h		; mask bsy and ready
	jnz	hdwrl
	; output c/h/s to ide interface
	call	wchs
	; send command, wait for drq to go active
	mvi	a, 030h		; write sector command
	out	idecmd
	lxi	h, hstbuf	; To here
wdrq	in	idestat		; get status
	mov	c, a
	ani	1
	jnz	wrerr		; jump on error flag
	mov	a, c
	ani	8
	jz	wdrq		; wait for drq
	mvi	b, 0		; transfer 256 words (512 bytes)
writel	mov	c, m		; get low byte to c
	inx	h
	mov	a, m		; get high byte to a (written first)
	inx	h
	out	idehi		; send high byte
	mov	a, c
	out	idedat		; send low byte
	dcr	b
	jnz	writel		; loop
	in	idestat		; get error flag
	ani	1		; error flag
	rz
wrerr	ori	0FFh
	ret
;
readhst:
	;hstdsk = host disk #, hsttrk = host track #,
	;hstsec = host sect #. read "hstsiz" bytes
	;into hstbuf and return error flag in erflag.
hdreadl	in	idestat		; get status of drive
	xri	040h		; invert ready bit
	ani	0c0h		; mask bsy and ready
	jnz	hdreadl
	; output c/h/s to ide interface
	call	wchs
	mvi	a, 020h		; Read command
	out	idecmd
	lxi	h, hstbuf	; To here
rdrq	in	idestat		; get status
	mov	c, a
	ani	1
	jnz	rderr		; jump on error flag
	mov	a, c
	ani	8
	jz	rdrq		; wait for drq to go active
	mvi	b, 0		; count 256 times in b
readl	in	idedat		; Get low byte
	mov	m, a
	inx	h	
	in	idehi		; write high byte to buffer
	mov	m, a
	inx	h
	dcr	b		; Count in b
	jnz	readl
	in	idestat
	ani	1
	rz
rderr	ori	0FFh		; return with non-zero in a on error
	ret
	
	; Send the C/H/S address to the drive
wchs	lda	hstsec
	inr	a
	out	idest		; Write sector
	mvi	a,0A0h		; IDE requirement
	out	idehead		; Write head
	lda	hsttrk		; Write low cylinder
	out	idecyll 
	lda	hsttrk+1
	ani	3		; Mask through two lowest bits
	out	idecylh
	mvi	a, 1		; 1 sector to transfer
	out	idesc
	ret
	
	; Print the message at HL
print	mov	c, m
	inx	h
	xra	a
	ora	c
	rz
	call	conout
	jmp	print

mesg	db	'62k CP/M 2.2',13,10,0



;	the remainder of the cbios is reserved uninitialized
;	data area, and does not need to be a Part of the
;	system	memory image (the space must be available,
;	however, between"begdat" and"enddat").
;
sekdsk:	ds	1		;seek disk number
sektrk:	ds	2		;seek track number
seksec:	ds	1		;seek sector number
;
hstdsk:	ds	1		;host disk number
hsttrk:	ds	2		;host track number
hstsec:	ds	1		;host sector number
;
sekhst:	ds	1		;seek shr secshf
hstact:	ds	1		;host active flag
hstwrt:	ds	1		;host written flag
;
unacnt:	ds	1		;unalloc rec cnt
unadsk:	ds	1		;last unalloc disk
unatrk:	ds	2		;last unalloc track
unasec:	ds	1		;last unalloc sector
;
erflag:	ds	1		;error reporting
rsflag:	ds	1		;read sector flag
readop:	ds	1		;1 if read operation
wrtype:	ds	1		;write operation type
dmaadr:	ds	2		;last dma address
hstbuf:	ds	hstsiz		;host buffer

alv00: 	ds	257		;allocation vector 0
dirbuf: 	ds	128 		;scratch directory area
	
	end
