; (c) Freddy 2015 antenn@land.ru
; For non-commercial use only! Do not redistribute.
; Do not republish in any media.
;
; Monitor for the 8080 v0.2.
;
; RAM is required for the STACK and BUFFER
; labels. Otherwise, the code is ROMable.
;
; CMDTAB is a table of commands. The format
; is one byte for the upper-case letter for
; the command, followed by a word for the
; address to jump to. On entry, DE contains
; the hexadecimal number entered after the
; command letter by the user. The table is
; terminated with a zero.

MONORG	EQU	0F800H

PROMPT	EQU	'.'		; Monitor prompt
INPLEN	EQU	5		; Length of input buffer

; Control characters
BEL	EQU	7
BS	EQU	8
LF	EQU	10
CR	EQU	13
ESC	EQU	27
DEL	EQU	127

TIM_CTRL	EQU	0FBH		;Timer control
TIMER0	EQU	0F8H		;Timer0 counter
TIMER1	EQU	0F9H		;Timet1 counter

CONSD	EQU	0F0H		; Console data
CONSC	EQU	0F1H		; Console control
AUXD	EQU	0F4H		; Aux data
AUXC	EQU	0F5H		; Aux control
TXD	EQU	1		; Transmit status
RXD	EQU	2		; Receive status

BANK	EQU	0FCH		;memory switch port

BSECT	EQU	07C00H		;boot sector offset

IDEDAT	EQU	0E8H	; IDE DATA REGISTER (LOW)
			; HIGH IS LATCHED INTO E4H ON READ
			; E4H MUST CONTAIN HIGH DATA ON WRITE
IDEERR	EQU	0E9H	; IDE ERROR REGISTER
IDESC	EQU	0EAH	; IDE SECTOR COUNT REGISTER
IDEST	EQU	0EBH	; IDE START SECTOR REGISTER
IDECYLL	EQU	0ECH	; IDE CYLINDER LOW REGISTER
IDECYLH	EQU	0EDH	; IDE CYLINDER HIGH REGISTER
IDEHEAD	EQU	0EEH	; IDE HEAD REGISTER
IDECMD	EQU	0EFH	; IDE COMMAND REGISTER
IDESTAT	EQU	0EFH	; IDE STATUS REGISTER
IDEHI	EQU	0E4H	; IDE HIGH BYTE REGISTER


	ORG	MONORG
	JMP	COLD

MESG:	DB	CR,LF,"8080 MONITOR",CR,LF,0
SERTAB:	DB	0,0,0,40H,4EH,35H

; Command table. Character followed by address.
CMDTAB:	DB	'B'
	DW	BOOT
	DB	'D'
	DW	DUMP
	DB	'J'
	DW	JUMPTO
	DB	'M'
	DW	MODIFY
	DB	'R'
	DW	READ
	DB	0		; End of table

; COUT sends a character to the console	
COUT:	IN	CONSC		; read status
	ANI	TXD		; mask transmit status
	JZ	COUT		; loop until non-zero
	MOV	A,C		; Send character in C
	OUT	CONSD
	RET

; CIN receives a character from the console
CIN:	IN	CONSC		; read status
	ANI	RXD		; mask receive status
	JZ	CIN		; wait for status to be true
	IN	CONSD		; receive character
	RET

; Console status, returns 0 if no character waiting
CSTAT:	IN	CONSC
	ANI	RXD
	RZ
	MVI	A,0FFH
	RET

; Initialize console USART
CINIT:	LXI	B,6		; B=0, C=6
	LXI	H,SERTAB		; Point at table
CINITL:	DCR	B		; Short delay
	JNZ	CINITL
	MOV	A,M		; Get next byte
	INX	H		; Increment to next byte
	OUT	CONSC		; Send to port
	DCR	C		; Count bytes
	JNZ	CINITL		; Loop
	IN	CONSD		; Clear any rubbish data
	RET
	
; Aux serial port output
AOUT:	IN	AUXC		; read status
	ANI	TXD		; mask transmit status
	JZ	AOUT		; loop until non-zero
	MOV	A,C		; Send character in C
	OUT	AUXD
	RET

; Aux serial port input
AIN:	IN	AUXC		; read status
	ANI	RXD		; mask receive status
	JZ	AIN		; wait for status to be true
	IN	AUXD		; receive character
	RET

; Console status, returns 0 if no character waiting
ASTAT:	IN	AUXC
	ANI	RXD
	RZ
	MVI	A,0FFH
	RET

; Initialize Aux USART
AINIT:	LXI	B,6		; B=0, C=6
	LXI	H,SERTAB		; Point at table
AINITL:	DCR	B		; Short delay
	JNZ	AINITL
	MOV	A,M		; Get next byte
	INX	H		; Increment to next byte
	OUT	AUXC		; Send to port
	DCR	C		; Count bytes
	JNZ	AINITL		; Loop
	IN	AUXD		; Clear any rubbish data
	RET
	
; Perform initialization
INIT:	CALL	CINIT
	JMP	AINIT
	
; PRINTs a string to the console	
PRINT:	MOV	C,M		; Retrieve character to C
	INX	H		; Move to next character
	XRA	A		; Clear accumulator for test
	ORA	C		; Test if C is zero
	RZ			; Return if it is
	CALL	COUT		; Print the character
	JMP	PRINT		; Loop until zero detected

; INP a string to the buffer
; Returns string length in B, string stored at HL (not returned)
INP:	MVI	B,0		; Count string length in B
INPL:	CALL	CIN		; Get next character
	CPI	DEL		; Delete?
	JZ	INPBS		; Backspace routine
	CPI	' '		; Control character?
	JC	INPC		; Jump if it is
	CPI	'a'		; Lower case?
	JC	INPU		; It isn't, skip conversion
	ANI	NOT 20h		; Remove upper case bit
INPU:	MOV	C,A		; Character to receive in C
	MOV	A,B		; Count string length
	CPI	INPLEN		; Compare string length to limit
	JNC	INPBEL		; Sound bell if too long
	INR	B		; Increment string length
	MOV	M,C		; Store character to RAM
	INX	H		; Increment to next memory character
	CALL	COUT		; Echo the character
	JMP	INPL		; Loop to next character

INPC:	CPI	BS		; Backspace?
	JZ	INPBS		; Yes, perform backspace
	CPI	CR		; End of line
	JZ	CRLF
	CPI	LF
	JZ	CRLF
	; Any other control characters are ignored
INPBEL:	MVI	C,BEL		; Sound bell if invalid character
	CALL	COUT
	JMP	INPL		; Loop to next character

INPBS:	MOV	A,B		; Is B zero?
	ORA	A
	JZ	INPBEL		; Can't backsapce nothing, sound bell
	DCR	B		; Retract string length
	DCX	H		; and buffer position
	MVI	C,BS		; Rubout screen
	CALL	COUT
	CALL	SPACE
	MVI	C,BS
	CALL	COUT
	JMP	INPL

CRLF:	MVI	C,LF		; Newline and return
	CALL	COUT
	MVI	C,CR
	JMP	COUT

; Print a space
SPACE:	MVI	C,' '
	JMP	COUT
	
; Print A as a hexadecimal number
PHEX:	PUSH	PSW
	RRC
	RRC
	RRC
	RRC
	CALL	PHEX1
	POP	PSW
PHEX1:	ANI	0FH
	CPI	0AH
	JC	PHEX2
	SBI	9
	ORI	040H
	JMP	PHEX3
PHEX2:	ORI	030H
PHEX3	MOV	C,A
	JMP	COUT
	
; Get operands from text buffer at HL to DE. String length in B.
GETOP:	LXI	D,0		; Operands to DE
OPERL:	MOV	A,M
	CPI	'A'
	JC	OPERN		; Dealing with numbers (less than A)
	SBI	7		; Subtract 7 to make it hexadecimal
OPERN:	XCHG			; Shift DE left by four bits
	DAD	H
	DAD	H
	DAD	H
	DAD	H
	XCHG
	ANI	0Fh		; Make it a nibble
	ORA	E		; Combine with DE
	MOV	E,A
	INX	H		; Move to next character
	DCR	B		; Loop until at end of string
	JNZ	OPERL
	RET

; Dump 256 bytes starting at DE
DUMP:	MVI	B,0		; Count 256 bytes in B
DUMPA:	MOV	A,B		; Check if CR/LF needed
	ANI	0FH
	JNZ	DUMPL		; Jump if not needed
	CALL	CRLF
	MOV	A,D		; Print address
	CALL	PHEX
	MOV	A,E
	CALL	PHEX
DUMPL:	MVI	C,' '		; Space between address and data
	MOV	A,B		; Check for address MOD 16 + 8
	ADI	8
	ANI	0FH
	JNZ	DUMPSP
	MVI	C,'-'		; Print a dash if at 8th byte
DUMPSP:	CALL	COUT
	LDAX	D		; Get data at DE
	INX	D
	CALL	PHEX
	DCR	B
	JNZ	DUMPA
	JMP	CRLF		; Return through CRLF

; Modify contents of memory.
MODIFY:	CALL	CRLF
	MOV	A,D		; Print address
	CALL	PHEX
	MOV	A,E
	CALL	PHEX
	CALL	SPACE
	LDAX	D		; Print data at address
	CALL	PHEX
	CALL	SPACE
	MVI	H,0		; H determnies hi/lo nibble state
MODIFL:	CALL	CIN		; Get input
	CPI	' '		; Less than space
	JC	CRLF		; Terminate through CRLF on control character
	CPI	'.'
	JZ	CRLF		; Terminate through CRLF on dot
	CPI	','		; Comma goes to previous address
	JZ	MODPRV
	CPI	'0'
	JC	MODIFL		; Ignore anything under 0
	CPI	'9'+1
	JC	MODNUM		; Is a number
	CPI	'A'
	JC	MODIFL		; Is invalid (between 3A and 40)
	CPI	'F'+1
	JC	MODCHR		; Is a letter
	CPI	'a'
	JC	MODIFL		; Is invalid (between 47 and 60)
	CPI	'f'+1
	JC	MODCHR		; Is a letter
	JMP	MODIFL		; Invalid
	
MODCHR:	ANI	NOT 020H	; Make upper case
	MOV	C,A		; Move to C to print
	CALL	COUT		; A is restored by COUT
	SBI	7		; Make hexadecimal
	JMP	MODN1
MODNUM:	MOV	C,A		; Move to C to print
	CALL	COUT		; A is restored by COUT
MODN1:	ANI	0FH		; Make it a nibble
	MOV	B,A		; Keep partial result in B
	XRA	A		; H = 0 or 1?
	ORA	H
	JNZ	MODLO		; Modify low nibble (second character)
	MVI	H,1		; Indicate low nibble state on next run
	MOV	A,B
	ADD	A		; Shift to high nibble
	ADD	A
	ADD	A
	ADD	A
	MOV	L,A		; Result in L
	JMP	MODIFL		; Get next character

MODLO:	MVI	H,0		; Set back to first state
	MOV	A,B		; Restore partial result
	ORA	L		; Combine high and low nibble
	STAX	D		; Store entry to DE
	INX	D		; and increment DE
	JMP	MODIFY		; Run next byte
	
MODPRV:	DCX	D
	JMP	MODIFY

	
; Read character from AUX port, warm booting if the ESC key is pressed
; on the console
GETAUX:	CALL	CSTAT		; Check for ESC
	JZ	GETA
	CALL	CIN		; Get character if pressed
	CPI	ESC		; Is it esc?
	RNZ			; Return if it isn't
	JMP	WARM		; Terminate if it is
GETA:	CALL	ASTAT		; Read A status
	JZ	GETAUX		; If nothing waiting, check console
	JMP	AIN		; Get character
	

; Get a HEX number of 1..4 digits (in C) to HL
GETNUM:	CALL	GETAUX
	CPI	':'
	JZ	BADREC		; Pop return address off stack, restart
	CPI	'9'+1
	JC	GETN1
	SUI	7
GETN1:	ANI	0FH
	DAD	H
	DAD	H
	DAD	H
	DAD	H
	ORA	L
	MOV	L,A
	DCR	C
	JNZ	GETNUM
	RET

BADREC:	POP	H		; Don't return to calling address
	MVI	C,'!'
	CALL	COUT		; Display mark to indicate bad record
	JMP	READB		; Colon found, read new record
	
; Read Intel HEX data from AUX port. Values are calculated on the
; fly because the 8080 may not be able to keep up with the incoming
; data.
READ:	CALL	GETAUX		; Get next character
	CPI	':'		; Colon indicates start of record
	JNZ	READ		; Wait until it is
READB:	MVI	C,2		; Read two bytes to HL
	CALL	GETNUM		; Get byte count to B
	MOV	B,L
	ORA	A		; L is also returned in A, check for zero
	JZ	CRLF		; Return through CRLF
	MVI	C,4
	CALL	GETNUM		; Get address to DE
	XCHG
	MVI	C,2		; Get record type
	CALL	GETNUM		; Ignore record type
READL:	MVI	C,2		; Get next byte
	CALL	GETNUM
	MOV	A,L		; Store that byte
	STAX	D
	INX	D		; Next address
	DCR	B		; Decrement byte count
	JNZ	READL
	MVI	C,2		; Checksum
	CALL	GETNUM
	MVI	C,'*'		; Indicate record received
	CALL	COUT
	JMP	READ
	
	
; Jumps to address specified
JUMPTO:	XCHG			; DE to HL so we can jump
	PCHL			; And go to it
	
	
COLD:	MVI	A,1		; Disable ROM, replace with RAM
	OUT	BANK		; gets switched to RAM space
	
	MVI	A,016H		;TIMER0 mode3,LOW byte load,binary
	OUT	TIM_CTRL
	MVI	A,056H		;TIMER1 mode3,LOW byte load,binary
	OUT	TIM_CTRL
	
	MVI	A,12		;SET 9600X16Hz
	OUT	TIMER0
	OUT	TIMER1
	
	LXI	SP, STACK	; Cold start prints message
	CALL	INIT
	LXI	H,MESG
	CALL	PRINT
WARM:	LXI	SP,STACK	; Warm start
	
MON:	MVI	C,PROMPT	; Display the prompt
	CALL	COUT
	LXI	H,BUFFER	; Retrieve to buffer
	CALL	INP
	XRA	A		; Check for any input
	ORA	B
	JZ	MON		; Redo input		
	LXI	H,BUFFER	; Get command from buffer
	MOV	C,M		; To C
	DCR	B		; Decrement string legnth
	INX	H		; Move to operands
	CALL	GETOP		; Get operand data to DE
	
	LXI	H, CMDTAB	; Get command table
SEARCH:	MOV	A,M
	INX	H		; Move to command address
	ORA	A		; End of command table?
	JZ	NOCMD		; No such command
	CMP	C		; Command character in C
	JZ	DISPAT		; Dispatch to that command
	INX	H		; Not at end of table
	INX	H		; Move to next command
	JMP	SEARCH

NOCMD:	MVI	C,BEL		; Bell indicates invalid command
	CALL	COUT
	JMP	MON
	
DISPAT:	LXI	B,MON		; Return to MON
	PUSH	B
	MOV	A,M		; Get low byte of address
	INX	H
	MOV	H,M		; Get high byte of address
	MOV	L,A		; HL = address
	PCHL

BOOT	MVI	A,1
	OUT	IDEST		; Write sector
	MVI	A,0A0H		; IDE requirement
	OUT	IDEHEAD		; Write head (always 0)
	MVI	A,0		; Write low cylinder
	OUT	IDECYLL
	MVI	A,0
	OUT	IDECYLH
	MVI	A, 1		; 1 sector to transfer
	OUT	IDESC
	
HDREADY	IN	IDESTAT		; get status of drive
	XRI	040H		; invert ready bit
	ANI	0C0H		; mask bsy and ready
	JNZ	HDREADY
	
	LXI	H, BSECT
	MVI	A, 020H		; read hard drive
	OUT	IDECMD
RDRQ	IN	IDESTAT		; get status
	MOV	C, A
	ANI	1
	JNZ	RWERR		; jump on error flag
	MOV	A, C
	ANI	8
	JZ	RDRQ		; wait for drq to go active
	MVI	B, 0		; count 256 times in b
HREADL	IN	IDEDAT		; Get low byte
	MOV	M, A
	INX	H	
	IN	IDEHI		; get high byte from buffer
	MOV	M, A
	INX	H
	DCR	B		; Count in b
	JNZ	HREADL
	IN	IDESTAT
	ANI	1
	RNZ
	JMP	BSECT
RWERR	IN	IDEERR		; return with non-zero in a on error
	RET

	DS	32
STACK	EQU	$
BUFFER	EQU	$

	END
	