/*  This file is part of UKNCBTL.
    UKNCBTL is free software: you can redistribute it and/or modify it under the terms
of the GNU Lesser General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.
    UKNCBTL is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Lesser General Public License for more details.
    You should have received a copy of the GNU Lesser General Public License along with
UKNCBTL. If not, see <http://www.gnu.org/licenses/>. */

#ifndef _ESCPARSER_H_
#define _ESCPARSER_H_

#include <iostream>
#include <vector>

extern unsigned short RobotronFont[];

/* Units of measure expressed in PostScript units */
#define INCH 72.0

/* The graphics modes **** not used by me **** */
#define GRAPHICS_60   0
#define GRAPHICS_120a 1
#define GRAPHICS_120b 2
#define GRAPHICS_240  3
#define GRAPHICS_80   4
#define GRAPHICS_72   5
#define GRAPHICS_90   6
#define GRAPHICS_24_60 32
#define GRAPHICS_24_120 33
#define GRAPHICS_24_90 38
#define GRAPHICS_24_180 39
#define GRAPHICS_24_360 40

// Indicates what type of graphics mode is on
#define	GRAPHICS_MODE_8_PIN		0 // ESC *, ESC K, ESC L, ESC Y, ESC Z
#define	GRAPHICS_MODE_9_PIN		1 // ESC ^

#define	FX80_PAGE_WIDTH		8.0
#define	FX80_PAGE_HEIGHT	11
#define	FX80_VERT_DPI		216
#define FX80_HORIZ_DPI		240

// in output to postscript
#define EMULATION_P6_INTERPRETATION	4
#define EMULATION_24PIN_UNITS		8
#define EMULATION_8IN_LINE			16		/* Narrow carriage */

/* Print colours */
#define COLOUR_BLACK	0
#define COLOUR_MAGENTA	1
#define COLOUR_CYAN		2
#define COLOUR_VIOLET	3
#define COLOUR_YELLOW	4
#define COLOUR_ORANGE	5
#define COLOUR_GREEN	6
#define COLOUR_BROWN	7

/* Truncate line longer than this */
#define BUFFER_SIZE		300

/* These are the master mode select codes */
#define MODE_3X_HORIZONTAL			1024	/* my hack */
#define MODE_4X_HORIZONTAL			2048	/* my hack */

#define MODE_2X_VERTICAL			4096	/* my hack */
#define MODE_4X_VERTICAL			8192	/* my hack */
#define MODE_2X_VERTICAL_BASELINE	16384	/* my hack */

#define MODE_15PITCH				512		/* my hack */
#define MODE_UNDERLINE				128
#define MODE_ITALIC					64
#define MODE_EXPANDED				32
#define MODE_DOUBLE_STRIKE			16
#define MODE_EMPHASIZED				8
#define MODE_CONDENSED				4
#define MODE_PROPORTIONAL			2
#define MODE_ELITE					1
#define MODE_PICA					0

// Base abstract class for output drivers
class OutputDriver
{
protected:
    std::ostream& m_output;

public:
    OutputDriver(std::ostream& output) : m_output(output) { }

public:
    // Write beginning of the document
    virtual void WriteBeginning(int pagestotal) { }  // Overwrite if needed
    // Write ending of the document
    virtual void WriteEnding() { }  // Overwrite if needed
    // Write beginning of the page
    virtual void WritePageBeginning(int pageno) { }  // Overwrite if needed
    // Write ending of the page
    virtual void WritePageEnding() { }  // Overwrite if needed
    // Write strike by one pin
    virtual void WriteStrike(float x, float y, float r) = 0;  // Always overwrite
};

// Stub driver, does nothing
class OutputDriverStub : public OutputDriver
{
public:
    OutputDriverStub(std::ostream& output) : OutputDriver(output) { };

public:
    virtual void WriteStrike(float x, float y, float r) { }
};

struct PdfXrefItem
{
    std::streamoff offset;
    int size;
    char flag;
public:
    PdfXrefItem(std::streamoff anoffset, int asize, char aflag)
    {
        offset = anoffset;
        size = asize;
        flag = aflag;
    }
};
// PDF driver with multipage support
class OutputDriverPdf : public OutputDriver
{
public:
    OutputDriverPdf(std::ostream& output) : OutputDriver(output) { };

public:
    virtual void WriteBeginning(int pagestotal);
    virtual void WriteEnding();
    virtual void WritePageBeginning(int pageno);
    virtual void WritePageEnding();
    virtual void WriteStrike(float x, float y, float r);

private:
    std::vector<PdfXrefItem> xref;
    std::string pagebuf;
    float strikesize;
};


//////////////////////////////////////////////////////////////////////
// ESC/P interpreter

class EscInterpreter
{
private:  // Input and output
    std::istream& m_input;
    OutputDriver& m_output;

private:
			// History
	char	dbuf[1024];
    char	*dp;

	
	// Current state
	float m_cr;			// Printer head dot size radius
double m_x, m_y;      // Current position		TBD: stored in wich units?
//	int m_curX, m_curY;						// Current pin location (in dots)
	int xpos, ypos; /* These variables describe the next desired print possition */
	
	bool m_marksMade;		// Indicate marks made on the page. Only emmit the line if it makes marks: if (someblack) {}

	bool DILswitch[7][1];
    bool m_printmode;   // 0 - draft, 1 - (N)LQ (Near) Letter Quality /// SLQ: 360 dpi
	bool m_expandedOneLine;					/* one line expanded mode */
    bool m_font_ex;      //  () .  ,  (elongated)
    bool m_font_ds;      //  () . double strike (NLQ simulated)
	bool m_font_dh;      //  () 
	bool m_font_pr;      //  
    bool m_font_fe;      //  
    bool m_font_ks;      //  , condensed, compressed - in SI mode
    bool m_font_el;      //  ""			m_elite
	bool m_font_mi;			// micron 
	bool m_font_it;			// Italic
    bool m_font_un;      // 
    bool m_super_script; //  
    bool m_sub_script;   //  
    unsigned char m_internationalcharset;  //   

	/* These constants describe the horizontal and vertical spacing units */
	int HORIZONTAL_UNITS;
	int VERTICAL_UNITS;
		/* Factor to convert 72-ths or 60-ths to VERTICAL_UNITS */
	int VFACTOR;
	int PTR_MM_METRIC;  // unit = 1/100 mm - i.e. 1 cm = 10 mm = 10 * 100 units


	/* The routine select_font() uses these to describe the character style to the output routine */
	int out_style;
	double out_hscale;
	double out_vscale;

	int graphic_mode_K;
	int graphic_mode_L;
	int graphic_mode_Y;
	int graphic_mode_Z;

	/* The current tabs in terms of VERTICAL_UNITS and HORIZONTAL_UNITS (mostly in inch?) */
	int				m_tabs[32];					// Up to 32 tabs supported - tabs_horizontal

	int				m_vertTabs[8][16];			// Eight channels of 16 tabs - tabs_vertical
	unsigned char	m_tabChannel;				// Currently selected tab channel
	char			m_escTabChannel;			// Actively updated Tab Channel

	int numHorizTabs;					// Number of configured tabs
	int numVertTabs;					// Number of configured tabs

	double m_cpi, actcpi;				// cpi value set by program and the actual one (taking in account font types)
	int m_dpi;							// dpi of the page


	char				m_skipPerf;					// Indicates if skip Perforation is on
	int perforation_skip;	/* amount to skip between pages */

	double				m_width;					// Width of printable region
	/* These variables describe the page length and width we think we are printing on in terms of the above units */ /// postscript
	double				m_formHeight;				// Height of Form (11 inch default)
	double				m_allocHeight;				// Height of Form allocation
	int page_width;
	int page_length;
	int page_height;
	/// Size of page (in inch)

	/* Margins of the page. All are expressed in terms of *_UNITS. If top_margin is zero then half of perforation_skip is used instead */
/// Defaults:
/// bottom margin: Either no margin or 1-inch margin, depending on the DIP-switch setting
///	left margin: The left-most column (column 1)
///	right margin: The right-most column
	double				m_leftMargin;				// Left margin, in inches
//	int					m_leftMarginX;				// Left margin, in dots
	double				m_rightMargin;				// Right margin, in inches
//	int					m_rightMarginX;				// Right margin, in dots
	double				m_topMargin;				// Top margin, in inches
//	int					m_topMarginY;				// Top margin, in dots
	double				m_bottomMargin;				// Bottom margin, in inches
//	int					m_bottomMarginY;			// Bottom margin, in dots

	int					m_vertDots;					// Number of vertical "dots"
	int					m_horizDots;				// Number of horizontal "dots"
	int					m_vertDpi;					// Vertical DPI
	int					m_horizDpi;					// Horizontal DPI
	int					m_bytesPerLine;				// Number of bytes per line on page


	/* This describes the current character spacing and line spacing in terms of the above units */
	double m_charSpacing_shift_x; // Shift for text printout
	double m_lineSpacing_shift_y; // Current line spacing setting - the size of one line (in inch)

	int m_txt_spacing_between_lines;
	int m_gra_spacing_between_lines;
	int line_feed_direction;				/* 1 or -1 */
	int line_spacing_multiplier;			/* 1 or 2 */

	/* 7-bit & printUpperControl */
	int set8th;
	int clear8th;
	
	bool m_highCodesPrinted;		// Indicates if high codes (upper command characters) should be printed
	bool m_lowCodesPrinted;			// Indicates if low codes are printed
	int upper_controls; // (variant of previouse) Are high bit control codes allowed? => Default: Codes 128 to 159 treated as control codes

	/* The printer language uses these to describe the current character syle */
	int current_charmode;					/* ESC "!" format */
	int script_mode;						/* Script mode (superscript, subscript) */
	int international_char_set;				/* International character set number */
	int nlq_font;							/* Roman or San-Serif */
	int justification;						/* Justification mode */
	int charset;							/* Italic or IBM */
	int hmi;							// Horizontal motion index (in inch; overrides CPI settings)
	int extraIntraSpace;				// Extra space between two characters (set by program, in inch)
	int extra_dot_spacing;					/* Extra space added to char spacing */
	
	int print_colour;						/* Print colour */
	bool colour_ok;							/* Should we emmit colour code? */
	bool uses_colour;
	
	bool m_zeroSlashed;
	bool m_zeroRedefined;

	/* Automatic line feed and carriage return */
	bool auto_cr;
	bool auto_lf;

	/* The character description structure */
	struct CHARs {
		int style;
		double hscale;
		double vscale;
		int movement;			/* difference between xpos and previous xpos */
		int xpos;				/* xpos of start in HORIZONTAL_UNITS */
		int fixed_spacing;
		int width;				/* current_char_spacing in HORIZONTAL_UNITS */
		int extra_space;		/* extra space after */
		int c;
		int colour;				/* desired colour */
		};
	int buffer_count;
	struct CHARs buffer[BUFFER_SIZE];

	/* Previous xpos for getting space width data */
	int predicted_xpos;

public:
    // Constructor
    EscInterpreter(std::istream& input, OutputDriver& output);
    // Interpret next character or escape sequense
    bool InterpretNext();
    // Interpret escape sequence
    bool InterpretEscape();
    // is the end of input stream reached
    bool IsEndOfFile() const { return m_input.eof(); }

	/* This variable is TRUE if a page is started */
	bool in_page;


protected:
    // Retrieve a next byte from the input
    unsigned char GetNextByte ();
    // Update m_charSpacing_shift_x according to current font settings
    void UpdateShiftX ();
    // Reset the printer settings
    void PrinterReset ();
    // Print graphics
    void printGR (int printing_gr_mode, int pins);
    // Print graphics
    void printGR24 (int printing_gr_mode, int width);
    // Print the symbol using current charset
    void PrintCharacter (unsigned char ch);
    // Draw strike made by one pin
    void DrawStrike (float x, float y);
	void reset_tabs (void);
	void vertical_tabs_set (int channel);
	void horizontal_tab_increment (int inc);
	void vertical_tab_increment (int inc);
	void line_feed (int how_far);
	void page_reset (int consumed);
	void buffer_delete (int howmuch);
	void empty_buffer (void);
	void buffer_add (int c);

	void SetVertChannelTabs (unsigned char byte);

	void newPage (void);
};


//////////////////////////////////////////////////////////////////////
#endif // _ESCPARSER_H_
