﻿/*  This file is part of UKNCBTL.
    UKNCBTL is free software: you can redistribute it and/or modify it under the terms
of the GNU Lesser General Public License as published by the Free Software Foundation,
either version 3 of the License, or (at your option) any later version.
    UKNCBTL is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Lesser General Public License for more details.
    You should have received a copy of the GNU Lesser General Public License along with
UKNCBTL. If not, see <http://www.gnu.org/licenses/>. */

#include "ESCParser.h"


//////////////////////////////////////////////////////////////////////

EscInterpreter::EscInterpreter (std::istream& input, OutputDriver& output) : m_output(output), m_input(input)
{
	PrinterReset();
}

unsigned char EscInterpreter::GetNextByte ()
{
    if (m_input.eof())
        return 0;
    unsigned char result = (unsigned char)m_input.get();
	return ((result & clear8th) | set8th); /* 7-битный режим во входных данных. All data is affected, including graphics data */
/*	
	// Test for printing of low codes
	if (m_lowCodesPrinted && (ch < ' ')) // Use codes 0...31
	{
		// For codes that are printable, return FALSE
		if ((ch < 7) || (ch == 16) || ((ch > 20) && (ch != 24) && (ch != 27)))
				return false;
	}
	
	// Test if it is a high code and high codes are printed
	if ((ch > 'z') && m_highCodesPrinted) // Use codes 128...159, 255
		return false;
false here are means that character are send to default:PrintCharacter(ch);
	switch (ch & 0x7F)
*/
}

void EscInterpreter::PrinterReset () // "Power Up" settings
{
	dp = dbuf;

	set8th = 0;
	clear8th = 0xFF;

	m_highCodesPrinted = false;			// Turn off printing of high codes // Default Codes 128 to 159 are treated as printable characters (esc/p/2) / Codes 128 to 159 are treated as control codes (9-pin esc/p)
	m_lowCodesPrinted = false;			// Turn off printing of low codes => Default: Codes 128 to 159 treated as control codes
	int upper_controls = false;			// => Default: Codes 128 to 159 treated as control codes

	m_x = m_y = 0;
	m_printmode = false; // Выбор качества: 180 (LQ) или 120 (draft)
	/* TBD: POINT commands actually select the CPI settings which are equivalent to the requested point size.
	These were included to provide as much compatability with files designed to be printer on laser printers
	NORMALPOINT	18			* Select 10CPI for normal point size
	POINT7		15			* Select 16.7CPI for 7 point size print
	POINT10		27 58		* Select 12CPI for 10 point size print
	POINT12		18			* Select 10CPI for 12 point size print
	POINT14		15 27 87 1	* Select 16.7CPI and expanded print mode which yields 14 point size print */
	m_cr = 5.9f;
/*
// Get DIP switch settings
m_defCharSet = IntlDipChanged();		// Get current settings
m_defCompressed = m_pPitchOff->value();
m_defEnhance = m_pPrintWeightOff->value();
m_defSkipPerf = m_pSkipPerfOff->value();
m_topOfForm = atof(m_pTopOfForm->value());

#define  XPSIZE		6		// horizontal paper size in inches
#define  XPSIZE		8		// horizontal paper size in inches
#define  XSIZE		(XDPI * XPSIZE) // number of horizontal dots

#define  YPSIZE		8		// vertical paper size in inches
#define  YPSIZE		11		// vertical paper size in inches
#define  YSIZE		(YDPI * YPSIZE)	// number of vertical dots


unidirectional=0;
page_management_units=360;
relative_horizontal_units=180;
absolute_horizontal_units=60;
relative_vertical_units=360;
absolute_vertical_units=360;
m_topMargin=120;
bottom_margin=page_height=22*360; // 22 inches is default ???
*/

    //TODO: Настраивать режимы по DIP-переключателям
	auto_cr = m_font_ex /* */ = m_expandedOneLine = m_font_pr /* */ = m_font_ds /* */ = m_font_dh = m_font_fe /* */ = m_font_ks /* */ = m_font_el /* */ = m_font_mi = m_font_it = m_font_un = false;
	
	// Note: 1/720 дюйма здесь используется как минимально адресуемый отрезок расстояния (1 decipoint = 1/720 inch; 1 centipoint = 1/7200 inch; 1 dot = 1/300 inch)
    
	m_zeroSlashed = true;
	m_zeroRedefined = false;	// Indicate we can do slashed zero

	if (DILswitch[7][1] == true)
		auto_cr = true; /* Automatic carriage return */

	auto_lf = false; /* Automatic line feed */

    m_super_script = m_sub_script = false;
    m_internationalcharset = 0; /* Set international character set from DIP switch (0 - USA) */
	extra_dot_spacing = extraIntraSpace = 0; /* No extra inter-character spacing */

	/* Reset the spacing */
	// Changing the line spacing does not affect previous settings for vertical tabs or page length
	m_lineSpacing_shift_y = 720 * 1.0 / 6.0;					/*  6 lines/inch; default: 6 LPI */ /// VERTICAL_UNITS / 6;
	UpdateShiftX();

	/* Reset all tabs */
	reset_tabs(); // Default tabs

	buffer_count = 0;
	predicted_xpos = 0;

	/* Select a dot matrix printer to emulate */
	/* Select whether to emulation a 9 or 24 pin printer */
///	if (!explicit_pins && uses_24pin_commands) /// If the user did not manually set pins and we saw 24 pin commands, change values to 24 pin mode
///		emulation |= EMULATION_24PIN_UNITS;
///	if (emulation & EMULATION_24PIN_UNITS)
	{
		VERTICAL_UNITS = 360;
		HORIZONTAL_UNITS = 360;
		VFACTOR = 6;
	}
///	else
	{
		/* They are set for 9-pin printers from HiRES family */
		VERTICAL_UNITS = 216; /* (3*72) */ // YDPI
		HORIZONTAL_UNITS = 240; // XDPI
		VFACTOR = 3;
	}
///	else
	{
		/* They are set for 9-pin printers from LoRES family */
		VERTICAL_UNITS = 72;	// YDPI
		HORIZONTAL_UNITS = 60; // XDPI
		//VFACTOR = ;
	}

		line_feed_direction = 1;					/* forward */
		line_spacing_multiplier = 1;				/* normal */

	m_txt_spacing_between_lines = 60.0; /// (#DEFINE text_param) В зависимости от настроек: 'System menu > OTHER OPTIONS: OPT:EPSON OGM>N (n/60) / OPT:EPSON OGM>Y (n/72)'
	m_gra_spacing_between_lines = 180.0; /// (#DEFINE gra_param) В зависимости от настроек: 'System menu > OTHER OPTIONS: OPT:EPSON OGM>N (n/180) / OPT:EPSON OGM>Y (n/216)'

	/* Return the graphics modes to their defaults */
	graphic_mode_K = 0;
	graphic_mode_L = 1;
	graphic_mode_Y = 2;
	graphic_mode_Z = 3;

	/// Horizontal 60, 72, 120 dpi
	// Graphics resolution
	// Horizontal: 480 (60 dpi), 576 (72 dpi), 640 (80 dpi), 720 (90 dpi), 960 (120 dpi), 1920 (240 dpi) dots per 8 inches
	// Vertical: 72 dpi
	
	// Set form height and bottom margin
	m_formHeight = FX80_PAGE_HEIGHT;			// Default to 11 inches
	m_width = FX80_PAGE_WIDTH;					// Default to 8" printable
	m_vertDpi = FX80_VERT_DPI;					// (216) The number of "Dots" on the page
	m_horizDpi = FX80_HORIZ_DPI;				// (240) The number of "Dots" on the page
	m_vertDots = (int) (m_vertDpi * m_formHeight);	// Calculate number of vertical dots
	m_horizDots = (int) (m_horizDpi * m_width);	// Calculate number of horizontal dots
	m_bytesPerLine = (m_horizDots + 7 ) / 8;

	int defaultPageWidth = 2500 / 10; // width /
	int defaultPageHeight = 1800 / 10; // height /

	m_rightMargin = page_width = defaultPageWidth;
	m_bottomMargin = page_height = defaultPageHeight;


	// Reset margins, cpi and clear pin posiitons
	m_leftMargin = 0.0;							// Default to no margin?
	m_rightMargin = 80.0;						// Default to max margin?
	m_topMargin = 120.0;						/* always zero for most printers !!!!!!!!!!!!!!!! */
{
/// if DIL[][]:
//	if (m_defSkipPerf)
		m_bottomMargin = 0.0;					// One inch margin when skip perf on
//	else
		m_bottomMargin = 0.0;
// m_bottomMarginY = (int) ((m_formHeight - m_bottomMargin) * m_vertDpi);
}
		m_bottomMargin = m_formHeight * 720;

	/* Reset the form length */
/// ... %%DocumentMedia ... !
double phys_pu_width = 612.0;									/* 612.0 for letter */
double phys_pu_height = 792.0;									/* 792.0 for letter */
	page_length = (int)(phys_pu_height / INCH * (double)VERTICAL_UNITS + 0.5);
	page_width = (int)(phys_pu_width / INCH * (double)HORIZONTAL_UNITS + 0.5);

	m_rightMargin = page_width;
	perforation_skip = 0; /// initial_perfskip * (HORIZONTAL_UNITS / 10); /* measured in lines */  /// if (initial_perfskip < 0 || initial_perfskip > 65) -> Value is unreasonable

	in_page = false;

	colour_ok = false; /* Is it ok to emmit colour code? */
	uses_colour = false;
}

// Update print density - Обновить значение m_charSpacing_shift_x в соответствии с выбранным шрифтом (compressed take priority over the selected font pitch)
void EscInterpreter::UpdateShiftX ()
{
	m_cpi = 10.0; // обычный (нормальный pica) 10-cpi интервал
	if (m_font_ks)
        m_cpi = 17.16;  // condenced: обычный (нормальный pica), сжатый 17.16 cpi интервал. Он же на некоторых принтерах (STAR, etc.): Compressed print (16.7 cpi) m_cpi = 16.7;
    if (m_font_el)
	{
        m_cpi = 12.0;  // элита (полусжатый) 12 cpi интервал
		if (m_font_ks)
			m_cpi = 20.0;  // Полусжатый, сжатый 20 cpi интервал
	}
	if (m_font_mi)
		m_cpi = 15.0; // even if condenced is active
	m_charSpacing_shift_x = 720 * 1.0 / m_cpi; // типа: HORIZONTAL_UNITS / 10 or m_horizDpi / 10.0

    if (m_font_ex || m_expandedOneLine)  // Шрифт вразрядку (растянутый)
        m_charSpacing_shift_x *= 2; // реальное значение cpi падает вдвое
}

/* Control Codes section. Parse a direct control bytes from printer data stream - Интерпретировать следующий Esc-токен */
bool EscInterpreter::InterpretNext ()
{
int moveTo;

    if (IsEndOfFile()) return false;

    unsigned char ch = GetNextByte();
	switch (ch)
    {
        case 0: /* NUL */
		case 7: /* BEL */ /// sounds the printer’s beeper for 1/10 second - звонок (вырабатывается короткий звуковой сигнал)
		case 17: /* DC1 */ /// printer active - Select printer after it has been deselected with the DC3 command - выбор принтера (вывод из автономного состояние), если он был переведен в автономное состояние командой DC3. Устройство, установленное пользователем в автономное состояние клавишей ON/OFF LINE, не выбирается
			/// The printer ignores this command if the user has set the printer off line by pressing the on-line button
		case 19: /* DC3 */ /// printer inactive - Deselect printer - невыбор принтера (перевод в автономное состояние до приема команды DC1). При этом устройство не может быть выбрано нажатием клавиши ON/ OFF LINE
			/// The printer ignores the ESC @ command (initialize printer) when it is deselected. The printer cannot be reselected by pressing the on-line button
			break;
        case 24: /* CAN */ /// cancel buffer (all printable text characters and bit-image graphics on the current line, but not affect (clear) control codes) - гашение буфера печати
            m_x = m_leftMargin; // Moves the print position to the left-margin position
			buffer_delete(0);
            break;

/* Vertical/Horizontal Motion - Подача бумаги, смещение позиции печати */
        case 8: /* BS */ // backspace - сдвиг на 1 символ назад
            m_x -= m_charSpacing_shift_x;
			if (m_x < m_leftMargin)
				m_x = m_leftMargin;
			// Moves the print position to the left a distance equal to one character in the current character pitch plus any additional intercharacter space
			// The printer ignores this command if it would move the print position to the left of the left margin
			empty_buffer();			/* <-- per epson manual */
            break;
        case 9: /* 0x09 HT */ // horizontal tab - Горизонтальная табуляция
			// Find tab
			moveTo = -1;
			// Loop through tabs and find next tab stop right to current pos
			for (int i = 0; i < numHorizTabs; i++)
				if (m_x < m_tabs[i])
				{
					moveTo = m_tabs[i]; // Find next tab location greater than the current position, if nothing found - ignore
					break;
				}
			if ((moveTo > 0))// && (moveTo < m_rightMargin))
				m_x = moveTo;
			empty_buffer();			/* <-- per epson manual */
			// Underlines are not printed between the current print position and the next tab when this command is sent
            break;
        case 10: /* LF */ // print and line feed - сдвиг к следующей строке
			// Update X and Y locations
			m_x = m_leftMargin; // Moves the horizontal print position to the left-margin position

			// Turn off Expanded One Line
	m_expandedOneLine = false;
	UpdateShiftX();
            m_y += m_lineSpacing_shift_y * line_spacing_multiplier; // Advances the vertical print position one line (in the currently set line spacing)
			if (m_y > m_bottomMargin)
			{
				newPage();
				return false; //Конец страницы
			}

			// if the LF command moves the print position below the bottom margin on continuous paper, the printer advances to the top-of-form position on the next page
///			ProcessLineFeed(); // Check for new page

			// if the LF command moves the print position beyond the end of the printable area on single-sheet paper, the printer ejects the paper

//			line_feed(m_lineSpacing_shift_y * line_feed_direction * line_spacing_multiplier);
			if (auto_cr)
				goto carriage_return;
			empty_buffer();			/* <-- per epson manual */
            break;
        case 11: /* VT */ // Vertical Tab - Вертикальная табуляция - Установить табуляцию по вертикали
			// Vertical tab advances the page by several lines without a carriage return
            m_y += m_lineSpacing_shift_y * line_spacing_multiplier;

	m_expandedOneLine = false;
	UpdateShiftX();

			if (numVertTabs == 0) // All tabs cancelled => Act like CR
				m_x = m_leftMargin;
			else if (numVertTabs == -1) // No tabs set since reset => Act like LF
			{ // bla-bla
				m_x = m_leftMargin;
				m_y += m_lineSpacing_shift_y;
				if (m_y > m_bottomMargin)
				{
					newPage();
					return false; //Конец страницы
				}
			}
			else
			{
				// Find tab below current pos
				moveTo = -1;
				for (int i = 0; i < numVertTabs; i++)
					if (m_vertTabs[m_tabChannel][i] > m_y)
						moveTo = m_vertTabs[m_tabChannel][i];

				// Nothing found => Act like FF
				if (moveTo > m_bottomMargin || moveTo < 0)
				{
					newPage();
					return false; //Конец страницы
				}
				else
					m_y = moveTo;
			}
/// or
			// Loop through tabs and find nex tab stop
			for (int c = 0; c < 16; c++)
			{
				// Find next tab location greater than the current position
				if (m_y < m_vertTabs[m_tabChannel][c])
				{
					m_y = m_vertTabs[m_tabChannel][c];
					break;
				}
			}
/// or

	for (int x = 0; x < 16; x++)
	{
		if (m_vertTabs[m_tabChannel][x] > m_y)
		{
			m_y = m_vertTabs[m_tabChannel][x];
			break;
		}
	}
			empty_buffer();			/* <-- per epson manual */
            break;
        case 12: /* FF */ // form feed - !!! доделать - Causes action defined by ESC O (Set Form Feed Type) and ESC N (Form Feed Interpretation)
            m_x = m_leftMargin; // Moves the horizontal print position to the left-margin position
			m_y = m_topMargin; // Advances the vertical print position on continuous paper to the top-margin position of the next page

			// Turn off Expanded One Line
	m_expandedOneLine = false;
	UpdateShiftX();

			newPage(); // Ejects single-sheet paper
			return false; //Конец страницы
			empty_buffer();			/* <-- per epson manual */
///			if (!in_page)
///				top_of_page();
///			bottom_of_page();
///			in_page = false;
///			page_reset(0);
            break;
        case 13: /* CR */ // print and carriage return - возврат каретки
            m_x = m_leftMargin; // Moves the print position to the left margin position

			/// Turn off Expanded One Line
	m_expandedOneLine = false;
	UpdateShiftX();

			// Test if automatic CR generation is enabled
			if (auto_cr)
			{
				// Check if auto CRLF is on
				m_y += m_lineSpacing_shift_y * line_spacing_multiplier; /* LF */
			}
			// Check for new page
			if (m_y > m_bottomMargin)
			{
				newPage();
				return false; //Конец страницы
			}

			empty_buffer();			/* per epson manual */

			if (auto_lf)				/* Auto line feed? */
				line_feed(m_lineSpacing_shift_y * line_feed_direction * line_spacing_multiplier);
carriage_return:		/* for auto_cr */
				m_x = m_leftMargin;
				/// select_font();			/* font might have changed */
			// Always send a CR command at the end of each line of text or graphics data
            break;

/* Print Size and Character Width */
			/// "Мир, Peace" необходимо послать следующие символы : SO	0155	0111	0122	054	040	SI	0120	0145	0141	0143	0145

		// Если наборы знаков КОИ7 H0 или КОИ7 H1 выбраны путем установки DIP переключателей, то меняется значение управляющих кодов SO (14) и SI (15):
		// SO предназначен для выборки набора знаков КОИ7 H1, код SI - для выборк набора КОИ7 H0
		/* Robotron: 0x0E if DIP[][] switch to RUS layout */
        case 14: /* 0x0e SO */ // one line expanded mode - Select double width for one line - Включение шрифта вразрядку - Выбрать печать шрифтом двойной ширины (одна строка)
            m_expandedOneLine = true; /// Expanded One Line mode
			UpdateShiftX();
            break;
		/* Robotron: 0x0F if DIP[][] switch to LAT layout */
        case 15: /* SI, ^O */ // Select condensed mode - Включение сжатого шрифта (fine print, 17.1 символов_знаков на дюйм) - Выбрать режим плотной печати
            m_font_ks = true;
            UpdateShiftX();
            break;
        case 18: /* DC2, ^R */ // Cancel condensed mode - Выключение сжатого шрифта - Отменить режим плотной печати - condensed mode off
            m_font_ks = false;
            UpdateShiftX();
            break;
        case 20: /* 0x14 DC4 */ // Cancel one line double width mode - Выключение шрифта вразрядку - Отменить печать шрифтом двойной ширины (одна строка)
			m_expandedOneLine = false; /* expanded mode off */
            UpdateShiftX();
            break;
		case 27: /* ESC */ //Expanded Function Codes
            return InterpretEscape(); // Started ESC sequence
		case 28: /* start of NEC FS code */
			break;
		case 127: /* DEL */ /// Delete last text character in buffer (but not control codes). only deletes printable characters; printer control codes are not affected
			// The printer ignores this command if it follows a command that moves the horizontal print position (ESC $, ESC \, or HT) - nothing_to_delete = true;
			buffer_delete(1);
            break;
        default: /* printable character - иначе "напечатать" символ */
            PrintCharacter(ch);
			if (m_font_pr)
				m_x += m_charSpacing_shift_x - 1; // run an advance Proportional_x_shift() function
			else
			{
				if (hmi < 0) // HMI overrides CPI settings
					m_x += m_charSpacing_shift_x; /// 1/(Real64)actcpi; 
				else
					m_x += hmi; /// (in inch) - correct to my case!
			}
			m_x += extraIntraSpace;

			if (!in_page)
			{
///				top_of_page(); // for postscript.c
				in_page = true;
			}
			/// buffer_add(ch); /* emulate printer line buffer */
            break;
    }
	/// Кобинация Esc вводится простым нажатием клавиши Esc, числовой код - с помощью малой цифровой клавиатуры (держать Alt и вводить число обязательно из трех цифр, при необходимости ввести лидирующие нули), комбинация типа ^R (она же $chr(18), она же DC2) вводится, удерживая клавишу Ctrl и нажимая R
    return true;
}

// Escape Seduences. Интерпретировать Escape-последовательность
bool EscInterpreter::InterpretEscape ()
{
    unsigned char ch = GetNextByte();
    switch (ch)
    {
		/* Vertical/Horizontal Motion - Подача бумаги */
		case '/': /// 27 47 c  -    ESC / c	 - Select vertical tab Channel nn for future vertical tab commands (c = 0...7)
			{
				int chn = GetNextByte();
				if (chn < 8) // Validate tab channel is in range
					m_tabChannel = chn; /// TBD: коррекция при беспорядке в заданиях значений по каналам
				else
					fprintf(stderr, "TBD: Illegal value in ESC channel /\n");
			}
			break;
		case 'A': /* text line spacing: ESC A n	- Select n/60-inch (or n/72 epson) line spacing (n = 0...85 (or 128)) */
			{
			int byte = GetNextByte();
				byte &= 0x7F;			// Mask upper bit
				if (byte > 85)			// 85/72" is max (85 / 60/72" is max)
					byte = 85;
				m_lineSpacing_shift_y = (720 * (int)byte / m_txt_spacing_between_lines); /// см. VFACTOR
			}
            break;
        case '2': // Select 6 lines per inch (set text line spacing to 1/6 inch) - Выбрать междустрочный интервал в 1/6" дюйма: (6 LPI)
            m_lineSpacing_shift_y = 720 / 6;
            break;
		case '0': // Select 8 lines per inch (set text line spacing to 1/8 inch) - Установка междустрочного интервала 1/8" (дюйма): (8 LPI)
            m_lineSpacing_shift_y = 720 / 8;
            break;
        case '1': // Select 7/72 inch vertical line spacing - Установка междустрочного интервала 7/72" дюйма: (10.2 LPI)
            m_lineSpacing_shift_y = 720 * 7 / 72;
            break;
			/// А есть ли 3, 4, 9, 12, 72, 144 LPI (1/*)?
        case '3': /* graphics line spacing: ESC 3 n - Select n/180 (or n/216 epson) inch line spacing (n = 0...255) */
            m_lineSpacing_shift_y = (720 * (int)GetNextByte() / m_gra_spacing_between_lines);
			/// epson: m_lineSpacing_shift_y = (double)GetNextByte() / 216.0;
            break;
        case '+': /* Set n/360-inch line spacing - ESC + n - Установить междустрочный интервал в n/360 дюйма (24 pin epson, Panasonic PX-P1124) */
			m_lineSpacing_shift_y = (720 * (int)GetNextByte() / 360);
            break;

		/* Формат страницы */
        case 'C': // Page Length
			{
				int byte = GetNextByte();
				if (byte == 0) /// 27 67 48 n   ESC C NUL n	- Select page length to n in inches (n = 1...22) - Установить длину страницы в дюймах
				{
					page_height = m_bottomMargin = GetNextByte();
					/// This command sets the page length in 1-inch increments only - ???
				}
				else /// 27 67 n      ESC C n - Select page length to n in lines (n = 1...127 | 0 < 'n' x (current line spacing) <= 22 inches) in the current line spacing - Установить длину страницы в строках
					page_height = m_bottomMargin = byte * m_lineSpacing_shift_y;
				m_topMargin = 0.0; // Setting the page length cancels the top and bottom margin settings
			}
		/// Set the page length before paper is loaded or when the print position is at the top-of-form position. Otherwise, the current print position becomes the top-of-form position
		/// Changing the line spacing does not affect the current page-length setting
            break;
        case 'N': // ESC N n - Sets the bottom margin on continuous paper to n lines (n = 1...127 | 0 < 'n' x (current line spacing) <= page length) in the current line spacing from the top-of-form position on the next page - Установить нижнее поле для ленты
			// This was formerly called the “Set skip-over-perforation” command (n = 1...127) - Установить прогон через перфорацию
			{
				int byte = GetNextByte();
				if ((double)byte * m_lineSpacing_shift_y < m_formHeight) // Ignore values greater than the form length
				{
					// Update bottom margin
					m_bottomMargin = (double) byte * m_lineSpacing_shift_y;
					/// m_bottomMarginY = (int) ((m_formHeight - m_bottomMargin) * m_vertDpi);
					m_topMargin = 0.0; // Sending this command cancels the top-margin setting
m_skipPerf = true; // The bottom margin set with the ESC N command is ignored when printing on single sheets (Turn on Skip Perforation)
				}
			}
            break;
        case 'O': // Cancel bottom (and top) margin - Отменить верхнее/нижнее поле для ленты
			// This was formerly called the “Cancel skip-over-perforation” command - Отменить прогон через перфорацию
			m_topMargin = 0.0;
			m_bottomMargin = page_height; /// ! Either no margin or 1-inch margin, depending on the DIP-switch setting
m_skipPerf = false; // Turn off skip perforations
			break;
		case 'l': // 27 108 n     ESC l n	  Set the left margin (n = 1...255 column) to n columns of current character width (in the current character pitch), as measured from the left-most printable column - Установить левое поле
			{
			int c = GetNextByte();
				if (c == 0)
					m_leftMargin = 0;
				else
					m_leftMargin = (c - 1.0) * m_charSpacing_shift_x;
				m_x = 0;
			}
///			else if ((m_leftMargin >= 0) && (m_x < m_leftMargin))
///				m_x = m_leftMargin;
			/// 0 <= (left margin) < (right margin { 80-column printers: 4.50 inches; 110-column printers: 7.00 inches; 136-column printers: 8.00 inches })
			// Set the left margin at the beginning of a line; the printer ignores any data preceding this command on the same line in the buffer
			// Always set the pitch at the beginning of a print job, and before setting the margins. Do not assume what the pitch setting will be (ESC X, ESC c, ESC P, ESC M, ESC g,ESC W, ESC p, ESC SP, SO, ESC ! and SI)
			// Always set the left margin to be at least one column (at 10 cpi) less than the right :)
			// The printer calculates the left margin based on 10 cpi if proportional spacing is selected with the ESC p command
			// Moving the left-margin position moves the tab settings by the same distance + m_leftMargin +
			break;
        case 'Q': // ESC Q n -  Set the right margin (n = column) of character width (in the current character pitch), as measured from the left-most printable column - Установить правое поле
			m_rightMargin = (GetNextByte() - 1.0) * m_charSpacing_shift_x;
			if (m_x > m_rightMargin)
				m_x = m_rightMargin; /// ?
			// if (m_elite) ((nbyte <= 96) && (nbyte >= 3))
			// else if (m_font_ks) ((nbyte <= 137) && (nbyte >= 4))
			// else if ((nbyte <= 80) && (nbyte >= 2))
			reset_tabs();
			// Always set the right margin to be at least two columns (at 10 cpi) greater than the left
            break;

		/* Graphics Character Sets - Bit image graphics mode */
        case '?': /// ESC ? s n	Reassign bit image graphics mode command: 's' = (K, L, Y or Z) to mode 'n' = (0...6)
			switch (GetNextByte()) /// re-define graphics modes
			{
				case 'K': /// Reassign 8-pin bit-image command ESC K to mode n (see ESC *)
					graphic_mode_K = GetNextByte();
					break;
				case 'L': /// Reassign 8-pin bit-image command ESC L to mode n (see ESC *)
					graphic_mode_L = GetNextByte();
					break;
				case 'Y': /// Reassign 8-pin bit-image command ESC Y to mode n (see ESC *)
					graphic_mode_Y = GetNextByte();
					break;
				case 'Z': /// Reassign 8-pin bit-image command ESC Z to mode n (see ESC *)
					graphic_mode_Z = GetNextByte();
					break;
				default:
					fprintf(stderr,"Invalid ESC ? command\n");
					break;
			}
			break;
		case 'K': /// 27 75 n1 n2 data...    ESC K n1 n2 data...	8-bit single density bit-image graphics (often 60 dpi), (same as ESC * NUL) or as redefined by ESC ?
			printGR(graphic_mode_K, GRAPHICS_MODE_8_PIN);
			break;
        case 'L': /// 27 76 n1 n2 data...    ESC L n1 n2 data...	8-bit double density bit-image graphics (often 120 dpi), 6"/sec speed, (same as ESC * SOH) or as redefined by ESC ?
			printGR(graphic_mode_L, GRAPHICS_MODE_8_PIN);
            break;
        case 'Y': /// 27 89 n1 n2 data...    ESC Y n1 n2 data...	8-bit double-density hi-speed graphics (often 120 dpi), 10"/sec speed, (same as ESC * STX) or as redefined by ESC ?
			printGR(graphic_mode_Y, GRAPHICS_MODE_8_PIN);
            break;
        case 'Z': /// 27 90 n1 n2 data...    ESC Z n1 n2 data...	8-bit quadruple density graphics (often 240 dpi), 6"/sec speed, (same as ESC * ETX) or as redefined by ESC ?
			printGR(graphic_mode_Z, GRAPHICS_MODE_8_PIN);
            break;
        case '*': /* Bit Image Graphics Mode - Выбрать печать растрового изображения */
			/// 27 42 m n1 n2 data...  -  ESC * m n1 n2 data... - Generate 8-pin bit-image of quality set by 'm'. Bytes per column = 1, number of dot columns = 'n1' + (256 * 'n2')
			printGR(GetNextByte(), GRAPHICS_MODE_8_PIN);
			break;
		case '^': /// ESC ^ m n1 n2 data… - Generate 9-pin graphics mode of quality set by 'm' (m = 0 or 1 only). Bytes per column = 2 - Выбрать печать 9-точечного растрового изображения. Adjacent dot printing: Yes
				/// 27 94 0 n1 n2 data…  -  ESC ^ NUL n1 n2 data… -  Single density 9-pin bit-image (60 hor., 72 vert.)
				/// 27 94 1 n1 n2 data…  - ESC ^ SOH n1 n2 data… -  Double density 9-pin bit-image (120 hor., 72 vert.)
			/// входные данные готовятся так: Each dot column requires two bytes of data. The first byte represents the top 8 dots in the print head. Bit 0 (the LSB) in the second byte represents the ninth (bottom) dot in the print head; the remaining 7 bits are ignored
			printGR(GetNextByte(), GRAPHICS_MODE_9_PIN);
			break;

        /* User Defined Characters - Символы, определяемые пользователем */
        case '&': /* this command downloads character sets to the printer */
			/// 27 38 0   -   ESC & NUL	-	Select user defined chars - Определить пользовательские символы
			/// 27 38 0 d0 d1 data…  - ESC & d0 d1 data… - Define user-defined characters from ASCII code d0 to d1. d0-d1: range of characters to be redefined
			GetNextByte(); /* eat zero */
			/* missing code */
			break;
        case '%': /* Activate character set: select/deselect download character code. Use RAM set */
			/// 27 37 0    ESC % NUL      User-defined character set off
			/// 27 37 1    ESC % SOH      User-defined character set on | Selects user defined set - Выбрать набор, определенный пользователем
			GetNextByte();
			break;
        case ':': /* this command copies the internal character set into the download area */ /// 27 58 0 0 0 - ESC : NUL NUL NUL -	Copy character ROM to user-defined characters (into RAM) - Скопировать содержимое ПЗУ в ОЗУ
            GetNextByte(); GetNextByte(); GetNextByte(); /// Copy ROM to RAM

			// Set pitch to 12 cpi on some printers
            break;

        /* MSB control. Old support for 7-bit systems */
        case '#': /* cancel control of 8th bit (as sent from host): clear most significant bit (MSB control sequence cancel) */
			/* Use bit 8 of received all input data, overriding ESC = or ESC > commands */
			set8th = 0;
			clear8th = 0xFF;
			break;
        case '=': /* set eight bit of all incoming data to 0: clear most significant bit (MSB = 0) | (epson: all characters non-italic) */
			set8th = 0;
			clear8th = 0x7F;
			break;
        case '>': /* set eight bit of all incoming data to 1: set most significant bit (MSB = 1) | (epson: all characters italic) */
			set8th = 0x80;
			clear8th = 0;
			break;

		
		/* Print Size and Character Width - Размер отпечатка и ширина символов (Группа функций character pitch - плотность печати) */
        case 'P':  // Select pica width print (10 cpi, 10.5-point) - Включение шрифта "пайк|пика" (прямой шрифт - цицеро) (выбрать шаг 10 символов_знаков на дюйм) (if condensed: 17.16 cpi)
            m_font_el = m_font_mi = false;
            UpdateShiftX();
            break;
		case 'M':  // Select elite width print (12 cpi, 10.5-point) - Включение шрифта "элита" (выбрать шаг 12 символо-знаков на дюйм) (if condensed: 20 cpi)
            m_font_el = true;
			m_font_mi = false;
			hmi = -1;
            UpdateShiftX();
            break;
        case 'g': // Select micron width print (15 cpi, 10.5-point) - Выбрать шаг 15 символов на дюйм (if condensed: 15 cpi too)
			m_font_mi = true;
			hmi = -1;
			UpdateShiftX();
            break;
		case 14: /* SO */ // Double wide width for one line (duplicate) - Включение шрифта вразрядку - Выбрать печать шрифтом двойной ширины (одна строка)
            m_expandedOneLine = true;
            UpdateShiftX();
            break;
		case 15: /* SI */ // Select condensed mode - Включение сжатого шрифта (fine print, 17.16, or 20 символов_знаков на дюйм) - Выбрать режим плотной печати
            m_font_ks = true;
            UpdateShiftX();
            break;
		case 'W': // Enlarged (double width) mode - Включение или выключение шрифта вразрядку (растянутый) - Включить/выключить печать шрифтом двойной ширины (expanded)
			switch (GetNextByte()) ///  ESC W 0	-  Cancel, ESC W 1	-  Select - ESC W 1/0
			{
				case 0:
				case '0':
					m_font_ex = false;
					break;
				case 1:
				case '1':
					m_font_ex = true;
					m_expandedOneLine = false; // Приоритет перед SO
					break;
			}
            UpdateShiftX();
            break;
        case 'w': /* turn double-height printing mode on/off - Включить/выключить печать шрифтом двойной высоты - ESC w 1/0 */
			switch (GetNextByte())
			{
				case 0:
				case '0':
					m_font_dh = false;
					line_spacing_multiplier = 1;
					break;
				case 1:
				case '1':
					m_font_dh = true;
					line_spacing_multiplier = 2;
					break;
				default:
					fprintf(stderr,"Invalid ESC w code\n");
					break;
			}
            break;
		case 'p':  // epson: 27 112      ESC p	  Select/cancel proportional mode (proportional spacing) - Включить/выключить пропорциональный режим печати - ESC p 1/0
			switch(GetNextByte())
			{
				case 0: /// 27 112 48  ESC p 0  - Proportional spacing off
				case 48:
					/// style &= (0xffff - STYLE_PROP);
					break;
				case 1:
				case 49: /// 27 112 49  ESC p 1  - Proportional spacing on
					/// style |= STYLE_PROP;
					m_printmode = true; // (QUALITY_LQ)
				break;
			}
			break;
		case 'i':  // epson: Select immediate print mode
			switch(GetNextByte())
			{
				case 0: /// Prints data on a line by line basis
				case 48:
					break;
				case 1:
				case 49: /// Prints data on a character by character basis. If no print data is sent for a short period, moves the vertical print position so that all print is visible
					break;
			}
			/// 27 105 0   ESC i NULL    • Typewriter mode off
			/// 27 105 1   ESC i SOH     • Typewriter mode on
		break;

		/* inter character space */
        case 32: /* SP */ /// set additional intercharacter spacing - epson 27 32 n     ESC SP n	  Space in n/72 inch following each NLQ char - Установить межбуквенную разбивку (разрядку)
            extra_dot_spacing = GetNextByte();
			extraIntraSpace = extra_dot_spacing * 720 / (m_printmode ? 180 : 120); /* учесть моду LQ или draft */
			hmi = -1;
			// Add n dots between characters
            break;
		case 'V':
			///	27 86 0    ESC V NUL     Standard character spacing
			/// 27 86 nn   ESC V nn      Change character spacing by nn
		break;

       /* Смещение позиции печати */
        case '$': /* ESC $ n1 n2 - set absolute dot position to n1+(n2*256) in 60-th of an inch from the left margin - Установить абсолютную горизонтальную позицию печати */
			{
				int moveTo;
				m_x = GetNextByte();
				m_x += 256 * (int)GetNextByte();
				moveTo = (int)m_x * 720 / 60.0; // + (int)(m_leftMargin)
			//	if (moveTo <= m_rightMargin)
					m_x = moveTo; // The printer ignores this command if the specified position is to the right of the right margin
		// Move to position = (n+(n*256))/60 inches from left margin. Max. values: n=(0...127), n=(0...255)
		// The default defined unit setting for this command is 1/60 inch (modified only in escp2)
		// i.e.: xposition = sh * (relative_horizontal_units / absolute_horizontal_units);
			}
            break;
		case '\\': /* ESC \ n1 n2 - set relative dot position to n1+(n2*256) in 120-/180-th of inch from the previous print head position - Установить относительную горизонтальную позицию печати */
			{
				/* build a length number from next two bytes */
				int shift = GetNextByte();
				shift += 256 * (int)GetNextByte();
				// Moves the horizontal print position left or right from the current position
				// For positive (right) movement
				// For negative (left) movement
				double length = (int)(shift * 720 / (m_printmode ? 180 : 120)); /* учесть моду (N)LQ или draft cases */
				/* If proposed position is between left and right margins, use it */
				if ((m_x + length) >= m_leftMargin) // && (m_x + length) <= m_rightMargin)
					m_x += length;
//				else
//					fprintf(stderr, "Warning! Move off left of region ignored\n"); // The printer ignores this command if it would move the print position outside the printable area
			}
	        break;
		case 'J': /* Advance print position vertically (variable line spacing). Immediate perform n/180 (n/216 epson) inch line feed (n = 0...255) - Выполнить междустрочный интервал (подать бумагу вперед) на n/180 (n/216) дюйма */
			/* Note: Acts as CR if n = 0 */
			/// Prints all data in the line buffer
			m_y += (int)GetNextByte() * 720 / m_gra_spacing_between_lines;
			if (m_y > m_bottomMargin)
			{
				/// Continuous paper? - Advances paper moves the vertical print position to the top-margin position on the next page
				/// Single-sheet paper? - Ejects the paper ...
				newPage();
			}
			// Ejects single-sheet paper if the ESC J command moves the print position beyond the end of the printable area (and paper was loaded by cut-sheet feeder)
			// Ejects single-sheet paper and advances the next single sheet the remaining distance if the ESC J command moves the print position beyond the end of the printable area (and paper was loaded manually)
			break;
		case 10: /* strange ESC-LF (line up?) */
			m_x = m_leftMargin;
			UpdateShiftX();
            m_y -= m_lineSpacing_shift_y * line_spacing_multiplier; // Advances the vertical print position one line (in the currently set line spacing)
			break;
		case 12: /* strange ESC-FF (page top?) */
			m_y = 0; // m_topMargin;
			break;
		case ']': /// IBM -	27 93      ESC ]         Reverse line feed
		case 'j': /// epson - 27 106 n     ESC j n	  Perform reverse paper linefeed (n/216 inch after printing all data in buffer) - Подать бумагу назад на n/180 дюйма
			m_y -= (int)GetNextByte() * 720 / 180; // if reverse-feed paper more than 1/2 inch the vertical print position may not be accurate
			break;
		case 'b': // Set Vertical Channel nn Tab Stops
			// epson -	27 98 b c 0  - ESC b c NUL  Clear vertical tabs in channel (c = 0...7)
			// epson -	27 98 c tabs - ESC b c tabs - Select up to 16 vertical tabs in channels (c = 0...7) where tabs are ascending values positions from 1...255 in ascending order (n1 n2) ending with NUL
			SetVertChannelTabs(GetNextByte());
			break;
        case 'B': // Set/Clear vertical tabs Channel 0 (all) - Select up to 16 vertical tabs where tabs are ascending values positions from 1...255 ending with NUL - Установить символы вертикальной табуляции
			/// 27 66 n1 n2 … 0 - ESC B n1 n2 … NUL         - Set vert tabs, n1, n2 …, 1 to 255 in ascending order
			/// 27 66 0    ESC B NUL     Cancel above tab settings
            vertical_tabs_set(GetNextByte());
            break;
        case 'D': // Set/Clear horizontal tab stops in ascending order - Установить символы горизонтальной табуляции
			/// 27 68 0      ESC D NUL	 - Clears all horizontal tables
			/// 27 68 tabs 0 ESC D tabs NUL - Sets up to 32 horizontal tabs with ascending values 1...137. NUL or a value less than previous tab ends command
			{
			int tab, c, lastc;

				lastc = 0;
				for (tab = 0; tab < numHorizTabs; tab++) /* Set up to 32 horizontal tabs */
				{
					c = GetNextByte();
					/* Check if done setting tab stops */
					if ((c == 0) || (c < lastc)) // (c < lastc) вроде аналогично (c < m_tabs[tab - 1])
						break;
					m_tabs[tab] = (int)((double) (c-1) * m_charSpacing_shift_x); /// + m_leftMargin + // Set next tab stop base on current cpi
					lastc = c;
				}
				while (tab < numHorizTabs) /* clear remaining tab stops */
					m_tabs[tab++] = 0;
			}
            break;
		case 'e': /* Set horizontal or vertical tab increments */
			///	epson - 27 101 48 (s)  ESC e 0 s	  Set horizontal tab to increments of 's'
			/// epson - 27 101 49 (s)  ESC e 1 s	  Set vertical tab to increments of 's'
			switch (GetNextByte())
			{
				case 0:
				case '0':
					horizontal_tab_increment(GetNextByte());
					break;
				case 1:
				case '1':
					vertical_tab_increment(GetNextByte());
					break;
				default:
					fprintf(stderr,"Invalid ESC e command\n");
			}
			break;
		case 'f': // Moves the print position
			switch (GetNextByte())
				{
					case 0: // epson - 27 102 48 s  ESC f 0 s	  Set horizontal skip to increments of 's' - Prints n spaces in the current pitch
					case 48:
						m_x += (GetNextByte() - 1) * m_charSpacing_shift_x; // Set to 's' column, or skip a 's' columns?
						// Underline is performed between the current and final print positions when this command is used to move the print position horizontally
						break;
					case 1: /// epson - 27 102 49 s  ESC f 1 s	  Set vertical skip to increments of 's' - Performs n line feeds, in the current line spacing moves the horizontal print position to the left-margin position
					case 49:
						/* Move to left margin */
						m_x = m_leftMargin;
						m_expandedOneLine = false; // Using this command to move the print position vertically (m = 1) cancels double-width printing selected with the SO or ESC SO command
						m_y += GetNextByte() * m_lineSpacing_shift_y;
						if (m_y > m_bottomMargin)
							newPage();
						///line_feed(GetNextByte() * m_lineSpacing_shift_y * line_feed_direction * line_spacing_multiplier);
						break;
				}
			break;
		case 'c': // ESC c - epson - Set horizontal motion index (HMI) - Установить шаг перемещения по горизонтали (in inch; overrides CPI settings)
			hmi = GetNextByte() / 360.0;
			extraIntraSpace = 0.0;
			break;


        /* CHARACTER CONTROL CODES - Print Enhancement - Улучшение шрифта */
		case '-': // underlining - Подчеркивание
			switch (GetNextByte()) // ESC - 0	  Cancel, ESC - 1	  Select underlining
			{
				case 0:
				case '0':
					m_font_un = false;
					break;
				case 1:
				case '1':
					m_font_un = true;
					break;
			}
			break;
		case 'E': // Select emphasized (bold-face) mode - Включить полужирный (утолщенный) шрифт - (overprinted with sideways shift)
            m_font_fe = true;
            UpdateShiftX();
            break;
        case 'F': // Cancel emphasized (bold-face) mode - Выключить полужирный (утолщенный) шрифт
            m_font_fe = false;
            UpdateShiftX();
		/// Another variant: /* selects forward line feed */
		/// line_feed_direction = 1;
            break;
        case 'G': // Select double strike (doublestroke) printing mode on - Включение двойной печати (режим печати: жирный шрифт с двойным ударом, boldfaced?)
            /* Note: overprinted after 1/216 inch line feed */
			m_font_ds = true;
            break;
        case 'H': // Cancel double strike mode - Выключение двойной печати (режим печати: жирный шрифт с двойным ударом, boldfaced?)
            m_font_ds = false;
            m_super_script = m_sub_script = false;
            break;
        case 'S': // Включение печати в верхней или нижней части строки (надстрочная/подстрочная печать)
			switch (GetNextByte())
			{
				case 0:
				case '0':
					m_super_script = true; /// 27 83 48  -  ESC S 0	 - Select superscript
					m_sub_script = false;
					break;
				case 1:
				case '1':
					m_sub_script = true; /// 27 83 49    ESC S 1	  Select subscript
					m_super_script = false;
					break;
			}
            break;
        case 'T': // 27 84	    ESC T	  Cancel superscript/subscript - Выключение печати в верхней или нижней части строки
            m_super_script = m_sub_script = false;
            break;

        /* italic print */
        case '4': /* set/select italic mode/font - Выбрать курсивный шрифт */
			m_font_it = true;
            break;
        case '5': /* clear/cancel italic mode/font - Отменить курсивный шрифт */
			m_font_it = false;
            break;

		case 'r': /* select print color */
			{
				int colourmode = GetNextByte();
				if (colourmode != COLOUR_BLACK) {
					uses_colour = true;
					/// LOG(LOG_MISC,LOG_WARN)("PRINTER: Color printing not supported");
				}
				if (colourmode > 7)
					fprintf(stderr,"Invalid ESC r command\n");
				else
					print_colour = colourmode; /// Print yellow first when overlapping colors
			}
		break;

		/* Printing Style - Общий стиль печати */
        case '!': // Select Master Print mode - Выбор вида шрифта | Установить режим печати where n is a combination of:
			{
                unsigned char fontbits = GetNextByte(); /// current_charmode = GetNextByte();
				m_font_mi = false; // изучить
                m_font_el = (fontbits & 1) != 0; // elite
				m_font_pr = (fontbits & 2) != 0; // proportional
                m_font_ks = ((fontbits & 4) != 0) && !m_font_el; // condensed
                m_font_fe = ((fontbits & 8) != 0) && !m_font_el; // emphasized (bold)
                m_font_ds = (fontbits & 16) != 0; // enhanced (double-strike)
                m_font_ex = (fontbits & 32) != 0; // enlarged (double-width)
				m_font_it = (fontbits & 64) != 0; // italic
				m_font_un = (fontbits & 128) != 0; // underline
				if ((fontbits == 0) || (m_font_ex)) // 0 pica - стандартный шрифт
					m_expandedOneLine = false;
                UpdateShiftX();
				/* Pica & Elite and Condensed/Emphasized are mutually exclusive */
			}
			break;
		case 'X': // ESC X nn - Select font by pitch and point - Выбрать шрифт по шагу и кеглю
			if (GetNextByte() > 0) {} // Set CPI
			if (GetNextByte() > 0) {} // Set points
			break;
        case 'x': // Выбор качества (высококачественная или черновая печать) ESC x 0 - Select draft mode, ESC x 1 - Select NLQ mode
			switch (GetNextByte())
			{
				case 0: /* draft mode */
				case 3: /* Genicom high speed mode */
				case 48:
					m_printmode = false; // normal print quality on
					break;
				case 1: /* NLQ mode */
				case 49:
					m_printmode = true; // letter quality on
					break;
				default:
					fprintf(stderr,"Invalid ESC x command\n");
					break;
			}
            break;
		case 'k': /// Select typeface epson: - Выбрать гарнитуру шрифта
			/// 27 107 48    ESC k 0	  Select NLQ Roman font - черновик (draft (Roman)) - мелкий шрифт пишущей машинки
			/// 27 107 49    ESC k 1	  Select NLQ Sans Serif font - текст    (text  (Sans Serif))
			/// 27 107 50    ESC k 2	  Select NLQ Courier font - курьер   (courier)
/*			enum Typeface
			{
				roman = 0,
				sansserif,
				courier,
				prestige,
				script,
				ocrb,
				ocra,
				orator,
				orators,
				scriptc,
				romant,
				sansserifh,
				svbusaba = 30,
				svjittra = 31
			};
			if (params[0] <= 11 || params[0] == 30 || params[0] == 31) 
				LQtypeFace = (Typeface)params[0]; */
			break;
		///	case 'q': break; /// epson: 				ESC q - Выбрать начертание шрифта
		case '(': /// epson
			///  27 40  -    ESC ( - High speed draft (HSD) mode
			/// ESC ( - - Выбрать линию/черту
			break;

		/* Upper control area */
        case '6': /* Enable printing of upper control codes - select upper character set - Задать старшие управляющие коды */
			/// epson: Enable printing of high ASCII character codes (128...159, 255)
			// Expand redefine area On
			m_highCodesPrinted = true; // Tells the printer to treat codes from 128 to 159 as printable characters instead of control codes
			upper_controls = false; /// !
			break;
        case '7': /* select lower character set - Отменить старшие управляющие коды */
			// Expand redefine area Off
			m_highCodesPrinted = false; // Inverting (i.e. canceling) the ESC 6 command
			break;
		case 'I': /// ESC I - epson Enable/disable printing of control (low) codes - Задать старшие управляющие коды
			///	27 73 48    ESC I 0	  Use codes 0...31 and 128...159, 255 as control codes (Cancel ESC I 1)
			///	27 73 49    ESC I 1	  Use codes 0...31 and 128...159, 255 as characters (Printable codes expansion)
			// Update low code printing flag
			switch (GetNextByte())
			{
				case 0:
				case '0':
					m_lowCodesPrinted = false; // Tells the printer to treat these codes as unprintable characters
					break;
				case 1:
				case '1':
					m_lowCodesPrinted = true; // Tells the printer to treat codes 0–6, 16, 17, 21–23, 25, 26, 28–31 (, and 128–159?) as printable characters
					break;
			}
			break;
		case 'm':	/* Select printing of upper control codes */
			/// has no effect when the italic character table is selected (no characters are defined for these codes in the italic character table)
			/// remains in effect even if you change the character table
			switch (GetNextByte())
				{
				case '0':
				case 0:
					upper_controls = true; // Tells the printer to treat codes from 128 to 159 as printable characters
					break;
				case '4':		/* graphics available */
				case 4:
					upper_controls = false; // Tells the printer to treat codes from 128 to 159 as control codes
					break;
				}
			break;

        /* character table sets (print table control) */
		case 'R': /* set international character set - Выбрать международный литерный набор */
			{
				int ich = GetNextByte();
				if (ich <= 12) // (20 max?)
					m_internationalcharset = ich; // Save the interntional character mode
				/* 27 82 n  -   ESC R n - Select International character set where numeric 'n' is:
								0  USA - US ASCII
								1  France
								2  Germany
								3  United Kingdom
								4  Denmark I
								5  Sweden
								6  Italy
								7  Spain I
								8  Japan
								9  Norway
								10  Denmark II
								11  Spain II
								12  Latin America */
				else
					fprintf(stderr, "Illegal value for ESC R\n");
		/// Another variant: /* selects reverse line feed */
		/// line_feed_direction = -1;
			}
            break;
        case 't': /* select character table - Выбрать таблицы символов */
			/*Epson:	27 116 0    ESC t NUL	 Select italic character set
						27 116 1    ESC t SOH	 Select Epson character set		*/
			m_internationalcharset = GetNextByte(); /* игнорировать */
            break;

        /* SYSTEM CONTROL CODES - Printer Operation - команды подготовки принтера к работе */
        case '@': /// Hard or soft reset to initialise printer with default tabs - инициализация принтера (как при включении питания). Текущая выдача не­медленно отменяется
            PrinterReset(); // ejects the existing page? This command does not affect user-defined characters or control panel (SelecType) settings
            break;
        case 25: /* EM */   /* Control paper loading/ejecting: cut-sheet feeder option (CSF) control - Управление загрузкой/выводом бумаги */
			// We are not really loading paper, so most commands can be ignored
			switch (GetNextByte())
			{
				case 'R': /// 27 25 82    ESC EM R    Cut Sheet Feeder (CSF) ejects one sheet of single-sheet paper
					newPage(); //  ejects the currently loaded single-sheet paper without printing datafrom the line buffer; this is not the equivalent of the FF command (which does printline-buffer data)
					break;
				/* cut-sheet feeder mode should beselected by DIP switch */
				///	27 25 52    ESC EM 4	Turn cut sheet feeder control on (Enters cut-sheet feeder mode)
				/// 27 25 48    ESC EM 0	Turn cut sheet feeder control off (Exits cut-sheet feeder mode)
				/// 27 25 49    ESC EM 1	Cut Sheet Feeder (CSF) bin 1 select
				/// 27 25 50    ESC EM 2	Cut Sheet Feeder (CSF) bin 2 select
				/// 27 25 73    ESC EM I	IBM - Cut Sheet Feeder (CSF) single sheet insert
				///				ESC EM B	Loads paper from the rear tractor
				/// 			ESC EM F	Loads paper from the front tractor
			}
            break;
		case 'U': // Печать в одном или двух направлениях ESC U n — включение (n = 1) или выключение (n = 0) режима однонаправленной печати для более точного позиционирования печатающей головки
            /// 27 85 48     ESC U 0	  Cancel unidirectional mode (print bidirectionally)
			///	27 85 49     ESC U 1	  Select unidirectional mode
			GetNextByte(); // Игнорируем
			/// If unidirectional is selected by DIP switch, you cannot select bidirectional printing with this command
            break;
        case '<': // one line unidirectional mode one line only (while we don't have a print head, so just ignore this)
			/* in epson: — Select unidirectional mode for one line - установка режима однонаправленной печати для текущей строки (отменяется возвратом каретки) с целью более точного позиционирования печатающей головки */
			/* IBM - home head */
            m_x = m_leftMargin;    /* repositions the print head to the left most column (Moves the print head to the extreme left position so the next line will print left to right) */
            break;
		case '8': /// ESC 8 — disable paper out sensor - запрет срабатывания датчика конца бумаги с целью печати до конца листа
			// The printer continues printing when the end of the paper is reached. No beeper sounds when the end of paper is reached, but the printer sets the PE (printer-error) signal to high and the parallel interface error signal to low
			break;
		case '9': /// ESC 9 — enable paper out sensor - разрешение срабатывания датчика конца бумаги, в результате чего примерно за 30 мм до конца бумажного листа выдается звуковой сигнал и печать приостанавливается до заправки следующей страницы
			// The printer stops printing when the end of the paper is reached. The beeper sounds when the end of paper is reached, and the printer sets the PE(printer-error) signal to high and the parallel interface error signal to low
			break;
		case 's': /// ESC s n — print head speed (select low-speed mode) - включение (n = 1) или выключение (n - 0) режима пониженной скорости для уменьшения уровня шума
			/// 27 115 48    ESC s 0	  Turn half speed mode off
			///	27 115 49    ESC s 1	  Turn half speed mode on
			/// Speed of 2 is 10 inches/sec; speed of 0 or 1 is 20
			break;
		case 'a': /*	ESC a - 27 97	n    - ESC a n	  - Selects word processing mode. NLQ justification where numeric 'n' is following values can be used:
							0  Releases word processing mode: NLQ left justification (default)
							1  Selects centering mode: NLQ centre justification
							2  Selects right alignment mode: NLQ right justification
							3  Selects justification mode: NLQ full or fill justification */
			break;
		case '~': // 0/1 - Print normal/slashed zero
			m_zeroSlashed = (GetNextByte() != 0);
			break;
		case '_':
			/// 27 95 48   ESC _ 0       Over-scoring off
			/// 27 95 49   ESC _ 1       Over-scoring on
		break;
    default:
		m_zeroSlashed = (GetNextByte() != 0);
		// Skip unrecognized escape code
		break;
	}
    return true;
}


void EscInterpreter::printGR (int printing_gr_mode, int m_graphicsMode)
{
int x_dx, width, width_max;
bool draft_pinskip = false;
unsigned char mask;
unsigned char pbyte[2];
unsigned char fbyte = 0xff;

	/* Команды K, L, Y, Z работают как terminator code: сначала отпечатывается содержимое буфера, а потом устанавливаются параметры для печати */

	width = GetNextByte(); // Количество заявленных слайсиков (вертикальных "кусочков" данных о изображении) number_of_dot_columns (это не обязательно столько же байт)
	width += 256 * (int)GetNextByte(); // Full 8-inch line, 1920 columns = CHR$(128)CHR$(7)

	switch (printing_gr_mode) // Set resolution based on current mode assignment
	{
		case 0: /* same as ESC K graphic command */
			x_dx = 2 * 6; // 720 / 60 = + 12 по х (x_dx = (m_horizDpi / m_graphicsDpi) + 0.5 /* pin size? */); m_graphicsDpi = 60.0; // Default graphics mode to 60
			width_max = 480; // допустимо 480 (хм..., колонок или байт?) на строку? (816 на широкоформатнике)
			break;
		case 1: /* same as ESC L graphic command */
			x_dx = 2 * 3;		// 720 / 120 = + 6 по х
			width_max = 960;	// допустимо 960 на строку
			break;
		case 2: /* same as ESC Y graphic command */
			draft_pinskip = true; // adjacent pixels not printing - принтер игнорирует каждую вторую из двух последовательных точек по горизонтали
			x_dx = 2 * 3;		// 720 / 120 = + 6 по х
			width_max = 960;	// допустимо 960 на строку
			break;
		case 3: /* same as ESC Z graphic command */
			draft_pinskip = true; // adjacent pixels not printing - принтер игнорирует каждую вторую из двух последовательных точек по горизонтали
			x_dx = 3;			// 720 / 240 = + 3 по х  /* 2*1.5 */
			width_max = 1920;	// допустимо 1920 на строку
			break;
		case 4: /* CRT graphic 1 (Semi-double density) */
			x_dx = 9;			// 720 / 80 = + 9 по х
			width_max = 360;	// добавлено по аналогии, 640 Dots per line
			break;
		case 5: /* Plotter graphic (Aspect ratio match) */
			x_dx = 10;			// 720 / 72 = + 10 по х
			width_max = 400;	// добавлено по аналогии
			break;
		case 6: /* CRT graphic 2 (DEC Screen), 8"/sec speed */
			x_dx = 8;			// 720 / 90 = + 8 по х
			width_max = 320;	// добавлено по аналогии
			break;
		case 7: /* Double density Plotter */
			x_dx = 5;			// 720 / 144 = + 5 по х
			width_max = 800;	// добавлено по аналогии
			break;
		default:
			printGR24(printing_gr_mode, width);
			return;
			break;
	}

	/* Проверка допустимого лимита объема данных точечных позиций (байтов данных) для одной строки заданного режима */
	/// Придумать формульный вариант определения width_max в зависимости от известной горизонтальной разрешающей способности принтера
	if (width > width_max * 2)
		width = width_max; // than returns to text mode
	/// Graphics data that would print beyond the right-margin position is ignored

    // Читать и выводить данные
	pbyte[1] = fbyte;
    for (int cwidth = 1; cwidth <= width; cwidth++)
    {
		mask = 0x80;
		pbyte[0] = fbyte;
		fbyte = GetNextByte();

		for (int i = 0; i < (m_graphicsMode?9:8); i++)
        {
			if (i == 8) // Must be 2-nd byte of 9-pin mode for the "bottom pin" printing
			{
				mask = 0x80; // Need to know: plot upper bit (MSB) 0x80 or LSB (0x01) only?
				pbyte[1] = 0xFF; //fbyte; // пока оставил некорректным - т.к. fbyte это предыдущий байт не по горизонтали, а по вертикали :(
				fbyte = GetNextByte();
			}

            if (fbyte & mask)
            {
				if ((draft_pinskip) && (cwidth % 2 == 0)) // В скоростных драфт-режимах чётные точки,
				{
					if ((fbyte & mask) != (pbyte[m_graphicsMode?1:0] & mask)) // если следуют сразу друг за другом по горизонтали, не печатаются
						DrawStrike(float(m_x), float(m_y + i * 12));
				}
				else 
					DrawStrike(float(m_x), float(m_y + i * 12));
                /* 12 соответствует 1/60 inch - расстояние по вертикали между иглами у 9-pin матричных принтеров.
					На самом деле расстояние у них = 1/72 inch (т.е. должно быть 10), но при эмуляции (по какой-то причине) принимается 1/60
				   Принять во внимание, что теперь есть возможность печати и 9-й иголкой! отрегулировать междупиновое расстояние - всё должно укладываться в корректное головко-место, как и на реальном принтере

					Кстати, сделать алерт-проверку на "размер иголки" (сейчас r="0.6"):
					// When page dpi is greater than graphics dpi, the drawn pixels get "bigger"
					Bitu pixsizeX = dpi/bitGraph.horizDens > 0?dpi/bitGraph.horizDens:1;
					Bitu pixsizeY = dpi/bitGraph.vertDens > 0?dpi/bitGraph.vertDens:1;
					*/
            }
            mask >>= 1;
        }
        m_x += x_dx;
    }
}

void EscInterpreter::printGR24 (int printing_gr_mode, int width)
{
int x_dx, width_max;

	/// bitGraph.vertDens = 180;
	/// bitGraph.bytesColumn = 3;
	switch (printing_gr_mode)
	{
		case 32:  /* High-resolution for ESC K */
			x_dx = 2 * 6;		// bitGraph.horizDens = 60;
			width_max = 60;
			break;
		case 33:  /* High-resolution for ESC L */
			x_dx = 2 * 3;		// bitGraph.horizDens = 120;
			width_max = 120;
			break;
		case 38:  /* CRT graphic 3 */
			x_dx = 2 * 4;		// bitGraph.horizDens = 90;
			width_max = 90;
			break;
		case 39:  /* high-resolution triple-density */
			x_dx = 2 * 2;		// bitGraph.horizDens = 180;
			width_max = 180;
			break;
		case 40:  /* high-resolution hex-density */
			x_dx = 2;			// bitGraph.horizDens = 360;
			width_max = 360;
			/// draft_pinskip = true; adjacent pixels not printing - принтер игнорирует каждую вторую из двух последовательных точек по горизонтали
			break;
	default:
		/// bitGraph.vertDens = 360;
		/// bitGraph.bytesColumn = 6;
	x_dx = 2; /// для 48-игольчатых принтеров расстояние между иглами ~ 300 точек/дюйм :)
/* 2 соответствует 1/360 inch - расстояние между иглами у __-pin матричных принтеров
	case 71:
		bitGraph.horizDens = 180;
		break;
	case 72:
		bitGraph.horizDens = 360;
		/// adjacent pixels not printing - принтер игнорирует каждую вторую из двух последовательных точек по горизонтали
		break;
	case 73:
		bitGraph.horizDens = 360;
	default:
		LOG(LOG_MISC,LOG_ERROR)("PRINTER: Unsupported bit image density %i", dens);
*/
			return;
			break;
	}

    // Читать и выводить данные
    for (; width > 0; width--)
    {
        for (unsigned char n = 0; n < 3; n++)
        {
            unsigned char fbyte = GetNextByte();
            unsigned char mask = 0x80;
            for (int i = 0; i < 8; i++)
            {
				if (fbyte & mask)
                {
					DrawStrike(float(m_x), float((m_y + (n * 4 * 8 /* игл */) + i * 4)));
					/* 4 соответствует 1/180 inch - расстояние между иглами у 24-pin матричных принтеров */
                }
                mask >>= 1;
            }
        }
        m_x += x_dx;
    }
}


void EscInterpreter::PrintCharacter (unsigned char ch)
{

	if (ch < 32 || ch > 254) return; // т.к. постоянно печатается 0xFF 255 в конце файла...
	if (ch < 160 && ch > 126) return;

	// Вычисляем символ знакогенератора по текущему набору символов
	int charset = m_internationalcharset ^ (ch > 128 ? 1 : 0);
	// if (m_font_it) ch |= 0x80;
//ch &= 0x7f;
	
	// Check for zeroSlash
	if (m_zeroSlashed && !m_zeroRedefined && (ch == '0'))
		ch = 127;

	int symbol = ch;
 //   if (ch >= (unsigned char)'@' && charset != 0)
 //       symbol += 68;

    // Получаем адрес символа в знакогенераторе
    const unsigned short* pchardata = RobotronFont + symbol * 9;

    float step = float(m_charSpacing_shift_x) / 12.0f;  // Шаг по горизонтали - тут делится на 11: это расстояние для знакоместа 9x11? где учет радиуса иглы (m_cr)?
	/* the ~ same is the 'scalar' var: calculate scalar for dot positioning base ond on CPI / DPI */
	float scalar = (double)m_horizDpi / ((double)m_cpi * 12.0); // 11 "dots" per character + extra half dot

    float y = float(m_y);
    if (m_sub_script) y += 4 * 12;

    // Цикл печати символа по строкам
    unsigned short data = 0, prevdata = 0;
    for (int line = 0; line < 9; line++) // Walk through each bit for the current byte
    {
        data = pchardata[line];

        // Особая обработка для над- и под-строчных символов
        if ((m_super_script || m_sub_script))
        {
            if ((line & 1) == 0)
            {
                prevdata = data;
                continue;
            }
            else
                data |= prevdata;  // Объединяем две строки символа в одну
        }
int italicshift = 0;
        for (int col = 0; col < 9; col++)  // Цикл печати точек строки
        {
            unsigned short bit = (data >> col) & 1;
            if (m_font_un && line == 8) bit = 1;
            if (!bit) continue; // If this pin does not "strike", move on to next pin
			
			if (m_font_it)
				italicshift = 8 - line;
            
			DrawStrike(m_x + (col + italicshift) * step , y);

			if (m_font_fe) // Bold => Print a second time (one pixel, etc) to the right
				DrawStrike(m_x + (col + italicshift + 1.0f) * step, y); // 2.0f looks very strange

			if (m_font_ex || m_expandedOneLine) // Expanded => Print a second time (one dot) to the right
				DrawStrike(m_x + (col + italicshift + 1.0f) * step, y);

			if (m_font_ds) // Doublestrike => Print a second time (one pixel, etc) below (логически бъется по тем же координатам?, но принтер механический и лист бумаги может немного уплывать (как по X, так и по Y?))
			{
				DrawStrike(m_x + (col + italicshift) * step, y + 5.0f); /// Double strike printing prints each line twice, with the second line slightly below the first to create a bold appearance
				if (m_font_ex || m_expandedOneLine)
					DrawStrike(m_x + (col + 1.0f + italicshift) * step, y + 1.0f);
			}

			if (m_font_dh) // Doubleheight => Print a second time below * 2
				DrawStrike(m_x + (col + italicshift) * step, y + 1.0f); // в данный момент, случае с m_font_fe = true, точка распечатается тремя! ударами как "Г"

			
			/// TBD: узнать реальные значение смещения головы (по X) и рулона (по Y) при ужирнениях (вместо минималки 1.0f)
			/// if (m_font_ex || m_expandedOneLine) { }; - в просто широких режимах точки тоже обпечатываются несколькими ударами?
			/// Вообще, такие режимы (в т.ч. NLQ) реализуются за два прохода:
			/// после первого прохода головки бумага протягивается на расстояние, соответствующее половинному размеру точки;
			/// затем совершается второй проход (с частичным перекрытием точек)
		}
		y += 12 * line_spacing_multiplier;  // 12 соответствует 1/60 inch (= 720 / 60)
	}

	// Для m_font_un добивать последнюю точку
	if (m_font_un)
	{
		DrawStrike(m_x + 9.0f * step, float(m_y * line_spacing_multiplier + 8 * 12));
		if (m_font_fe) // Bold => Print a second time (one pixel, etc) to the right
			DrawStrike(m_x + (9.0f + 1.0f) * step, y);
	}
}

void EscInterpreter::DrawStrike (float x, float y)
{
    float cx = float(m_leftMargin) + x;
    float cy = float(m_topMargin) + y;

    m_output.WriteStrike(cx, cy, m_cr);
}


/* Reset all tab stops to factory default */
void EscInterpreter::reset_tabs (void)
{
int tab;

	numHorizTabs = 32;
	/* Clear the horizontal tabs to 8 character intervals (each eight characters) */
	tab = 0;
	for (int x = 8; x < 80; x += 8) // 80 - max char count for current pitch mode and printer possibility
		m_tabs[tab++] = (int) ((double) (x-1) * m_charSpacing_shift_x);

	// Fill remaining tabs with zero
	while (tab < numHorizTabs)
		m_tabs[tab++] = 0;
	/// Определиться что мы храним - готовые координаты по Х или чисто номер позиции символа для следующей табуляции, а потом его каждый раз пересчитываем? пересчет при изменении m_cpi, да?


	/* Clear the vertical tabs */
	for (int y = 0; y < 8; y++)
		for (int x = 0; x < 16; x++)
			m_vertTabs[y][x] = 0;
	numVertTabs = -1;

int		c, channel;
int		stop;

	// Restore to vertical Tab Channel zero
	m_tabChannel = 0;
	m_escTabChannel = 0;

	// Loop for all 16 vertical tab stops
	for (c = 0; c < 16; c++)
	{
		stop = (int) (m_vertDpi * (double) ((c * 2) + 1) / 12.0);
		for (channel = 0; channel < 8; channel++)
			m_vertTabs[channel][c] = stop;
	}

}

//////////////////////////////////////////////////////////////////////
/* Set vertical channel tab stops */
void EscInterpreter::SetVertChannelTabs (unsigned char byte)
{
// not init:
int m_escParamsRcvd = 0;
int m_escCmd;

	// Check if done setting tab stops
	if (m_escParamsRcvd == 0)
	{
		// Save the tab channel number
		m_escTabChannel = byte;
		m_escParamsRcvd++;
		return;
	}
	else if ((byte == 0) || ((m_escParamsRcvd > 1) && (byte < m_vertTabs[m_escTabChannel][m_escParamsRcvd-2])))
	{
		m_escCmd = m_escParamsRcvd = 0;
		return;
	}

	// Set next tab stop base on current cpi
	if (m_escParamsRcvd < 17)
	{
		if (m_escTabChannel < 8)
		{
			m_vertTabs[m_escTabChannel][m_escParamsRcvd++] = (int) ((double) byte * m_lineSpacing_shift_y);
		}
		else
			m_escParamsRcvd++;
	}

	return;
}
//////////////////////////////////////////////////////////////////////

void EscInterpreter::vertical_tabs_set (int channel)
{
int x;
int c, lastc;

	lastc = 0;
	for (x = 0; x < 16; x++)
	{
		c = GetNextByte();

		if (c < lastc || c == 0)
			break;

		m_vertTabs[channel][x] = c * m_lineSpacing_shift_y;

		lastc = c;
	}

	for (; x < 16; x++)					/* clear remaining tab stops */
		m_vertTabs[channel][x] = 0;

// not init:
int m_escParamsRcvd = 0;
int m_escCmd;
				/* ??? correct? set vertical tabs, channel 0 */ /// while (GetNextByte() != 0);

			// Check if done setting tab stops
			if ((c == 0) || ((m_escParamsRcvd > 0) && (c < m_vertTabs[0][m_escParamsRcvd-1])))
			{
				m_escCmd = m_escParamsRcvd = 0;
				return;
			}

			// Set next tab stop base on current cpi
			if (m_escParamsRcvd < 16)
				m_vertTabs[0][m_escParamsRcvd++] = (int) ((double) c * m_lineSpacing_shift_y);
			if (m_escParamsRcvd == 16)
				m_escCmd = m_escParamsRcvd = 0;
}

void EscInterpreter::horizontal_tab_increment (int inc)
{
int x;

	for (x = 0; x < 32; x++)
		m_tabs[x] = (int) ((double) x * (double) inc * m_charSpacing_shift_x - m_charSpacing_shift_x);
}

void EscInterpreter::vertical_tab_increment (int inc)
{
	/* don't totally understand this command usage, implementation like horizontal? */
}


void EscInterpreter::newPage (void)
{
	m_y = m_topMargin;
}


/* Do a line feed of the specified number of VERTICAL_UNITS */
void EscInterpreter::line_feed (int how_far)
{
//	empty_buffer();		/* per Epson manual */

	ypos -= how_far;	/* Move down specified amount */

///	m_y -= how_far;
	/* Move to next page if necessary */
	if (ypos < (m_topMargin ? perforation_skip : (perforation_skip / 2)))
	{
		///bottom_of_page();		/* close out this page */
		in_page = false;			/* no page currently in progress */

		///if (ypos < 0)			/* !!! This is close to correct */
		///	page_reset(-ypos);
		///else
		///	page_reset(0);
	}
}

/* Reset at top of page. The argument "consumed" is the amount of space already eaten up at the start of the page by an uncompleted vertical movement command */
void EscInterpreter::page_reset (int consumed)
{
	/* move to top of page */
	if (m_topMargin == 0)									/* If we don't use top margin, */
		m_y = page_length - (perforation_skip / 2);		/* top margin is half of perforation skip. */
	else												/* If explicit top magin, */
		m_y = page_length - m_topMargin;				/* use it. */

	m_y -= consumed;

	/* Move to left margin */
	m_x = m_leftMargin;
}

//////////////////////////////////////////////////////////////////////

/* Delete characters from the buffer, 0 means all, 1 means one */
void EscInterpreter::buffer_delete (int howmuch)
{
	if (howmuch == 0)
		buffer_count = 0;
	if (howmuch == 1 && buffer_count)
		buffer_count--;
}

/*
** Add a character to the line output buffer
*/
void EscInterpreter::buffer_add (int c)
{
	if (buffer_count == 0)
		predicted_xpos = 0;

	if (buffer_count < BUFFER_SIZE)
	{
		struct CHARs *p = &buffer[buffer_count++];

		p->style = out_style;
		p->hscale = out_hscale;
		p->vscale = out_vscale;
		p->movement = m_x - predicted_xpos;
		p->xpos = m_x;
		p->extra_space = (extra_dot_spacing * (HORIZONTAL_UNITS / 120));
		p->c = c;
		p->colour = print_colour;
		p->fixed_spacing = m_charSpacing_shift_x;

///		/* If proportional spacing is currently in effect, the width comes from the proportional spacing table */
///		if(out_style & OSTYLE_PROPORTIONAL)
///			p->width = (int)(((double)width(c,out_style & OSTYLE_OBLIQUE)*out_hscale)+0.5);
///		else
			p->width = m_charSpacing_shift_x;

		/* Advance to next position */
		m_x += p->width;
		m_x += p->extra_space;

		/* Save this xpos for next time */
		predicted_xpos = m_x;
	}
}

/* Empty the print buffer */
void EscInterpreter::empty_buffer (void)
{
// By "empty" we mean to emmit PostScript code which represents the contents and then to clear it, not to simply throw the contents away.
// This is called at the end of the line and at certain other times

}


/* 'reverse_bit_order'
 *
 * reverse the bit order in an array of bytes - does not reverse byte order!
 */
void reverse_bit_order (unsigned char *buff, int n)
{
  int i;
  unsigned char a;
  if (!n) return; /* nothing to do */

  for (i = 0; i < n; i++) {
    a = buff[i];
    buff[i] =
      (a & 0x01) << 7 |
      (a & 0x02) << 5 |
      (a & 0x04) << 3 |
      (a & 0x08) << 1 |
      (a & 0x10) >> 1 |
      (a & 0x20) >> 3 |
      (a & 0x40) >> 5 |
      (a & 0x80) >> 7;
  }
}
/*
int
dgloc2int( char arg1, char arg2, char arg3 )
{
    int n = 0 ;

    if ( dgunix ) {
        n += unixhex2int(arg1) * 256;
        n += unixhex2int(arg2) * 16;
        n += unixhex2int(arg3);
    }
    else {
        n += (arg1 - '@')%32 * 1024;
        n += (arg2 - '@')%32 * 32;
        n += (arg3 - '@')%32 ;
    }
    return n;
}

void
dgint2loc( int n, char * arg1, char * arg2, char * arg3 )
{
    if ( dgunix ) {
        n %= 3072;                              // max DG-Location value
        *arg1 = int2unixhex(n/256);
        n %= 256 ;
        *arg2 = int2unixhex(n/16);
        n %= 16 ;
        *arg3 = int2unixhex(n);
    }
    else {
        n %= 3072;                              // max DG-Location value
        *arg1 = '@' + n / 1024;
        n %= 1024 ;
        *arg2 = '@' + n / 32;
        n %= 32 ;
        *arg3 = '@' + n;
    }
}
*/