unit ZXAsmFastMem;

interface

uses Windows, KOL;

function AllocMemFast( size: Integer ): Pointer;
function GetMemFast( size: Integer ): Pointer;
procedure FreeAllAllocated;
function ReallocateMemFast( OldPtr: Pointer; NewSize: Integer ): Pointer;

type
  PName = ^ TName;
  TName = packed record
    NameStart: PChar;
    NameLen: Integer;
    Data: DWORD;
    Chk: WORD;
  end;

  PBlock = ^TBlock;
  TBlock = packed record
    Data: array[ 0..255 ] of TName;
  end;

type
  PNames = ^TNames;
  TNames = object( TObj )
  private
    function GetData(idx: Integer): DWORD;
    procedure SetData(idx: Integer; const Value: DWORD);
    function GetItems(idx: Integer): String;
    function GetNames(idx: Integer): PName;
  protected
    procedure Init; virtual;
  public
    Blocks: PList;
    FCount: Integer;
    Hash: array[ 0..2047 ] of DWORD;
    DESTRUCTOR Destroy; virtual;
    property Count: Integer read FCount;
    property Items[ idx: Integer ]: String read GetItems;
    function NameEq( idx: Integer; StrStart: PChar; StrLen: Integer ): Boolean;
    function NameMayBePresent( StrStart: PChar; StrLen: Integer; var k: Word ): Boolean;
    function IndexOf( StrStart: PChar; StrLen: Integer ): Integer;
    property Objects[ idx: Integer ]: DWORD read GetData write SetData;
    function AddObject( StrStart: PChar; StrLen: Integer; Data: DWORD ): Integer;
    procedure DelFrom( idx: Integer );
    procedure Delete( idx: Integer );
    procedure Clear;
    procedure DeleteLast;
  public
    property Names[ idx: Integer ]: PName read GetNames;
  end;

function NewNames: PNames;

implementation

var PoolList: PList;
    CurFreePos: Pointer;
    RestInBlock: Integer;

function AllocMemFast( size: Integer ): Pointer;
begin
  Result := GetMemFast( size );
  FillChar( Result^, size, 0 );
end;

function GetMemFast( size: Integer ): Pointer;
var Pool: Pointer;
    getsz: Integer;
begin
  if PoolList = nil then
    PoolList := NewList;
  if RestInBlock < size then
  begin
    getsz := max( size, 1 shl 20 );
    GetMem( Pool, getsz );
    PoolList.Add( Pool );
    CurFreePos := Pool;
    RestInBlock := getsz;
  end;
  Result := CurFreePos;
  CurFreePos := Pointer( Integer( CurFreePos ) + size );
  dec( RestInBlock, size );
end;

function ReallocateMemFast( OldPtr: Pointer; NewSize: Integer ): Pointer;
var OldSize: Integer;
begin
  OldSize := DWORD( CurFreePos ) - DWORD( OldPtr );
  Assert( OldSize > 0 );
  inc( RestInBlock, OldSize );
  CurFreePos := OldPtr;
  Result := GetMemFast( NewSize );
  if Result <> OldPtr then
    Move( OldPtr^, Result^, Min( OldSize, NewSize ) );
end;

procedure FreeAllAllocated;
begin
  if PoolList <> nil then
    PoolList.Release;
  PoolList := nil;
  CurFreePos := nil;
  RestInBlock := 0;
end;

function NewNames: PNames;
begin
  new( Result, Create );
end;

{ TNames }

function TNames.AddObject(StrStart: PChar; StrLen: Integer;
  Data: DWORD): Integer;
var blk: PBlock;
    i, L: Integer;
    k: Word;
    s: PChar;
    {$IFDEF USE_DEFNAMES}
    def: PDef;
    {$ENDIF}
begin
  if FCount = 0 then
    Clear;
  if FCount and 255 <> 0 then
  begin
    blk := Blocks.Items[ FCount shr 8 ];
    i := FCount and 255;
  end
    else
  begin
    GetMem( blk, Sizeof( blk^ ) );
    Blocks.Add( blk );
    i := 0;
  end;
  Result := FCount;
  inc( FCount );
  blk.Data[ i ].NameStart := StrStart;
  blk.Data[ i ].NameLen := StrLen;
  blk.Data[ i ].Data := Data;
  k := 0;
  s := StrStart;
  L := StrLen;
  while L > 1 do
  begin
    k := ((k shl 1) or (k shr 15)) xor PWORD( s )^;
    inc( s, 2 );
    dec( L, 2 );
  end;
  if L > 0 then
    k := ((k shl 1) or (k shr 15)) xor PByte( s )^;
  blk.Data[ i ].Chk := k;
  Hash[ k shr 5 ] := Hash[ k shr 5 ] or (1 shl (k and 31));
  {$IFDEF USE_DEFNAMES}
  def := Pointer( Data );
  if def <> nil then
  begin
    def.FTokenStart := StrStart;
    def.FTokenLen := StrLen;
  end;
  {$ENDIF}
end;

procedure TNames.Clear;
begin
  DelFrom( 0 );
  FillChar( Hash, Sizeof( Hash ), 0 );
end;

procedure TNames.Delete(idx: Integer);
begin
  if idx = Count-1 then DeleteLast
  else Names[ idx ].NameStart := nil;
end;

procedure TNames.DeleteLast;
begin
  DelFrom( Count-1 );
end;

procedure TNames.DelFrom(idx: Integer);
begin
  if idx >= FCount then Exit;
  while Blocks.Count * 256 - idx >= 256 do
  begin
    FreeMem( Blocks.Items[ Blocks.Count-1 ] );
    Blocks.Delete( Blocks.Count-1 );
    FCount := min( FCount, Blocks.Count * 256 );
  end;
  FCount := (FCount and not 255) or (idx) and 255;
end;

destructor TNames.Destroy;
begin
  DelFrom( 0 );
  Blocks.Free;
  inherited;
end;

function TNames.GetData(idx: Integer): DWORD;
var blk: PBlock;
begin
  blk := Blocks.Items[ idx shr 8 ];
  Result := blk.Data[ idx and 255 ].Data;
end;

function TNames.GetItems(idx: Integer): String;
var blk: PBlock;
begin
  blk := Blocks.Items[ idx shr 8 ];
  SetString( Result, blk.Data[ idx and 255 ].NameStart,
                     blk.Data[ idx and 255 ].NameLen );
end;

function TNames.GetNames(idx: Integer): PName;
var blk: PBlock;
begin
  blk := Blocks.Items[ idx shr 8 ];
  Result := @ blk.Data[ idx and 255 ];
end;

function TNames.IndexOf(StrStart: PChar; StrLen: Integer): Integer;
var i, j: Integer;
    blk: PBlock;
    cnt: Integer;
    k: Word;
    s: PChar;
begin
  Result := -1;

  k := 0;
  i := StrLen;
  s := StrStart;
  while i > 1 do
  begin
    k := ((k shl 1) or (k shr 15)) xor PWORD( s )^;
    inc( s, 2 );
    dec( i, 2 );
  end;
  if i > 0 then
    k := ((k shl 1) or (k shr 15)) xor PByte( s )^;
  if Hash[ k shr 5 ] and (1 shl (k and 31)) = 0 then Exit;

  cnt := -1;
  for i := 0 to Blocks.Count-1 do
  begin
    blk := Blocks.Items[ i ];
    for j := 0 to 255 do
    begin
      inc( cnt );
      if cnt >= FCount then Exit;
      if (blk.Data[ j ].NameLen = StrLen) and
         (blk.Data[ j ].NameStart^ = StrStart^) and
         (StrLComp( blk.Data[ j ].NameStart, StrStart, StrLen ) = 0) then
      begin
        Result := cnt; Exit;
      end;
    end;
  end;
end;

procedure TNames.Init;
begin
  Blocks := NewList;
end;

function TNames.NameEq(idx: Integer; StrStart: PChar;
  StrLen: Integer): Boolean;
var blk: PBlock;
    i, L: Integer;
begin
  blk := Blocks.Items[ idx shr 8 ];
  i := idx and 255;
  L := blk.Data[ i ].NameLen;
  Result := (L = StrLen) and
            (StrStart^ = blk.Data[ i ].NameStart^) and
            (StrLComp( StrStart, blk.Data[ i ].NameStart, L ) = 0);
end;

function TNames.NameMayBePresent(StrStart: PChar;
  StrLen: Integer; var k: Word): Boolean;
var s: PChar;
    i: Integer;
begin
  Result := FALSE;
  k := 0;
  i := StrLen;
  s := StrStart;
  while i > 1 do
  begin
    k := ((k shl 1) or (k shr 15)) xor PWORD( s )^;
    inc( s, 2 );
    dec( i, 2 );
  end;
  if i > 0 then
    k := ((k shl 1) or (k shr 15)) xor PByte( s )^;
  if Hash[ k shr 5 ] and (1 shl (k and 31)) = 0 then Exit;
  Result := TRUE;
end;

procedure TNames.SetData(idx: Integer; const Value: DWORD);
var blk: PBlock;
    {$IFDEF USE_DEFNAMES}
    def: PDef;
    {$ENDIF}
begin
  blk := Blocks.Items[ idx shr 8 ];
  blk.Data[ idx and 255 ].Data := Value;
  {$IFDEF USE_DEFNAMES}
  def := Pointer( Value );
  if (def <> nil) and (def.FTokenStart = nil) then
  begin
    def.FTokenStart := blk.Data[ idx and 255 ].NameStart;
    def.FTokenLen := blk.Data[ idx and 255 ].NameLen;
  end;
  {$ENDIF}
end;

end.
