library LoadZ80;

uses
  Windows, KOL, PluginUnit;

{$I KOLDEF.INC}

type
  TZ80 = packed record
    A: Byte;
    F: Byte;
    BC: Word;
    HL: Word;
    PC: Word;
    SP: Word;
    I: Byte;
    R0_6: Byte;
    R7_Border_SamRom_Compress: Byte; // FF = 01
    DE: Word;
    BCalt: Word;
    DEalt: Word;
    HLalt: Word;
    Aalt: Byte;
    Falt: Byte;
    IY: Word;
    IX: Word;
    IFF1: Byte;
    IFF2: Byte;
    IM_Speed_Joy: Byte;
  end;

  THardwareMode2_01 = ( hm_48K, hm_48K_If1, hm_SamRom, hm_128K, hm_128K_If1 );
  THardwareMode3_00 = ( hm3_48K, hm3_48K_If1, hm3_48K_MGT, hm3_SamRom, hm3_128K, hm3_128K_If1, hm3_128K_MGT );

  TZ80New = packed record
    ExtraHeaderLen: Word;
    PC: Word;
    HardwareMode: Byte;
    Last7FFD: Byte;
    Interface1: Byte; // FF  ROM Interface1
    Emulation_R_LDIR: Byte;
    LastFFFD: Byte;
    SoundChipRegs: array[ 0..15 ] of Byte;
    //  -  23 ,     54  55
    TStateCounterLo: Word;
    TStateCounterHi: Byte;
    SpectatorFlag: Byte;
    MGTRom: Byte;
    MultifaceROM: Byte;
    RAM0000_1FFF: Byte;
    RAM2000_3FFF: Byte;
    KeyboardMapJoy: array[ 0..4 ] of Word;
    KeyAsciiMapJoy: array[ 0..4 ] of Word;
    MGTType: Byte;
    DiscipleInhibitButton: Byte;
    DiscipleInhibitFlag: Byte;
    Unknown55: Byte;
  end;

  TMemPageHdr = packed record
    DataLen: Word;
    PageNum: Byte;
  end;

procedure RegisterLoadSpectrum( LoadFileTypes, SaveFileTypes: PChar;
          Descriptions: PChar; CanSaveS128: PByte ); stdcall;
begin
  StrCopy( LoadFileTypes, '*.Z80;*.SLT' );
  StrCopy( SaveFileTypes, '*.Z80' );
  StrCopy( Descriptions, 'Z80 raw snap format;Z80 raw snap format;SLT snap format (with additional levels);Z80 raw snap format' );
end;

function DecodeMemBlock( Dst, Src: PStream; Len, MaxDataLen: Integer ): Boolean;
var WaitMarker: Boolean;
    B1, B2: Byte;
    Was00: Boolean;
begin
  Result := FALSE;
  Dst.Size := 0;
  Dst.Position := 0;
  WaitMarker := Len = 0;
  if WaitMarker then Len := MaxInt;
  Was00 := FALSE;
  while (Len > 0) and ((MaxDataLen <= 0) or (Dst.Size < DWORD( MaxDataLen ))) do
  begin
    if Src.Read( B1, 1 ) <> 1 then Exit;
    Dec( Len );
    if B1 <> $ED then
    begin
      Dst.Write( B1, 1 );
      Was00 := B1 = 0;
    end
      else
    begin
      if Src.Read( B2, 1 ) <> 1 then Exit;
      Dec( Len );
      if B2 <> $ED then
      begin
        Dst.Write( B1, 1 );
        Dst.Write( B2, 1 );
      end
        else
      begin
        if Src.Read( B1, 1 ) <> 1 then Exit;
        Dec( Len );
        if WaitMarker and Was00 and (B1 = 0) then
        begin
          Result := TRUE;
          Exit;
        end;
        if Src.Read( B2, 1 ) <> 1 then Exit;
        Dec( Len );
        while B1 > 0 do
        begin
          Dst.Write( B2, 1 );
          Dec( B1 );
        end;
      end;
      Was00 := FALSE;
    end;
  end;
  Result := (Len = 0) or (MaxDataLen > 0) and (Dst.Size >= DWORD( MaxDataLen ));
end;

procedure LoadSLT( var Chunks: PStream; FS: PStream );
type
  PSLTHdr = ^TSLTHdr;
  TSLTHdr = packed record
    DataType: Word;
    IdWord: Word;
    Len: DWORD;
  end;
var SLTseparator: array[ 0..3 ] of Char;
    Headers: PList;
    H: PSLTHdr;
    H0: TSLTHdr;
    Present: Boolean;
    I, L: Integer;
    Chunk: array[ 0..3 ] of Char;
    MS, From: PStream;
begin
  SLTseparator[ 3 ] := #0;
  if FS.Read( SLTseparator, 3 ) < 3 then Exit;
  if SLTseparator <> 'SLT' then Exit;
  Headers := NewList;
  TRY
    FillChar( H0, Sizeof( H0 ), 0 );
    Present := FALSE;
    while TRUE do
    begin
      H := AllocMem( Sizeof( H^ ) );
      Headers.Add( H );
      if FS.Read( H^, Sizeof( H^ ) ) < Sizeof( H^ ) then Exit;
      if CompareMem( H, @ H0, Sizeof( H0 ) ) then break;
      if (H.DataType = 1) and (H.Len <> 0) then
        Present := TRUE;
    end;
    if not Present then Exit; // no levels
    if Chunks = nil then
      Chunks := NewMemoryStream;
    for I := 0 to Headers.Count-2 do
    begin
      H := Headers.Items[ I ];
      if ((H.DataType = 1) or (H.DataType = 3)) and
         (H.Len <> 0) then
      begin
        MS := NewMemoryStream;
        From := NewExMemoryStream( Pointer( DWORD( FS.Memory ) + FS.Position ), H.Len );
        TRY
          if DecodeMemBlock( MS, From, H.Len, 0 ) then
          begin
            Chunk := 'SLT ';
            Chunks.Write( Chunk, Sizeof( Chunk ) );
            L := 4 + MS.Size;
            Chunks.Write( L, 4 );
            L := H.DataType;
            Chunks.Write( L, 2 );
            L := H.IdWord;
            Chunks.Write( L, 2 );
            MS.Position := 0;
            Stream2Stream( Chunks, MS, MS.Size );
          end;
        FINALLY
          MS.Free;
          From.Free;
        END;
        FS.Position := FS.Position + H.Len;
      end
        else
      begin
        FS.Position := FS.Position + H.Len;
      end;
    end;
  FINALLY
    Headers.Release;
  END;
end;

function LoadSpectrum( FileData: Pointer; FileSize: Integer; FileExt: PChar;
         Data: PSpecData; ScreenOnly: Boolean ): Boolean; stdcall;
var FS: PStream;
    Hdr: TZ80;
    Hdr2: TZ80New;
    MemHdr: TMemPageHdr;
    Mem48, Mem16: PStream;
    Is128: Boolean;
    I, L, T_Hi, T_Lo: Integer;
    Pos: DWORD;
    IgnoreCompressFlag: Boolean;
    PgC000, PgCount: Integer;
    Chunks: PStream;
    Mem: Pointer;
    Chunk: array[ 0..3 ] of Char;
    Hz80: THardwareZ80;
begin
  Result := FALSE;
  FS := NewMemoryStream;
  FS.Size := FileSize;
  Chunks := nil;
  TRY
    Move( FileData^, FS.Memory^, FileSize );
    FillChar( Hdr, Sizeof( Hdr ), 0 );
    if FS.Read( Hdr, Sizeof( Hdr ) ) <> Sizeof( Hdr ) then Exit;
    if Hdr.R7_Border_SamRom_Compress = $FF then
       Hdr.R7_Border_SamRom_Compress := 1;
    Data.State.AF := Hdr.A shl 8 or Hdr.F;
    Data.State.BC := Hdr.BC;
    Data.State.DE := Hdr.DE;
    Data.State.HL := Hdr.HL;
    Data.State.IX := Hdr.IX;
    Data.State.IY := Hdr.IY;
    Data.State.AFalt := Hdr.Aalt shl 8 or Hdr.Falt;
    Data.State.BCalt := Hdr.BCalt;
    Data.State.DEalt := Hdr.DEalt;
    Data.State.HLalt := Hdr.HLalt;
    Data.State.I := Hdr.I;
    Data.State.R := Hdr.R0_6 and $7F or (Hdr.R7_Border_SamRom_Compress and 1 shl 7);
    Data.State.PC := Hdr.PC;
    Data.State.SP := Hdr.SP;
    Data.State.IFF1 := Hdr.IFF1 <> 0;
    Data.State.IFF2 := Hdr.IFF2 <> 0;
    Data.State.ImMode := Hdr.IM_Speed_Joy and 3;
    Data.State.BorderColor := (Hdr.R7_Border_SamRom_Compress shr 1) and 7;
    for I := 0 to 7 do
      FillChar( Data.RAMs[ I ], Sizeof( Data.RAMs[ 0 ] ), 0 );
    if Hdr.PC <> 0 then
    begin // old format - 48K only:
      Result := TRUE;
      Data.Lock7FFD := TRUE;
      Data.State.BankROM_0000 := 1;
      Data.State.BankRAM_C000 := 7;
      Data.State.BankVideo := 0;
      Mem48 := NewMemoryStream;
      TRY
        if ((Hdr.R7_Border_SamRom_Compress and (1 shl 5)) <> 0) or
           (FS.Size - FS.Position <> 16384 * 3) then
        begin
          I := 3 * 16384;
          if ScreenOnly then
            I := $1B00;
          Result := DecodeMemBlock( Mem48, FS, 0, I );
        end
        else
          Stream2Stream( Mem48, FS, FS.Size - FS.Position );
        Mem48.Position := 0;
        Move( Mem48.Memory^, Data.RAMs[ 5 ], Min( Mem48.Size, 16384 ) );
        if Mem48.Size > 16384 then
          Move( PByte( Integer( Mem48.Memory ) + 16384 )^, Data.RAMs[ 2 ],
                Min( Mem48.Size - 16384, 16384 ) );
        if Mem48.Size > 32768 then
          Move( PByte( Integer( Mem48.Memory ) + 32768 )^, Data.RAMs[ 7 ],
                Min( Mem48.Size - 32768, 16384 ) );
      FINALLY
        Mem48.Free;
      END;
    end
      else
    begin // new format - 48K or 128K:
      FillChar( Hdr2, Sizeof( Hdr2 ), 0 );
      if FS.Read( Hdr2.ExtraHeaderLen, 2 ) <> 2 then Exit;
      if not IntIn( Hdr2.ExtraHeaderLen, [ 23, 54, 55 ] ) then Exit;
      if FS.Read( Hdr2.PC, Hdr2.ExtraHeaderLen ) <> Hdr2.ExtraHeaderLen then Exit;
      Data.State.PC := Hdr2.PC;

      Chunks := NewMemoryStream;
      Chunk := 'HARD';
      Chunks.Write( Chunk, 4 );
      L := Sizeof( THardwareZ80 );
      Chunks.Write( L, 4 );
      if Hdr2.HardwareMode = 128 then
        Hz80.Hardware := hTimexTS2068
      else
      if (Hdr2.ExtraHeaderLen = 23) and (Hdr2.HardwareMode <= 4) then
        CASE Hdr2.HardwareMode OF
        0: Hz80.Hardware := h48K;
        1: Hz80.Hardware := h48K_IF1;
        2: Hz80.Hardware := hSamRAM;
        3: Hz80.Hardware := h128K;
        4: Hz80.Hardware := h128K_IF1;
        END
      else
      if Hdr2.HardwareMode <= 14 then
        Hz80.Hardware := THardware( Hdr2.HardwareMode );
      if Hdr2.Emulation_R_LDIR and $80 <> 0 then
      begin
        CASE Hz80.Hardware OF
        h48K, h48K_IF1, h48K_MGT:
          Hz80.Hardware := h16K;
        hPlus3, hPlus3_8:
          Hz80.Hardware := hPlus2a;
        //h128K, h128K_IF1, h128K_MGT:
        else if not(Hz80.Hardware in [hPlus2a {, hPlus3} ]) then
          Hz80.Hardware := hPlus2;
        END;
      end;

      CASE Hz80.Hardware OF
      hSamRAM:
        begin
          Hz80.SamRAM_74ls259 := Hdr2.Last7FFD;
        end;
      hTimexTC2048, hTimexTS2068:
        begin
          Hz80.LastPortF5 := Hdr2.Last7FFD;
          Hz80.LastPortFF := Hdr2.Interface1;
        end;
      hPlus3:
        begin
          if Hdr2.ExtraHeaderLen = 55 then
            Hz80.Last1FFD := Hdr2.Unknown55;
        end;
      h48K_IF1, h128K_IF1:
        begin
          if Hdr2.Interface1 = $FF then
            Hz80.IF1_ROM_Paged := TRUE;
        end;
      h48K_MGT, h128K_MGT:
        if Hdr2.ExtraHeaderLen > 23 then
        begin
          if Hdr2.MGTRom = $FF then
            Hz80.MGT_ROM_Paged := TRUE;
          if Hdr2.RAM0000_1FFF = 0 then
            Hz80.RAM_at_0000_1FFF := TRUE;
          if Hdr2.RAM2000_3FFF = 0 then
            Hz80.RAM_at_2000_3FFF := TRUE;
          CASE Hdr2.MGTType OF
          0, 1: Hz80.MGT_Type := TMGTType( Hdr2.MGTType );
          16:   Hz80.MGT_Type := mgtPlusD;
          END;
          Hz80.DiscipleInhibitButton_IN := Hdr2.DiscipleInhibitButton = $FF;
          Hz80.DiscipleInhibitROM_NOTpageble := Hdr2.DiscipleInhibitFlag = $FF;
        end;
      END;
      Chunks.Write( Hz80, Sizeof( Hz80 ) );

      if Hdr2.ExtraHeaderLen = 23 then
      begin
        if THardwareMode2_01( Hdr2.HardwareMode ) = hm_SamRom then Exit;
        Is128 := (Hdr2.HardwareMode >= 5) and (Hdr2.HardwareMode <> 128) OR
           (THardwareMode2_01( Hdr2.HardwareMode ) in [ hm_128K, hm_128K_If1 ]);
      end
        else
      begin
        //if THardwareMode3_00( Hdr2.HardwareMode ) = hm3_SamRom then Exit;
        Is128 := (Hdr2.HardwareMode >= 7) and (Hdr2.HardwareMode <> 14) OR
           (THardwareMode3_00( Hdr2.HardwareMode ) in [ hm3_128K, hm3_128K_If1, hm3_128K_MGT ]);
      end;
      if Is128 then
      begin
        Data.State.BankRAM_C000 := Hdr2.Last7FFD and 7;
        Data.State.BankROM_0000 := (Hdr2.Last7FFD shr 4) and 1;
        Data.State.BankVideo := (Hdr2.Last7FFD shr 3) and 1;
        Data.Lock7FFD := (Hdr2.Last7FFD and $20 <> 0);
      end
        else
      begin
        Data.Lock7FFD := TRUE;
        Data.State.BankROM_0000 := 1;
      end;
      Data.Sound.LastFFFD := Hdr2.LastFFFD;
      for I := 0 to 15 do
        Data.Sound.Regs[ I ] := Hdr2.SoundChipRegs[ I ];

      if Hdr2.ExtraHeaderLen > 23 then
      begin
        //T_Hi := (Hdr2.TStateCounterHi + 1) and 3;
        T_Hi := Hdr2.TStateCounterHi;
        if Is128 then
        begin
          //T_Lo := 17726 - Hdr2.TStateCounterLo;
          T_Lo := Hdr2.TStateCounterLo;
          //if T_Lo < 0 then T_Lo := 0;
          Data.State.TactsFromLastInt := T_Hi * 17727 + T_Lo;
        end
        else
        begin
          //T_Lo := 17471 - Hdr2.TStateCounterLo;
          //if T_Lo < 0 then T_Lo := 0;
          T_Lo := Hdr2.TStateCounterLo;
          Data.State.TactsFromLastInt := T_Hi * 17472 + T_Lo;
        end;
      end
        else
        Data.State.TactsFromLastInt := 0;

      Mem16 := NewMemoryStream;
      TRY
        PgC000 := -1;
        PgCount := 0;
        while FS.Position < FS.Size do
        begin
          Mem16.Size := 0;
          Mem16.Position := 0;
          if FS.Read( MemHdr, Sizeof( MemHdr ) ) <> Sizeof( MemHdr ) then Exit;
          Pos := FS.Position;
          if MemHdr.DataLen <> 0 then
          begin
            if Is128 then
              if Data.State.BankVideo = 0 then
                I := 8
              else
                I := 10
            else
                I := 8;

            IgnoreCompressFlag := FALSE;
            if MemHdr.DataLen = 65535 then
            begin
              MemHdr.DataLen := 16384;
              IgnoreCompressFlag := TRUE;
            end;

            if ScreenOnly and (MemHdr.PageNum <> I) then
            begin
              FS.Position := Pos + MemHdr.DataLen;
              continue;
            end;

            L := 16384;
            if ScreenOnly then L := $1B00;
            if not IgnoreCompressFlag and
               ( ((Hdr.R7_Border_SamRom_Compress and (1 shl 5)) <> 0) or
                 (MemHdr.DataLen < 16384) ) then
            begin
              if not DecodeMemBlock( Mem16, FS, MemHdr.DataLen, L ) then Exit;
            end
              else Stream2Stream( Mem16, FS, L );

            Inc( PgCount );
            if Is128 then
            begin
              I := MemHdr.PageNum - 3;
              if I = Data.State.BankRAM_C000 then
                PgC000 := I
              else
              if (I = 2) and (PgC000 < 0) then
                PgC000 := I;

              CASE MemHdr.PageNum OF
              3..10: Move( Mem16.Memory^, Data.RAMs[ I ], Min( Mem16.Size, L ) );
              else ;
              END
            end
              else
            begin
              CASE MemHdr.PageNum OF
              4: I := 2;
              5: I := Data.State.BankRAM_C000;
              8: I := 5;
              else I := -1;
              END;
              if I >= 0 then
                Move( Mem16.Memory^, Data.RAMs[ I ], Min( Mem16.Size, L ) );
            end;
          end
            else
          begin
            if (MemHdr.DataLen = 0) and (MemHdr.PageNum = 0) then
              break; // MemHdr.DataLen = 0 => end of Z80, start of SLT
          end;
          FS.Position := Pos + MemHdr.DataLen;
        end;
        if (PgCount <= 3) and (PgC000 >= 0) and (PgC000 <> Data.State.BankRAM_C000) then
          Move( Data.RAMs[ PgC000 ], Data.RAMs[ Data.State.BankRAM_C000 ], 16384 );
        Result := TRUE;
      FINALLY
        Mem16.Free;
      END;
    end;
    if FS.Position < FS.Size - 6 then
    begin // possible SLT extension follows here
      LoadSLT( Chunks, FS );
    end;
    if (Chunks <> nil) and (Chunks.Size > 0) then
    begin
      GetMem( Mem, Chunks.Size );
      Move( Chunks.Memory^, Mem^, Chunks.Size );
      Data.Tape.TapeImgData := Mem;
      Data.Tape.TapeImgLen := Chunks.Size;
    end;
  FINALLY
    FS.Free;
    Chunks.Free;
  END;
end;

type
  TByteArray = array[0..1000000] of Byte;
  PByteArray = ^TByteArray;

function CompressMemory( Dst: PStream; Mem: PByteArray; Len: Integer;
         EndMarker: Boolean ): Boolean;
  function DoCompress: Integer;
  var Src: PByteArray;
      L, I, N: Integer;
      ED: Byte;
      Pos0: DWORD;
      AfterED: Boolean;
  begin
    Pos0 := Dst.Position;
    ED := $ED;
    Src := Mem;
    L := Len;
    AfterED := FALSE;
    while L > 0 do
    begin
      if not AfterED and
         (L > 4) and (Src[ 0 ] = Src[ 1 ]) and (Src[ 1 ] = Src[ 2 ]) and
         (Src[ 2 ] = Src[ 3 ]) and (Src[ 3 ] = Src[ 4 ]) or
         (L > 1) and (Src[ 0 ] = ED) and (Src[ 1 ] = ED) then
      begin
        Dst.Write( ED, 1 );
        Dst.Write( ED, 1 );
        N := 2;
        for I := 2 to L-1 do
        begin
          if Src[ I ] <> Src[ 0 ] then break;
          Inc( N );
          if N = 255 then break;
        end;
        Dst.Write( N, 1 );
        Dst.Write( Src[ 0 ], 1 );
        Src := @ Src[ N ];
        Dec( L, N );
        AfterED := FALSE;
      end
        else
      begin
        AfterED := Src[ 0 ]=ED;
        Dst.Write( Src[ 0 ], 1 );
        Dec( L );
        Src := @ Src[ 1 ];
      end;
    end;
    if EndMarker then
    begin
      L := 0;
      Dst.Write( L, 1 );
      Dst.Write( ED, 1 );
      Dst.Write( ED, 1 );
      Dst.Write( L, 1 );
    end;
    Result := Dst.Position - Pos0;
  end;

var Pos: DWORD;
begin
  Pos := Dst.Position;
  Result := DoCompress >= Len;
  if not Result then
  begin
    Dst.Size := Pos + DWORD( Len );
    Move( Mem^, Pointer( DWORD( Dst.Memory ) + Pos )^, Len );
    Dst.Position := Dst.Size;
    Result := FALSE;
  end;
end;

function SaveSpectrum( FilePath: PChar; Data: PSpecData ): Boolean; stdcall;
var FS, CMem48, Mem48: PStream;
    Hdr: TZ80;
    Hdr2: TZ80New;
    MemHdr: TMemPageHdr;
    I, L, T_Hi, T_Lo: Integer;

    Chunks: PStream;
    Hz80: THardwareZ80;
    Chunk: array[ 0..3 ] of Char;
    Is128: Boolean;
begin
  Result := FALSE;
  FS := NewWriteFileStream( FilePath );
  CMem48 := NewMemoryStream;
  Mem48 := NewMemoryStream;
  TRY
    if FS.Handle = INVALID_HANDLE_VALUE then Exit;
    Result := TRUE;

    FillChar( Hdr, Sizeof( Hdr ), 0 );
    Hdr.A := Data.State.AF shr 8;
    Hdr.F := Data.State.AF and $FF;
    Hdr.BC := Data.State.BC;
    Hdr.HL := Data.State.HL;
    if Data.Lock7FFD then
      Hdr.PC := Data.State.PC;
    Hdr.SP := Data.State.SP;
    Hdr.I := Data.State.I;
    Hdr.R0_6 := Data.State.R;
    Hdr.R7_Border_SamRom_Compress := Data.State.R shr 7 or
      (Data.State.BorderColor and 7) shl 1;

    Mem48.Size := 3 * 16384;
    Move( Data.RAMs[ 5 ], Mem48.Memory^, 16384 );
    Move( Data.RAMs[ 2 ], Pointer( DWORD( Mem48.Memory ) + 16384 )^, 16384 );
    Move( Data.RAMs[ Data.State.BankRAM_C000 ],
          Pointer( DWORD( Mem48.Memory ) + 2 * 16384 )^, 16384 );

    if Data.Lock7FFD and
       CompressMemory( CMem48, Mem48.Memory, 3 * 16384, TRUE ) then
      Hdr.R7_Border_SamRom_Compress := Hdr.R7_Border_SamRom_Compress or 32
    else
    begin
      CMem48.Position := 0;
      Stream2Stream( CMem48, Mem48, 3 * 16384 );
    end;

    Hdr.DE := Data.State.DE;
    Hdr.BCalt := Data.State.BCalt;
    Hdr.DEalt := Data.State.DEalt;
    Hdr.HLalt := Data.State.HLalt;
    Hdr.Aalt := Data.State.AFalt shr 8;
    Hdr.Falt := Data.State.AFalt and $FF;
    Hdr.IY := Data.State.IY;
    Hdr.IX := Data.State.IX;
    if Data.State.IFF1 then Hdr.IFF1 := 1
                       else Hdr.IFF1 := 0;
    if Data.State.IFF2 then Hdr.IFF2 := 1
                       else Hdr.IFF2 := 0;
    Hdr.IM_Speed_Joy := Data.State.ImMode and 3;

    Is128 := not Data.Lock7FFD;

    if not Is128 and (Data.Tape.TapeImgData = nil) then
    begin
      FS.Write( Hdr, Sizeof( Hdr ) );
      CMem48.Position := 0;
      Stream2Stream( FS, CMem48, CMem48.Size );
    end
    else
    begin
      Hdr.PC := 0;
      FS.Write( Hdr, Sizeof( Hdr ) );
      FillChar( Hdr2, Sizeof( Hdr2 ), 0 );
      Hdr2.ExtraHeaderLen := 23;
      Hdr2.Emulation_R_LDIR := 3;
      Hdr2.PC := Data.State.PC;
      Hdr2.HardwareMode := 3;
      Hdr2.Last7FFD := Data.State.BankRAM_C000 or
        Data.State.BankVideo shl 3 or
        Data.State.BankROM_0000 shl 4;
      if Data.Lock7FFD then
        Hdr2.Last7FFD := Hdr2.Last7FFD or $20;
      Hdr2.LastFFFD := Data.Sound.LastFFFD;
      for I := 0 to 15 do
        Hdr2.SoundChipRegs[ I ] := Data.Sound.Regs[ I ];

      if Data.Tape.TapeImgData <> nil then
      begin
        Chunks := NewExMemoryStream( Data.Tape.TapeImgData, Data.Tape.TapeImgLen );
        TRY
          while Chunks.Position < Chunks.Size do
          begin
            Chunks.Read( Chunk, 4 );
            Chunks.Read( L, 4 );
            if Chunk = 'HARD' then
            begin
              FillChar( Hz80, Sizeof( Hz80 ), 0 );
              Chunks.Read( Hz80, Min( L, Sizeof( Hz80 ) ) );

              //if Data.Lock7FFD then
              //  Hdr2.Last7FFD := Hdr2.Last7FFD or $20;

              Is128 := Hz80.Hardware in [ h128K, h128K_IF1, h128K_MGT, hPlus2,
                    hPlus2a, hPlus3, hPentagon, hScorpion, hTimexTC2048,
                    hTimexTS2068 ];

              if (Hz80.Hardware in [ hPlus3 ]) or (Data.State.TactsFromLastInt <> 0) or
                 Hz80.MGT_ROM_Paged or Hz80.RAM_at_0000_1FFF or Hz80.RAM_at_2000_3FFF or
                 (Hz80.MGT_Type > mgtDisciple_Epson) or (not Hz80.DiscipleInhibitROM_NOTpageble) then
              begin
                Hdr2.ExtraHeaderLen := 54;
                if Hz80.Hardware = hPlus3 then
                begin
                  Hdr2.ExtraHeaderLen := 55;
                  Hdr2.Unknown55 := Hz80.Last1FFD;
                end;
              end;
              CASE Hz80.Hardware OF
              hTimexTS2068: Hdr2.HardwareMode := 128;
              h16K:         begin
                              Hdr2.HardwareMode := 0;
                              Hdr2.Emulation_R_LDIR := Hdr2.Emulation_R_LDIR or $80;
                            end;
              else          Hdr2.HardwareMode := Byte( Hz80.Hardware );
              END;
              CASE Hz80.Hardware OF
              hSamRAM:
                begin
                  Hdr2.Last7FFD := Hz80.SamRAM_74ls259;
                end;
              hTimexTC2048, hTimexTS2068:
                begin
                  Hdr2.Last7FFD := Hz80.LastPortF5;
                  Hdr2.Interface1 := Hz80.LastPortFF;
                end;
              hPlus3:
                begin
                  Hdr2.Unknown55 := Hz80.Last1FFD;
                end;
              h48K_IF1, h128K_IF1:
                begin
                  if Hz80.IF1_ROM_Paged then
                    Hdr2.Interface1 := $FF;
                end;
              h48K_MGT, h128K_MGT:
                begin
                  if Hz80.MGT_ROM_Paged then
                    Hdr2.MGTRom := $FF;
                  if Hz80.RAM_at_0000_1FFF
                    then Hdr2.RAM0000_1FFF := 0
                    else Hdr2.RAM0000_1FFF := $FF;
                  if Hz80.RAM_at_2000_3FFF
                    then Hdr2.RAM2000_3FFF := 0
                    else Hdr2.RAM2000_3FFF := $FF;
                  CASE Hz80.MGT_Type OF
                  mgtDisciple_Epson: Hdr2.MGTType := 0;
                  mgtDisciple_HP   : Hdr2.MGTType := 1;
                  mgtPlusD         : Hdr2.MGTType := 16;
                  END;
                  if Hz80.DiscipleInhibitButton_IN then
                    Hdr2.DiscipleInhibitButton := $FF;
                  if Hz80.DiscipleInhibitROM_NOTpageble then
                    Hdr2.DiscipleInhibitFlag := $FF;
                end;
              END;
              break;
            end
              else
              Chunks.Position := Chunks.Position + DWORD( L );
          end;
        FINALLY
          Chunks.Free;
        END;
      end;

      if Is128 then
      begin
        {T_Hi := (Data.State.TactsFromLastInt div 17727 - 1) and 3;
        T_Lo := Data.State.TactsFromLastInt mod 17727;
        T_Lo := 17726 - T_Lo;}
        T_Hi := Data.State.TactsFromLastInt div 17727;
        T_Lo := Data.State.TactsFromLastInt mod 17727;
      end
        else
      begin
        {T_Hi := (Data.State.TactsFromLastInt div 17472 - 1) and 3;
        T_Lo := Data.State.TactsFromLastInt mod 17472;
        T_Lo := 17471 - T_Lo;}
        T_Hi := Data.State.TactsFromLastInt div 17472;
        T_Lo := Data.State.TactsFromLastInt mod 17472;
      end;
      if T_Lo < 0 then T_Lo := 0;
      Hdr2.TStateCounterLo := T_Lo;
      Hdr2.TStateCounterHi := T_Hi;

      FS.Write( Hdr2, 2 + Hdr2.ExtraHeaderLen );
      for I := 0 to 7 do
      begin
        CMem48.Size := 0;

        if not Is128 then
          CASE I OF
          5: MemHdr.PageNum := 8;
          2: MemHdr.PageNum := 4;
          else
            if I = Data.State.BankRAM_C000 then
              MemHdr.PageNum := 5
            else
              CONTINUE;
          END
        else
          MemHdr.PageNum := I + 3;
        if not CompressMemory( CMem48, @ Data.RAMs[ I ], 16384, FALSE ) then
        begin
          MemHdr.DataLen := CMem48.Size;
          FS.Write( MemHdr, Sizeof( MemHdr ) );
          CMem48.Position := 0;
          Stream2Stream( FS, CMem48, CMem48.Size );
        end
        else
        begin
          MemHdr.DataLen := 65535;
          FS.Write( MemHdr, Sizeof( MemHdr ) );
          FS.Write( Data.RAMs[ I ], 16384 );
        end;
      end;
    end;

  FINALLY
    CMem48.Free;
    Mem48.Free;
    FS.Free;
  END;
end;

procedure ReleaseData( MemAddr: Pointer ); stdcall;
begin
  FreeMem( MemAddr );
end;

exports RegisterLoadSpectrum, LoadSpectrum, SaveSpectrum, ReleaseData;

begin
  //UseDelphiMemoryManager;
end.
