module speccy2007_vid (
	input	wire	CLK,			// 14MHz input clock
	input	wire	nCPU_Reset,		// CPU reset
	// CPU signals
	output	reg 	z80_clk = 1,	// CPU clock 3.5 MHz
	input	wire	rd_n,
	input	wire	wr_n,
	input	wire	iorq_n,
	input	wire	mreq_n,
	input	wire	m1_n,
	output	wire 	int_n,
	// AVR
	input	wire	AVR_NOINT,
	input	wire	AVR_PROM,
	input	wire	AVR_WAIT,
	output	wire	AVR_INT,
	output	wire	AVR_TRDOS,
	// buffers
	output	wire	nADR_GATE_EN,
	output	wire 	WR_BUF,
	output	wire 	nRD_BUF_EN,
	output	wire 	nWR_GATE_EN,
	// speaker/tape_in
	output	wire	SPEAKER,
	input	wire	TAPE_IN,
//	output	reg		TAPE_OUT = 0,
	// RAM/ROM select
	output 	wire	ROM_CS,
	output 	wire	ROM_A14,
	output 	wire	ROM_A15,
	output	wire	mRD,
	output	wire	mWR,
	output	wire	RAM_A14,
	output	wire	RAM_A15,
	output	wire	RAM_A16,
	// Video
	output	wire	SYNC,
	output	reg		Red = 0,
	output	reg		Green = 0,
	output	reg		Blue = 0,
	output	reg		Bright = 0,
	// AY
	output	wire	AY_CLK,
	output	wire	AY_BC1,
	output	wire	AY_BDIR,
    // Address/Data Bus
	input	wire	[7:0] mD,		// Data Bus
	inout	wire	[13:0] mA,		// Address bus
	input	wire	A14,
	input	wire	A15
);

//---------------------------------------------------
// pixel/attr fetch (last 3 bits of hcnt, 0-7 states)
	parameter pixel_cnt = 3'b000;
	parameter attr_cnt = 3'b001;
	parameter pixel_shift = attr_cnt - pixel_cnt; // we need to shift pixels to match attributes
//---------------------------------------------------
// CLASSIC

	parameter HLineEnd = 447;
	parameter HBorderStart = 256; //X resolution
	parameter HSynStart  = 320;
	parameter HSynEnd  = 347;
	parameter HBlancStart  = 304;
	parameter HBlancEnd  = 399;

	parameter VBorderStart = 192; //Y Resolution
	parameter VBlancStart = 242;
	parameter VBlancEnd = 249;
	parameter VSynStart = 242;
	parameter VSynEnd = 249;
	parameter VLineEnd = 311;

	parameter VIntStart = 247;
	parameter HIntStart  = 320;
	parameter HIntEnd  = 389;

//---------------------------------------------------
//---------------------------------------------------
// PENTAGON
/*
	parameter HLineEnd = 447;
	parameter HBorderStart = 256; //X resolution
	parameter HSynStart  = 320;
	parameter HSynEnd  = 347;
	parameter HBlancStart  = 304;
	parameter HBlancEnd  = 399;

	parameter VBorderStart = 192; //Y Resolution
	parameter VBlancStart = 250;
	parameter VBlancEnd = 265;
	parameter VSynStart = 250;
	parameter VSynEnd = 265;
	parameter VLineEnd = 319;

	parameter VIntStart = 240-1;
	parameter HIntStart  = 320;
	parameter HIntEnd  = 389;
*/
////-----------------------------------------------------
wire CPU_MODE = hcnt[2:0] == pixel_cnt || hcnt[2:0] == attr_cnt ? 1'b0 : 1'b1;
// hcnt[2:0] = (with current parameters)
// 000	(CPU_MODE = 0)	video processing pixels
// 001	(CPU_MODE = 0)	video processing attributes
// 010	(CPU_MODE = 1)	CPU
// 011	(CPU_MODE = 1) 	CPU	//WR_BUF
// 100	(CPU_MODE = 1) 	CPU
// 101	(CPU_MODE = 1) 	CPU	//WR_BUF
// 110	(CPU_MODE = 1)	CPU
// 111	(CPU_MODE = 1) 	CPU	//WR_BUF

reg pixel_clock			= 0;			// 7MHz pixel clock
reg [8:0] hcnt			= 9'b000000000;	// Horizontal counter
reg	[8:0] vcnt			= 9'b000000000;	// Vertical counter
reg [7:0] attrs;						// attributes register
reg	[7+pixel_shift:0] pixels;			// pixel data from data bus
reg [2:0] border_attr	= 3'b000;		// border attributes
reg	[5:0] port_7ffd		= 6'b000000;	// port 7FFD data
reg [4:0] flash_cnt		= 0;			// flash counter
reg hsync 				= 1;
reg vsync 				= 1;
reg hblanc 				= 0;
reg vblanc 				= 0;
reg vbord 				= 0;
reg hbord 				= 0;
reg border_r 			= 0;
reg blank_r 			= 0;
reg TRDOS_FLG			= 0;
reg SOUND_OUT			= 0;

wire ROMADR 			= A15 | A14;
assign ROM_CS			= mreq_n | ROMADR | TRDOS_TGL;
wire RAM_CS 			= mreq_n | ~ROMADR;
wire [1:0] ROM_PAGE;//		= 2'b00;
wire [2:0] RAM_PAGE;//		= 3'b000;
reg AVR_PORT 			= 0;

// select AY
reg AY_SEL = 0;
always @(negedge CLK)
	if( CPU_MODE == 1 )
		AY_SEL <= A15 == 1 && mA[13] == 1 && mA[1:0] == 2'b01;


always_comb
begin
	casex({TRDOS_FLG,AVR_PROM,port_7ffd[4]})
		'b1xx: ROM_PAGE = 'b11;
		'b01x: ROM_PAGE = 'b10;
		'b001: ROM_PAGE = 'b01;
		'b000: ROM_PAGE = 'b00;
	endcase

	case({A14,A15})
		'b00 : RAM_PAGE = 'b000;
		'b10 : RAM_PAGE = 'b101;
		'b01 : RAM_PAGE = 'b010;
		'b11 : RAM_PAGE = port_7ffd[2:0];
	endcase

	ROM_A14 = ROM_PAGE[0];
	ROM_A15 = ROM_PAGE[1];
	RAM_A14 = CPU_MODE == 1 ? RAM_PAGE[0] : 1'b1;
	RAM_A15 = CPU_MODE == 1 ? RAM_PAGE[1] : port_7ffd[3];
	RAM_A16 = CPU_MODE == 1 ? RAM_PAGE[2] : 1'b1;

	// buffers enable
	nADR_GATE_EN = ~CPU_MODE;
	nRD_BUF_EN = RAM_CS == 0 && rd_n == 0 ? 1'b0 : 1'b1;
	nWR_GATE_EN = CPU_MODE == 1 && ((RAM_CS == 0 || (iorq_n == 0 && m1_n == 1)) && wr_n == 0) ? 1'b0 : 1'b1;
	WR_BUF = CPU_MODE == 1 && hcnt[0] == 1 && CLK == 0 && pixel_clock == 1;
	
	mWR = CPU_MODE == 1 && RAM_CS == 0 && wr_n == 0 ? 1'b0 : 1'b1;
	mRD = (CPU_MODE == 1 && rd_n == 0 && mreq_n == 0 ) || CPU_MODE == 0 ? 1'b0 : 1'b1;

	// Memory address selector, select CPU address (Z-state) or generate address from counters
	case(hcnt[2:0])
		pixel_cnt: mA = {1'b0,vcnt[7:6],vcnt[2:0],vcnt[5:3],hcnt[7:3]}; // pixels
		attr_cnt: mA = {4'b0110,vcnt[7:3],hcnt[7:3]};					// attributes
		default: mA = 14'bZZZZZZZZZZZZZZ;								// Z80 address bus
	endcase

	SPEAKER = SOUND_OUT ^ TAPE_IN;
	AVR_INT = ~AVR_PORT;
	AVR_TRDOS = TRDOS_FLG;

	// AY
	AY_CLK = hcnt[1];
	AY_BC1 = m1_n == 1 && iorq_n == 0 && AY_SEL == 1 && A14 == 1 && RAM_CS == 1 ? 1'b1 : 1'b0;
	AY_BDIR = m1_n == 1 && iorq_n == 0 && AY_SEL == 1 && wr_n == 0 && RAM_CS == 1 ? 1'b1 : 1'b0;

end


// stop generating CPU clock when using keyboard or other AVR ports (AVR has slow clock)
wire pport_wait	= AVR_WAIT == AVR_INT || TRDOS_TGL == 1 ? 1'b1 : 1'b0;
 

// TR-DOS conditions
wire TRDOS_TGL = CPU_MODE == 1 && rd_n == 0 && m1_n == 0 && mreq_n == 0 && pixel_clock == 0 && (
			( TRDOS_FLG == 0 && port_7ffd[4] == 1 && {A15,A14,mA[13:8]} == 'h3D) // enter TR-DOS condition
		||	( TRDOS_FLG == 0 && AVR_PROM == 0 && port_7ffd[4] == 1 && {A15,A14,mA[13:0]} == 'h66 ) 
		||	( TRDOS_FLG == 1 && ( A15 | A14 | AVR_PROM == 1 )	)		 // return from TR-DOS
		) ? 1'b1 : 1'b0;


assign SYNC = ~(hsync ^ vsync);
assign int_n = vcnt == VIntStart && hcnt >= HIntStart && hcnt <= HIntEnd ? 1'b0 : 1'b1;

	
// generate clocks
always @(posedge CLK)
begin
	// generate 7MHz pixelclock
	pixel_clock <= ~pixel_clock;
	// generate Z80 CLOCK 3.5 MHz, disable if AVR is busy
	if(pixel_clock == 1)
		z80_clk <= hcnt[0] | pport_wait;
end

	
// --------------------------- VIDEO COUNTERS -------------------------------
// horisontal counters
always @(posedge CLK)
	if( pixel_clock == 0 )
	begin
		// horisontal counter increase or reset
		if(hcnt < HLineEnd)
		begin
			hcnt <= hcnt + 9'd1;
			if(hcnt == HSynEnd)
			begin
				vsync <= vcnt >= VSynStart-1 && vcnt <= VSynEnd-1 ? 1'b1 : 1'b0;
				vbord <= vcnt >= VBorderStart-1 && vcnt < VLineEnd ? 1'b1 : 1'b0;
				vblanc <= vcnt >= VBlancStart-1 && vcnt <= VBlancEnd-1 ? 1'b1 : 1'b0;
			end
		end
		else
		begin
			hcnt <= 9'd0;
			// vertical counter increase or reset
			if( vcnt < VLineEnd )
				vcnt <= vcnt + 9'd1;
			else
			begin
				vcnt <= 9'd0;
				// flash generator
				flash_cnt <= flash_cnt + 1'b1;
			end
		end
		// horisontal signals
		hsync <= hcnt >= HSynStart && hcnt <= HSynEnd ? 1'b1 : 1'b0;
		hbord <= hcnt >= HBorderStart ? 1'b1 : 1'b0;
		hblanc <= hcnt >= HBlancStart && hcnt <= HBlancEnd ? 1'b1 : 1'b0;			
	end
// --------------------------------------------------------------------------

								

// fetch attributes or shift pixel, generate border/blank
always @(negedge CLK)
	if(pixel_clock == 0)
	begin
		border_r <= hbord | vbord;
		blank_r <= hblanc | vblanc;
		pixels <= pixels << 1;
		case(hcnt[2:0])
			pixel_cnt: pixels[7:0] <= mD;	// get pixels data from video ram
			attr_cnt: attrs <= mD;			// get attributes data from video ram
		endcase
	end


// video out signals generation
always @(posedge CLK)
	if(pixel_clock == 1)
	begin
		if( border_r == 0 )
		begin
			// show pixels
			if( pixels[7+pixel_shift] ^ ( attrs[7] & flash_cnt[4] ) == 1 )
			begin // pixels output
				Blue <= attrs[0];
				Red <= attrs[1];
				Green <= attrs[2];
			end
			else
			begin // attributes output
				Blue <= attrs[3];
				Red <= attrs[4];
				Green <= attrs[5];
			end
			Bright <= attrs[6] == 1  ? 1'b1 : 1'b0;
		end
		else
		begin
			// show border
			if(blank_r == 1 )
			begin // blank output
				Blue <= 1'bZ;
				Red <= 1'bZ;
				Green <= 1'bZ;
				Bright <= 1'b0;
			end
			else
			begin // border output
				Blue <= border_attr[0];
				Red <= border_attr[1];
				Green <= border_attr[2];
				Bright <= 1'b0;
			end
		end
	end


always @(negedge CLK or negedge nCPU_Reset) //posedge pixel_clock
	if( !nCPU_Reset)
	begin
		TRDOS_FLG <= 0;
		AVR_PORT <= 0;
		port_7ffd <= 6'b000000;
		SOUND_OUT <= 0;
		//TAPE_OUT <= 0;
	end
	else
	begin
		if( TRDOS_TGL == 1 )
			TRDOS_FLG <= ~TRDOS_FLG;
		// ports ----------------------------------
		if(pixel_clock == 1)
		if( iorq_n == 0 )
		begin
			if( CPU_MODE == 1 && m1_n == 1 && hcnt[0] == 1 )
			begin
				// read/write ports using AVR
				// port FE
				if( rd_n == 0 && mA[7] == 1 && mA[4:3] == 'b11 && mA[1:0] == 'b10 && AVR_NOINT == 1 )
					AVR_PORT <= 1;
				// port 1F
				else if( ( rd_n == 0 || wr_n == 0 ) && mA[7:0] == 'h1F )
					AVR_PORT <= 1;

				// Write to ports
				if( wr_n == 0 )
				begin //read ports
					// port #FD & #7FFD (memory manager port)
					if( mA[1] == 0 && A15 == 0 && (mA[13] ^ mA[9]) == 0 && port_7ffd[5] == 0 )
						port_7ffd <= mD[5:0];
					// port #FE (border, speaker, tapeout)
					if( mA[7] == 1 && mA[4:3] == 'b11 && mA[1:0] == 'b10 )
					begin
						border_attr <= mD[2:0];
						//TAPE_OUT <= mD[3];
						SOUND_OUT <= mD[4];
					end
				end
			end
		end
		else
			AVR_PORT <= 0;
		// ----------------------------------------
	end

endmodule