#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdint.h>

const char tzx_signature[9] = {'Z','X','T','a','p','e','!',0x1A,1};
#define BUFSIZE 4096

#define SAMPLE_RATE_CSW 44100
#define SAMPLE_RATE_WAV 44100
// Our data pulse duration is aligned to the sample rate and hence
// deviates by a few T-states from the nominal 875
#define PULSE_SAMS_CSW (SAMPLE_RATE_CSW/4000)
#define PULSE_SAMS_WAV (SAMPLE_RATE_WAV/4000)

#define WAV_VALUE_0 64
#define WAV_VALUE_1 192

#pragma pack(push,1)
struct STANDARD_DATA_BLOCK_HEADER
{
    unsigned short pause;
    unsigned short datalen;
};

struct GENERALIZED_DATA_BLOCK_HEADER
{
    unsigned block_length;
    unsigned short pause;
    unsigned totp;
    unsigned char npp;
    unsigned char asp;
    unsigned totd;
    unsigned char npd;
    unsigned char asd;
};

struct SYMDEF_1
{
    unsigned char flag;
    unsigned short length;
};

struct SYMDEF_2
{
    unsigned char flag;
    unsigned short length[2];
};

struct PRLE
{
    unsigned char symbol;
    unsigned short length;
};

struct CSW_1_01_FILE_HEADER
{
    char csw_signature[23];
    unsigned char rev_major;
    unsigned char rev_minor;
    unsigned short sample_rate;
    unsigned char compr_type;
    unsigned char flags;
    unsigned char reserved[3];
};

struct TZX_CSW_HEADER
{
    unsigned char block_type;
    unsigned block_length;
    unsigned short pause;
    unsigned char sample_rate_bytes[3];
    unsigned char compr_type;
    unsigned n_pulses;
};

struct TZX_FILE_HEADER
{
    char tzx_signature[8];
    unsigned char rev_major;
    unsigned char rev_minor;
};

#pragma pack(pop)

const struct SYMDEF_2 symdef_pilot_sync[2] = {{0,{1750,0}},{0,{875,875}}};
const struct SYMDEF_1 symdef_data[2] = {{2,875},{3,875}};

const struct PRLE pilot_sync_long[2] = {0,10000}; // 5s leader for flags 00-7F ("header")
const struct PRLE pilot_sync_short[2] = {1,4000}; // 2s leader for flags 80-FF ("data")

//------- Encoder stats / signal quality checking
struct ENCODER_STATS
{
    int min_rds;
    int max_rds;
    int cur_rds;
    int lastbits;
    int runlength;
    int max_runlength;
};

//------- Coding tables for the 8b/10b format
const unsigned char coding_tbl_5b6b[32] =
{
    0x9D, 0x75, 0xB5, 0xC6, 0xD5, 0xA6, 0x66, 0xE0, 0xE5, 0x96, 0x56, 0xD2, 0x36, 0xB2, 0x72, 0x5D,
    0x6D, 0x8E, 0x4E, 0xCA, 0x2E, 0xAA, 0x6A, 0xE9, 0xCD, 0x9A, 0x5A, 0xD9, 0x3A, 0xB9, 0x79, 0xAD
};

const unsigned char coding_tbl_3b4b_p[8] =
{
    0xB3, 0x98, 0x58, 0xC2, 0xD3, 0xA8, 0x68, 0xE3
};

const unsigned char coding_tbl_3b4b_a[8] =
{
    0xB3, 0x98, 0x58, 0xC2, 0xD3, 0xA8, 0x68, 0x73
};

//------- Coding tables for the 8b/12b format
const unsigned char coding_tbl_4b6b[16] =
{
    0xD0, 0xC8, 0xC4, 0xB0, 0x70, 0x56, 0x5A, 0x66, 0x6A, 0x96, 0x9A, 0xA6, 0xAA, 0xD5, 0xCD, 0xB5
};


//------- File error handling

static void print_write_error(FILE* ft)
{
    if(ferror(ft))
    {
        perror("Output file write error");
    }
    else
    {
        fprintf(stderr,"Output file write error\n");
    }
}

static void print_read_error(FILE* fs)
{
    if(ferror(fs))
    {
        perror("Input file read error");
    }
    else if(feof(fs))
    {
        fprintf(stderr,"Unexpected end of input file\n");
    }
    else
    {
        fprintf(stderr,"Input file_read_error\n");
    }
}

static void print_seek_error(const char* s)
{
    fprintf(stderr,"%s file seek error\n",s);
}


//---------- Encoder routines

typedef int(*encode_func_type)(int c, int* rd);

static int encode_byte_8b10b(int c, int* rd)
{
    int d6,d4,tmp;
    int p7a7_flag = 0; // Flag to use the Primary or Alternative coding for D.x7
    // Encode the 5 MSBs
    d6 = coding_tbl_5b6b[c>>3];
    // Check if the codeword affects RD or requires optional inversion
    if(d6&2)
    {
        // parity-neutral codeword - check P7/A7 condition
        tmp = ~d6;
        if(*rd)
            tmp = ~tmp; // invert if current RD=+1;
        if(!(tmp&0x0C))
        {
            // If two last data bits, inverted and RD-inverted, are zeros, use the A7 code
            p7a7_flag = 1;
        }
    }
    else
    {
        // not a parity-neutral codeword or D.07, update RD and use optional inversion
        if(*rd)
            d6 = ~d6; // invert if current RD=+1
        // After the inversion, new RD can be taken from bit 0 which tells whether to update it or not
        *rd = d6&1;
    }
    d6 = (d6>>2) & 0x3F; // Extract data bits only

    // Encode the 3 LSBs, depending on P7/A7 flag
    if(p7a7_flag)
        d4 = coding_tbl_3b4b_a[c&7]; // Alternate encoding for D.x7
    else
        d4 = coding_tbl_3b4b_p[c&7]; // Primary encoding for D.x7
    if(d4&2)
    {
        // Not a parity-neutral codeword or D.x3, update RD and use optional inversion
        if(*rd)
            d4 = ~d4; // invert if current RD=+1
        // After the inversion, new RD can be taken from bit 0 which tells whether to update it or not
        *rd = d4&1;
    }
    d4 = (d4>>4) & 0x0F; // Extract data bits only

    return (d6<<4) | d4; // Return combined codeword
}

//-----------   8b/12b encoder -------------

static int encode_nibble_4b6b(int c, int* rd)
{
    int d6;
    // Encode the 4 MSBs
    d6 = coding_tbl_4b6b[c];
    // Check if the codeword affects RD or requires optional inversion
    if(d6&2)
    {
        // optional inversion is not required
    }
    else
    {
        // optional inversion is required and update of RD may be required
        if(*rd)
            d6 = ~d6; // invert if current RD=+1
        // After the inversion, new RD can be taken from bit 0 which tells whether to update it or not
        *rd = d6&1;
    }
    d6 = (d6>>2) & 0x3F; // Extract data bits only

    return d6; // Return combined codeword
}

static int encode_byte_4b6b(int c, int* rd)
{
    int d;

    d = encode_nibble_4b6b(c>>4,rd);
    d = (d<<6) | encode_nibble_4b6b(c&0x0F,rd);
    return d;
}

//--------- 1b/2b encoder
static int encode_bit_1b2b(int c, int* rd)
{
    int d2;

    d2 = c ? 3 : 2;
    if(*rd)
        d2 = d2 ^ 3;
    if(c)
        *rd = *rd ^ 1;
    return d2;
}

static int encode_byte_1b2b(int c, int* rd)
{
    int d,i;

    d = 0;
    for(i=0; i<8; i++)
    {
        d = (d<<2) | encode_bit_1b2b((c>>7) & 1,rd);
        c <<= 1;
    }
    return d;
}

//--------- Encoder stats routines

void initialize_encoder_stats(struct ENCODER_STATS* a)
{
    // Initialize the encoder stats to the state that exists after the pilot/sync
    a->min_rds = 0;
    a->max_rds = 2;
    a->cur_rds = 0;
    a->lastbits = 0;
    a->runlength = 1;
    a->max_runlength = 2;
}

void print_encoder_stats(struct ENCODER_STATS* a)
{
    printf("Encoder statistics:\n"
        "min_rds = %d\n"
        "max_rds = %d\n"
        "cur_rds = %d\n"
        "lastbits = %08X\n"
        "runlength = %d\n"
        "max_runlength = %d\n",
        a->min_rds,a->max_rds,a->cur_rds,a->lastbits,a->runlength,a->max_runlength);
}

void update_encoder_stats(struct ENCODER_STATS* a, int d, int codeword_bits)
{
    int i;
    int curbit;
    int lbs;

    lbs = a->lastbits;
    for(i=0; i<codeword_bits; i++)
    {
        curbit = (d>>(codeword_bits-1))&1;
        if(curbit != (lbs&1))
        {
            // Different bit - runlength ends here
            a->runlength = 1;
        }
        else
        {
            // Same bit - runlength accumulates
            a->runlength++;
            if(a->runlength > a->max_runlength)
                a->max_runlength = a->runlength;
        }
        // Accumulate the RDS
        if(curbit)
        {
            a->cur_rds++;
            if(a->cur_rds > a->max_rds)
                a->max_rds = a->cur_rds;
        }
        else
        {
            a->cur_rds--;
            if(a->cur_rds < a->min_rds)
                a->min_rds = a->cur_rds;
        }
        lbs = (lbs<<1) | curbit;
        d<<=1;
    }
    a->lastbits = lbs; // Return back the current sequence of last bits
}


//--------- Bit stream handling routines

static int write_bits_buf(FILE* ft, int d, int* bits_buf)
{
    int bb = *bits_buf;
    do
    {
        if(putc((d>>(bb-8))&0xFF,ft)==EOF)
            return 1; // write error
        bb-=8;
    } while(bb>=8);
    *bits_buf = bb;
    return 0;
}

// Fletcher checksum

static void update_fletcher(unsigned* fletcher_1, unsigned* fletcher_2, unsigned char c)
{
    *fletcher_1 += c;
    *fletcher_1 = (*fletcher_1 + (*fletcher_1>>8))&0xFF;
    *fletcher_2 += *fletcher_1;
    *fletcher_2 = (*fletcher_2 + (*fletcher_2>>8))&0xFF;
}



enum input_type
{
    itBin = 0,
    itTzx
};

enum output_type
{
    otCsw = 0,
    otTzxgen,
    otTzxcsw,
	otWav
};



static int write_csw_header(FILE* ft)
{
    struct CSW_1_01_FILE_HEADER hdr;

    memset(&hdr,0,sizeof(struct CSW_1_01_FILE_HEADER));
    strcpy(hdr.csw_signature,"Compressed Square Wave\x1A");
    hdr.rev_major = 1;
    hdr.rev_minor = 1;
    hdr.sample_rate = SAMPLE_RATE_CSW;
    hdr.compr_type = 1;
    hdr.flags = 0;
    if(fwrite(&hdr,sizeof(struct CSW_1_01_FILE_HEADER),1,ft)!=1)
    {
        print_write_error(ft);
        return 1;
    }
    return 0;
}

static int write_csw_leader_sync(FILE* ft, unsigned* n_pulses, int leader_duration)
{
    int i;

    // Write the 2s leader (0011 bit pattern)
    for(i=0; i<leader_duration*SAMPLE_RATE_CSW/(2*PULSE_SAMS_CSW); i++)
    {
        if(putc(2*PULSE_SAMS_CSW,ft) == EOF)
            goto return_write_error;
        (*n_pulses)++;
    }
    // Write one sync pulse. The other will be added at the start of data
    if(putc(PULSE_SAMS_CSW,ft) == EOF)
        goto return_write_error;
    (*n_pulses)++;

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int write_csw_data_word(FILE* ft, int d, unsigned char* lastbits, unsigned* n_pulses, int wordbits)
{
    int i;
    unsigned char curbit;
    unsigned char lbs = *lastbits; // Cache locally for speed and source clarity
    unsigned char lbs2;
    int runlength;
    // Write the bits of the data word
    for(i=0; i<wordbits; i++)
    {
        curbit = (d>>(wordbits-1))&1;
        if(curbit != (lbs&1))
        {
            // Different bit - write the accumulated pulse length
            lbs2 = lbs;
            if(lbs&1)
                lbs2 = ~lbs2;
            // Find the first one in the bit sequence
            runlength = 0;
            do
            {
                lbs2 >>= 1;
                runlength++;
            } while(runlength<6 && !(lbs2&1));
            if(runlength==6)
            {
                printf("Internal encoding error - runlength>=6\n");
                return 3;
            }
            if(putc(runlength*PULSE_SAMS_CSW,ft) == EOF)
                goto return_write_error;
            (*n_pulses)++;
        }
        lbs = (lbs<<1) | curbit;
        d<<=1;
    }
    *lastbits = lbs; // Return back the current sequence of last bits

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int finalize_csw(FILE* ft, unsigned char lastbits, unsigned* n_pulses)
{
    unsigned char lbs2;
    int runlength;
    // The last bit of a CSW stream is 0. If there was a train of previous '1's,
    // end it here.
    if(lastbits&1)
    {
        lbs2 = ~lastbits;
        // Find the first one in the bit sequence
        runlength = 0;
        do
        {
            lbs2 >>= 1;
            runlength++;
        } while(runlength<6 && !(lbs2&1));
        if(runlength==6)
        {
            printf("Internal encoding error - runlength>=6\n");
            return 3;
        }
        if(putc(runlength*PULSE_SAMS_CSW,ft) == EOF)
            goto return_write_error;
        (*n_pulses)++;
    }
    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int part_to_csw_data(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                            struct ENCODER_STATS* stats,
                            int part_length, unsigned char flag, unsigned* n_pulses)
{
    int c;
    int retval;
    unsigned char lastbits = 2; // Last data bits (from the sync pulse) - "10"
    int rd = 0; // Running disparity - starts at -1
    unsigned fletcher_1 = 1; // Initial values of the two Fletcher's checksum bytes
    unsigned fletcher_2 = 1;
    int d; // current data word (n bits)
    int leader_duration; // Number of seconds in the leader tone

    // Write the leader pulses
    if(flag&0x80)
        leader_duration = 2;
    else
        leader_duration = 5;
    retval = write_csw_leader_sync(ft,n_pulses,leader_duration);
    if(retval)
        return retval; // Error return

    // Write the flag byte
    update_fletcher(&fletcher_1,&fletcher_2,flag);
    d = encode_func(flag,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses,codeword_bits);
    if(retval)
        return retval; // Error return
    // Process data bytes
    while(part_length--)
    {
        if((c=getc(fs))==EOF)
            goto return_read_error;
        update_fletcher(&fletcher_1,&fletcher_2,c);
        d = encode_func(c,&rd);
        update_encoder_stats(stats, d, codeword_bits);
        retval = write_csw_data_word(ft,d,&lastbits,n_pulses,codeword_bits);
        if(retval)
            return retval; // Error return
    }
    // Write the Fletcher's checksum
    d = encode_func(fletcher_1,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses,codeword_bits);
    if(retval)
        return retval; // Error return
    d = encode_func(fletcher_2,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_csw_data_word(ft,d,&lastbits,n_pulses,codeword_bits);
    if(retval)
        return retval; // Error return
    // Finalize the CSW stream (the final state of the mic bit must be 0)
    retval = finalize_csw(ft,lastbits,n_pulses);
    if(retval)
        return retval; // Error return

    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}

static int part_to_tzx_gen(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                           struct ENCODER_STATS* stats,
                           int part_length, unsigned char flag, unsigned short pause)
{
    struct GENERALIZED_DATA_BLOCK_HEADER gh;
    int c;
    int d = 0;
    int rd = 0; // Running disparity - starts at -1
    unsigned fletcher_1 = 1; // Initial values of the two Fletcher's checksum bytes
    unsigned fletcher_2 = 1;
    int bits_buf = 0;

    // Fill in the generalized header
    memset(&gh,0,sizeof(struct GENERALIZED_DATA_BLOCK_HEADER));
    gh.block_length = sizeof(struct GENERALIZED_DATA_BLOCK_HEADER)-4+2*sizeof(struct SYMDEF_2)+2*sizeof(struct PRLE)+2*sizeof(struct SYMDEF_1)+(int)ceil(((part_length+3)*codeword_bits*1.0)/8.0);
    gh.pause = pause;
    gh.totp = 2;
    gh.npp = 2;
    gh.asp = 2;
    gh.totd = (part_length+3)*codeword_bits; // Data length (n encoded bits per byte) + 3 extra bytes for the flag and Fletcher's checksum
    gh.npd = 1;
    gh.asd = 2; // 2 possible data symbols, bits 0 and 1
    // Write the generalized data block tag
    if(putc(0x19,ft)==EOF)
        goto return_write_error;
    // Write the fixed part of the generalized header
    if(fwrite(&gh,sizeof(struct GENERALIZED_DATA_BLOCK_HEADER),1,ft)!=1)
        goto return_write_error;
    // Write the pilot/sync definition table
    if(fwrite(symdef_pilot_sync,sizeof(struct SYMDEF_2),2,ft)!=2)
        goto return_write_error;
    // Write the pilot/sync data stream, depending on the header/data flag value
    if(flag&0x80)
    {
        // Short leader
        if(fwrite(pilot_sync_short,sizeof(struct PRLE),2,ft)!=2)
            goto return_write_error;
    }
    else
    {
        // Long leader
        if(fwrite(pilot_sync_long,sizeof(struct PRLE),2,ft)!=2)
            goto return_write_error;
    }
    // Write the data symbols definition table
    if(fwrite(symdef_data,sizeof(struct SYMDEF_1),2,ft)!=2)
        goto return_write_error;

    // Write the flag
    update_fletcher(&fletcher_1,&fletcher_2,flag);
    d = d<<codeword_bits | encode_func(flag,&rd);
    update_encoder_stats(stats, d&((1<<(codeword_bits-1))-1), codeword_bits);
    bits_buf+=codeword_bits;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    // Convert data bytes
    while(part_length--)
    {
        if((c=getc(fs))==EOF)
            goto return_read_error;
        // Update the Fletcher's checksum
        update_fletcher(&fletcher_1,&fletcher_2,c);
        d = d<<codeword_bits | encode_func(c,&rd);
        update_encoder_stats(stats, d&((1<<(codeword_bits-1))-1), codeword_bits);
        bits_buf+=codeword_bits;
        if(write_bits_buf(ft,d,&bits_buf))
            goto return_write_error;
    }
    // Add 2-byte Fletcher's checksum
    d = d<<codeword_bits | encode_func(fletcher_1,&rd);
    update_encoder_stats(stats, d&((1<<(codeword_bits-1))-1), codeword_bits);
    bits_buf+=codeword_bits;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    d = d<<codeword_bits | encode_func(fletcher_2,&rd);
    update_encoder_stats(stats, d&((1<<(codeword_bits-1))-1), codeword_bits);
    bits_buf+=codeword_bits;
    if(write_bits_buf(ft,d,&bits_buf))
        goto return_write_error;
    if(bits_buf)
    {
        // Save the remaining bits, if any
        if(putc((d<<(8-bits_buf))&0xFF,ft)==EOF)
            goto return_write_error;
    }

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
return_read_error:
    print_read_error(fs);
    return 1;
}


static int write_wav_leader_sync(FILE* ft, int leader_duration)
{
    int i,j;

    // Write the the leader (0011 bit pattern)
    for(i=0; i<leader_duration*SAMPLE_RATE_WAV/(4*PULSE_SAMS_WAV); i++)
    {
    	for(j=0; j<2*PULSE_SAMS_WAV; j++)
    	{
    		if(putc(WAV_VALUE_0,ft) == EOF)
    			goto return_write_error;
    	}
    	for(j=0; j<2*PULSE_SAMS_WAV; j++)
    	{
    		if(putc(WAV_VALUE_1,ft) == EOF)
    			goto return_write_error;
    	}
    }
    // Write the sync bits
    for(j=0; j<PULSE_SAMS_WAV; j++)
    {
		if(putc(WAV_VALUE_0,ft) == EOF)
			goto return_write_error;
    }
	for(j=0; j<PULSE_SAMS_WAV; j++)
	{
		if(putc(WAV_VALUE_1,ft) == EOF)
			goto return_write_error;
	}

    return 0;
// Error handling return paths
return_write_error:
    print_write_error(ft);
    return 1;
}

static int write_wav_data_word(FILE* ft, int d, unsigned char* lastbits, int wordbits)
{
    int i, j;
    int msk = 1<<wordbits;
    uint8_t samval;

    for(i=0; i<wordbits; i++)
    {
    	d<<=1;
    	samval = d&msk ? WAV_VALUE_1 : WAV_VALUE_0;
    	for(j=0; j<PULSE_SAMS_WAV; j++)
    	{
    		if(putc(samval,ft) == EOF)
    			goto return_write_error;
    	}
    }

    return 0;
// Error handling return path
return_write_error:
    print_write_error(ft);
    return 1;
}



static int part_to_wav_data(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                            struct ENCODER_STATS* stats,
                            int part_length, unsigned char flag)
{
    int c;
    int retval;
    unsigned char lastbits = 2; // Last data bits (from the sync pulse) - "10"
    int rd = 0; // Running disparity - starts at -1
    unsigned fletcher_1 = 1; // Initial values of the two Fletcher's checksum bytes
    unsigned fletcher_2 = 1;
    int d; // current data word (n bits)
    int leader_duration; // Number of seconds in the leader tone

    // Write the leader pulses
    if(flag&0x80)
        leader_duration = 2;
    else
        leader_duration = 5;
    retval = write_wav_leader_sync(ft,leader_duration);
    if(retval)
        return retval; // Error return

    // Write the flag byte
    update_fletcher(&fletcher_1,&fletcher_2,flag);
    d = encode_func(flag,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_wav_data_word(ft,d,&lastbits,codeword_bits);
    if(retval)
        return retval; // Error return
    // Process data bytes
    while(part_length--)
    {
        if((c=getc(fs))==EOF)
            goto return_read_error;
        update_fletcher(&fletcher_1,&fletcher_2,c);
        d = encode_func(c,&rd);
        update_encoder_stats(stats, d, codeword_bits);
        retval = write_wav_data_word(ft,d,&lastbits,codeword_bits);
        if(retval)
            return retval; // Error return
    }
    // Write the Fletcher's checksum
    d = encode_func(fletcher_1,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_wav_data_word(ft,d,&lastbits,codeword_bits);
    if(retval)
        return retval; // Error return
    d = encode_func(fletcher_2,&rd);
    update_encoder_stats(stats, d, codeword_bits);
    retval = write_wav_data_word(ft,d,&lastbits,codeword_bits);
    if(retval)
        return retval; // Error return

    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}




static int convert_file_part(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                             struct ENCODER_STATS* stats,
                             int output_format, int part_length, unsigned char flag, unsigned short pause)
{
    int retval;
    unsigned n_pulses = 0; // Number of pulses counter
    long filepos_hdr;
    long filepos_next;
    struct TZX_CSW_HEADER tchdr;
    unsigned pause_sam; // Number of CSW samples in a pause
    uint32_t i;

    if(output_format == otTzxcsw)
    {
        // Write a TZX/CSW data block header and remember its file position
        filepos_hdr = ftell(ft);
        memset(&tchdr,0,sizeof(struct TZX_CSW_HEADER));
        tchdr.block_type = 0x18;
        tchdr.compr_type = 1; // RLE compression
        tchdr.sample_rate_bytes[0] = SAMPLE_RATE_CSW & 0xFF;
        tchdr.sample_rate_bytes[1] = (SAMPLE_RATE_CSW>>8) & 0xFF;
        tchdr.sample_rate_bytes[2] = (SAMPLE_RATE_CSW>>16) & 0xFF;
        tchdr.pause = pause;
        if(fwrite(&tchdr,sizeof(struct TZX_CSW_HEADER),1,ft) != 1)
            goto return_write_error;
        // Write CSW data (we don't know its size yet)
        retval = part_to_csw_data(fs,ft,encode_func,codeword_bits,stats,part_length,flag,&n_pulses);
        if(retval)
            return retval;
        // Remember the file position to restore it after updating the header
        filepos_next = ftell(ft);
        // Update the header
        if(fseek(ft,filepos_hdr,SEEK_SET))
            goto return_out_seek_error;
        tchdr.n_pulses = n_pulses;
        tchdr.block_length = n_pulses + sizeof(struct TZX_CSW_HEADER) - 5;
        if(fwrite(&tchdr,sizeof(struct TZX_CSW_HEADER),1,ft) != 1)
            goto return_write_error;
        // Seek the file back to where data writing has finished
        if(fseek(ft,filepos_next,SEEK_SET))
            goto return_out_seek_error;
    }
    else if(output_format == otTzxgen)
    {
        // Generalized Data block (0x19) output
        retval = part_to_tzx_gen(fs,ft,encode_func,codeword_bits,stats,part_length,flag,pause);
        if(retval)
            return retval;
    }
    else if(output_format == otCsw)
    {
        // CSW output
        retval = part_to_csw_data(fs,ft,encode_func,codeword_bits,stats,part_length,flag,&n_pulses);
        if(retval)
            return retval; // Error return
        // Append a pause, if it was specified by the caller
        if(pause>0)
        {
            // Add one bit interval before the pause so that the MIC bit is 0 during the pause
            if(putc(PULSE_SAMS_CSW,ft) == EOF)
                goto return_write_error;
            pause_sam = pause*SAMPLE_RATE_CSW/1000;
            if(putc(0,ft)==EOF)
                goto return_write_error;
            if(fwrite(&pause_sam,sizeof(unsigned),1,ft)!=1)
                goto return_write_error;
        }
    }
    else if(output_format == otWav)
    {
    	// WAV output
    	retval = part_to_wav_data(fs, ft, encode_func, codeword_bits, stats, part_length, flag);
    	if(retval)
    		return retval; // Error return
    	// Append a pause, if it was specified by the caller
    	if(pause>0)
    	{
    		pause_sam = pause*SAMPLE_RATE_WAV;
    		for(i=0; i<pause_sam; i++)
    		{
    			if(putc(128,ft)==EOF)
    				goto return_write_error;
    		}
    	}
    }

    return 0;
// Error handling return path
return_write_error:
    print_write_error(ft);
    return 1;
return_out_seek_error:
    print_seek_error("Output");
    return 1;
}

static int convert_from_bin(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                            int output_format)
{
    int retval;
    long file_length;
    struct ENCODER_STATS stats;

    // Determine the input file length
    if(fseek(fs,0,SEEK_END))
    {
        print_seek_error("Input");
        return 1;
    }
    file_length = ftell(fs);
    rewind(fs);
    if(file_length > 65535)
    {
        fprintf(stderr,"Input file is too big, must be <65536 bytes\n");
        return 2;
    }

    initialize_encoder_stats(&stats);
    retval = convert_file_part(fs,ft,encode_func,codeword_bits,&stats,output_format,(int)file_length,0xFF,0);
    if(retval)
        return retval; // Error return
    print_encoder_stats(&stats);

    return 0;
}

// A function to convert a standard TZX data block
int convert_tzx_data_block(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                           int output_format)
{
    struct STANDARD_DATA_BLOCK_HEADER sbh;
    int c;
    int retval;
    struct ENCODER_STATS stats;

    initialize_encoder_stats(&stats);
    // Read block header
    if(fread(&sbh,sizeof(struct STANDARD_DATA_BLOCK_HEADER),1,fs)!=1)
        goto return_read_error;
    if(sbh.datalen < 2)
    {
        fprintf(stderr,"Data blocks of length <2 are not supported\n");
        return 1;
    }
    // Read the first data byte ("header/data flag")
    if((c=getc(fs))==EOF)
        goto return_read_error;
    // Convert the contiguous data stream by a common routine
    retval = convert_file_part(fs,ft,encode_func,codeword_bits,&stats,output_format,sbh.datalen-2,c,sbh.pause);
    if(retval)
        return retval;
    print_encoder_stats(&stats);
    // Read and discard the last data byte (parity)
    if((c=getc(fs))==EOF)
        goto return_read_error;

    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}

static int convert_from_tzx(FILE* fs, FILE* ft, encode_func_type encode_func, int codeword_bits,
                            int output_format)
{
    char buf[BUFSIZE];
    int retval;
    int c;

    // Check input file signature
    if(fread(buf,1,10,fs)!=10)
    {
        goto return_read_error;
    }
    if(memcmp(buf,tzx_signature,9))
    {
        fprintf(stderr,"Input file format error\n");
        return 1;
    }
    
    // Process data blocks until the end of file
    while((c=getc(fs))!=EOF)
    {
        if(c==0x10)
        {
            // Standard data block - convert into 8b/10b
            retval = convert_tzx_data_block(fs,ft,encode_func,codeword_bits,output_format);
            if(retval)
                return retval; // Error encountered - break
        }
        else
        {
            // Unsupported block type encountered
            fprintf(stderr,"Unsupported block type 0x%02X. Only standard data blocks are supported\n",c);
            return 1;
        }
    }
    return 0;
// Error handling return path
return_read_error:
    print_read_error(fs);
    return 1;
}


static int write_tzx_file_header(FILE* ft)
{
    struct TZX_FILE_HEADER hdr;

    memset(&hdr,0,sizeof(struct TZX_FILE_HEADER));
    strcpy(hdr.tzx_signature,"ZXTape!\x1A");
    hdr.rev_major = 1;
    hdr.rev_minor = 20;
    if(fwrite(&hdr,sizeof(struct TZX_FILE_HEADER),1,ft)!=1)
    {
        print_write_error(ft);
        return 1;
    }
    return 0;
}

#pragma pack(push,1)
struct MZ_WAVEFORMATEX_NOSIZE
{
	uint16_t wFormatTag;
	uint16_t nChannels;
	uint32_t nSamplesPerSec;
	uint32_t nAvgBytesPerSec;
	uint16_t nBlockAlign;
	uint16_t wBitsPerSample;
};

struct MZ_WAV_HEADER
{
	uint32_t riff_tag;
	uint32_t riff_size;
	uint32_t wave_tag;
	uint32_t fmt_tag;
	uint32_t fmt_size;
	struct MZ_WAVEFORMATEX_NOSIZE fmt;
	uint32_t data_tag;
	uint32_t data_size;
};
#pragma pack(pop)

static struct MZ_WAV_HEADER wav_hdr =
{
	.riff_tag = 'FFIR',  //'RIFF' backwards
	.riff_size = 0, 	 //RIFF size not known yet
	.wave_tag = 'EVAW',
	.fmt_tag = ' tmf', 	 //'fmt ' backwards
	.fmt_size = sizeof(struct MZ_WAVEFORMATEX_NOSIZE),
	.fmt =
	{
		.wFormatTag = 1, // WAVE_FORMAT_PCM
		.nChannels = 1,  // mono
		.nSamplesPerSec = SAMPLE_RATE_WAV,
		.nAvgBytesPerSec = SAMPLE_RATE_WAV,
		.nBlockAlign = 1,
		.wBitsPerSample = 8
	},
	.data_tag = 'atad',  //'data' backwards
	.data_size = 0       // Data size not known yet
};

static int write_wav_header(FILE* ft)
{
	if(fwrite(&wav_hdr, sizeof(struct MZ_WAV_HEADER), 1, ft)!=1)
	{
		print_write_error(ft);
		return 1;
	}
    return 0;
}

static int finalize_wav(FILE* ft)
{
	long wavlen = ftell(ft); // Get the file size from its current write position
	if(fseek(ft, 0, SEEK_SET))// Seek the file to the beginning to overwrite the header
	{
	    print_seek_error("Output");
	    return 1;
	}

	// Reset the WAV header fields
	wav_hdr.riff_size = wavlen-8; // Subtract the size of the RIFF header itself
	wav_hdr.data_size = wavlen-sizeof(struct MZ_WAV_HEADER);
	// Overwrite the header
	return write_wav_header(ft);
}



static void print_usage(void)
{
    printf("Usage: tzx_8b10b [-from input_type] [-to output_type] [-encoding encoding] input_file output_file\n\n"
        "input_type can be one of the following:\n"
        " tzx (default): normal blocks (header/data) only are supported\n"
        " bin: raw data is encoded without any parsing\n"
        "  as data blocks (flag=0xFF)\n\n"
        "output_type can be one of the following:\n"
        " tzxgen: TZX Generalized data block (code 0x19) - unsupported by many emulators\n"
        " tzxcsw: TZX encapsulated CSW recording (code 0x18) - also rarely supported\n"
    	" wav: WAV file, mono, 8-bit, 44kHz\n"
        " csw (default): CSW file\n\n"
        "encoding can be one of the following:\n"
        " 8b10b: (default)IBM 8b/10b encoding\n"
        " 4b6b: MB 4b/6b encoding\n"
        " 1b2b: MB 1b/2b encoding (same as S/PDIF)\n");
}

int main(int argc, char* argv[])
{
    FILE *fs, *ft;
    int retval = 1;
    int input_format = itTzx; // Default input format is tzx
    int output_format = otCsw; // Default output format is CSW
    int cur_arg = 1; // Starting to parse argument 1
    int arg_ifn = 0; // Argument index for the input file name
    int arg_ofn = 0; // Argument index for the output file name
    encode_func_type encode_func = encode_byte_8b10b; // Encoding function, default 8b/10b
    int codeword_bits = 10; // Length of codeword in bits, default 10

    printf("ZX Spectrum enhanced tape format encoder/converter V1.02\n"
        "(c) 2013, Michael Borisov\n\n");

    while(cur_arg<argc && (arg_ifn==0 || arg_ofn==0))
    {
        if(strcmp(argv[cur_arg],"-from")==0)
        {
            cur_arg++;
            if(cur_arg>=argc)
            {
                // A switch is given but nothing follows - syntax error
                print_usage();
                return 2;
            }
            if(strcmp(argv[cur_arg],"tzx")==0)
            {
                input_format = itTzx;
            }
            else if(strcmp(argv[cur_arg],"bin")==0)
            {
                input_format = itBin;
            }
            else
            {
                // Unknown input type - error
                print_usage();
                return 2;
            }
        }
        else if(strcmp(argv[cur_arg],"-encoding")==0)
        {
            cur_arg++;
            if(cur_arg>=argc)
            {
                // A switch is given but nothing follows - syntax error
                print_usage();
                return 2;
            }
            if(strcmp(argv[cur_arg],"8b10b")==0)
            {
                encode_func = encode_byte_8b10b;
                codeword_bits = 10;
            }
            else if(strcmp(argv[cur_arg],"4b6b")==0)
            {
                encode_func = encode_byte_4b6b;
                codeword_bits = 12;
            }
            else if(strcmp(argv[cur_arg],"1b2b")==0)
            {
                encode_func = encode_byte_1b2b;
                codeword_bits = 16;
            }
            else
            {
                // Unknown encoding - error
                print_usage();
                return 2;
            }
        }
        else if(strcmp(argv[cur_arg],"-to")==0)
        {
            cur_arg++;
            if(cur_arg>=argc)
            {
                // A switch is given but nothing follows - syntax error
                print_usage();
                return 2;
            }
            if(strcmp(argv[cur_arg],"csw")==0)
            {
                output_format = otCsw;
            }
            else if(strcmp(argv[cur_arg],"tzxgen")==0)
            {
                output_format = otTzxgen;
            }
            else if(strcmp(argv[cur_arg],"tzxcsw")==0)
            {
                output_format = otTzxcsw;
            }
            else if(strcmp(argv[cur_arg],"wav")==0)
            {
            	output_format = otWav;
            }
            else
            {
                // Unknown output format - error
                print_usage();
                return 2;
            }
        }
        else if(arg_ifn==0)
        {
            arg_ifn = cur_arg;
        }
        else
        {
            arg_ofn = cur_arg;
        }
        cur_arg++;
    }
    if(!(arg_ifn && arg_ofn))
    {
        // Not enough input arguments
        print_usage();
        return 2;
    }

    printf("Converting \"%s\" -> \"%s\"\n",argv[arg_ifn],argv[arg_ofn]);

    fs = fopen(argv[arg_ifn],"rb");
    if(!fs)
    {
        perror("Error opening input file");
        return 1;
    }
    ft = fopen(argv[arg_ofn],"wb");
    if(!ft)
    {
        perror("Error opening output file");
        goto cleanup_fs;
    }

    // Write a CSW or TZX file header, depending on the chosen output format
    if(output_format == otTzxgen || output_format == otTzxcsw)
    {
        retval = write_tzx_file_header(ft);
    }
    else if(output_format == otCsw)
    {
        retval = write_csw_header(ft);
    }
    else
    {
    	retval = write_wav_header(ft);
    }
    if(retval)
        return retval; // Error return

    if(input_format == itTzx)
    {
        retval = convert_from_tzx(fs,ft,encode_func,codeword_bits,output_format);
        if(retval)
            goto cleanup_ft; // Error encountered - break
    }
    else
    {
        retval = convert_from_bin(fs,ft,encode_func,codeword_bits,output_format);
        if(retval)
            goto cleanup_ft; // Error encountered - break
    }

    if(output_format == otWav)
    {
    	retval = finalize_wav(ft);
    	if(retval)
    		goto cleanup_ft;
    }


    printf("Complete.\n");
    retval = 0;

cleanup_ft:
    fclose(ft);
cleanup_fs:
    fclose(fs);
    return retval;
}
