#include <windows.h>
#include <stdio.h>
#include <stdlib.h>

#include "cpmplg.h"
#include "log.h"
#include "cpmhdd.h"


#define CPM_TYPE        0x02// ⨯ ࠧ CP/M
#define MAX_DIR         0x10// ᨬ쭮 ⢮ ஢  ४
#define MAX_BLK         1024// ࠧ ᨢ   ४਩  

#define ELEM_MAXNAMELEN 32

#pragma pack (push)
#pragma pack (1)

typedef struct {
    UINT8   Active;         // 0x80 - active partion
    UINT8   Side;           // head
    UINT16  Addr;           // cylinder/sector
    UINT8   Type;           // type partion
    UINT8   SideEnd;
    UINT16  AddrEnd;
    UINT32  RelAddr;        // ⭮⥫  
    UINT32  Size;           // ࠧ ࠧ  ᥪ
} PARTION;


typedef struct {
    UINT16  SPT;            // sectors per track
    UINT8   BSH;            // block shift
    UINT8   BLM;            // block mask
    UINT8   EXM;            // extent mask
    UINT16  DSM;            // disk maximum
    UINT16  DRM;            // directory maximum
    UINT8   AL0;            // allocation vector
    UINT8   AL1;
    UINT16  CKS;            // checksum vector size
    UINT16  OFF;            // track offset
    UINT8   res;
} DPB;

//  ᥪ ᪮ ᪠ CP/M
typedef struct {
    char    Sign[8];        // ᨣ "CP/M"
    DPB     dpb;
    UINT8   res[486];       // १
    UINT16  parSign;        // 0xAA55;
} SYSSEC;

//  ४
typedef struct {
    char    user;           //  짮⥫᪮  (user 0..15)
    char    name[8];        // 訥  - ਡ
    char    ext[3];         // 訥  - ਡ (ext[0] - R/W, ext[1] - SysFile] )
    char    ex;             // ⥪騩 ⥭
    UINT16  res;
    char    rc;             // ᫮ ᥩ  ᫥ ⥭
    UINT16  map[8];
} DIRREC;


#define ELEM_DEVICE     1
#define ELEM_DISK       2
#define ELEM_USER       3
#define ELEM_FILE       4

// "ॢ"

typedef struct _elem {
    char    type;                   // ⨯  (ELEM_XXXX)
    DWORD   attrib;                 // ਡ - ⠫  䠩 (FILE_ATTRIBUTE_DIRECTORY)
    char    name[ELEM_MAXNAMELEN];  //  
    struct _elem  *next_elem;       // ᫥騩 . ⮣ ஢
    struct _elem  *prev_elem;
    struct _elem  *next_lev;        // ᫥騩 ஢
    struct _elem  *prev_lev;
} ELEM;

typedef struct {
    ELEM    elem;
    char    Orig[12];       // ਣ쭮   ᪥
    int     Size;           // ࠧ
} CPMFILE;

//  짮⥫᪮ 
typedef struct {
    ELEM    elem;
    int     user_no;        //  짮⥫᪮ 
} USER;

typedef struct {
    int     size;           // ࠧ  ( ࠧ⮬ )
    int     free;           // ⢮ ᢮ ⮢
    char    map[];          // ⮢ 
} BMAP;

//  ᪮ ᪠ CP/M
typedef struct {
    ELEM    elem;
    UINT32  AbsAddr;        // ᮫  ᪮ ᪠
    UINT32  StartSector;    // 砫 ᥪ CP/M ᪠
    UINT16  NumBlocks;      // ࠧ ᪠  
    UINT16  BlockSize;      // ࠧ   
    UINT16  MaxDirRec;      // . ⢮ ᥩ  ४ਨ
    UINT16  DirBlocks;      // ⢮ ⢥   
    BMAP   *BlockMap;       //  ᢮ ஢ ᪠
    BMAP   *DirMap;         //  ४ ᥩ
} DISK;

// 䨧᪨ 
typedef struct {
    ELEM    elem;
    HANDLE  handle;         //  ᪠
    int     device_id;      // 浪   ( 0)
} DEVICE;

typedef struct {
    ELEM   *elem;           // 㪠⥫  ।  ᯨ᪠
    char    path[MAX_PATH]; // 
} LASTFIND;

#define DIRINSEC        (512 / sizeof(DIRREC))

#pragma pack (pop)

DEVICE *Root = NULL;          // ७



DISK *disk_Mount(DEVICE *d, UINT32 absAddr);


//============================================================================
//
// DISK IO 
//
//============================================================================

char ide_ReadSector(HANDLE handle, UINT32 Sector, void* buff)
{
    DWORD nReads;

    long long int AbsAddr = Sector * 512;
    long loAddr = AbsAddr & 0xFFFFFFFF;
    long hiAddr = AbsAddr >> 32;

    if (SetFilePointer(handle, loAddr, &hiAddr, FILE_BEGIN) == INVALID_SET_FILE_POINTER)
        return 0;
    if (!ReadFile(handle, buff, 512, &nReads, NULL))
        return 0;
    return -1;
}

char ide_WriteSector(HANDLE handle, UINT32 Sector, void* buff)
{
    DWORD nWritten;

    long long int AbsAddr = Sector * 512;
    long loAddr = AbsAddr & 0xFFFFFFFF;
    long hiAddr = AbsAddr >> 32;

    if (SetFilePointer(handle, loAddr, &hiAddr, FILE_BEGIN) == INVALID_SET_FILE_POINTER)
        return 0;
    if (!WriteFile(handle, buff, 512, &nWritten, NULL))
        return 0;
    return -1;
}

/*
஢ઠ   稥 ᨣ ⥬ ᥪ
*/
char ide_CheckSign(char buff[])
{
    if ((buff[0x1FE] != 0x55) || (buff[0x1FF] != 0xAA))
        return 0;
    else
        return -1;
}


/*
᪠஢ SMBR  । ᪨ ᪮ CP/M
 室:
    relAdd - ⭮⥫   砫 ࠧ SMBR
 室: - ⢮   ᬮ஢ ࠧ CP/M
*/
int ide_ParseSMBR(DEVICE *dev, UINT32 relAddr)
{
    PARTION    *part;
    PARTION    *next;
    UINT8       buff[512];
    UINT32      base = relAddr;     // ᮫  砫 SMBR
    DISK       *disk;
    int         num_disks;

    num_disks = 0;
    do
    {
        // 㦠 SMBR
        if (!ide_ReadSector(dev->handle, relAddr, &buff))
        {
            log_Print("  *error ide_ParseSMBR(base: 0x%08X) - can't read SMBR at 0x%08X, with code: %u\n", base, relAddr, GetLastError());
            break;
        }
        if (!ide_CheckSign(buff))
            break;
        part = (PARTION *) &buff[0x1BE];
        next = (PARTION *) &buff[0x1CE];
        if (part->Type == CPM_TYPE)
        {
            disk = disk_Mount(dev, part->RelAddr+relAddr);
            if (disk)
            {
                num_disks++;
                log_Print("  -found CP/M disk [%s] at: 0x%08X, size: %u Kb, dirs: %u\n", disk->elem.name, disk->StartSector, ((int)disk->NumBlocks*disk->BlockSize)/1024, disk->MaxDirRec);
            }
        }
        relAddr = next->RelAddr+base;
    } while (next->Type != 0);
    return num_disks;
}



/*
  ஢ ᪮ CP/M (  ७ ࠧ dos)
 室:
    d       - ⥪騩 䨧᪨ 
    filename-  䠩 ࠧ 䨧. ᪠
 室:  - ⢮   ᬮ஢ ࠧ CP/M
*/
int ide_ParseMBR(DEVICE *dev)
{
    PARTION    *par;
    UINT8       buff[512];
    int         i, num_disks;

    // 㦠 MBR
    if (!ide_ReadSector(dev->handle, 0, &buff))
    {
        log_Print("  *error ide_ParseMBR() - can't read sector!\n");
        return 0;
    }
    if (!ide_CheckSign(buff))
        return 0;
    // ᪠㥬 樨  MBR
    num_disks = 0;
    par = (PARTION *) &buff[0x1BE];
    for(i = 0; i < 4; i++)
    {
        if (par->Active)
        {
            log_Print("  -skip primary DOS partion.\n");
        } else {
            if ((par->Type == 0x05) || (par->Type == 0x0C) || (par->Type == 0x0F))
            {
                num_disks += ide_ParseSMBR(dev, par->RelAddr);
            }
        }
        par++;
    }
    return num_disks;
}




/*
  ⪮ ᪠  ᯨ᮪ ன
 室:
    model   -   "⪮ ᪠"
    dev_no  -  䨧᪮ ன⢠ (PhysicalDrive)
    filename-  䠩  ࠧ 
 室:
*/
BOOL ide_AppendDevice(HANDLE Handle, char *model)
{
    DEVICE *dev, *t;
    char    ch;
    char   *mdl = model;

    // ᮧ   DEVICE
    if ((dev = malloc(sizeof(DEVICE))) == NULL)
    {
        log_Print("  *error ide_AppendDevice() - not enought memory!\n");
        return FALSE;
    }
    memset(dev, 0, sizeof(DEVICE));
    dev->elem.type = ELEM_DEVICE;
    dev->elem.attrib = FILE_ATTRIBUTE_DIRECTORY;
    dev->handle = Handle;
    // dev->elem.next_elem = NULL;
    // dev->elem.prev_elem = NULL;
    // dev->elem.next_lev  = NULL;
    // dev->elem.prev_lev  = NULL;
    // 饬 CP/M ᪨
    if (ide_ParseMBR(dev) == 0)
    {
        free(dev);
        return FALSE;
    }
    // 塞  ᯨ᮪
    ch = 0;
    t = Root;
    if (t == NULL)
    {
        Root = dev;
    } else {
        while (t->elem.next_elem != NULL)
        {
            t = (DEVICE *) t->elem.next_elem;
            ch++;
        }
        ch++;
        t->elem.next_elem = (ELEM *) dev;
        dev->elem.prev_elem = (ELEM *) t;
    }
    if ((!model) || (strlen(model) == 0))
        mdl = "Noname";
    dev->device_id = ch;
    sprintf(dev->elem.name, "%u:", ch);
    strncat(dev->elem.name, mdl, ELEM_MAXNAMELEN-(3+1)-1);
    return TRUE;
}


void elem_Delete(void *elem)
{
    ELEM *n;
    ELEM *t = elem;
    while (t)
    {
        n = t;
        if (t->next_lev)
            elem_Delete(t->next_lev);
        t = t->next_elem;
        switch (n->type){
            case ELEM_DEVICE:
                CloseHandle( ((DEVICE *)n)->handle);
                break;
            case ELEM_DISK:
                free(((DISK *)n)->DirMap);
                free(((DISK *)n)->BlockMap);
                break;
            case ELEM_USER:
                break;
            case ELEM_FILE:
                break;
        }
        free(n);
    }
}


void ide_Done()
{
    elem_Delete(Root);
    Root = NULL;
}



//============================================================================
//
// UTIL 
//
//============================================================================

// 砥 / 
BOOL map_Set(BMAP* map, UINT16 nRec)
{
    UINT16 n = 1 << (nRec % 8);
    if (map)
        if (nRec < map->size)
            if ((map->map[nRec/8] & n) == 0)
            {
                map->map[nRec/8] |= n;
                map->free--;
                return TRUE;
            }
    return FALSE;
}

// 砥 / ᢮
BOOL map_Free(BMAP *map, UINT16 nRec)
{
    UINT16 n = 1 << (nRec % 8);
    if (map)
        if (nRec < map->size)
            if ((map->map[nRec/8] & n) > 0)
            {
                map->map[nRec/8] &= ~n;
                map->free++;
                return TRUE;
            }
    return FALSE;
}
/*
頥  /: 0      - ᢮
                                   1..80h - 
                                   -1     - 訡
*/
char map_Get(BMAP *map, UINT16 nRec)
{
    if (map)
    {
        if (nRec < map->size)
        {
            return map->map[nRec / 8] & (1 << (nRec % 8));
        } else {
            return -1;
        }
    } else {
        return -1;
    }
}

// ஢ઠ  䠩
//
BOOL FileExist(char *LocalName)
{
    FILE *f;
    BOOL b = FALSE;
    if ((f = fopen(LocalName, "r")) != NULL)
    {
        b = TRUE;
        fclose(f);
    }
    return b;
}

// ⥭ 䠩  
//  室:
//    LocalName -  䠩
//  室:
//    nReads    - ⢮ ⠭ 
char *FileLoad(char *LocalName, DWORD *nReads)
{
    HANDLE  file;
    char   *buff;
    DWORD   size;
    DWORD  nTemp;

    if ((file = CreateFile(LocalName, GENERIC_READ, 0, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE)
    {
        log_Print("    *error FileLoad() - can`t open file \"%s\"\n", LocalName);
        return NULL;
    }
    size = GetFileSize(file, NULL);
    if ((buff = malloc(size+512*2)) == NULL)
    {
        log_Print("    *error FileLoad() - not enought memory\n");
        return NULL;
    }
    memset(buff, 0, size+512*2);
    ReadFile(file, buff, size, &nTemp, NULL);
    CloseHandle(file);
    if (nReads)
        *nReads = nTemp;
    return buff;
}

// ࠧ ப     䠩
BOOL SplitPath(char *FullPath, char *Path, char *FileName)
{
    int  lendir;
    char dir[_MAX_DIR];
    char fname[_MAX_FNAME];
    char ext[_MAX_EXT];

    _splitpath(FullPath , NULL, dir, fname, ext );
    lendir = strlen(dir);
    if (lendir != 0)
    {
        lendir--;
        if (dir[lendir] == '\\')
            dir[lendir] = '\0';
    }
    strcpy(Path, dir);
    strcpy(FileName, fname);
    if (strlen(fname) > 0)
    {
        strcat(FileName, ext);
        return TRUE;
    }
    return FALSE;
}



//    CP/M 䠩 ਡ 
// ८ࠧ   -
//
DWORD fcpm_GetAttrib(char *cpmname)
{
    DWORD attr = 0;

    if (cpmname[8] & 0x80)
        attr |= FILE_ATTRIBUTE_READONLY;
    if (cpmname[9] & 0x80)
        attr |= FILE_ATTRIBUTE_HIDDEN; //   FILE_ATTRIBUTE_SYSTEM;

    if (attr == 0)
        attr = FILE_ATTRIBUTE_NORMAL;
    return attr;
}

void fcpm_SetAttrib(char *cpmname, DWORD Attr)
{
    // ⠭ ਡ
    if (Attr & FILE_ATTRIBUTE_READONLY)
        cpmname[8] |= 0x80;
    if (Attr & (FILE_ATTRIBUTE_HIDDEN | FILE_ATTRIBUTE_SYSTEM))
        cpmname[9] |= 0x80;
}


// ८ࠧ CP/M  䠩  ⥪⮢
//
void fcpm_CPM2Ansi(char *ansi, char *cpm)
{
    int   i;
    char  ch;
    int  id = 0;

    for (i = 0; i < 8; i++)
    {
        ch = cpm[i] & 0x7F;
        if (ch <= ' ')
            break;
        ansi[id++] = ch;
    }
    ansi[id] = '.';
    for (i = 8; i < 11; i++)
    {
        ch = cpm[i] & 0x7F;
        if (ch <= ' ')
            break;
        ansi[++id] = ch;
    }
    if (ansi[id] != '.')
        id++;
    ansi[id] = '\0';
}

/*
८ࠧ  䠩  ଠ CP/M
*/
void fcpm_Ansi2CPM(char *cpm, char *ansi)
{
    int     i;
    int     is = 0;
    char    ch;

    for (i = 0; i < 8; i++)
    {
        ch = ansi[is];
        if ((ch > ' ') && (ch != '.'))
        {
            cpm[i] = ch;
            is++;
        } else {
            cpm[i] = ' ';
        }
    }
    if (ansi[is] != '\0')
        is++;
    for (i = 8; i < 11; i++)
    {
        ch = ansi[is];
        if ((ch > ' ') && (ch != '.'))
        {
            cpm[i] = ch;
            is++;
        } else {
            cpm[i] = ' ';
        }
    }
}

/*
ࠢ   䠩 CP/M
 室:
    name1, name2 -   CP/M-ଠ: 11 ᨬ  窨
*/
BOOL fcpm_CompareName(char *name1, char *name2)
{
    int i;
    for (i = 0; i < 11; i++)
    {
        if ((*name1++ & 0x7F) != (*name2++ & 0x7F))
            return FALSE;
    }
    return TRUE;
}




/*
頥 㪠⥫   ⨯  
*/
void *elem_Get(char type, void *elem)
{
    ELEM *t = elem;
    while (t)
    {
        if (t->type == type)
            return t;
        t = t->prev_lev;
    }
    // ஡㥬 ᪠  㣮 ࠢ
    t = elem;
    while (t)
    {
        if (t->type == type)
            return t;
        t = t->next_lev;
    }
    return NULL;
}

/*
㭨⮦   ᯨ᪠
*/
void elem_DeleteFile(CPMFILE *elem)
{
    ELEM *paren;
    ELEM *prev;
    ELEM *next;

    if (!elem)
        return;
    if (elem->elem.type != ELEM_FILE)
        return;
    paren = elem->elem.prev_lev;
    prev  = elem->elem.prev_elem;
    next  = elem->elem.next_elem;

    // ४㥬 㪠⥫
    if (next)
    {
        next->prev_elem = prev;
    }
    if (prev)
    {
        prev->next_elem = next;
    } else {
        if (paren)
            paren->next_lev = next;
    }
    free(elem);
}

// 頥 㪠⥫  ᫥  
ELEM *elem_GetLast(char *Path)
{
    int  t;
    char buf[MAX_PATH];
    ELEM *n;
    ELEM *k = (ELEM *) Root;
    char *s = Path;

    while (*s != '\0')
    {
        n = k;
        // 뤥塞 । ப 
        s = strchr(s, '\\');
        if (s == NULL)
            return n;
        s++;
        t = strcspn(s, "\\");
        if (t == 0)
            return NULL;            // 訡窠 諠
        memcpy(buf, s, t);
        buf[t] = '\0';
        //  饬 ᮢ
        while (n)
        {
            if (stricmp(n->name, buf) == 0)
                break;
            n = n->next_elem;
        }
        if (!n)
            return NULL;            // 祣  諨
        // 室  ᫥饩 ப
        s += t;
        k = n->next_lev;
    }
    return n;
}



//============================================================================
//
// DISK CP/M IO 
//
//============================================================================

/*
⪠ १ࢨ஢ ४ ᥩ
*/
void disk_FreeDir(DISK *disk, DIRREC *dir, int nDirs)
{
    DIRREC *r = dir;

    if ((!disk) || (!dir))
        return;

    while (nDirs > 0)
    {
        if (r->res != 0)
            map_Free(disk->DirMap, r->res);
        r++;
        nDirs--;
    }
}

/*
⪠ १ࢨ஢ ஢
*/
void disk_FreeBlock(DISK *disk, DIRREC *dir, int nDirs)
{
    DIRREC *r = dir;
    int     i;

    if ((!disk) || (!dir))
        return;

    while (nDirs > 0)
    {
        for (i = 0; i < 8; i++)
            if (r->map[i] != 0)
                map_Free(disk->BlockMap, r->map[i]);
        r++;
        nDirs--;
    }
}


/*
१ࢨ   ४ 
*/
DIRREC *disk_AllocDir(DISK *disk, int nDirs, char *name, DWORD Attr)
{
    int     i, n = nDirs;
    char    cpmname[16];
    DIRREC *r, *dir;

    // 뤥塞   DIRREC
    if ((dir = (DIRREC *) malloc(nDirs * sizeof(DIRREC))) == NULL)
    {
        log_Print("    *error disk_AllocDir(\"%s\") - insufficient memory\n", name);
        return NULL;
    }
    r = dir;
    memset(dir, 0, nDirs * sizeof(DIRREC));
    fcpm_Ansi2CPM(&cpmname, name);
    fcpm_SetAttrib(&cpmname, Attr);
    i = 0;
    while ((i < disk->MaxDirRec) && (n > 0))
    {
        if (map_Get(disk->DirMap, i) == 0)
        {
            memcpy(r->name, cpmname, 11);
            // 諨 ᢮   ४ਨ
            r->res = i;                 // ६   
            map_Set(disk->DirMap, i);
            n--;
            r++;
        }
        i++;
    }
    if (n > 0)
    {
        log_Print("    *error disk_AllocDir(\"%s\") - insufficient directory space\n", name);
        disk_FreeDir(disk, dir, nDirs);
        free(dir);
        dir = NULL;
    }
    return dir;
}

/*
뤥   ᪥  頥 ந樠஢ ᨢ DIRREC
*/
DIRREC *disk_AllocSpace(DISK *disk, char nUser, char *filename, int nSize, DWORD Attr)
{
    UINT16  i, j;
    DIRREC *r, *dir;
    int     nBlocks;
    UINT16  nDirs;
    UINT16  curBlock;

    if ((!disk) || (nUser >= 16) || (!filename))
    {
        log_Print("    *error disk_AllocSpace() - bad parametr\n");
        return NULL;
    }
    nBlocks = (nSize+disk->BlockSize-1) / disk->BlockSize;
    nDirs = (nBlocks + 7) / 8;

    if ((disk->BlockMap->free < nBlocks) || (disk->DirMap->free < nDirs))
    {
        log_Print("    *error disk_AllocSpace() - disk full\n");
        return NULL;
    }
    // 뤥塞   DIRREC
    if ((dir = disk_AllocDir(disk, nDirs, filename, Attr)) == NULL)
    {
        return NULL;
    }
    r = dir;
    curBlock = disk->DirBlocks;

    for (i = 0; i < nDirs; i++)             //   ४୮ 
    {
        r->user = nUser;
        j = 0;
        while ((j < 8) && (nBlocks > 0))
        {
            while (curBlock < disk->NumBlocks)
            {
                if (map_Get(disk->BlockMap, curBlock) == 0)
                {
                    // 諨 ᢮ 
                    map_Set(disk->BlockMap, curBlock);
                    r->map[j] = curBlock;
                    nBlocks--;
                    break;
                }
                curBlock++;
            }
            j++;
        }
        r++;
    }
    if (curBlock >= disk->NumBlocks)
    {
        log_Print("    *error disk_AllocSpace() - can`t alloc blocks\n");
        disk_FreeDir(disk, dir, nDirs);
        disk_FreeBlock(disk, dir, nDirs);
        free(dir);
        dir = NULL;
    }
    return dir;
}



/*
祭 ࠬ஢ ࠧ CP/M
 室:
    p       -  ⪮ ᪠
    n       -  ⥪饣 ࠧ CP/M
    AbsSec  - ᮫  砫 ᪠ CP/M
*/
BOOL disk_GetParam(UINT32 AbsSec, DISK *disk)
{
    SYSSEC  sec;
    DEVICE  *dev = elem_Get(ELEM_DEVICE, disk);

    if (!dev)
        return FALSE;
    // 뢠  ࠬ஢ ᪠
    if (!ide_ReadSector(dev->handle, AbsSec, &sec))
    {
        log_Print("  *error disk_GetParam() - can't read sector at 0x%08X!\n", AbsSec);
        return FALSE;
    }
    if ((!ide_CheckSign((char *) &sec)) || (memcmp(sec.Sign, "CP/M    ", 8) != 0))
    {
        log_Print("  *info disk_GetParam() - disk is not CP/M!\n");
        return FALSE;
    }
    disk->AbsAddr = AbsSec;
    disk->StartSector = ((sec.dpb.OFF*sec.dpb.SPT)*128) / 512 + AbsSec + 1;
    disk->NumBlocks   = sec.dpb.DSM + 1;
    disk->BlockSize   = (sec.dpb.BLM + 1) * 128;
    disk->MaxDirRec   = sec.dpb.DRM + 1;
    disk->DirBlocks   = disk->MaxDirRec / (disk->BlockSize/sizeof(DIRREC));
    // ᮧ ᨢ  ⮢     ४ ᥩ
    disk->DirMap = malloc(sizeof(BMAP)+(disk->MaxDirRec+7)/8);
    disk->BlockMap = malloc(sizeof(BMAP)+(disk->NumBlocks+7)/8);
    if ((!disk->DirMap) || (!disk->BlockMap))
    {
        free(disk->DirMap);
        free(disk->BlockMap);
        return FALSE;
    }
    memset(disk->BlockMap, 0, sizeof(BMAP)+(disk->NumBlocks+7)/8);
    memset(disk->DirMap, 0, sizeof(BMAP)+(disk->MaxDirRec+7)/8);
    disk->BlockMap->size = disk->NumBlocks;
    disk->BlockMap->free = disk->NumBlocks;
    disk->DirMap->size   = disk->MaxDirRec;
    disk->DirMap->free   = disk->MaxDirRec;
    return TRUE;
}


/*
ᮧ ⠫ USER0-USER15,  樨 짮⥫᪨ ࠧ
 室:
    disk    - ⥪騩  CP/M
*/
USER *disk_NewUsersRec(DISK *disk)
{
    int   i;
    USER *user;
    USER *prev = NULL;

    // ᮧ ⠫ USER
    for (i=0; i<16; i++)
    {
        if ((user = malloc(sizeof(USER))) == NULL)
            break;
        user->elem.type = ELEM_USER;
        user->user_no = i;
        user->elem.attrib = FILE_ATTRIBUTE_DIRECTORY;
        sprintf(user->elem.name, "%s%02i", "user", i);
        user->elem.next_lev  = NULL;
        user->elem.prev_lev  = (ELEM *) disk;
        user->elem.next_elem = NULL;
        user->elem.prev_elem = (ELEM *) prev;
        // 塞  ᯨ᮪ ᪠
        if (prev)
        {
            prev->elem.next_elem = (ELEM *) user;
        } else {  //  
            disk->elem.next_lev = (ELEM *) user;
        }
        prev = user;
    }
    if (i != 16)
    {
        //  墠⨫ 
        log_Print("    *error disk_NewUsersRec() - insufficient memory!\n");
        prev = (USER *) disk->elem.next_lev;
        disk->elem.next_lev = NULL;
        while (prev)
        {
            user = prev;
            prev = (USER *) prev->elem.next_elem;
            free(user);
        }
    }
    return (USER *) disk->elem.next_lev;
}

void disk_UserUpCase(USER *user)
{
    if (user)
        if ((user->elem.next_lev) && (user->elem.type == ELEM_USER))
        {
            sprintf(user->elem.name, "%s%02i", "USER", user->user_no);
        } else {
            sprintf(user->elem.name, "%s%02i", "user", user->user_no);
        }
}

/*
ᮧ    䠩
*/
CPMFILE *disk_NewFileRec(DIRREC *dir)
{
    CPMFILE *file = malloc(sizeof(CPMFILE));

    if (file)
    {
        file->elem.type = ELEM_FILE;
        file->Size = dir->ex * 16384 + (dir->rc * 128);
        // ࠭塞 ਣ쭮 
        memcpy(file->Orig, dir->name, 11);
        // 㥬   ப 
        fcpm_CPM2Ansi(file->elem.name, dir->name);
        //  ਡ   䠩
        file->elem.attrib = fcpm_GetAttrib(dir->name);
        file->elem.next_elem = NULL;
        file->elem.prev_elem = NULL;
        file->elem.next_lev = NULL;
        file->elem.prev_lev = NULL;
    }
    return file;
}


void disk_InsertFile(DISK *disk, DIRREC *dir)
{
    CPMFILE *file, *first;
    USER *user = elem_Get(ELEM_USER, disk);

    // 饬 뫪  USER  dir->user
    while (user)
    {
        if (user->user_no == dir->user)
            break;
        user = (USER *) user->elem.next_elem;
    }
    if (!user)
        return;

    if (!user->elem.next_lev)
    {
        //   ᯨ᪥
        file = disk_NewFileRec(dir);
        file->elem.prev_lev    = (ELEM *) user;
        user->elem.next_lev = (ELEM *) file;
    } else {
        // 饬 䠩  ᯨ᪥ 䠩 USER
        file = elem_Get(ELEM_FILE, user);
        while (file)
        {
            if (fcpm_CompareName(file->Orig, dir->name))
                break;
            file = (CPMFILE *) file->elem.next_elem;
        }
        if (file)
        {
            // 䠩 㦥 , 塞  ࠧ
            file->Size = dir->ex * 16384 + (dir->rc * 128);
        } else {
            // 塞 䠩  ᯨ᮪
            file = disk_NewFileRec(dir);
            first = (CPMFILE *) user->elem.next_lev;  // । 䠩 
            first->elem.prev_elem = (ELEM *) file;
            file->elem.next_elem  = (ELEM *) first;
            file->elem.prev_lev   = (ELEM *) user;
            user->elem.next_lev   = (ELEM *) file;
        }
    }
}


/* ᪠஢ ४ਨ ᪠  ᮧ ᯨ᪠ 䠩 */
BOOL disk_CreateFileList(DISK *disk)
{
    UINT16  i, j;
    USER   *user;
    DIRREC  dir[DIRINSEC];
    int     nDir;
    UINT16  nSec = 0;
    DEVICE *dev  = elem_Get(ELEM_DEVICE, disk);

    if (!dev)
    {
        log_Print("    *error disk_CreateFileList() - bad disk '%s'\n", disk->elem.name);
        return FALSE;
    }
    if (!dev->handle)
    {
        log_Print("    *error disk_CreateFileList() - bad handle\n");
        return FALSE;
    }
    // 砥 ⠫ USER
    if ((user = disk_NewUsersRec(disk)) == NULL)
        return FALSE;

    // ࠧ 砥 ,   ४
    for (i = 0; i < disk->DirBlocks; i++)
        map_Set(disk->BlockMap, i);

    nDir = disk->MaxDirRec;
    // ᪠㥬 ४਩
    while (nDir > 0)
    {
        // 㦠 । ᥪ
        if (!ide_ReadSector(dev->handle, disk->StartSector + nSec, &dir))
        {
            log_Print("    *error disk_CreateFileList() - can't read directory sector at 0x%08X\n", disk->StartSector+nSec);
            return FALSE;
        }
        // ᪠㥬 । ᥪ
        i = 0;
        while ((i < DIRINSEC) && (nDir > 0))
        {
            if (dir[i].user != 0xE5)
            {
                // 諨 䠩 ( ), 砥  
                map_Set(disk->DirMap, disk->MaxDirRec-nDir);
                for (j = 0; j < 8; j++)
                    if (dir[i].map[j] >= disk->DirBlocks)
                        if (map_Get(disk->BlockMap, dir[i].map[j]) == 0)
                            map_Set(disk->BlockMap, dir[i].map[j]);
                //  塞 䠩  ᯨ᮪
                disk_InsertFile(disk, &dir[i]);
            }
            nDir--;
            i++;
        }
        nSec++;
    } // while

    // 塞 ॣ ᨬ   
    while (user)
    {
        disk_UserUpCase(user);
        user = (USER *) user->elem.next_elem;
    }
    return TRUE;
}


/*
   CP/M
 室:
 室:
    dev     -  饣 ⪮ ᪠
    absAddr - ᮫  砫 ࠧ ᪮ ᪠ CP/M
 室:  -   ᪠  NULL  訡
*/
DISK *disk_Mount(DEVICE *dev, UINT32 absAddr)
{
    DISK *disk, *last;
    char ch;

    if ((disk = malloc(sizeof(DISK))) == NULL)
        return NULL;
    disk->elem.type = ELEM_DISK;
    disk->elem.attrib = FILE_ATTRIBUTE_DIRECTORY;
    disk->elem.next_elem = NULL;
    disk->elem.prev_elem = NULL;
    disk->elem.next_lev = NULL;
    disk->elem.prev_lev = (ELEM *) dev;

    // 砥 ࠬ ᪠
    if (disk_GetParam(absAddr, disk) == 0)
    {
        free(disk);
        return NULL;
    }
    // 砥   ᯨ᮪
    last = elem_Get(ELEM_DISK, dev);
    ch = 0;
    if (last == NULL)
    {
        //    ᯨ᪥
        dev->elem.next_lev = (ELEM *) disk;
    } else {
        while (last->elem.next_elem != NULL)
        {
            last = (DISK *) last->elem.next_elem;
            ch++;
        }
        ch++;
        last->elem.next_elem = (ELEM *) disk;
        disk->elem.prev_elem = (ELEM *) last;
    }
    sprintf(disk->elem.name, "%c", ch+'A');

    // 塞 ᯨ᮪ 䠩
    if (!disk_CreateFileList(disk))
        return NULL;

    return disk;
}





//============================================================================
//
// FILE CP/M IO 
//
//============================================================================



/*
 室:
  file    - 䠩
  dstBuff - , ᫨ NULL,  뤥   
                512 
 室:
          -  (ᮧ  dstBuff)
*/
char *file_Read(CPMFILE *file, char *dstBuff)
{
    DEVICE *dev   = elem_Get(ELEM_DEVICE, file);
    DISK   *disk  = elem_Get(ELEM_DISK, file);
    USER   *user  = elem_Get(ELEM_USER, file);
    int     nDir;
    int     nSize;
    UINT16  nSec  = 0;
    UINT16  i, j, k;
    char   *b;
    DIRREC  dir[DIRINSEC];
    UINT16  nSecPerBlock;

    if ((!file) || (!dev) || (!disk) || (!user))
        return NULL;

    nDir  = disk->MaxDirRec;
    nSize = file->Size;
    nSecPerBlock = disk->BlockSize / 512;
    if (!dstBuff)
    {
        dstBuff = malloc(nSize+512);
        if (!dstBuff)
        {
            log_Print("    *error file_Read(\"%s\") - not enought memory\n", file->elem.name);
            return NULL;
        }
    }
    b = dstBuff;
    memset(b, 0, nSize+512);

    while ((nDir > 0) && (nSize > 0))
    {
        // 㦠 । ᥪ
        if (!ide_ReadSector(dev->handle, disk->StartSector + nSec, &dir))
        {
            log_Print("    *error file_Read(\"%s\") - can't read directory sector at %u\n", file->elem.name, disk->StartSector + nSec);
            free(b);
            return NULL;
        }
        // ᪠㥬 । ᥪ
        i = 0;
        while ((i < DIRINSEC) && (nDir > 0) && (nSize > 0))
        {
            if ((dir[i].user == user->user_no) && (fcpm_CompareName(dir[i].name, file->Orig)))
            {
                j = 0;
                while ((j < 8) && (nSize > 0))
                {
                    // ⠥  , ᫨  室  । ᪠
                    if ((dir[i].map[j] >= disk->DirBlocks) && (dir[i].map[j] < disk->NumBlocks))
                    {
                        if (map_Get(disk->BlockMap, dir[i].map[j]) > 0)
                        {
                            k = 0;
                            while ((k < nSecPerBlock) && (nSize > 0))
                            {
                                ide_ReadSector(dev->handle, disk->StartSector+(dir[i].map[j]*nSecPerBlock)+k, b);
                                if (nSize >= 512)
                                {
                                    b += 512;
                                    nSize -= 512;
                                } else {
                                    b += nSize;
                                    nSize = 0;
                                }
                                k++;
                            }
                        } else {
                             log_Print("    *error file_Read(\"%s\") - sector %u is empty\n", file->elem.name, dir[i].map[j]);
                        }
                    } else {
                        log_Print("    *error file_Read(\"%s\") - sector %u is not bound\n", file->elem.name, dir[i].map[j]);
                    }
                    j++;
                }           // j
            }
            nDir--;
            i++;
        }                   // i
        nSec++;
    }                       // while
    return dstBuff;
}

//   ४୮    ᪠
BOOL file_WriteDir(DISK *disk, UINT16 nDir, DIRREC *dir)
{
    DEVICE *dev = elem_Get(ELEM_DEVICE, disk);
    DIRREC  buff[DIRINSEC];
    UINT32  nSec;
    UINT16  nRec;

    if ((!disk) || (!dev) || (!dir))
    {
        log_Print("    *error file_WriteDir() - bad parametrs!\n");
        return FALSE;
    }

    if (nDir >= disk->MaxDirRec)
    {
        log_Print("    *error file_WriteDir() - bad directory number!\n");
        return FALSE;
    }
    nSec = disk->StartSector + (nDir / DIRINSEC);
    nRec = nDir % DIRINSEC;
    ide_ReadSector(dev->handle, nSec, &buff);
    memcpy(&buff[nRec], dir, sizeof(DIRREC));
    ide_WriteSector(dev->handle, nSec, &buff);
    return TRUE;
}

BOOL file_Write(USER *user, char *fname, char *buff, DWORD nSize, DWORD Attr)
{
    DEVICE *dev     = elem_Get(ELEM_DEVICE, user);
    DISK   *disk    = elem_Get(ELEM_DISK, user);
    UINT8   ex      = 0;
    UINT32  total   = 0;
    char   *b       = buff;
    DIRREC *dir;
    DIRREC *r;
    UINT16  nBlocks;
    UINT16  nDirs;
    UINT16  nSecPerBlock;
    UINT16  j, k;

    if ((!dev) || (!disk) || (!buff) || (!fname))
    {
        log_Print("    *error file_Write() - bad parameters\n");
        return FALSE;
    }
    strupr(fname);
    if ((dir = disk_AllocSpace(disk, user->user_no, fname, nSize, Attr)) == NULL)
    {
        log_Print("    *error file_Write(\"%s\") - can`t allocate space for file\n", fname);
        return FALSE;
    }
    r       = dir;
    nBlocks = (nSize+disk->BlockSize-1) / disk->BlockSize;
    nDirs   = (nBlocks + 7) / 8;
    nSecPerBlock = disk->BlockSize / 512;

    // 襬  
    while ((nDirs) && (nSize > 0))
    {
        // 襬   
        j = 0;
        while ((j < 8) && (nSize > 0))
        {
            // 襬   . 
            if ((r->map[j] >= disk->DirBlocks) && (r->map[j] < disk->NumBlocks))
            {
                if (map_Get(disk->BlockMap, r->map[j] > 0))
                {
                    // 襬 । 
                    k = 0;
                    while ((k < nSecPerBlock) && (nSize > 0))
                    {
                        ide_WriteSector(dev->handle, disk->StartSector+(r->map[j]*nSecPerBlock)+k, b);
                        b += 512;
                        if (nSize >= 512)
                        {
                            total += 512;
                            nSize -= 512;
                        } else {
                            total += nSize;
                            nSize = 0;
                        }
                        if (total > 16384)
                        {
                            total -= 16384;
                            ex++;
                        }
                        k++;
                    } // while
                } else {
                     log_Print("    *fatal error file_Write(\"%s\") - sector %u is empty\n", fname, r->map[j]);
                }
            } else {
                log_Print("    *fatal error file_Write(\"%s\") - sector %u is not bound\n", fname, r->map[j]);
            }

            j++;
        } // for j
        r->ex = ex;
        r->rc = (total+127) / 128;
        k = r->res;                    // n -  ४ਨ
        r->res = 0;
        // 뢠 ᠬ ४ 
        if (!file_WriteDir(disk, k, r))
        {
            log_Print("    *error file_Write(\"%s\") - can`t write directory rec\n", fname);
            return FALSE;
        }
        // ᨬ  ᯨ᮪
        disk_InsertFile(disk, r);
        r++;
        nDirs--;
    } // for i
    return TRUE;
}



/*
砥 ,   㪠⥫     䠩 祭
 室:
    RemoteName -    䠩  ன⢥ CP/M
 室:
               - 頥   䠩 (USER *)
    Name       -  䠩  ன⢥ CP/M
*/
USER *SplitRemoteName(char *RemoteName, char *Name)
{
    char Path[MAX_PATH];
    USER *user;

    // ᮧ  祭
    if (!SplitPath(RemoteName, &Path, Name))
    {
        log_Print("    *error SplitRemoteName() - bad destination: \"%s\"!\n", RemoteName);
        return NULL;
    }
    if (!(user = (USER *) elem_GetLast(Path)))
    {
        log_Print("    *error SplitRemoteName() - bad path: \"%s\"!\n", Path);
        return NULL;
    }
    if (user->elem.type != ELEM_USER)
    {
        log_Print("    *error SplitRemoteName() - path must be only USER!\n");
        return NULL;
    }
    return user;
}






//============================================================================
//
// INTERFACE WITH PLUGIN TC 
//
//============================================================================

HANDLE plg_FindFirst(char *Path, WIN32_FIND_DATA *FindData)
{
    LASTFIND *lf;

    if (Root == NULL)
        return INVALID_HANDLE_VALUE;

    lf = malloc(sizeof(LASTFIND));
    if (lf == NULL)
        return INVALID_HANDLE_VALUE;

    if (strcmp(Path, "\\") == 0)
    {
        // 稭 뢮 
        lf->elem = (ELEM *) Root;

        FindData->dwFileAttributes=FILE_ATTRIBUTE_DIRECTORY;
        FindData->ftLastWriteTime.dwHighDateTime=0xFFFFFFFF;
        FindData->ftLastWriteTime.dwLowDateTime=0xFFFFFFFE;
        strcpy(FindData->cFileName, lf->elem->name);
        strcpy(lf->path, Path);
    } else {
        if ((lf->elem = elem_GetLast(Path)) == NULL)
            return INVALID_HANDLE_VALUE;                //   
        FindData->ftLastWriteTime.dwHighDateTime = 0xFFFFFFFF;
        FindData->ftLastWriteTime.dwLowDateTime = 0xFFFFFFFE;
        if (lf->elem->next_lev != NULL)
        {
            lf->elem = lf->elem->next_lev;
            FindData->dwFileAttributes = lf->elem->attrib;
            strcpy(FindData->cFileName, lf->elem->name);
            if (lf->elem->type == ELEM_FILE)
            {
                CPMFILE *t = (CPMFILE *) lf->elem;
                FindData->nFileSizeLow = t->Size;
            }
        } else {
            // ⠫ 
            FindData->dwFileAttributes = FILE_ATTRIBUTE_DIRECTORY;
            strcpy(FindData->cFileName, "..");
            lf->elem = lf->elem->next_lev;
        }
        strcpy(lf->path, Path);
    }
    return (HANDLE) lf;
}


BOOL plg_FindNext(HANDLE Hdl, WIN32_FIND_DATA *FindData)
{
    LASTFIND *lf;

    lf = (LASTFIND *) Hdl;
    if (!lf->elem)
        return FALSE;
    lf->elem = lf->elem->next_elem;
    if (!lf->elem)
        return FALSE;

    strcpy(FindData->cFileName, lf->elem->name);
    FindData->dwFileAttributes = lf->elem->attrib;
    if (lf->elem->type == ELEM_FILE)
    {
        CPMFILE *t = (CPMFILE *) lf->elem;
        FindData->nFileSizeLow = t->Size;
    }
    return TRUE;
}

/*
㤠 䠩
*/
BOOL plg_DeleteFile(char* RemoteName)
{
    DISK  *disk;
    DEVICE *dev;
    USER  *user;
    int    nDir;
    DIRREC dir[DIRINSEC];
    UINT16 i, j;
    BOOL   update;
    UINT16 nSec = 0;
    CPMFILE *file = (CPMFILE *) elem_GetLast(RemoteName);

    if ((!file) || (file->elem.type != ELEM_FILE))
    {
        log_Print("    *error file_Delete() - bad path \"%s\"\n", RemoteName);
        return FALSE;
    }
    user = elem_Get(ELEM_USER, file);
    disk = elem_Get(ELEM_DISK, file);
    dev = elem_Get(ELEM_DEVICE, file);
    if ((!user) || (!disk) || (!dev))
    {
        log_Print("    *error file_Delete() - bad path \"%s\"\n", RemoteName);
        return FALSE;
    }
    nDir = disk->MaxDirRec;
    while (nDir > 0)
    {
        // 㦠 । ᥪ
        if (!ide_ReadSector(dev->handle, disk->StartSector + nSec, &dir))
        {
            log_Print("    *error file_Delete() - can't read directory sector at 0x%08X\n", disk->StartSector+nSec);
            return FALSE;
        }
        update = FALSE;
        // ᪠㥬 । ᥪ
        i = 0;
        while ((i < DIRINSEC) && (nDir > 0))
        {
            if (dir[i].user == user->user_no)
            {
                if (fcpm_CompareName(dir[i].name, file->Orig))
                {
                    // ᢮  
                    update = TRUE;          // ᥪ 㦭 १
                    dir[i].user = 0xE5;     // 砥 ४  ⢨⥫쭮
                    map_Free(disk->DirMap, disk->MaxDirRec-nDir);
                    for (j = 0; j < 8; j++)
                        if (dir[i].map[j] >= disk->DirBlocks)
                        {
                            if (map_Get(disk->BlockMap, dir[i].map[j]) > 0)
                                map_Free(disk->BlockMap, dir[i].map[j]);
                        }
                }
            }
            nDir--;
            i++;
        }
        if (update)
            ide_WriteSector(dev->handle, disk->StartSector + nSec, &dir);
        nSec++;
    } // while
    // ᪫砥  ᯨ᪠
    elem_DeleteFile(file);
    disk_UserUpCase(user);
    return TRUE;
}



int plg_GetFile(char* RemoteName, char* LocalName, int CopyFlags, RemoteInfoStruct* ri)
{
    CPMFILE *src;
    HANDLE  *dst;
    char    *buff;
    DWORD    nWritten;

    if (!(src = (CPMFILE *) elem_GetLast(RemoteName)))
        return FS_FILE_NOTFOUND;

    if ( (FileExist(LocalName) && !(CopyFlags & (FS_COPYFLAGS_OVERWRITE | FS_COPYFLAGS_RESUME))))
        return FS_FILE_EXISTSRESUMEALLOWED;


    if (CopyFlags & FS_COPYFLAGS_OVERWRITE)
    {
        // 室 ந १
        dst = CreateFile(LocalName, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS , ri->Attr, NULL);
        if (dst == INVALID_HANDLE_VALUE)
        {
            log_Print("    *error plg_GetFile() - can`t overwrite file \"%s\"\n", LocalName);
            return FS_FILE_OK;
        }

    } else if(CopyFlags & FS_COPYFLAGS_RESUME) {
        // 室 ந 
        dst = CreateFile(LocalName, GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
        if (dst == INVALID_HANDLE_VALUE)
        {
            log_Print("    *error plg_GetFile() - can`t open file \"%s\"\n", LocalName);
            return FS_FILE_OK;
        }
    } else {
        //⢨,  室 ந  ᯮ  १  
        dst = CreateFile(LocalName, GENERIC_WRITE, 0, NULL, CREATE_NEW, ri->Attr, NULL);
        if (dst == INVALID_HANDLE_VALUE)
        {
            log_Print("    *error plg_GetFile() - can`t create file \"%s\"\n", LocalName);
            return FS_FILE_OK;
        }
    }

    // 㥬
    if ((buff = file_Read(src, NULL)) == NULL)
    {
        log_Print("    *error plg_GetFile() - can`t read from \"%s\"\n", RemoteName);
        CloseHandle(dst);
        return FS_FILE_READERROR;
    }

    WriteFile(dst, buff, src->Size, &nWritten, NULL);
    CloseHandle(dst);
    free(buff);
    // ४㥬  ਡ
    SetFileAttributes(LocalName, fcpm_GetAttrib(src->Orig));

    if (CopyFlags & FS_COPYFLAGS_MOVE)
    {
        plg_DeleteFile(RemoteName);
    }
    return FS_FILE_OK;
}


int plg_PutFile(char* LocalName, char* RemoteName, int CopyFlags)
{
    char Name[MAX_PATH];

    CPMFILE *fil = (CPMFILE *) elem_GetLast(RemoteName);
    USER    *path;
    char    *lBuff;
    DWORD    lSize;
    DWORD    Attr;

    //   ࠧ 室饣  ᪥ 䠩
    char    *rBuff = NULL;

    log_Print("  *info plg_PutFile(%s)\n", RemoteName);
    // ஢塞 稥 饣 䠩  ᪥
    if ((fil) && (!(CopyFlags & (FS_COPYFLAGS_OVERWRITE | FS_COPYFLAGS_RESUME))))
        return FS_FILE_EXISTSRESUMEALLOWED;

    // ᮧ  祭
    if ((path = SplitRemoteName(RemoteName, &Name)) == NULL)
        return FS_FILE_WRITEERROR;

    // 㦠 室 䠩  ᪠  砥  ਡ
    if ((lBuff = FileLoad(LocalName, &lSize)) == NULL)
        return FS_FILE_READERROR;
    Attr = GetFileAttributes(LocalName);

    if (lSize == 0)
        return FS_FILE_OK;

    if (CopyFlags & FS_COPYFLAGS_RESUME)
    {
        // 塞   饣
        if ((rBuff = malloc(lSize + fil->Size + 512*2)) == NULL)
        {
            log_Print("    *error plg_PutFile() - not enought memory!\n");
            free(lBuff);
            return FS_FILE_OK;
        }
        memset(rBuff, 0, lSize + fil->Size + 512*2);
        if (file_Read(fil, rBuff) == NULL)
        {
            log_Print("    *error plg_PutFile() - can`t read file \"%s\"!\n", fil->elem.name);
            free(lBuff);
            free(rBuff);
            return FS_FILE_OK;
        }
        memcpy(rBuff+fil->Size, lBuff, lSize);
        free(lBuff);
        lBuff = rBuff;
        lSize += fil->Size;
        Attr = fil->elem.attrib;        // 㤥 ᯮ짮 ਡ 饣
        plg_DeleteFile(RemoteName);
    } else if (CopyFlags & FS_COPYFLAGS_OVERWRITE)
    {
        // १뢠 騩 䠩
        plg_DeleteFile(RemoteName);
    }

    // 襬  
    if (!file_Write(path, Name, lBuff, lSize, Attr))
        return FS_FILE_WRITEERROR;

    disk_UserUpCase(path);
    return FS_FILE_OK;
}


/*
஢/६饭 䠩
*/
int plg_RenMovFile(char* OldName, char* NewName, BOOL Move, BOOL OverWrite, RemoteInfoStruct* ri)
{
    char Name[MAX_PATH];
    CPMFILE *src = (CPMFILE *) elem_GetLast(OldName);
    CPMFILE *dst = (CPMFILE *) elem_GetLast(NewName);
    USER    *srcPath;
    USER    *dstPath;
    char    *buff;
    DWORD    nSize;
    DWORD    Attr;

    if ((!src) || (src->elem.type != ELEM_FILE))
        return FS_FILE_NOTSUPPORTED;

    Attr = src->elem.attrib;

    if (dst && !OverWrite)
        return FS_FILE_EXISTS;

    srcPath = elem_Get(ELEM_USER, src);

    // ᮧ  祭
    if ((dstPath = SplitRemoteName(NewName, &Name)) == NULL)
        return FS_FILE_WRITEERROR;

    // 㦠 室 䠩  ᪠
    buff = file_Read(src, NULL);
    nSize= src->Size;

    if (OverWrite)
    {
        // १뢠 騩 䠩
        plg_DeleteFile(NewName);
    }

    // 㥬
    if (!file_Write(dstPath, Name, buff, nSize, Attr))
        return FS_FILE_WRITEERROR;

    if (Move)
    {
        plg_DeleteFile(OldName);
    }

    disk_UserUpCase(srcPath);
    disk_UserUpCase(dstPath);
    return FS_FILE_OK;
}


