#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Keyboard emulation (part of sPycialist - Specialist PC Emulator)
# (C) Stanislav Yudin (CityAceE)
# http://zx-pk.ru

import numpy as np


keys = {
  # Key  [   C O L U M N    ] [  R O W  ]
    282: [0b1000, 0b00000000, 0b10000000],  # F - F1
    283: [0b0100, 0b00000000, 0b10000000],  # HELP - F2
    284: [0b0010, 0b00000000, 0b10000000],  # NEW - F3
    285: [0b0001, 0b00000000, 0b10000000],  # LOAD - F4
    286: [0b0000, 0b10000000, 0b10000000],  # SAVE - F5
    287: [0b0000, 0b01000000, 0b10000000],  # RUN - F6
    288: [0b0000, 0b00100000, 0b10000000],  # STOP - F7
    289: [0b0000, 0b00010000, 0b10000000],  # CONT - F8
    290: [0b0000, 0b00001000, 0b10000000],  # EDIT - F9
    291: [0b0000, 0b00000100, 0b10000000],  # СФ - F10
    292: [0b0000, 0b00000010, 0b10000000],  # ТФ - F11
    293: [0b0000, 0b00000001, 0b10000000],  # НФ - F12

    61:  [0b1000, 0b00000000, 0b01000000],  # ; + - +
    49:  [0b0100, 0b00000000, 0b01000000],  # 1 ! - 1
    50:  [0b0010, 0b00000000, 0b01000000],  # 2 " - 2
    51:  [0b0001, 0b00000000, 0b01000000],  # 3 # - 3
    52:  [0b0000, 0b10000000, 0b01000000],  # 4 $ - 4
    53:  [0b0000, 0b01000000, 0b01000000],  # 5 % - 5
    54:  [0b0000, 0b00100000, 0b01000000],  # 6 & - 6
    55:  [0b0000, 0b00010000, 0b01000000],  # 7 ' - 7
    56:  [0b0000, 0b00001000, 0b01000000],  # 8 ( - 8
    57:  [0b0000, 0b00000100, 0b01000000],  # 9 ) - 9
    48:  [0b0000, 0b00000010, 0b01000000],  # 0 - 0
    45:  [0b0000, 0b00000001, 0b01000000],  # - = - -

    113: [0b1000, 0b00000000, 0b00100000],  # Й J - Й
    119: [0b0100, 0b00000000, 0b00100000],  # Ц C - Ц
    101: [0b0010, 0b00000000, 0b00100000],  # У U - У
    114: [0b0001, 0b00000000, 0b00100000],  # К K - К
    116: [0b0000, 0b10000000, 0b00100000],  # Е E - Е
    121: [0b0000, 0b01000000, 0b00100000],  # Н N - Н
    117: [0b0000, 0b00100000, 0b00100000],  # Г G - Г
    105: [0b0000, 0b00010000, 0b00100000],  # Ш [ - Ш
    111: [0b0000, 0b00001000, 0b00100000],  # Щ ] - Щ
    112: [0b0000, 0b00000100, 0b00100000],  # З Z - З
    91:  [0b0000, 0b00000010, 0b00100000],  # Х H - H
    93:  [0b0000, 0b00000001, 0b00100000],  # : * - Ъ

    97:  [0b1000, 0b00000000, 0b00010000],  # Ф F - Ф
    115: [0b0100, 0b00000000, 0b00010000],  # Ы Y - Ы
    100: [0b0010, 0b00000000, 0b00010000],  # В W - В
    102: [0b0001, 0b00000000, 0b00010000],  # А A - А
    103: [0b0000, 0b10000000, 0b00010000],  # П P - П
    104: [0b0000, 0b01000000, 0b00010000],  # Р R - Р
    106: [0b0000, 0b00100000, 0b00010000],  # О O - О
    107: [0b0000, 0b00010000, 0b00010000],  # Л L - Л
    108: [0b0000, 0b00001000, 0b00010000],  # Д D - Д
    59:  [0b0000, 0b00000100, 0b00010000],  # Ж V - Ж
    39:  [0b0000, 0b00000010, 0b00010000],  # Э \ - Э
    92:  [0b0000, 0b00000001, 0b00010000],  # . > - \

    122: [0b1000, 0b00000000, 0b00001000],  # Я Q - Я
    120: [0b0100, 0b00000000, 0b00001000],  # Ч ^ - Ч
    99:  [0b0010, 0b00000000, 0b00001000],  # С S - С
    118: [0b0001, 0b00000000, 0b00001000],  # М M - М
    98:  [0b0000, 0b10000000, 0b00001000],  # И I - И
    110: [0b0000, 0b01000000, 0b00001000],  # Т T - Т
    109: [0b0000, 0b00100000, 0b00001000],  # Ь X - Ь
    44:  [0b0000, 0b00010000, 0b00001000],  # Б B - Б
    46:  [0b0000, 0b00001000, 0b00001000],  # Ю @ - Ю
    47:  [0b0000, 0b00000100, 0b00001000],  # , < - ?
    303: [0b0000, 0b00000010, 0b00001000],  # / ? - Right Shift
    8:   [0b0000, 0b00000001, 0b00001000],  # ЗБ - Backspace

    301: [0b1000, 0b00000000, 0b00000100],  # НРФ - CapsLock
    277: [0b0100, 0b00000000, 0b00000100],  # HOME - Home
    273: [0b0010, 0b00000000, 0b00000100],  # ВВЕРХ - Up
    274: [0b0001, 0b00000000, 0b00000100],  # ВНИЗ - Down
    000: [0b0000, 0b10000000, 0b00000100],  #
    000: [0b0000, 0b01000000, 0b00000100],  #
    32:  [0b0000, 0b00100000, 0b00000100],  # ПРОБЕЛ - Space
    276: [0b0000, 0b00010000, 0b00000100],  # ВЛЕВО - Left
    307: [0b0000, 0b00001000, 0b00000100],  # ПВ - Rigth Alt
    275: [0b0000, 0b00000100, 0b00000100],  # ВПРАВО - Right
    281: [0b0000, 0b00000010, 0b00000100],  # ПС - PageDown
    13:  [0b0000, 0b00000001, 0b00000100],  # ВК - Enter

    304: [0b0000, 0b00000000, 0b00000010]}  # НР - Left Shift


kb_matrix = [[0, 0],
             [0, 0],
             [0, 0],
             [0, 0],
             [0, 0],
             [0, 0],
             [0, 0]]

vv55a_mode = 0x82
# kb_ports = bytearray([0x00, 0x00, 0xf0])
kb_mem = bytearray([0x00, 0x00, 0x00])

def converter(code):
    if code == 2:
        return 6
    elif code == 4:
        return 5
    elif code == 8:
        return 4
    elif code == 16:
        return 3
    elif code == 32:
        return 2
    elif code == 64:
        return 1
    elif code == 128:
        return 0


def keydown(code):
    if code in keys:
        i = converter(keys[code][2])
        kb_matrix[i][0] |= keys[code][0]
        kb_matrix[i][1] |= keys[code][1]


def keyup(code):
    if code in keys:
        if code in keys:
            i = converter(keys[code][2])
            kb_matrix[i][0] &= 255 - keys[code][0]
            kb_matrix[i][1] &= 255 - keys[code][1]


def kb_ports(port):
    if not port:
        byte = 0
        for i in range(6):
            byte |= kb_matrix[i][1]
        return byte
    elif port == 1:
        byte = 0
        if kb_matrix[0][0] or kb_matrix[0][1]:
            byte += 128
        if kb_matrix[1][0] or kb_matrix[1][1]:
            byte += 64
        if kb_matrix[2][0] or kb_matrix[2][1]:
            byte += 32
        if kb_matrix[3][0] or kb_matrix[3][1]:
            byte += 16
        if kb_matrix[4][0] or kb_matrix[4][1]:
            byte += 8
        if kb_matrix[5][0] or kb_matrix[5][1]:
            byte += 4
        if kb_matrix[6][0] or kb_matrix[6][1]:
            byte += 2
        return byte
    elif port == 2:
        byte = 0
        for i in range(6):
            byte |= kb_matrix[i][0]
        return byte


def read_kb_ports(addr):
    if vv55a_mode == 0x91:
        if not (addr % 4):
            if kb_ports(1) & kb_mem[1]:
                byte = 0
                if (kb_mem[1] & 0b10000000) and kb_matrix[0][1]:
                    byte = kb_matrix[0][1]
                if (kb_mem[1] & 0b01000000) and kb_matrix[1][1]:
                    byte |= kb_matrix[1][1]
                if (kb_mem[1] & 0b00100000) and kb_matrix[2][1]:
                    byte |= kb_matrix[2][1]
                if (kb_mem[1] & 0b00010000) and kb_matrix[3][1]:
                    byte |= kb_matrix[3][1]
                if (kb_mem[1] & 0b00001000) and kb_matrix[4][1]:
                    byte |= kb_matrix[4][1]
                if (kb_mem[1] & 0b00000100) and kb_matrix[5][1]:
                    byte |= kb_matrix[5][1]
                return 0xff - byte
            else:
                return 0xff
        elif addr % 4 == 1:
            return 0xff - kb_mem[1]
        elif addr % 4 == 2:
            if kb_ports(1) & kb_mem[1]:
                byte = 0
                if (kb_mem[1] & 0b10000000) and kb_matrix[0][0]:
                    byte = kb_matrix[0][0]
                if (kb_mem[1] & 0b01000000) and kb_matrix[1][0]:
                    byte |= kb_matrix[1][0]
                if (kb_mem[1] & 0b00100000) and kb_matrix[2][0]:
                    byte |= kb_matrix[2][0]
                if (kb_mem[1] & 0b00010000) and kb_matrix[3][0]:
                    byte |= kb_matrix[3][0]
                if (kb_mem[1] & 0b00001000) and kb_matrix[4][0]:
                    byte |= kb_matrix[4][0]
                if (kb_mem[1] & 0b00000100) and kb_matrix[5][0]:
                    byte |= kb_matrix[5][0]
                return 0xff - byte
            else:
                return 0x0f
    elif vv55a_mode == 0x82:
        if not (addr % 4):
            return 0xff - kb_mem[0]
        elif addr % 4 == 1:
            if kb_ports(0) & kb_mem[0]:
                byte = 0
                if kb_mem[0] & kb_matrix[0][1]:
                    byte += 128
                if kb_mem[0] & kb_matrix[1][1]:
                    byte += 64
                if kb_mem[0] & kb_matrix[2][1]:
                    byte += 32
                if kb_mem[0] & kb_matrix[3][1]:
                    byte += 16
                if kb_mem[0] & kb_matrix[4][1]:
                    byte += 8
                if kb_mem[0] & kb_matrix[5][1]:
                    byte += 4
                if kb_mem[0] & kb_matrix[6][1]:
                    byte += 2
                return 0xff - byte
            elif (kb_ports(2) & kb_mem[2]) % 0x10:
                byte = 0
                if kb_mem[2] & kb_matrix[0][0]:
                    byte += 128
                if kb_mem[2] & kb_matrix[1][0]:
                    byte += 64
                if kb_mem[2] & kb_matrix[2][0]:
                    byte += 32
                if kb_mem[2] & kb_matrix[3][0]:
                    byte += 16
                if kb_mem[2] & kb_matrix[4][0]:
                    byte += 8
                if kb_mem[2] & kb_matrix[5][0]:
                    byte += 4
                if kb_mem[2] & kb_matrix[6][0]:
                    byte += 2
                return 0xff - byte
            else:
                return 0xff - (kb_ports(1) & 2)
        elif addr % 4 == 2:
            return 0xff - kb_mem[2]


def write_kb_ports(addr, byte):
    global vv55a_mode
    if addr % 4 != 3:
        kb_mem[(addr % 4)] = 0xff - byte
    elif byte == 0x91:
        vv55a_mode = byte
        kb_mem[1] = 0xff
    elif byte == 0x82:
        vv55a_mode = byte
        kb_mem[0] = 0xff
        kb_mem[2] = 0xff
