﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace AYPlayer
{
    /**
         * Z80 Class emulates a Z80's CPU behaviour
         * By Elfic One (2006)
         *
         *  V1.00	+ Port to CSharp based on version 1.50 of CPUZ80 written in 2001 by E.Duijs as a part of JEmu
	 *  V2.00	More corrections + fixed undocumented by EvgenRU 2014-2019
         * 
         */

    public delegate void OutB(byte porth, byte portl, byte value);
    public delegate byte InB(byte porth, byte portl);


    public class Z80
    {
        public static OutB outb = null;
        public static InB inb = null;

        /// registers
        public byte I, R, A, F, B, C, D, E, H, L, IXH, IXL, IYH, IYL;
        public UInt16 AF1, BC1, DE1, HL1;
        public UInt16 PC;
        public UInt16 SP = 0;

        /// 64K Memory
        public byte[] memory = new byte[0x10000];

        /// interrupt
        public byte IM = 0; 	// interrupt mode
        public bool IFF0 = false; // IRQ interrupt flip-flop
        public bool IFF1 = false; // NMI interrupt flip-flop
        public bool IRQ = false; // interrupt request
        public bool NMI = false; // Non-maskable interrupt

        // Current cycle of execution
        public byte cycle = 0;
        public int cycleCounter = 0;
        public int totalCounter = 0;
        public bool halted = false;
        public bool INTblocked = false;

        // Misc
        private static UInt16 tmp1;
        private static int result;
        private static byte tmp;

        // flag tables
        private static byte[] SZ = new byte[256];
        private static byte[] SZP = new byte[256];
        private static byte[] SZHV_inc = new byte[256];
        private static byte[] SZHV_dec = new byte[256];

        // FLAGS BITS
        private static byte SF = 0x80;
        private static byte ZF = 0x40;
        //private static byte YF = 0x20;
        private static byte HF = 0x10;
        //private static byte XF = 0x08;
        private static byte VPF = 0x04;
        private static byte NF = 0x02;
        private static byte CF = 0x01;
        private static byte CHF = 0x11;
        private static byte CVPF = 0x05;
        //private static byte ZHF = 0x50;
        //private static byte SNF = 0x82;
        //private static byte ZNF = 0x42;
        private static byte HNF = 0x12;
        private static byte ZVPF = 0x44;
        private static byte SZVPF = (byte)(SF | ZF | VPF);
        private static byte SZCF = (byte)(SF | ZF | CF);
        private static byte XYF = 0x28;
        //private static byte HXYF = 0x38;
        // INVERTED FLAGS BITS
        private static byte NSF = 0x7F;
        private static byte NZF = 0xBF;
        private static byte NYF = 0xDF;
        private static byte NHF = 0xEF;
        private static byte NXF = 0xF7;
        private static byte NVPF = 0xFB;
        private static byte NNF = 0xFD;
        private static byte NCF = 0xFE;
        private static byte NZNF = (byte)(NZF & NNF);
        private static byte NHNF = (byte)(NHF & NNF);
        private static byte NCHNF = (byte)(NCF & NHNF);
        private static byte NXYF = (byte)(NXF & NYF);
        private static byte NSZHF = (byte)(NSF & NZF & NHF);

        /// ////////////////////////////////////////
        public UInt16 BC
        {
            get { return (UInt16)((B << 8) | C); }
            set { B = (byte)(value >> 8); C = (byte)value; }
        }
        public UInt16 DE
        {
            get { return (UInt16)((D << 8) | E); }
            set { D = (byte)(value >> 8); E = (byte)value; }
        }
        public UInt16 HL
        {
            get { return (UInt16)((H << 8) | L); }
            set { H = (byte)(value >> 8); L = (byte)value; }
        }
        public UInt16 IX
        {
            get { return (UInt16)((IXH << 8) | IXL); }
            set { IXH = (byte)(value >> 8); IXL = (byte)value; }
        }
        public UInt16 IY
        {
            get { return (UInt16)((IYH << 8) | IYL); }
            set { IYH = (byte)(value >> 8); IYL = (byte)value; }
        }
        public UInt16 AF
        {
            get { return (UInt16)((A << 8) | F); }
            set { A = (byte)(value >> 8); F = (byte)value; }
        }
        /// ////////////////////////////////////////


        static Z80()
        {
            ////////////////////////////////////////////////////////////////////////////////////////////
            /// PRE-CALCULATE FLAGS FOR FAST ALU OPERATIONS
            ////////////////////////////////////////////////////////////////////////////////////////////
            for (Int16 i = 0; i <= 255; i++)
            {
                // parity calculation
                int p = i ^ (i >> 4);
                p ^= (p >> 2);
                p ^= p >> 1;
                p = ((p & 1) << 2) ^ VPF;

                SZ[i] = (i != 0) ? (byte)(i & (SF|XYF)) : ZF;
                SZP[i] = (byte)(SZ[i] | p);

                SZHV_inc[i] = SZ[i];
                if (i == 0x80) SZHV_inc[i] |= VPF;
                if ((i & 0x0f) == 0x00) SZHV_inc[i] |= HF;

                SZHV_dec[i] = (byte)(SZ[i] | NF);
                if (i == 0x7f) SZHV_dec[i] |= VPF;
                if ((i & 0x0f) == 0x0f) SZHV_dec[i] |= HF;
            }
        }
        ////////////////////////////////////////////////////////////////////////////////////////////

        ///  Default constructor
        public Z80() {}

        /// irq request
        public void Irq()
        {
            IRQ = true;
            checkInterrupt();
        }

        /// Non-Maskable irq request
        public void Nmi()
        {
            this.NMI = true;
        }

        /// Reset CPU - Only resets the registers
        public void Reset()
        {
            PC = 0; I = R = 0;
            BC = DE = HL = BC1 = DE1 = HL1 = 0;
            AF1 = AF = SP = IX = IY = 0xFFFF;
            cycle = 0;
            cycleCounter = 0;
            halted = false;
            INTblocked = IRQ = IFF0 = IFF1 = false;
            totalCounter = 0;
        }


        /// Execute one op-code
        public void Exec(int cycles)
        {
            while (((cycles==0 || cycleCounter < cycles) && !halted) || INTblocked)
            {
                //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                cycle = 0;
                INTblocked = false;
                ////////////////////////////////////////////////////////
                /// MAIN OPCODES
                switch (memory[PC++])
                {
                    case 0x00: cycle += 4; break;                                               // NOP
                    case 0x01: cycle += 10; C = memory[PC++]; B = memory[PC++]; break; 	        // LD BC,nn
                    case 0x02: cycle += 7; memory[(B << 8) | C] = A; break; 	                // LD (BC),A
                    case 0x03: cycle += 6; if (C == 255) { B++; C = 0; } else C++; break; 	    // INC BC
                    case 0x04: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++B]); break;         // INC B
                    case 0x05: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--B]); break;		    // DEC B
                    case 0x06: cycle += 7; B = memory[PC++]; break;	                            // LD B,n
                    case 0x07: cycle += 4; tmp = ((A & 0x80) != 0) ? CF : (byte)0; A = (byte)((A << 1) | tmp); F = (byte)((F & SZVPF) | (A & XYF) | tmp); break;		                                // RLCA
                    case 0x08: cycle += 4; tmp1 = AF; AF = AF1; AF1 = tmp1; break;	            // EX AF,AF'
                    case 0x09: cycle += 11; tmp1 = (UInt16)((B << 8) | C); result = ((H << 8) | L) + tmp1; F = (byte)((F & SZVPF) | (((((H << 8) | L) ^ result ^ tmp1) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF)); L = (byte)result; H = (byte)(result >> 8); break;	            // ADD HL,BC
                    case 0x0a: cycle += 7; A = memory[(B<<8) | C]; break;	                    // LD A,(BC)
                    case 0x0b: cycle += 6; if (C == 0) { B--; C = 255; } else C--; break;	    // DEC BC
                    case 0x0C: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++C]); break;		    // INC C
                    case 0x0D: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--C]); break;		    // DEC C
                    case 0x0e: cycle += 7; C = memory[PC++]; break;	                            // LD C,n
                    case 0x0f: cycle += 4; tmp = (byte)(A & CF); A = (byte)((A >> 1) | (tmp << 7)); F = (byte)((F & SZVPF) | (A & XYF) | tmp); break;		                                // RRCA
                    case 0x10: cycle += 8; if (--B != 0) { cycle += 5; PC += (UInt16)(sbyte)memory[PC]; } PC++; break;        // DJNZ,n
                    case 0x11: cycle += 10; E = memory[PC++]; D = memory[PC++]; break; 	        // LD DE,nn
                    case 0x12: cycle += 7; memory[(D << 8) | E]=A; break;	                                // LD (DE),A
                    case 0x13: cycle += 6; if (E == 255) { D++; E = 0; } else E++; break;       // INC DE
                    case 0x14: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++D]); break;		    // INC D
                    case 0x15: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--D]); break;		    // DEC D
                    case 0x16: cycle += 7; D = memory[PC++]; break;	                            // LD D,n
                    case 0x17: cycle += 4; tmp = ((A & 0x80) != 0) ? CF : (byte)0; A = (byte)((A << 1) | (F & CF)); F = (byte)((F & SZVPF) | (A & XYF) | tmp); break;		                                // RLA
                    case 0x18: cycle += 12; PC += (UInt16)(sbyte)memory[PC]; PC++; break;                      // JR e
                    case 0x19: cycle += 11; tmp1 = (UInt16)((D << 8) | E); result = ((H << 8) | L) + tmp1; F = (byte)((F & SZVPF) | (((((H << 8) | L) ^ result ^ tmp1) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF)); HL = (UInt16)result; break;	                                // ADD HL,DE
                    case 0x1a: cycle += 7; A = memory[(D << 8) | E]; break;	                            // LD A,(DE)
                    case 0x1b: cycle += 6; if (E == 0) { D--; E = 255; } else E--; break; 	    // DEC DE
                    case 0x1C: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++E]); break;         // INC E
                    case 0x1D: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--E]); break; 	    // DEC E
                    case 0x1e: cycle += 7; E = memory[PC++]; break; 	                        // LD E,n
                    case 0x1f: cycle += 4; tmp = (byte)(A & CF); A = (byte)((A >> 1) | ((F & CF) << 7)); F = (byte)((F & SZVPF) | (A & XYF) | tmp); break; 		                                // RRA
                    case 0x20: cycle += 7; if ((F & ZF) == 0) { cycle += 5; PC += (UInt16)(sbyte)memory[PC]; } PC++; break;// JR NZ,e
                    case 0x21: cycle += 10; L = memory[PC++]; H = memory[PC++]; break; 	        // LD HL,nn
                    case 0x22: cycle += 16; pokew(peekw(PC), (UInt16)((H << 8) | L)); PC += 2; break;	            // LD (nn),HL
                    case 0x23: cycle += 6; if (L == 255) { H++; L = 0; } else L++; break; 	    // INC HL
                    case 0x24: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++H]); break;         // INC H
                    case 0x25: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--H]); break;		    // DEC H
                    case 0x26: cycle += 7; H = memory[PC++]; break; 	                        // LD H,n
                    case 0x27: cycle += 4; daa(); break;		                                // DAA
                    case 0x28: cycle += 7; if ((F & ZF) != 0) { cycle += 5; PC += (UInt16)(sbyte)memory[PC]; } PC++; break;	  // JR Z,e
                    case 0x29: cycle += 11; tmp1 = (UInt16)((H << 8) | L); result = ((H << 8) | L) + tmp1; F = (byte)((F & SZVPF) | (((((H << 8) | L) ^ result ^ tmp1) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF)); HL = (UInt16)result; break;	            // ADD HL,HL
                    case 0x2a: cycle += 16; HL = peekw(peekw(PC)); PC += 2; break;	            // LD HL,(nn) correct!!!
                    case 0x2b: cycle += 6; if (L == 0) { H--; L = 255; } else L--; break; 	    // DEC HL
                    case 0x2c: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++L]); break;         // INC L
                    case 0x2d: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--L]); break; 	    // DEC L
                    case 0x2e: cycle += 7; L = memory[PC++]; break; 	                        // LD L,n
                    case 0x2f: cycle += 4; A ^= 0xFF; F = (byte)((F & (SZVPF | CF)) | HNF | (A & XYF)); break;		                                // CPL
                    case 0x30: cycle += 7; if ((F & CF) == 0) { cycle += 5; PC += (UInt16)(sbyte)memory[PC]; } PC++; break;	 // JR NC,e
                    case 0x31: cycle += 10; SP = (UInt16)(memory[PC++] | (memory[PC] << 8)); PC++; break; 	                // LD SP,nn
                    case 0x32: cycle += 13; memory[peekw(PC)]=A; PC += 2; break;                // LD (nn),A
                    case 0x33: cycle += 6; SP++; break;	                                        // INC SP
                    case 0x34: cycle += 11; F = (byte)((F & CF) | SZHV_inc[++memory[(H << 8) | L]]); break; // INC (HL)
                    case 0x35: cycle += 11; F = (byte)((F & CF) | SZHV_dec[--memory[(H << 8) | L]]); break; // DEC (HL)
                    case 0x36: cycle += 10; memory[(H << 8) | L] = memory[PC++]; break; 	                    // LD (HL),n
                    case 0x37: cycle += 4; F = (byte)((F & (SZVPF)) | CF | (A & XYF)); break;	// SCF
                    case 0x38: cycle += 7; if ((F & CF) != 0) { cycle += 5; PC += (UInt16)(sbyte)memory[PC]; } PC++; break;    // JR C,e
                    case 0x39: cycle += 11; result = ((H << 8) | L) + SP; F = (byte)((F & SZVPF) | (((((H << 8) | L) ^ result ^ SP) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF)); HL = (UInt16)result; break;                                   // ADD HL,SP
                    case 0x3a: cycle += 13; A = memory[peekw(PC)]; PC += 2; break;	            // LD A,(nn) correct!!!
                    case 0x3b: cycle += 6; SP--; break; 	                                    // DEC SP
                    case 0x3C: cycle += 4; F = (byte)((F & CF) | SZHV_inc[++A]); break;         // INC A
                    case 0x3D: cycle += 4; F = (byte)((F & CF) | SZHV_dec[--A]); break;		    // DEC A
                    case 0x3e: cycle += 7; A = memory[PC++]; break;	                            // LD A,n
                    case 0x3f: cycle += 4; F = (byte)(((F & (SZVPF | CF)) | (A & XYF) | ((F << 4) & HF)) ^ CF); break;		                                // CCF
                    case 0x40: cycle += 4; break;	                                            // LD B,B
                    case 0x41: cycle += 4; B = C; break; 	                                    // LD B,C
                    case 0x42: cycle += 4; B = D; break; 	                                    // LD B,D
                    case 0x43: cycle += 4; B = E; break; 	                                    // LD B,E
                    case 0x44: cycle += 4; B = H; break; 	                                    // LD B,H
                    case 0x45: cycle += 4; B = L; break; 	                                    // LD B,L
                    case 0x46: cycle += 7; B = memory[(H << 8) | L]; break; 	                            // LD B,(HL)
                    case 0x47: cycle += 4; B = A; break; 	                                    // LD B,A
                    case 0x48: cycle += 4; C = B; break;	                                    // LD C,B
                    case 0x49: cycle += 4; break; 	                                            // LD C,C
                    case 0x4a: cycle += 4; C = D; break; 	                                    // LD C,D
                    case 0x4b: cycle += 4; C = E; break; 	                                    // LD C,E
                    case 0x4c: cycle += 4; C = H; break;                                        // LD C,H
                    case 0x4d: cycle += 4; C = L; break;                                        // LD C,L
                    case 0x4e: cycle += 7; C = memory[(H << 8) | L]; break;                               // LD C,(HL)
                    case 0x4f: cycle += 4; C = A; break;                                        // LD C,A
                    case 0x50: cycle += 4; D = B; break;                                        // LD D,B
                    case 0x51: cycle += 4; D = C; break;                                        // LD D,C
                    case 0x52: cycle += 4; break;	                                            // LD D,D
                    case 0x53: cycle += 4; D = E; break;	                                    // LD D,E
                    case 0x54: cycle += 4; D = H; break;                                        // LD D,H
                    case 0x55: cycle += 4; D = L; break;                                        // LD D,L
                    case 0x56: cycle += 7; D = memory[(H << 8) | L]; break; 	                            // LD D,(HL)
                    case 0x57: cycle += 4; D = A; break;	                                    // LD D,A
                    case 0x58: cycle += 4; E = B; break; 	                                    // LD E,B
                    case 0x59: cycle += 4; E = C; break; 	                                    // LD E,C
                    case 0x5a: cycle += 4; E = D; break; 	                                    // LD E,D
                    case 0x5b: cycle += 4; break; 	                                            // LD E,E
                    case 0x5c: cycle += 4; E = H; break; 	                                    // LD E,H
                    case 0x5d: cycle += 4; E = L; break; 	                                    // LD E,L
                    case 0x5e: cycle += 7; E = memory[(H << 8) | L]; break; 	                            // LD E,(HL)
                    case 0x5f: cycle += 4; E = A; break; 	                                    // LD E,A
                    case 0x60: cycle += 4; H = B; break;	                                    // LD H,B
                    case 0x61: cycle += 4; H = C; break;	                                    // LD H,C
                    case 0x62: cycle += 4; H = D; break;	                                    // LD H,D
                    case 0x63: cycle += 4; H = E; break;	                                    // LD H,E
                    case 0x64: cycle += 4; break; 	                                            // LD H,H
                    case 0x65: cycle += 4; H = L; break;                                        // LD H,L
                    case 0x66: cycle += 7; H = memory[(H << 8) | L]; break;                               // LD H,(HL)
                    case 0x67: cycle += 4; H = A; break;                                        // LD H,A
                    case 0x68: cycle += 4; L = B; break;                                        // LD L,B
                    case 0x69: cycle += 4; L = C; break;                                        // LD L,C
                    case 0x6a: cycle += 4; L = D; break;                                        // LD L,D
                    case 0x6b: cycle += 4; L = E; break;                                        // LD L,E
                    case 0x6c: cycle += 4; L = H; break;                                        // LD L,H
                    case 0x6d: cycle += 4; break; 	                                            // LD L,L
                    case 0x6e: cycle += 7; L = memory[(H << 8) | L]; break;                               // LD L,(HL)
                    case 0x6f: cycle += 4; L = A; break;                                        // LD L,A
                    case 0x70: cycle += 7; memory[(H << 8) | L] = B; break;                                 // LD (HL),B
                    case 0x71: cycle += 7; memory[(H << 8) | L] = C; break;                                 // LD (HL),C
                    case 0x72: cycle += 7; memory[(H << 8) | L] = D; break;                                 // LD (HL),D
                    case 0x73: cycle += 7; memory[(H << 8) | L] = E; break;                                 // LD (HL),E
                    case 0x74: cycle += 7; memory[(H << 8) | L] = H; break;                                 // LD (HL),H
                    case 0x75: cycle += 7; memory[(H << 8) | L] = L; break;                                 // LD (HL),L
                    case 0x76: cycle += 4; halted = true; PC--; break;		                    // HALT
                    case 0x77: cycle += 7; memory[(H << 8) | L] = A; break;                                 // LD (HL),A
                    case 0x78: cycle += 4; A = B; break; 	                                    // LD A,B
                    case 0x79: cycle += 4; A = C; break; 	                                    // LD A,C
                    case 0x7a: cycle += 4; A = D; break; 	                                    // LD A,D
                    case 0x7b: cycle += 4; A = E; break; 	                                    // LD A,E
                    case 0x7c: cycle += 4; A = H; break; 	                                    // LD A,H
                    case 0x7d: cycle += 4; A = L; break; 	                                    // LD A,L
                    case 0x7e: cycle += 7; A = memory[(H << 8) | L]; break; 	                            // LD A,(HL)
                    case 0x7f: cycle += 4; break;                                               // LD A,A
                    case 0x80: cycle += 4; addA(B); break;	                                    // ADD A,B
                    case 0x81: cycle += 4; addA(C); break; 	                                    // ADD A,C
                    case 0x82: cycle += 4; addA(D); break; 	                                    // ADD A,D
                    case 0x83: cycle += 4; addA(E); break; 	                                    // ADD A,E
                    case 0x84: cycle += 4; addA(H); break; 	                                    // ADD A,H
                    case 0x85: cycle += 4; addA(L); break; 	                                    // ADD A,L
                    case 0x86: cycle += 7; addA(memory[(H << 8) | L]); break; 	                        // ADD A,(HL)
                    case 0x87: cycle += 4; addA(A); break; 	                                    // ADD A,A
                    case 0x88: cycle += 4; adcA(B); break;	                                    // ADC A,B
                    case 0x89: cycle += 4; adcA(C); break;	                                    // ADC A,C
                    case 0x8a: cycle += 4; adcA(D); break;	                                    // ADC A,D
                    case 0x8b: cycle += 4; adcA(E); break;	                                    // ADC A,E
                    case 0x8c: cycle += 4; adcA(H); break;	                                    // ADC A,H
                    case 0x8d: cycle += 4; adcA(L); break;	                                    // ADC A,L
                    case 0x8e: cycle += 7; adcA(memory[(H << 8) | L]); break; 	                        // ADC A,(HL)
                    case 0x8f: cycle += 4; adcA(A); break;	                                    // ADC A,A
                    case 0x90: cycle += 4; subA(B); break; 	                                    // SUB A,B
                    case 0x91: cycle += 4; subA(C); break; 	                                    // SUB A,C
                    case 0x92: cycle += 4; subA(D); break; 	                                    // SUB A,D
                    case 0x93: cycle += 4; subA(E); break; 	                                    // SUB A,E
                    case 0x94: cycle += 4; subA(H); break; 	                                    // SUB A,H
                    case 0x95: cycle += 4; subA(L); break; 	                                    // SUB A,L
                    case 0x96: cycle += 7; subA(memory[(H << 8) | L]); break; 	                        // SUB A,(HL)
                    case 0x97: cycle += 4; subA(A); break; 	                                    // SUB A,A
                    case 0x98: cycle += 4; sbcA(B); break; 	                                    // SBC A,B
                    case 0x99: cycle += 4; sbcA(C); break; 	                                    // SBC A,C
                    case 0x9a: cycle += 4; sbcA(D); break; 	                                    // SBC A,D
                    case 0x9b: cycle += 4; sbcA(E); break; 	                                    // SBC A,E
                    case 0x9c: cycle += 4; sbcA(H); break; 	                                    // SBC A,H
                    case 0x9d: cycle += 4; sbcA(L); break; 	                                    // SBC A,L
                    case 0x9e: cycle += 7; sbcA(memory[(H << 8) | L]); break; 	                        // SBC A,(HL)
                    case 0x9f: cycle += 4; sbcA(A); break; 	                                    // SBC A,A
                    case 0xa0: cycle += 4; A &= B; F = (byte)(SZP[A] | HF); break; 	            // AND B
                    case 0xa1: cycle += 4; A &= C; F = (byte)(SZP[A] | HF); break; 	            // AND C
                    case 0xa2: cycle += 4; A &= D; F = (byte)(SZP[A] | HF); break; 	            // AND D
                    case 0xa3: cycle += 4; A &= E; F = (byte)(SZP[A] | HF); break; 	            // AND E
                    case 0xa4: cycle += 4; A &= H; F = (byte)(SZP[A] | HF); break; 	            // AND H
                    case 0xa5: cycle += 4; A &= L; F = (byte)(SZP[A] | HF); break; 	            // AND L
                    case 0xa6: cycle += 7; A &= memory[(H << 8) | L]; F = (byte)(SZP[A] | HF); break;     // AND (HL)
                    case 0xa7: cycle += 4; F = (byte)(SZP[A] | HF); break; 	                    // AND A
                    case 0xa8: cycle += 4; A ^= B; F = SZP[A]; break; 	                        // XOR B
                    case 0xa9: cycle += 4; A ^= C; F = SZP[A]; break; 	                        // XOR C
                    case 0xaa: cycle += 4; A ^= D; F = SZP[A]; break; 	                        // XOR D
                    case 0xab: cycle += 4; A ^= E; F = SZP[A]; break; 	                        // XOR E
                    case 0xac: cycle += 4; A ^= H; F = SZP[A]; break; 	                        // XOR H
                    case 0xad: cycle += 4; A ^= L; F = SZP[A]; break; 	                        // XOR L
                    case 0xae: cycle += 7; A ^= memory[(H << 8) | L]; F = SZP[A]; break; 	                // XOR (HL)
                    case 0xaf: cycle += 4; A = 0; F = SZP[A]; break;                            // XOR A
                    case 0xb0: cycle += 4; A |= B; F = SZP[A]; break; 	                        // OR B
                    case 0xb1: cycle += 4; A |= C; F = SZP[A]; break; 	                        // OR C
                    case 0xb2: cycle += 4; A |= D; F = SZP[A]; break; 	                        // OR D
                    case 0xb3: cycle += 4; A |= E; F = SZP[A]; break; 	                        // OR E
                    case 0xb4: cycle += 4; A |= H; F = SZP[A]; break; 	                        // OR H
                    case 0xb5: cycle += 4; A |= L; F = SZP[A]; break; 	                        // OR L
                    case 0xb6: cycle += 7; A |= memory[(H << 8) | L]; F = SZP[A]; break;                  // OR (HL)
                    case 0xb7: cycle += 4; F = SZP[A]; break; 	                                // OR A
                    case 0xb8: cycle += 4; result = A - B; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ B) & HF)); if (B > A) F |= CVPF; break;	                                    // CP B
                    case 0xb9: cycle += 4; result = A - C; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ C) & HF)); if (C > A) F |= CVPF; break;	                                    // CP C
                    case 0xba: cycle += 4; result = A - D; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ D) & HF)); if (D > A) F |= CVPF; break;	                                    // CP D
                    case 0xbb: cycle += 4; result = A - E; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ E) & HF)); if (E > A) F |= CVPF; break;	                                    // CP E
                    case 0xbc: cycle += 4; result = A - H; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ H) & HF)); if (H > A) F |= CVPF; break;	                                    // CP H
                    case 0xbd: cycle += 4; result = A - L; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ L) & HF)); if (L > A) F |= CVPF; break;	                                    // CP L
                    case 0xbe: cycle += 7; cpA(memory[(H << 8) | L]); break;  	                        // CP A,(HL)
                    case 0xbf: cycle += 4; F = (byte)(ZF | NF); break;	// CP A
                    case 0xc0: cycle += 5; if ((F & ZF) == 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;     // RET NZ
                    case 0xc1: cycle += 10; C = memory[SP++]; B = memory[SP++]; break; 	                // POP BC
                    case 0xc2: cycle += 10; if ((F & ZF) == 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	 // JP NZ,nn
                    case 0xc3: cycle += 10; PC = peekw(PC); break; 	                            // JP nn
                    case 0xc4: cycle += 10; if ((F & ZF) == 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;  // CALL NZ,nn
                    case 0xc5: cycle += 11; memory[--SP] = B; memory[--SP] = C; break;                  // PUSH BC
                    case 0xc6: cycle += 7; addA(memory[PC++]); break; 	                        // ADD A,n
                    case 0xc7: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x00; break; 	        // RST $00
                    case 0xc8: cycle += 5; if ((F & ZF) != 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;	// RET Z
                    case 0xc9: cycle += 10; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); break; 	 // RET
                    case 0xca: cycle += 10; if ((F & ZF) != 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; // JP Z,nn
                    //////////////////////////////////
                    // Opcodes with $CB prefix
                    case 0xcb:
                        {
                            //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                            cycle += 8;
                            switch (memory[PC++])
                            {
                                /////////////////////////////
                                /// RLC instructions
                                ///
                                case 0x00: B = rlc(B); break;
                                case 0x01: C = rlc(C); break;
                                case 0x02: D = rlc(D); break;
                                case 0x03: E = rlc(E); break;
                                case 0x04: H = rlc(H); break;
                                case 0x05: L = rlc(L); break;
                                case 0x06: cycle += 7; memory[(H << 8) | L] = rlc(memory[(H << 8) | L]); break;
                                case 0x07: A = rlc(A); break;
                                /////////////////////////////
                                /// RRC instructions
                                ///
                                case 0x08: B = rrc(B); break;
                                case 0x09: C = rrc(C); break;
                                case 0x0a: D = rrc(D); break;
                                case 0x0b: E = rrc(E); break;
                                case 0x0c: H = rrc(H); break;
                                case 0x0d: L = rrc(L); break;
                                case 0x0e: cycle += 7; memory[(H << 8) | L] = rrc(memory[(H << 8) | L]); break;
                                case 0x0f: A = rrc(A); break;
                                /////////////////////////////
                                /// RL instructions
                                ///
                                case 0x10: B = rl(B); break;
                                case 0x11: C = rl(C); break;
                                case 0x12: D = rl(D); break;
                                case 0x13: E = rl(E); break;
                                case 0x14: H = rl(H); break;
                                case 0x15: L = rl(L); break;
                                case 0x16: cycle += 7; memory[(H << 8) | L] = rl(memory[(H << 8) | L]); break;
                                case 0x17: A = rl(A); break;
                                /////////////////////////////
                                /// RR instructions
                                ///
                                case 0x18: B = rr(B); break;
                                case 0x19: C = rr(C); break;
                                case 0x1A: D = rr(D); break;
                                case 0x1B: E = rr(E); break;
                                case 0x1C: H = rr(H); break;
                                case 0x1D: L = rr(L); break;
                                case 0x1E: cycle += 7; memory[(H << 8) | L] = rr(memory[(H << 8) | L]); break;
                                case 0x1F: A = rr(A); break;
                                /////////////////////////////
                                /// SLA instructions
                                ///
                                case 0x20: B = sla(B); break;
                                case 0x21: C = sla(C); break;
                                case 0x22: D = sla(D); break;
                                case 0x23: E = sla(E); break;
                                case 0x24: H = sla(H); break;
                                case 0x25: L = sla(L); break;
                                case 0x26: cycle += 7; memory[(H << 8) | L] = sla(memory[(H << 8) | L]); break;
                                case 0x27: A = sla(A); break;
                                /////////////////////////////
                                /// SRA instructions
                                ///
                                case 0x28: B = sra(B); break;
                                case 0x29: C = sra(C); break;
                                case 0x2a: D = sra(D); break;
                                case 0x2b: E = sra(E); break;
                                case 0x2c: H = sra(H); break;
                                case 0x2d: L = sra(L); break;
                                case 0x2e: cycle += 7; memory[(H << 8) | L] = sra(memory[(H << 8) | L]); break;
                                case 0x2f: A = sra(A); break;
                                /////////////////////////////
                                /// SLL instructions
                                ///
                                case 0x30: B = sll(B); break;
                                case 0x31: C = sll(C); break;
                                case 0x32: D = sll(D); break;
                                case 0x33: E = sll(E); break;
                                case 0x34: H = sll(H); break;
                                case 0x35: L = sll(L); break;
                                case 0x36: cycle += 7; memory[(H << 8) | L] = sll(memory[(H << 8) | L]); break;
                                case 0x37: A = sll(A); break;
                                /////////////////////////////
                                /// SRL instructions
                                ///
                                case 0x38: tmp = (byte)(B & CF); B >>= 1; F = (byte)(SZP[B] | tmp); break;
                                case 0x39: tmp = (byte)(C & CF); C >>= 1; F = (byte)(SZP[C] | tmp); break;
                                case 0x3a: tmp = (byte)(D & CF); D >>= 1; F = (byte)(SZP[D] | tmp); break;
                                case 0x3b: tmp = (byte)(E & CF); E >>= 1; F = (byte)(SZP[E] | tmp); break;
                                case 0x3c: tmp = (byte)(H & CF); H >>= 1; F = (byte)(SZP[H] | tmp); break;
                                case 0x3d: tmp = (byte)(L & CF); L >>= 1; F = (byte)(SZP[L] | tmp); break;
                                case 0x3e: cycle += 7; tmp1 = (UInt16)((H << 8) | L); tmp = (byte)(memory[tmp1] & CF); memory[tmp1] >>= 1; F = (byte)(SZP[memory[tmp1]] | tmp); break;
                                case 0x3f: tmp = (byte)(A & CF); A >>= 1; F = (byte)(SZP[A] | tmp); break;
                                /////////////////////////////
                                /// BIT instructions
                                ///
                                case 0x40: bit(0x01, B); break;                             //BIT 0,B
                                case 0x41: bit(0x01, C); break;
                                case 0x42: bit(0x01, D); break;
                                case 0x43: bit(0x01, E); break;
                                case 0x44: bit(0x01, H); break;
                                case 0x45: bit(0x01, L); break;
                                case 0x46: cycle += 4; bit(0x01, memory[(H << 8) | L]); break;
                                case 0x47: bit(0x01, A); break;
                                case 0x48: bit(0x02, B); break;                             //BIT 1,B
                                case 0x49: bit(0x02, C); break;
                                case 0x4a: bit(0x02, D); break;
                                case 0x4b: bit(0x02, E); break;
                                case 0x4c: bit(0x02, H); break;
                                case 0x4d: bit(0x02, L); break;
                                case 0x4e: cycle += 4; bit(0x02, memory[(H << 8) | L]); break;
                                case 0x4f: bit(0x02, A); break;
                                case 0x50: bit(0x04, B); break;                             //BIT 2,B
                                case 0x51: bit(0x04, C); break;
                                case 0x52: bit(0x04, D); break;
                                case 0x53: bit(0x04, E); break;
                                case 0x54: bit(0x04, H); break;
                                case 0x55: bit(0x04, L); break;
                                case 0x56: cycle += 4; bit(0x04, memory[(H << 8) | L]); break;
                                case 0x57: bit(0x04, A); break;
                                case 0x58: bit(0x08, B); break;                             //BIT 3,B
                                case 0x59: bit(0x08, C); break;
                                case 0x5a: bit(0x08, D); break;
                                case 0x5b: bit(0x08, E); break;
                                case 0x5c: bit(0x08, H); break;
                                case 0x5d: bit(0x08, L); break;
                                case 0x5e: cycle += 4; bit(0x08, memory[(H << 8) | L]); break;
                                case 0x5f: bit(0x08, A); break;
                                case 0x60: bit(0x10, B); break;                             //BIT 4,B
                                case 0x61: bit(0x10, C); break;
                                case 0x62: bit(0x10, D); break;
                                case 0x63: bit(0x10, E); break;
                                case 0x64: bit(0x10, H); break;
                                case 0x65: bit(0x10, L); break;
                                case 0x66: cycle += 4; bit(0x10, memory[(H << 8) | L]); break;
                                case 0x67: bit(0x10, A); break;
                                case 0x68: bit(0x20, B); break;                             //BIT 5,B
                                case 0x69: bit(0x20, C); break;
                                case 0x6a: bit(0x20, D); break;
                                case 0x6b: bit(0x20, E); break;
                                case 0x6c: bit(0x20, H); break;
                                case 0x6d: bit(0x20, L); break;
                                case 0x6e: cycle += 4; bit(0x20, memory[(H << 8) | L]); break;
                                case 0x6f: bit(0x20, A); break;
                                case 0x70: bit(0x40, B); break;                             //BIT 6,B
                                case 0x71: bit(0x40, C); break;
                                case 0x72: bit(0x40, D); break;
                                case 0x73: bit(0x40, E); break;
                                case 0x74: bit(0x40, H); break;
                                case 0x75: bit(0x40, L); break;
                                case 0x76: cycle += 4; bit(0x40, memory[(H << 8) | L]); break;
                                case 0x77: bit(0x40, A); break;
                                case 0x78: bit(0x80, B); break;                             //BIT 7,B
                                case 0x79: bit(0x80, C); break;                             //BIT 7,C
                                case 0x7a: bit(0x80, D); break;                             //BIT 7,D
                                case 0x7b: bit(0x80, E); break;                             //BIT 7,E
                                case 0x7c: bit(0x80, H); break;                             //BIT 7,H
                                case 0x7d: bit(0x80, L); break;                             //BIT 7,L
                                case 0x7e: cycle += 4; bit(0x80, memory[(H << 8) | L]); break;                    //BIT 7,(HL)
                                case 0x7f: bit(0x80, A); break;                             //BIT 7,A
                                /////////////////////////////
                                /// RES instructions
                                ///
                                case 0x80: B &= 0xFE; break;                                // RES 0,B
                                case 0x81: C &= 0xFE; break;                                // RES 0,C
                                case 0x82: D &= 0xFE; break;                                // RES 0,D
                                case 0x83: E &= 0xFE; break;                                // RES 0,E
                                case 0x84: H &= 0xFE; break;                                // RES 0,H
                                case 0x85: L &= 0xFE; break;                                // RES 0,L
                                case 0x86: cycle += 7; memory[(H << 8) | L] &= 0xFE; break;     // RES 0,(HL)
                                case 0x87: A &= 0xFE; break;                                // RES 0,A
                                case 0x88: B &= 0xFD; break;                                // RES 1,B
                                case 0x89: C &= 0xFD; break;                                // RES 1,C
                                case 0x8a: D &= 0xFD; break;                                // RES 1,D
                                case 0x8b: E &= 0xFD; break;                                // RES 1,E
                                case 0x8c: H &= 0xFD; break;                                // RES 1,H
                                case 0x8d: L &= 0xFD; break;                                // RES 1,L
                                case 0x8e: cycle += 7; memory[(H << 8) | L] &= 0xFD; break;     // RES 1,(HL)
                                case 0x8f: A &= 0xFD; break;                                // RES 1,A
                                case 0x90: B &= 0xFB; break;                                // RES 2,B
                                case 0x91: C &= 0xFB; break;                                // RES 2,C
                                case 0x92: D &= 0xFB; break;                                // RES 2,D
                                case 0x93: E &= 0xFB; break;                                // RES 2,E
                                case 0x94: H &= 0xFB; break;                                // RES 2,H
                                case 0x95: L &= 0xFB; break;                                // RES 2,L
                                case 0x96: cycle += 7; memory[(H << 8) | L] &= 0xFB; break;     // RES 2,(HL)
                                case 0x97: A &= 0xFB; break;                                // RES 2,A
                                case 0x98: B &= 0xF7; break;                                // RES 3,B
                                case 0x99: C &= 0xF7; break;                                // RES 3,C
                                case 0x9a: D &= 0xF7; break;                                // RES 3,D
                                case 0x9b: E &= 0xF7; break;                                // RES 3,E
                                case 0x9c: H &= 0xF7; break;                                // RES 3,H
                                case 0x9d: L &= 0xF7; break;                                // RES 3,L
                                case 0x9e: cycle += 7; memory[(H << 8) | L] &= 0xF7; break;     // RES 3,(HL)
                                case 0x9f: A &= 0xF7; break;                                // RES 3,A
                                case 0xa0: B &= 0xEF; break;                                // RES 4,B
                                case 0xa1: C &= 0xEF; break;                                // RES 4,C
                                case 0xa2: D &= 0xEF; break;                                // RES 4,D
                                case 0xa3: E &= 0xEF; break;                                // RES 4,E
                                case 0xa4: H &= 0xEF; break;                                // RES 4,H
                                case 0xa5: L &= 0xEF; break;                                // RES 4,L
                                case 0xa6: cycle += 7; memory[(H << 8) | L] &= 0xEF; break;     // RES 4,(HL)
                                case 0xa7: A &= 0xEF; break;                                // RES 4,A
                                case 0xa8: B &= 0xDF; break;                                // RES 5,B
                                case 0xa9: C &= 0xDF; break;                                // RES 5,C
                                case 0xaa: D &= 0xDF; break;                                // RES 5,D
                                case 0xab: E &= 0xDF; break;                                // RES 5,E
                                case 0xac: H &= 0xDF; break;                                // RES 5,H
                                case 0xad: L &= 0xDF; break;                                // RES 5,L
                                case 0xae: cycle += 7; memory[(H << 8) | L] &= 0xDF; break;     // RES 5,(HL)
                                case 0xaf: A &= 0xDF; break;                                // RES 5,A
                                case 0xb0: B &= 0xBF; break;                                // RES 6,B
                                case 0xb1: C &= 0xBF; break;                                // RES 6,C
                                case 0xb2: D &= 0xBF; break;                                // RES 6,D
                                case 0xb3: E &= 0xBF; break;                                // RES 6,E
                                case 0xb4: H &= 0xBF; break;                                // RES 6,H
                                case 0xb5: L &= 0xBF; break;                                // RES 6,L
                                case 0xb6: cycle += 7; memory[(H << 8) | L] &= 0xBF; break;     // RES 6,(HL)
                                case 0xb7: A &= 0xBF; break;                                // RES 6,A
                                case 0xb8: B &= 0x7F; break;                                // RES 7,B
                                case 0xb9: C &= 0x7F; break;                                // RES 7,C
                                case 0xba: D &= 0x7F; break;                                // RES 7,D
                                case 0xbb: E &= 0x7F; break;                                // RES 7,E
                                case 0xbc: H &= 0x7F; break;                                // RES 7,H
                                case 0xbd: L &= 0x7F; break;                                // RES 7,L
                                case 0xbe: cycle += 7; memory[(H << 8) | L] &= 0x7F; break;     // RES 7,(HL)
                                case 0xbf: A &= 0x7F; break;                                // RES 7,A
                                /////////////////////////////
                                /// SET instructions
                                ///
                                case 0xc0: B |= 0x01; break;                                // SET 0,B
                                case 0xc1: C |= 0x01; break;                                // SET 0,C
                                case 0xc2: D |= 0x01; break;                                // SET 0,D
                                case 0xc3: E |= 0x01; break;                                // SET 0,E
                                case 0xc4: H |= 0x01; break;                                // SET 0,H
                                case 0xc5: L |= 0x01; break;                                // SET 0,L
                                case 0xc6: cycle += 7; memory[(H << 8) | L] |= 0x01; break;     // SET 0,(HL)
                                case 0xc7: A |= 0x01; break;                                // SET 0,A
                                case 0xc8: B |= 0x02; break;                                // SET 1,B
                                case 0xc9: C |= 0x02; break;                                // SET 1,C
                                case 0xca: D |= 0x02; break;                                // SET 1,D
                                case 0xcb: E |= 0x02; break;                                // SET 1,E
                                case 0xcc: H |= 0x02; break;                                // SET 1,H
                                case 0xcd: L |= 0x02; break;                                // SET 1,L
                                case 0xce: cycle += 7; memory[(H << 8) | L] |= 0x02; break;     // SET 1,(HL)
                                case 0xcf: A |= 0x02; break;                                // SET 1,A
                                case 0xd0: B |= 0x04; break;                                // SET 2,B
                                case 0xd1: C |= 0x04; break;                                // SET 2,C
                                case 0xd2: D |= 0x04; break;                                // SET 2,D
                                case 0xd3: E |= 0x04; break;                                // SET 2,E
                                case 0xd4: H |= 0x04; break;                                // SET 2,H
                                case 0xd5: L |= 0x04; break;                                // SET 2,L
                                case 0xd6: cycle += 7; memory[(H << 8) | L] |= 0x04; break;     // SET 2,(HL)
                                case 0xd7: A |= 0x04; break;                                // SET 2,A
                                case 0xd8: B |= 0x08; break;                                // SET 3,B
                                case 0xd9: C |= 0x08; break;                                // SET 3,C
                                case 0xda: D |= 0x08; break;                                // SET 3,D
                                case 0xdb: E |= 0x08; break;                                // SET 3,E
                                case 0xdc: H |= 0x08; break;                                // SET 3,H
                                case 0xdd: L |= 0x08; break;                                // SET 3,L
                                case 0xde: cycle += 7; memory[(H << 8) | L] |= 0x08; break;     // SET 3,(HL)
                                case 0xdf: A |= 0x08; break;                                // SET 3,A
                                case 0xe0: B |= 0x10; break;                                // SET 4,B
                                case 0xe1: C |= 0x10; break;                                // SET 4,C
                                case 0xe2: D |= 0x10; break;                                // SET 4,D
                                case 0xe3: E |= 0x10; break;                                // SET 4,E
                                case 0xe4: H |= 0x10; break;                                // SET 4,H
                                case 0xe5: L |= 0x10; break;                                // SET 4,L
                                case 0xe6: cycle += 7; memory[(H << 8) | L] |= 0x10; break;     // SET 4,(HL)
                                case 0xe7: A |= 0x10; break;                                // SET 4,A
                                case 0xe8: B |= 0x20; break;                                // SET 5,B
                                case 0xe9: C |= 0x20; break;                                // SET 5,C
                                case 0xea: D |= 0x20; break;                                // SET 5,D
                                case 0xeb: E |= 0x20; break;                                // SET 5,E
                                case 0xec: H |= 0x20; break;                                // SET 5,H
                                case 0xed: L |= 0x20; break;                                // SET 5,L
                                case 0xee: cycle += 7; memory[(H << 8) | L] |= 0x20; break;     // SET 5,(HL)
                                case 0xef: A |= 0x20; break;                                // SET 5,A
                                case 0xf0: B |= 0x40; break;                                // SET 6,B
                                case 0xf1: C |= 0x40; break;                                // SET 6,C
                                case 0xf2: D |= 0x40; break;                                // SET 6,D
                                case 0xf3: E |= 0x40; break;                                // SET 6,E
                                case 0xf4: H |= 0x40; break;                                // SET 6,H
                                case 0xf5: L |= 0x40; break;                                // SET 6,L
                                case 0xf6: cycle += 7; memory[(H << 8) | L] |= 0x40; break;     // SET 6,(HL)
                                case 0xf7: A |= 0x40; break;                                // SET 6,A
                                case 0xf8: B |= 0x80; break;                                // SET 7,B
                                case 0xf9: C |= 0x80; break;                                // SET 7,C
                                case 0xfa: D |= 0x80; break;                                // SET 7,D
                                case 0xfb: E |= 0x80; break;                                // SET 7,E
                                case 0xfc: H |= 0x80; break;                                // SET 7,H
                                case 0xfd: L |= 0x80; break;                                // SET 7,L
                                case 0xfe: cycle += 7; memory[(H << 8) | L] |= 0x80; break;     // SET 7,(HL)
                                case 0xff: A |= 0x80; break;                                // SET 7,A
                            }
                        }
                        break;
                    //////////////////////////////////
                    case 0xcc: cycle += 10; if ((F & ZF) != 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;	                                            // CALL Z,nn
                    case 0xcd: cycle += 17; SP-=2; pokew(SP,(UInt16)(PC+2));PC=peekw(PC);break; // CALL nn
                    case 0xce: cycle += 7; adcA(memory[PC++]); break; 	                        // ADC A,n
                    case 0xcf: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x08; break;           // RST $08
                    case 0xd0: cycle += 5; if ((F & CF) == 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;	                                                // RET NC
                    case 0xd1: cycle += 10; E = memory[SP++]; D = memory[SP++]; break;	                    // POP DE
                    case 0xd2: cycle += 10; if ((F & CF) == 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                                            // JP NC,nn
                    case 0xd3: cycle += 11; portOut(A, memory[PC++], A); break;	                // OUT (n),A
                    case 0xd4: cycle += 10; if ((F & CF) == 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;                                             // CALL NC,nn
                    case 0xd5: cycle += 11; memory[--SP] = D; memory[--SP] = E; break; 	                    // PUSH DE
                    case 0xd6: cycle += 7; subA(memory[PC++]); break; 	                        // SUB A,n
                    case 0xd7: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x10; break;           // RST $10
                    case 0xd8: cycle += 5; if ((F & CF) != 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;		                                            // RET C
                    case 0xd9: cycle += 4; tmp1 = (UInt16)((B << 8) | C); BC = BC1; BC1 = tmp1; tmp1 = (UInt16)((D << 8) | E); DE = DE1; DE1 = tmp1; tmp1 = HL; HL = HL1; HL1 = tmp1; break; 		                                // EXX
                    case 0xda: cycle += 10; if ((F & CF) != 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                                            // JP C,nn
                    case 0xdb: cycle += 11; A = portIn_A(memory[PC++]); break;	                // IN A,(n)
                    case 0xdc: cycle += 10; if ((F & CF) != 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;                                              // CALL C,nn
                    //////////////////////////////////
                    // Opcodes with $DD prefix
                    case 0xdd:
                        {
                            switch (memory[PC++])
                            {
                                case 0x09: cycle += 15; addIX((UInt16)((B << 8) | C)); break;	                    // ADD IX,BC
                                case 0x19: cycle += 15; addIX((UInt16)((D << 8) | E)); break;	                                        // ADD IX,DE
                                case 0x21: cycle += 14; IX = peekw(PC); PC += 2; break; 	                        // LD IX,nn
                                case 0x22: cycle += 20; pokew(peekw(PC), (UInt16)(IXH << 8 | IXL)); PC += 2; break;	                    // LD (nn),IX
                                case 0x23: cycle += 10; if (IXL == 255) { IXH++; IXL = 0; } else IXL++; break; 	    // INC IX
                                case 0x24: cycle += 8; F = (byte)((F & CF) | SZHV_inc[++IXH]); break;               // INC IXh
                                case 0x25: cycle += 8; F = (byte)((F & CF) | SZHV_dec[--IXH]); break;		        // DEC IXh
                                case 0x26: cycle += 11; IXH = memory[PC++]; break; 	                                // LD IXh,n
                                case 0x29: cycle += 15; addIX((UInt16)(IXH << 8 | IXL)); break;	                                        // ADD IX,IX
                                case 0x2a: cycle += 20; IX = peekw(peekw(PC)); PC += 2; break;	                    // LD IX,(nn) correct!!!
                                case 0x2b: cycle += 10; if (IXL == 0) { IXH--; IXL = 255; } else IXL--; break;      // DEC IX
                                case 0x2c: cycle += 8; F = (byte)((F & CF) | SZHV_inc[++IXL]); break; 	            // INC IXL
                                case 0x2d: cycle += 8; F = (byte)((F & CF) | SZHV_dec[--IXL]); break; 	            // DEC IXL
                                case 0x2e: cycle += 11; IXL = memory[PC++]; break; 	                                // LD L,n
                                case 0x34: cycle += 23; F = (byte)((F & CF) | SZHV_inc[++memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]]); break;   // INC (IX+d)
                                case 0x35: cycle += 23; F = (byte)((F & CF) | SZHV_dec[--memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]]); break; 	// DEC (IX+d)
                                case 0x36: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = memory[PC++]; break; 	            // LD (IX+d),n
                                case 0x39: cycle += 15; addIX(SP); break;                                           // ADD IX,SP
                                case 0x44: cycle += 8; B = IXH; break; 	                                            // LD B,IXH
                                case 0x45: cycle += 8; B = IXL; break; 	                                            // LD B,IXL
                                case 0x46: cycle += 19; B = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break;              // LD B,(IX+d)
                                case 0x4c: cycle += 8; C = IXH; break;                                              // LD C,IXH
                                case 0x4d: cycle += 8; C = IXL; break;                                              // LD C,IXL
                                case 0x4e: cycle += 19; C = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break;              // LD C,(IX+d)
                                case 0x54: cycle += 8; D = IXH; break;                                              // LD D,IXH
                                case 0x55: cycle += 8; D = IXL; break;                                              // LD D,IXL
                                case 0x56: cycle += 19; D = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break; 	            // LD D,(IX+d)
                                case 0x5c: cycle += 8; E = IXH; break; 	                                            // LD E,IXH
                                case 0x5d: cycle += 8; E = IXL; break; 	                                            // LD E,IXL
                                case 0x5e: cycle += 19; E = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break;              // LD E,(IX+d)
                                case 0x60: cycle += 8; IXH = B; break;	                                            // LD IXH,B
                                case 0x61: cycle += 8; IXH = C; break;	                                            // LD IXH,C
                                case 0x62: cycle += 8; IXH = D; break;	                                            // LD IXH,D
                                case 0x63: cycle += 8; IXH = E; break;	                                            // LD IXH,E
                                case 0x64: cycle += 8; break;                                                       // LD IXH,IXH
                                case 0x65: cycle += 8; IXH = IXL; break;                                            // LD IXH,IXL
                                case 0x66: cycle += 19; H = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break;              // LD H,(IX+d)
                                case 0x67: cycle += 8; IXH = A; break;                                              // LD IXH,A
                                case 0x68: cycle += 8; IXL = B; break;                                              // LD IXL,B
                                case 0x69: cycle += 8; IXL = C; break;                                              // LD IXL,C
                                case 0x6a: cycle += 8; IXL = D; break;                                              // LD IXL,D
                                case 0x6b: cycle += 8; IXL = E; break;                                              // LD IXL,E
                                case 0x6c: cycle += 8; IXL = IXH; break;                                            // LD IXL,IXH
                                case 0x6d: cycle += 8; break;                                                       // LD IXL,IXL
                                case 0x6e: cycle += 19; L = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break;              // LD L,(IX+d)                
                                case 0x6f: cycle += 8; IXL = A; break;                                              // LD IXL,A
                                case 0x70: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = B; break;               // LD (IX+d),B
                                case 0x71: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = C; break;               // LD (IX+d),C
                                case 0x72: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = D; break;               // LD (IX+d),D
                                case 0x73: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = E; break;               // LD (IX+d),E
                                case 0x74: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = H; break;               // LD (IX+d),H
                                case 0x75: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = L; break;               // LD (IX+d),L
                                case 0x77: cycle += 19; memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])] = A; break;               // LD (IX+d),A
                                case 0x7c: cycle += 8; A = IXH; break; 	                                        // LD A,IXH
                                case 0x7d: cycle += 8; A = IXL; break; 	                                        // LD A,IXL
                                case 0x7e: cycle += 19; A = memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; break; 	        // LD A,(IX+d)
                                case 0x84: cycle += 8; addA(IXH); break; 	                                    // ADD A,IXH
                                case 0x85: cycle += 8; addA(IXL); break; 	                                    // ADD A,IXL
                                case 0x86: cycle += 19; addA(memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]); break; 	    // ADD A,(IX+d)
                                case 0x8c: cycle += 8; adcA(IXH); break;	                                    // ADC A,IXH
                                case 0x8d: cycle += 8; adcA(IXL); break;	                                    // ADC A,IXL
                                case 0x8e: cycle += 19; adcA(memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]); break; 	    // ADC A,(IX+d)
                                case 0x94: cycle += 8; subA(IXH); break; 	                                    // SUB A,IXH
                                case 0x95: cycle += 8; subA(IXL); break; 	                                    // SUB A,IXL
                                case 0x96: cycle += 19; subA(memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]); break; 	    // SUB A,(IX+d)
                                case 0x9c: cycle += 8; sbcA(IXH); break; 	                                    // SBC A,IXH
                                case 0x9d: cycle += 8; sbcA(IXL); break; 	                                    // SBC A,IXL
                                case 0x9e: cycle += 19; sbcA(memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]); break; 	    // SBC A,(IX+d)
                                case 0xa4: cycle += 8; A &= IXH; F = (byte)(SZP[A] | HF); break; 	                                    // AND IXH
                                case 0xa5: cycle += 8; A &= IXL; F = (byte)(SZP[A] | HF); break; 	                                    // AND IXL
                                case 0xa6: cycle += 19; A &= memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; F = (byte)(SZP[A] | HF); break;        // AND (IX+d)
                                case 0xac: cycle += 8; A ^= IXH; F = SZP[A]; break; 	                                    // XOR IXH
                                case 0xad: cycle += 8; A ^= IXL; F = SZP[A]; break; 	                                    // XOR IXL
                                case 0xae: cycle += 19; A ^= memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; F = SZP[A]; break; 	    // XOR (IX+d)
                                case 0xb4: cycle += 8; A |= IXH; F = SZP[A]; break; 	                        // OR IXH
                                case 0xb5: cycle += 8; A |= IXL; F = SZP[A]; break; 	                            // OR IXL
                                case 0xb6: cycle += 19; A |= memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]; F = SZP[A]; break; 	    // OR (IX+d)
                                case 0xbc: cycle += 8; result = A - IXH; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ IXH) & HF)); if (IXH > A) F |= CVPF; break;	// CP IXH
                                case 0xbd: cycle += 8; result = A - IXL; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ IXL) & HF)); if (IXL > A) F |= CVPF; break;	// CP IXL
                                case 0xbe: cycle += 19; cpA(memory[(UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++])]); break;  	        // CP (IX+d)
                                //////////////////////////////////
                                // Opcodes with $CB prefix
                                case 0xcb:
                                    {
                                        //increase_R();
                                        UInt16 IXd = (UInt16)((IXH << 8 | IXL) + (sbyte)memory[PC++]);
                                        switch (memory[PC++])
                                        {
                                            /////////////////////////////
                                            /// RLC instructions
                                            ///
                                            case 0x00: cycle += 23; B = memory[IXd]; B = rlc(B); memory[IXd] = B; break; // RLC (IXd),B
                                            case 0x01: cycle += 23; C = memory[IXd]; C = rlc(C); memory[IXd] = C; break; // RLC (IXd),C
                                            case 0x02: cycle += 23; D = memory[IXd]; D = rlc(D); memory[IXd] = D; break; // RLC (IXd),D
                                            case 0x03: cycle += 23; E = memory[IXd]; E = rlc(E); memory[IXd] = E; break; // RLC (IXd),E
                                            case 0x04: cycle += 23; H = memory[IXd]; H = rlc(H); memory[IXd] = H; break; // RLC (IXd),H
                                            case 0x05: cycle += 23; L = memory[IXd]; L = rlc(L); memory[IXd] = L; break; // RLC (IXd),L
                                            case 0x06: cycle += 23; memory[IXd] = rlc(memory[IXd]); break;               // RLC (IXd)
                                            case 0x07: cycle += 23; A = memory[IXd]; A = rlc(A); memory[IXd] = A; break; // RLC (IXd),A
                                            /////////////////////////////
                                            /// RRC instructions
                                            ///
                                            case 0x08: cycle += 23; B = memory[IXd]; B = rrc(B); memory[IXd] = B; break; // RRC (IXd),B
                                            case 0x09: cycle += 23; C = memory[IXd]; C = rrc(C); memory[IXd] = C; break; // RRC (IXd),C
                                            case 0x0A: cycle += 23; D = memory[IXd]; D = rrc(D); memory[IXd] = D; break; // RRC (IXd),D
                                            case 0x0B: cycle += 23; E = memory[IXd]; E = rrc(E); memory[IXd] = E; break; // RRC (IXd),E
                                            case 0x0C: cycle += 23; H = memory[IXd]; H = rrc(H); memory[IXd] = H; break; // RRC (IXd),H
                                            case 0x0D: cycle += 23; L = memory[IXd]; L = rrc(L); memory[IXd] = L; break; // RRC (IXd),L
                                            case 0x0E: cycle += 23; memory[IXd] = rrc(memory[IXd]); break;               // RRC (IXd)
                                            case 0x0F: cycle += 23; A = memory[IXd]; A = rrc(A); memory[IXd] = A; break; // RRC (IXd),A
                                            /////////////////////////////
                                            /// RL instructions
                                            ///
                                            case 0x10: cycle += 23; B = memory[IXd]; B = rl(B); memory[IXd] = B; break; // RL (IXd),B
                                            case 0x11: cycle += 23; C = memory[IXd]; C = rl(C); memory[IXd] = C; break; // RL (IXd),C
                                            case 0x12: cycle += 23; D = memory[IXd]; D = rl(D); memory[IXd] = D; break; // RL (IXd),D
                                            case 0x13: cycle += 23; E = memory[IXd]; E = rl(E); memory[IXd] = E; break; // RL (IXd),E
                                            case 0x14: cycle += 23; H = memory[IXd]; H = rl(H); memory[IXd] = H; break; // RL (IXd),H
                                            case 0x15: cycle += 23; L = memory[IXd]; L = rl(L); memory[IXd] = L; break; // RL (IXd),L
                                            case 0x16: cycle += 23; memory[IXd] = rl(memory[IXd]); break;              // RL (IXd)
                                            case 0x17: cycle += 23; A = memory[IXd]; A = rl(A); memory[IXd] = A; break; // RL (IXd),A
                                            /////////////////////////////
                                            /// RR instructions
                                            ///
                                            case 0x18: cycle += 23; B = memory[IXd]; B = rr(B); memory[IXd] = B; break; // RR (IXd),B
                                            case 0x19: cycle += 23; C = memory[IXd]; C = rr(C); memory[IXd] = C; break; // RR (IXd),C
                                            case 0x1A: cycle += 23; D = memory[IXd]; D = rr(D); memory[IXd] = D; break; // RR (IXd),D
                                            case 0x1B: cycle += 23; E = memory[IXd]; E = rr(E); memory[IXd] = E; break; // RR (IXd),E
                                            case 0x1C: cycle += 23; H = memory[IXd]; H = rr(H); memory[IXd] = H; break; // RR (IXd),H
                                            case 0x1D: cycle += 23; L = memory[IXd]; L = rr(L); memory[IXd] = L; break; // RR (IXd),L
                                            case 0x1E: cycle += 23; memory[IXd] = rr(memory[IXd]); break;               // RR (IXd)
                                            case 0x1F: cycle += 23; A = memory[IXd]; A = rr(A); memory[IXd] = A; break; // RR (IXd),A
                                            /////////////////////////////
                                            /// SLA instructions
                                            ///
                                            case 0x20: cycle += 23; B = memory[IXd]; B = sla(B); memory[IXd] = B; break; // SLA (IXd),B
                                            case 0x21: cycle += 23; C = memory[IXd]; C = sla(C); memory[IXd] = C; break; // SLA (IXd),C
                                            case 0x22: cycle += 23; D = memory[IXd]; D = sla(D); memory[IXd] = D; break; // SLA (IXd),D
                                            case 0x23: cycle += 23; E = memory[IXd]; E = sla(E); memory[IXd] = E; break; // SLA (IXd),E
                                            case 0x24: cycle += 23; H = memory[IXd]; H = sla(H); memory[IXd] = H; break; // SLA (IXd),H
                                            case 0x25: cycle += 23; L = memory[IXd]; L = sla(L); memory[IXd] = L; break; // SLA (IXd),L
                                            case 0x26: cycle += 23; memory[IXd] = sla(memory[IXd]); break;               // SLA (IXd)
                                            case 0x27: cycle += 23; A = memory[IXd]; A = sla(A); memory[IXd] = A; break; // SLA (IXd),A
                                            /////////////////////////////
                                            /// SRA instructions
                                            ///
                                            case 0x28: cycle += 23; B = memory[IXd]; B = sra(B); memory[IXd] = B; break; // SRA (IXd),B
                                            case 0x29: cycle += 23; C = memory[IXd]; C = sra(C); memory[IXd] = C; break; // SRA (IXd),C
                                            case 0x2A: cycle += 23; D = memory[IXd]; D = sra(D); memory[IXd] = D; break; // SRA (IXd),D
                                            case 0x2B: cycle += 23; E = memory[IXd]; E = sra(E); memory[IXd] = E; break; // SRA (IXd),E
                                            case 0x2C: cycle += 23; H = memory[IXd]; H = sra(H); memory[IXd] = H; break; // SRA (IXd),H
                                            case 0x2D: cycle += 23; L = memory[IXd]; L = sra(L); memory[IXd] = L; break; // SRA (IXd),L
                                            case 0x2E: cycle += 23; memory[IXd] = sra(memory[IXd]); break;               // SRA (IXd)
                                            case 0x2F: cycle += 23; A = memory[IXd]; A = sra(A); memory[IXd] = A; break; // SRA (IXd),A
                                            /////////////////////////////
                                            /// SLL instructions
                                            ///
                                            case 0x30: cycle += 23; B = memory[IXd]; B = sll(B); memory[IXd] = B; break; // SLL (IXd),B
                                            case 0x31: cycle += 23; C = memory[IXd]; C = sll(C); memory[IXd] = C; break; // SLL (IXd),C
                                            case 0x32: cycle += 23; D = memory[IXd]; D = sll(D); memory[IXd] = D; break; // SLL (IXd),D
                                            case 0x33: cycle += 23; E = memory[IXd]; E = sll(E); memory[IXd] = E; break; // SLL (IXd),E
                                            case 0x34: cycle += 23; H = memory[IXd]; H = sll(H); memory[IXd] = H; break; // SLL (IXd),H
                                            case 0x35: cycle += 23; L = memory[IXd]; L = sll(L); memory[IXd] = L; break; // SLL (IXd),L
                                            case 0x36: cycle += 23; memory[IXd] = sll(memory[IXd]); break;               // SLL (IXd)
                                            case 0x37: cycle += 23; A = memory[IXd]; A = sll(A); memory[IXd] = A; break; // SLL (IXd),A
                                            /////////////////////////////
                                            /// SRL instructions
                                            ///
                                            case 0x38: cycle += 23; B = memory[IXd]; tmp = (byte)(B & CF); B >>= 1; F = (byte)(SZP[B] | tmp); memory[IXd] = B; break; // SRL (IXd),B
                                            case 0x39: cycle += 23; C = memory[IXd]; tmp = (byte)(C & CF); C >>= 1; F = (byte)(SZP[C] | tmp); memory[IXd] = C; break; // SRL (IXd),C
                                            case 0x3A: cycle += 23; D = memory[IXd]; tmp = (byte)(D & CF); D >>= 1; F = (byte)(SZP[D] | tmp); memory[IXd] = D; break; // SRL (IXd),D
                                            case 0x3B: cycle += 23; E = memory[IXd]; tmp = (byte)(E & CF); E >>= 1; F = (byte)(SZP[E] | tmp); memory[IXd] = E; break; // SRL (IXd),E
                                            case 0x3C: cycle += 23; H = memory[IXd]; tmp = (byte)(H & CF); H >>= 1; F = (byte)(SZP[H] | tmp); memory[IXd] = H; break; // SRL (IXd),H
                                            case 0x3D: cycle += 23; L = memory[IXd]; tmp = (byte)(L & CF); L >>= 1; F = (byte)(SZP[L] | tmp); break; // SRL (IXd),L
                                            case 0x3E: cycle += 23; tmp = (byte)(memory[IXd] & CF); memory[IXd] >>= 1; F = (byte)(SZP[memory[IXd]] | tmp); break;               // SRL (IXd)
                                            case 0x3F: cycle += 23; A = memory[IXd]; tmp = (byte)(A & CF); A >>= 1; F = (byte)(SZP[A] | tmp); memory[IXd] = A; break; // SRL (IXd),A
                                            /////////////////////////////
                                            /// BIT instructions
                                            ///
                                            case 0x40:
                                            case 0x41:
                                            case 0x42:
                                            case 0x43:
                                            case 0x44:
                                            case 0x45:
                                            case 0x46:
                                            case 0x47: cycle += 20; bit(0x01, memory[IXd]); break;         //BIT 0,IXd
                                            case 0x48:
                                            case 0x49:
                                            case 0x4a:
                                            case 0x4b:
                                            case 0x4c:
                                            case 0x4d:
                                            case 0x4e:
                                            case 0x4f: cycle += 20; bit(0x02, memory[IXd]); break;         //BIT 1,IXd
                                            case 0x50:
                                            case 0x51:
                                            case 0x52:
                                            case 0x53:
                                            case 0x54:
                                            case 0x55:
                                            case 0x56:
                                            case 0x57: cycle += 20; bit(0x04, memory[IXd]); break;         //BIT 2,IXd
                                            case 0x58:
                                            case 0x59:
                                            case 0x5a:
                                            case 0x5b:
                                            case 0x5c:
                                            case 0x5d:
                                            case 0x5e:
                                            case 0x5f: cycle += 20; bit(0x08, memory[IXd]); break;         //BIT 3,IXd
                                            case 0x60:
                                            case 0x61:
                                            case 0x62:
                                            case 0x63:
                                            case 0x64:
                                            case 0x65:
                                            case 0x66:
                                            case 0x67: cycle += 20; bit(0x10, memory[IXd]); break;         //BIT 4,IXd
                                            case 0x68:
                                            case 0x69:
                                            case 0x6a:
                                            case 0x6b:
                                            case 0x6c:
                                            case 0x6d:
                                            case 0x6e:
                                            case 0x6f: cycle += 20; bit(0x20, memory[IXd]); break;         //BIT 5,IXd
                                            case 0x70:
                                            case 0x71:
                                            case 0x72:
                                            case 0x73:
                                            case 0x74:
                                            case 0x75:
                                            case 0x76:
                                            case 0x77: cycle += 20; bit(0x40, memory[IXd]); break;         //BIT 6,IXd
                                            case 0x78:
                                            case 0x79:
                                            case 0x7a:
                                            case 0x7b:
                                            case 0x7c:
                                            case 0x7d:
                                            case 0x7e:
                                            case 0x7f: cycle += 20; bit(0x80, memory[IXd]); break;         //BIT 7,IXd
                                            /////////////////////////////
                                            /// RES instructions
                                            ///
                                            case 0x80: cycle += 23; B = (byte)(memory[IXd] & 0xFE); memory[IXd] = B; break;      // RES 0,(XYd),B
                                            case 0x81: cycle += 23; C = (byte)(memory[IXd] & 0xFE); memory[IXd] = C; break;      // RES 0,(XYd),C
                                            case 0x82: cycle += 23; D = (byte)(memory[IXd] & 0xFE); memory[IXd] = D; break;      // RES 0,(XYd),D
                                            case 0x83: cycle += 23; E = (byte)(memory[IXd] & 0xFE); memory[IXd] = E; break;      // RES 0,(XYd),E
                                            case 0x84: cycle += 23; H = (byte)(memory[IXd] & 0xFE); memory[IXd] = H; break;      // RES 0,(XYd),H
                                            case 0x85: cycle += 23; L = (byte)(memory[IXd] & 0xFE); memory[IXd] = L; break;      // RES 0,(XYd),L
                                            case 0x86: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xFE); break;             // RES 0,(XYd)
                                            case 0x87: cycle += 23; A = (byte)(memory[IXd] & 0xFE); memory[IXd] = A; break;      // RES 0,(XYd),A                
                                            case 0x88: cycle += 23; B = (byte)(memory[IXd] & 0xFD); memory[IXd] = B; break;      // RES 1,(XYd),B
                                            case 0x89: cycle += 23; C = (byte)(memory[IXd] & 0xFD); memory[IXd] = C; break;      // RES 1,(XYd),C
                                            case 0x8A: cycle += 23; D = (byte)(memory[IXd] & 0xFD); memory[IXd] = D; break;      // RES 1,(XYd),D
                                            case 0x8B: cycle += 23; E = (byte)(memory[IXd] & 0xFD); memory[IXd] = E; break;      // RES 1,(XYd),E
                                            case 0x8C: cycle += 23; H = (byte)(memory[IXd] & 0xFD); memory[IXd] = H; break;      // RES 1,(XYd),H
                                            case 0x8D: cycle += 23; L = (byte)(memory[IXd] & 0xFD); memory[IXd] = L; break;      // RES 1,(XYd),L
                                            case 0x8E: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xFD); break;             // RES 1,(XYd)
                                            case 0x8F: cycle += 23; A = (byte)(memory[IXd] & 0xFD); memory[IXd] = A; break;      // RES 1,(XYd),A                
                                            case 0x90: cycle += 23; B = (byte)(memory[IXd] & 0xFB); memory[IXd] = B; break;      // RES 2,(XYd),B
                                            case 0x91: cycle += 23; C = (byte)(memory[IXd] & 0xFB); memory[IXd] = C; break;      // RES 2,(XYd),C
                                            case 0x92: cycle += 23; D = (byte)(memory[IXd] & 0xFB); memory[IXd] = D; break;      // RES 2,(XYd),D
                                            case 0x93: cycle += 23; E = (byte)(memory[IXd] & 0xFB); memory[IXd] = E; break;      // RES 2,(XYd),E
                                            case 0x94: cycle += 23; H = (byte)(memory[IXd] & 0xFB); memory[IXd] = H; break;      // RES 2,(XYd),H
                                            case 0x95: cycle += 23; L = (byte)(memory[IXd] & 0xFB); memory[IXd] = L; break;      // RES 2,(XYd),L
                                            case 0x96: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xFB); break;             // RES 2,(XYd)
                                            case 0x97: cycle += 23; A = (byte)(memory[IXd] & 0xFB); memory[IXd] = A; break;      // RES 2,(XYd),A                
                                            case 0x98: cycle += 23; B = (byte)(memory[IXd] & 0xF7); memory[IXd] = B; break;      // RES 3,(XYd),B
                                            case 0x99: cycle += 23; C = (byte)(memory[IXd] & 0xF7); memory[IXd] = C; break;      // RES 3,(XYd),C
                                            case 0x9A: cycle += 23; D = (byte)(memory[IXd] & 0xF7); memory[IXd] = D; break;      // RES 3,(XYd),D
                                            case 0x9B: cycle += 23; E = (byte)(memory[IXd] & 0xF7); memory[IXd] = E; break;      // RES 3,(XYd),E
                                            case 0x9C: cycle += 23; H = (byte)(memory[IXd] & 0xF7); memory[IXd] = H; break;      // RES 3,(XYd),H
                                            case 0x9D: cycle += 23; L = (byte)(memory[IXd] & 0xF7); memory[IXd] = L; break;      // RES 3,(XYd),L
                                            case 0x9E: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xF7); break;             // RES 3,(XYd)
                                            case 0x9F: cycle += 23; A = (byte)(memory[IXd] & 0xF7); memory[IXd] = A; break;      // RES 3,(XYd),A                
                                            case 0xA0: cycle += 23; B = (byte)(memory[IXd] & 0xEF); memory[IXd] = B; break;      // RES 4,(XYd),B
                                            case 0xA1: cycle += 23; C = (byte)(memory[IXd] & 0xEF); memory[IXd] = C; break;      // RES 4,(XYd),C
                                            case 0xA2: cycle += 23; D = (byte)(memory[IXd] & 0xEF); memory[IXd] = D; break;      // RES 4,(XYd),D
                                            case 0xA3: cycle += 23; E = (byte)(memory[IXd] & 0xEF); memory[IXd] = E; break;      // RES 4,(XYd),E
                                            case 0xA4: cycle += 23; H = (byte)(memory[IXd] & 0xEF); memory[IXd] = H; break;      // RES 4,(XYd),H
                                            case 0xA5: cycle += 23; L = (byte)(memory[IXd] & 0xEF); memory[IXd] = L; break;      // RES 4,(XYd),L
                                            case 0xA6: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xEF); break;             // RES 4,(XYd)
                                            case 0xA7: cycle += 23; A = (byte)(memory[IXd] & 0xEF); memory[IXd] = A; break;      // RES 4,(XYd),A                
                                            case 0xA8: cycle += 23; B = (byte)(memory[IXd] & 0xDF); memory[IXd] = B; break;      // RES 5,(XYd),B
                                            case 0xA9: cycle += 23; C = (byte)(memory[IXd] & 0xDF); memory[IXd] = C; break;      // RES 5,(XYd),C
                                            case 0xAA: cycle += 23; D = (byte)(memory[IXd] & 0xDF); memory[IXd] = D; break;      // RES 5,(XYd),D
                                            case 0xAB: cycle += 23; E = (byte)(memory[IXd] & 0xDF); memory[IXd] = E; break;      // RES 5,(XYd),E
                                            case 0xAC: cycle += 23; H = (byte)(memory[IXd] & 0xDF); memory[IXd] = H; break;      // RES 5,(XYd),H
                                            case 0xAD: cycle += 23; L = (byte)(memory[IXd] & 0xDF); memory[IXd] = L; break;      // RES 5,(XYd),L
                                            case 0xAE: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xDF); break;             // RES 5,(XYd)
                                            case 0xAF: cycle += 23; A = (byte)(memory[IXd] & 0xDF); memory[IXd] = A; break;      // RES 5,(XYd),A                
                                            case 0xB0: cycle += 23; B = (byte)(memory[IXd] & 0xBF); memory[IXd] = B; break;      // RES 6,(XYd),B
                                            case 0xB1: cycle += 23; C = (byte)(memory[IXd] & 0xBF); memory[IXd] = C; break;      // RES 6,(XYd),C
                                            case 0xB2: cycle += 23; D = (byte)(memory[IXd] & 0xBF); memory[IXd] = D; break;      // RES 6,(XYd),D
                                            case 0xB3: cycle += 23; E = (byte)(memory[IXd] & 0xBF); memory[IXd] = E; break;      // RES 6,(XYd),E
                                            case 0xB4: cycle += 23; H = (byte)(memory[IXd] & 0xBF); memory[IXd] = H; break;      // RES 6,(XYd),H
                                            case 0xB5: cycle += 23; L = (byte)(memory[IXd] & 0xBF); memory[IXd] = L; break;      // RES 6,(XYd),L
                                            case 0xB6: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0xBF); break;             // RES 6,(XYd)
                                            case 0xB7: cycle += 23; A = (byte)(memory[IXd] & 0xBF); memory[IXd] = A; break;      // RES 6,(XYd),A                
                                            case 0xB8: cycle += 23; B = (byte)(memory[IXd] & 0x7F); memory[IXd] = B; break;      // RES 7,(XYd),B
                                            case 0xB9: cycle += 23; C = (byte)(memory[IXd] & 0x7F); memory[IXd] = C; break;      // RES 7,(XYd),C
                                            case 0xBA: cycle += 23; D = (byte)(memory[IXd] & 0x7F); memory[IXd] = D; break;      // RES 7,(XYd),D
                                            case 0xBB: cycle += 23; E = (byte)(memory[IXd] & 0x7F); memory[IXd] = E; break;      // RES 7,(XYd),E
                                            case 0xBC: cycle += 23; H = (byte)(memory[IXd] & 0x7F); memory[IXd] = H; break;      // RES 7,(XYd),H
                                            case 0xBD: cycle += 23; L = (byte)(memory[IXd] & 0x7F); memory[IXd] = L; break;      // RES 7,(XYd),L
                                            case 0xBE: cycle += 23; memory[IXd] = (byte)(memory[IXd] & 0x7F); break;             // RES 7,(XYd)
                                            case 0xBF: cycle += 23; A = (byte)(memory[IXd] & 0x7F); memory[IXd] = A; break;      // RES 7,(XYd),A                
                                            /////////////////////////////
                                            /// SET instructions
                                            ///
                                            case 0xC0: cycle += 23; B = (byte)(memory[IXd] | 0x01); memory[IXd] = B; break;      // SET 0,(XYd),B
                                            case 0xC1: cycle += 23; C = (byte)(memory[IXd] | 0x01); memory[IXd] = C; break;      // SET 0,(XYd),C
                                            case 0xC2: cycle += 23; D = (byte)(memory[IXd] | 0x01); memory[IXd] = D; break;      // SET 0,(XYd),D
                                            case 0xC3: cycle += 23; E = (byte)(memory[IXd] | 0x01); memory[IXd] = E; break;      // SET 0,(XYd),E
                                            case 0xC4: cycle += 23; H = (byte)(memory[IXd] | 0x01); memory[IXd] = H; break;      // SET 0,(XYd),H
                                            case 0xC5: cycle += 23; L = (byte)(memory[IXd] | 0x01); memory[IXd] = L; break;      // SET 0,(XYd),L
                                            case 0xC6: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x01); break;             // SET 0,(XYd)
                                            case 0xC7: cycle += 23; A = (byte)(memory[IXd] | 0x01); memory[IXd] = A; break;      // SET 0,(XYd),A
                                            case 0xC8: cycle += 23; B = (byte)(memory[IXd] | 0x02); memory[IXd] = B; break;      // SET 1,(XYd),B
                                            case 0xC9: cycle += 23; C = (byte)(memory[IXd] | 0x02); memory[IXd] = C; break;      // SET 1,(XYd),C
                                            case 0xCA: cycle += 23; D = (byte)(memory[IXd] | 0x02); memory[IXd] = D; break;      // SET 1,(XYd),D
                                            case 0xCB: cycle += 23; E = (byte)(memory[IXd] | 0x02); memory[IXd] = E; break;      // SET 1,(XYd),E
                                            case 0xCC: cycle += 23; H = (byte)(memory[IXd] | 0x02); memory[IXd] = H; break;      // SET 1,(XYd),H
                                            case 0xCD: cycle += 23; L = (byte)(memory[IXd] | 0x02); memory[IXd] = L; break;      // SET 1,(XYd),L
                                            case 0xCE: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x02); break;             // SET 1,(XYd)
                                            case 0xCF: cycle += 23; A = (byte)(memory[IXd] | 0x02); memory[IXd] = A; break;      // SET 1,(XYd),A
                                            case 0xD0: cycle += 23; B = (byte)(memory[IXd] | 0x04); memory[IXd] = B; break;      // SET 2,(XYd),B
                                            case 0xD1: cycle += 23; C = (byte)(memory[IXd] | 0x04); memory[IXd] = C; break;      // SET 2,(XYd),C
                                            case 0xD2: cycle += 23; D = (byte)(memory[IXd] | 0x04); memory[IXd] = D; break;      // SET 2,(XYd),D
                                            case 0xD3: cycle += 23; E = (byte)(memory[IXd] | 0x04); memory[IXd] = E; break;      // SET 2,(XYd),E
                                            case 0xD4: cycle += 23; H = (byte)(memory[IXd] | 0x04); memory[IXd] = H; break;      // SET 2,(XYd),H
                                            case 0xD5: cycle += 23; L = (byte)(memory[IXd] | 0x04); memory[IXd] = L; break;      // SET 2,(XYd),L
                                            case 0xD6: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x04); break;             // SET 2,(XYd)
                                            case 0xD7: cycle += 23; A = (byte)(memory[IXd] | 0x04); memory[IXd] = A; break;      // SET 2,(XYd),A
                                            case 0xD8: cycle += 23; B = (byte)(memory[IXd] | 0x08); memory[IXd] = B; break;      // SET 3,(XYd),B
                                            case 0xD9: cycle += 23; C = (byte)(memory[IXd] | 0x08); memory[IXd] = C; break;      // SET 3,(XYd),C
                                            case 0xDA: cycle += 23; D = (byte)(memory[IXd] | 0x08); memory[IXd] = D; break;      // SET 3,(XYd),D
                                            case 0xDB: cycle += 23; E = (byte)(memory[IXd] | 0x08); memory[IXd] = E; break;      // SET 3,(XYd),E
                                            case 0xDC: cycle += 23; H = (byte)(memory[IXd] | 0x08); memory[IXd] = H; break;      // SET 3,(XYd),H
                                            case 0xDD: cycle += 23; L = (byte)(memory[IXd] | 0x08); memory[IXd] = L; break;      // SET 3,(XYd),L
                                            case 0xDE: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x08); break;             // SET 3,(XYd)
                                            case 0xDF: cycle += 23; A = (byte)(memory[IXd] | 0x08); memory[IXd] = A; break;      // SET 3,(XYd),A
                                            case 0xE0: cycle += 23; B = (byte)(memory[IXd] | 0x10); memory[IXd] = B; break;      // SET 4,(XYd),B
                                            case 0xE1: cycle += 23; C = (byte)(memory[IXd] | 0x10); memory[IXd] = C; break;      // SET 4,(XYd),C
                                            case 0xE2: cycle += 23; D = (byte)(memory[IXd] | 0x10); memory[IXd] = D; break;      // SET 4,(XYd),D
                                            case 0xE3: cycle += 23; E = (byte)(memory[IXd] | 0x10); memory[IXd] = E; break;      // SET 4,(XYd),E
                                            case 0xE4: cycle += 23; H = (byte)(memory[IXd] | 0x10); memory[IXd] = H; break;      // SET 4,(XYd),H
                                            case 0xE5: cycle += 23; L = (byte)(memory[IXd] | 0x10); memory[IXd] = L; break;      // SET 4,(XYd),L
                                            case 0xE6: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x10); break;             // SET 4,(XYd)
                                            case 0xE7: cycle += 23; A = (byte)(memory[IXd] | 0x10); memory[IXd] = A; break;      // SET 4,(XYd),A
                                            case 0xE8: cycle += 23; B = (byte)(memory[IXd] | 0x20); memory[IXd] = B; break;      // SET 5,(XYd),B
                                            case 0xE9: cycle += 23; C = (byte)(memory[IXd] | 0x20); memory[IXd] = C; break;      // SET 5,(XYd),C
                                            case 0xEA: cycle += 23; D = (byte)(memory[IXd] | 0x20); memory[IXd] = D; break;      // SET 5,(XYd),D
                                            case 0xEB: cycle += 23; E = (byte)(memory[IXd] | 0x20); memory[IXd] = E; break;      // SET 5,(XYd),E
                                            case 0xEC: cycle += 23; H = (byte)(memory[IXd] | 0x20); memory[IXd] = H; break;      // SET 5,(XYd),H
                                            case 0xED: cycle += 23; L = (byte)(memory[IXd] | 0x20); memory[IXd] = L; break;      // SET 5,(XYd),L
                                            case 0xEE: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x20); break;             // SET 5,(XYd)
                                            case 0xEF: cycle += 23; A = (byte)(memory[IXd] | 0x20); memory[IXd] = A; break;      // SET 5,(XYd),A
                                            case 0xF0: cycle += 23; B = (byte)(memory[IXd] | 0x40); memory[IXd] = B; break;      // SET 6,(XYd),B
                                            case 0xF1: cycle += 23; C = (byte)(memory[IXd] | 0x40); memory[IXd] = C; break;      // SET 6,(XYd),C
                                            case 0xF2: cycle += 23; D = (byte)(memory[IXd] | 0x40); memory[IXd] = D; break;      // SET 6,(XYd),D
                                            case 0xF3: cycle += 23; E = (byte)(memory[IXd] | 0x40); memory[IXd] = E; break;      // SET 6,(XYd),E
                                            case 0xF4: cycle += 23; H = (byte)(memory[IXd] | 0x40); memory[IXd] = H; break;      // SET 6,(XYd),H
                                            case 0xF5: cycle += 23; L = (byte)(memory[IXd] | 0x40); memory[IXd] = L; break;      // SET 6,(XYd),L
                                            case 0xF6: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x40); break;             // SET 6,(XYd)
                                            case 0xF7: cycle += 23; A = (byte)(memory[IXd] | 0x40); memory[IXd] = A; break;      // SET 6,(XYd),A
                                            case 0xF8: cycle += 23; B = (byte)(memory[IXd] | 0x80); memory[IXd] = B; break;      // SET 7,(XYd),B
                                            case 0xF9: cycle += 23; C = (byte)(memory[IXd] | 0x80); memory[IXd] = C; break;      // SET 7,(XYd),C
                                            case 0xFA: cycle += 23; D = (byte)(memory[IXd] | 0x80); memory[IXd] = D; break;      // SET 7,(XYd),D
                                            case 0xFB: cycle += 23; E = (byte)(memory[IXd] | 0x80); memory[IXd] = E; break;      // SET 7,(XYd),E
                                            case 0xFC: cycle += 23; H = (byte)(memory[IXd] | 0x80); memory[IXd] = H; break;      // SET 7,(XYd),H
                                            case 0xFD: cycle += 23; L = (byte)(memory[IXd] | 0x80); memory[IXd] = L; break;      // SET 7,(XYd),L
                                            case 0xFE: cycle += 23; memory[IXd] = (byte)(memory[IXd] | 0x80); break;             // SET 7,(XYd)
                                            case 0xFF: cycle += 23; A = (byte)(memory[IXd] | 0x80); memory[IXd] = A; break;      // SET 7,(XYd),A
                                        }
                                    }
                                    break;
                                //////////////////////////////////
                                case 0xe1: cycle += 14; IXL = memory[SP++]; IXH = memory[SP++]; break; 	                        // POP IX
                                case 0xe3: cycle += 23; tmp1 = (UInt16)(IXH << 8 | IXL); IX = peekw(SP); pokew(SP, tmp1); break;          // EX (SP),IX
                                case 0xe5: cycle += 15; memory[--SP] = IXH; memory[--SP] = IXL; break;	                            // PUSH IX
                                case 0xe9: cycle += 8; PC = (UInt16)(IXH << 8 | IXL); break;// ABSOLUTELY CORRECT! 	                    // JP (IX)
                                case 0xf9: cycle += 10; SP = (UInt16)(IXH << 8 | IXL); break;	                                            // LD SP,IX
                                //////////////////////////////////
                                case 0xdd: cycle += 4; INTblocked = true; break; // just return, not increase R
                                case 0xfd: cycle += 4; INTblocked = true; break; // just return, not increase R
                                default: cycle += 4; INTblocked = true; break; // just remove prefix and start as normal opcode
                            }
                            //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                        }
                        break;
                    //////////////////////////////////
                    case 0xde: cycle += 7; sbcA(memory[PC++]); break; 	                        // SBC A,n
                    case 0xdf: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x18; break;           // RST $18
                    case 0xe0: cycle += 5; if ((F & VPF) == 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;	                                                // RET PO
                    case 0xe1: cycle += 10; L = memory[SP++]; H = memory[SP++]; break; 	                // POP HL
                    case 0xe2: cycle += 10; if ((F & VPF) == 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                                            // JP PO,nn
                    case 0xe3: cycle += 19; tmp1 = (UInt16)((H << 8) | L); HL = peekw(SP); pokew(SP, tmp1); break;  // EX (SP),HL
                    case 0xe4: cycle += 10; if ((F & VPF) == 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;                                             // CALL PO,nn
                    case 0xe5: cycle += 11; memory[--SP] = H; memory[--SP] = L; break;	                    // PUSH HL
                    case 0xe6: cycle += 7; A &= memory[PC++]; F = (byte)(SZP[A] | HF); break;	                        // AND A,n
                    case 0xe7: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x20; break;	        // RST $20
                    case 0xe8: cycle += 5; if ((F & VPF) != 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;	                                                // RET PE
                    case 0xe9: cycle += 4; PC = (UInt16)((H << 8) | L); break;// ABSOLUTELY CORRECT! 	            // JP (HL)
                    case 0xea: cycle += 10; if ((F & VPF) != 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                                            // JP PE,nn
                    case 0xeb: cycle += 4; tmp = D; D = H; H = tmp; tmp = E; E = L; L = tmp; break; 	            // EX DE,HL
                    case 0xec: cycle += 10; if ((F & VPF) != 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else  PC += 2; break;                                             // CALL PE,nn
                    //////////////////////////////////
                    // Opcodes with $ED prefix
                    case 0xed:
                        {
                            //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                            switch (memory[PC++])
                            {
                                case 0x40: cycle += 12; B = portIn(B, C); break;	                            // IN B,(C)
                                case 0x41: cycle += 12; portOut(B, C, B); break;                                // OUT (C),B
                                case 0x42: cycle += 15; sbcHL((UInt16)((B << 8) | C)); break; 		              // SBC HL,BC
                                case 0x43: cycle += 20; pokew(peekw(PC), (UInt16)((B << 8) | C)); PC += 2; break; // LD (nn),BC
                                case 0x44: cycle += 8; neg(); break;                                            // NEG
                                case 0x45: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	        // RET N                     
                                case 0x46: cycle += 8; IM = 0; break; 			                                // IM 0
                                case 0x47: cycle += 9; I = A; break;			                                // LD I,A
                                case 0x48: cycle += 12; C = portIn(B, C); break;	                            // IN C,(C)
                                case 0x49: cycle += 12; portOut(B, C, C); break; 	                            // OUT (C),C
                                case 0x4a: cycle += 15; adcHL((UInt16)((B << 8) | C)); break; 		            // ADC HL,BC
                                case 0x4b: cycle += 20; BC = peekw(peekw(PC)); PC += 2; break;                  // LD BC,(nn)
                                case 0x4c: cycle += 8; neg(); break;                                            // NEG
                                case 0x4d: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break; 		    // RETI
                                case 0x4e: cycle += 8; IM = 3; break; 				                            // IM 0/1
                                case 0x4f: cycle += 9; R = A; break; 					                        // LD R,A
                                case 0x50: cycle += 12; D = portIn(B, C); break;			                    // IN D,(C)
                                case 0x51: cycle += 12; portOut(B, C, D); break; 			                    // OUT (C),D
                                case 0x52: cycle += 15; sbcHL((UInt16)((D << 8) | E)); break; 			                            // SBC HL,DE
                                case 0x53: cycle += 20; pokew(peekw(PC), (UInt16)((D << 8) | E)); PC += 2; break;                   // LD (nn),DE
                                case 0x54: cycle += 8; neg(); break;                                            // NEG
                                case 0x55: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	        // RET N
                                case 0x56: cycle += 8; IM = 1; break; 				                            // IM 1
                                case 0x57: cycle += 9; A = I; F = (byte)((F & CF) | SZ[A] | (A & XYF)); if (IFF1) F |= VPF; break;					                    // LD A,I
                                case 0x58: cycle += 12; E = portIn(B, C); break;			                    // IN E,(C)
                                case 0x59: cycle += 12; portOut(B, C, E); break; 			                    // OUT (C),E 
                                case 0x5a: cycle += 15; adcHL((UInt16)((D << 8) | E)); break; 			                            // ADC HL,DE
                                case 0x5b: cycle += 20; DE = peekw(peekw(PC)); PC += 2; break;                  // LD DE,(nn)
                                case 0x5c: cycle += 8; neg(); break;                                            // NEG
                                case 0x5d: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	        // RET N
                                case 0x5e: cycle += 8; IM = 2; break; 				                            // IM 2
                                case 0x5f: cycle += 9; A = R; F = (byte)((F & CF) | SZ[A] | (A & XYF)); if (IFF1) F |= VPF; break; 				                        // LD A,R
                                case 0x60: cycle += 12; H = portIn(B, C); break;			                    // IN H,(C)
                                case 0x61: cycle += 12; portOut(B, C, H); break; 			                    // OUT (C),H
                                case 0x62: cycle += 15; sbcHL((UInt16)((H << 8) | L)); break; 			                            // SBC HL,HL
                                case 0x63: cycle += 20; pokew(peekw(PC), (UInt16)((H << 8) | L)); PC += 2; break;	                // LD (nn),HL
                                case 0x64: cycle += 8; neg(); break;                                            // NEG
                                case 0x65: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	        // RET N
                                case 0x66: cycle += 8; IM = 0; break; 				                            // IM 0
                                case 0x67: cycle += 18; rrd(); break;		                                    // RRD
                                case 0x68: cycle += 12; L = portIn(B, C); break;			                    // IN L,(C)
                                case 0x69: cycle += 12; portOut(B, C, L); break; 			                    // OUT (C),L
                                case 0x6a: cycle += 15; adcHL((UInt16)((H << 8) | L)); break; 			                            // ADC HL,HL
                                case 0x6b: cycle += 20; HL = peekw(peekw(PC)); PC += 2; break;      // LD HL,(nn)
                                case 0x6c: cycle += 8; neg(); break;                                // NEG
                                case 0x6d: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	            // RET N
                                case 0x6e: cycle += 8; IM = 3; break; 				                // IM 0/1
                                case 0x6f: cycle += 18; rld(); break; 		                        // RLD
                                case 0x70: cycle += 12; portIn(B, C); break;                        // IN (c)
                                case 0x71: cycle += 12; portOut(B, C, 0); break; 			        // OUT (C),0
                                case 0x72: cycle += 15; sbcHL(SP); break; 			                // SBC HL,SP
                                case 0x73: cycle += 20; pokew(peekw(PC), SP); PC += 2; break; 	    // LD (nn),SP
                                case 0x74: cycle += 8; neg(); break;                                // NEG
                                case 0x75: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	            // RET N
                                case 0x76: cycle += 8; IM = 1; break; 				                // IM 1
                                case 0x78: cycle += 12; A = portIn(B, C); break;			        // IN A,(C)
                                case 0x79: cycle += 12; portOut(B, C, A); break; 			        // OUT (C),A
                                case 0x7a: cycle += 15; adcHL(SP); break; 			                // ADC HL,SP
                                case 0x7b: cycle += 20; SP = peekw(peekw(PC)); PC += 2; break;	    // LD SP,(nn)
                                case 0x7c: cycle += 8; neg(); break;                                // NEG
                                case 0x7d: cycle += 14; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); IFF0 = IFF1; break;	            // RET N
                                case 0x7e: cycle += 8; IM = 2; break; 				                // IM 2
                                case 0xa0: cycle += 16; memory[DE++] = memory[HL++]; F &= SZCF; if (C == 0) B--; C--; if (B != 0 || C != 0) F |= VPF; /*F |= (byte)(((byte)(outval + A) << 2) & XYF);*/ break;						        // LDI                    
                                case 0xa1: cycle += 21; cpi(); break;					            // CPI
                                case 0xa2: cycle += 16; ini(); break;					            // INI
                                case 0xa3: cycle += 16; tmp = memory[HL++]; F = (byte)((F & CF) | SZHV_dec[--B]); portOut(B, C, tmp); F = (byte)((F & NCHNF & NVPF) | ((tmp & SF) >> 6)); /*if ((SZP[(((tmp + L) & 7) ^ B)] & VPF) != 0) F |= VPF; if (tmp + L > 255) { F |= CF; F |= HF; }*/ break;					            // OUTI
                                case 0xa8: cycle += 16; memory[DE--] = memory[HL--]; F &= SZCF; if (C == 0) B--; C--; if (B != 0 || C != 0) F |= VPF; /* F |= (byte)(((byte)(tmp + A) << 2) & XYF); */ break;						        // LDD
                                case 0xa9: cycle += 16; cpd(); break;						        // CPD
                                case 0xaa: cycle += 16; ind(); break;					            // IND
                                case 0xab: cycle += 16; tmp = memory[HL--]; F = (byte)((F & CF) | SZHV_dec[--B]); portOut(B, C, tmp); F = (byte)((F & NCHNF & NVPF) | ((tmp & SF) >> 6)); /*if (outval + L > 255) F |= CHF; F |= (byte)(SZP[((((outval) + L) & 7) ^ B)] & VPF);*/ break;					            // OUTD
                                case 0xb0: cycle += 21; memory[DE++] = memory[HL++]; F &= SZCF; if (C == 0) B--; C--; if (B != 0 || C != 0) { F |= VPF; PC -= 2; cycle -= 5; } /*F |= (byte)(((byte)(tmp + A) << 2) & XYF);*/ break;                              // LDIR
                                case 0xb1: cycle += 21; cpir(); break;                              // CPIR
                                case 0xb2: cycle += 21; inir(); break;                              // INIR
                                case 0xb3: cycle += 21; otir(); break;  	                        // OTIR
                                case 0xb8: cycle += 21; lddr(); break;                              // LDDR
                                case 0xb9: cycle += 21; cpdr(); break;          			        // CPDR
                                case 0xba: cycle += 21; indr(); break;                              // INDR
                                case 0xbb: cycle += 21; otdr(); break;  	                        // OTDR
                                default: cycle += 8; break; // NO EFFECT *NOP* twice
                            }
                        }
                        break;
                    //////////////////////////////////
                    case 0xee: cycle += 7; A ^= memory[PC++]; F = SZP[A]; break; 	            // XOR A,n
                    case 0xef: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x28; break;           // RST $28
                    case 0xf0: cycle += 5; if ((F & SF) == 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;		                                            // RET P
                    case 0xf1: cycle += 10; F = memory[SP++]; A = memory[SP++]; break;	                    // POP AF
                    case 0xf2: cycle += 10; if ((F & SF) == 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                                            // JP P,nn
                    case 0xf3: cycle += 4; IFF0=IFF1=false; IRQ=false; INTblocked=true; break;  // DI
                    case 0xf4: cycle += 10; if ((F & SF) == 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;                                              // CALL P,nn
                    case 0xf5: cycle += 11; memory[--SP] = A; memory[--SP] = F; break;	                    // PUSH AF
                    case 0xf6: cycle += 7; A |= memory[PC++]; F = SZP[A]; break;		        // OR A,n
                    case 0xf7: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x30; break;           // RST $30
                    case 0xf8: cycle += 5; if ((F & SF) != 0) { cycle += 6; PC = (UInt16)(memory[SP++] | (memory[SP++] << 8)); } break;		                // RET M
                    case 0xf9: cycle += 6; SP = (UInt16)((H << 8) | L); break;	                                    // LD SP,HL
                    case 0xfa: cycle += 10; if ((F & SF) != 0) { PC = (UInt16)(memory[PC++] | (memory[PC] << 8)); } else PC += 2; break; 	                // JP M,nn
                    case 0xfb: cycle += 4; IFF0 = IFF1 = true; IRQ = false; INTblocked = true; break;		                                                // EI
                    case 0xfc: cycle += 10; if ((F & SF) != 0) { cycle += 7; SP -= 2; pokew(SP, (UInt16)(PC + 2)); PC = peekw(PC); } else PC += 2; break;	// CALL M,nn
                    //////////////////////////////////
                    // Opcodes with $FD prefix
                    case 0xfd:
                        {
                            switch (memory[PC++])
                            {
                                case 0x09: cycle += 15; addIY((UInt16)((B << 8) | C)); break;	                    // ADD IY,BC
                                case 0x19: cycle += 15; addIY((UInt16)((D << 8) | E)); break;	                                        // ADD IY,DE
                                case 0x21: cycle += 14; IY = peekw(PC); PC += 2; break; 	                        // LD IY,nn
                                case 0x22: cycle += 20; pokew(peekw(PC), IY); PC += 2; break;	                    // LD (nn),IY
                                case 0x23: cycle += 10; if (IYL == 255) { IYH++; IYL = 0; } else IYL++; break;      // INC IY
                                case 0x24: cycle += 8; F = (byte)((F & CF) | SZHV_inc[++IYH]); break;               // INC IYh
                                case 0x25: cycle += 8; F = (byte)((F & CF) | SZHV_dec[--IYH]); break;               // DEC IYh
                                case 0x26: cycle += 11; IYH = memory[PC++]; break; 	                                // LD IYh,n
                                case 0x29: cycle += 15; addIY(IY); break;	                                        // ADD IY,IY
                                case 0x2a: cycle += 20; IY = peekw(peekw(PC)); PC += 2; break;	                    // LD IY,(nn) correct!!!
                                case 0x2b: cycle += 10; if (IYL == 0) { IYH--; IYL = 255; } else IYL--; break; 	    // DEC IY
                                case 0x2c: cycle += 8; F = (byte)((F & CF) | SZHV_inc[++IYL]); break; 	            // INC IYL
                                case 0x2d: cycle += 8; F = (byte)((F & CF) | SZHV_dec[--IYL]); break;               // DEC IYL
                                case 0x2e: cycle += 11; IYL = memory[PC++]; break; 	                                // LD L,n
                                case 0x34: cycle += 23; F = (byte)((F & CF) | SZHV_inc[++memory[(UInt16)(IY + (sbyte)memory[PC++])]]); break; // INC (IY+d)
                                case 0x35: cycle += 23; F = (byte)((F & CF) | SZHV_dec[--memory[(UInt16)(IY + (sbyte)memory[PC++])]]); break; 	                // DEC (IY+d)
                                case 0x36: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = memory[PC++]; break; 	// LD (IY+d),n
                                case 0x39: cycle += 15; addIY(SP); break;                                           // ADD IY,SP
                                case 0x44: cycle += 8; B = IYH; break; 	                                            // LD B,IYH
                                case 0x45: cycle += 8; B = IYL; break; 	                                            // LD B,IYL
                                case 0x46: cycle += 19; B = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break;              // LD B,(IY+d)
                                case 0x4c: cycle += 8; C = IYH; break;                                              // LD C,IYH
                                case 0x4d: cycle += 8; C = IYL; break;                                              // LD C,IYL
                                case 0x4e: cycle += 19; C = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break;              // LD C,(IY+d)
                                case 0x54: cycle += 8; D = IYH; break;                                              // LD D,IYH
                                case 0x55: cycle += 8; D = IYL; break;                                              // LD D,IYL
                                case 0x56: cycle += 19; D = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break; 	            // LD D,(IY+d)
                                case 0x5c: cycle += 8; E = IYH; break; 	                                            // LD E,IYH
                                case 0x5d: cycle += 8; E = IYL; break; 	                                            // LD E,IYL
                                case 0x5e: cycle += 19; E = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break;              // LD E,(IY+d)
                                case 0x60: cycle += 8; IYH = B; break;	                                            // LD IYH,B
                                case 0x61: cycle += 8; IYH = C; break;	                                            // LD IYH,C
                                case 0x62: cycle += 8; IYH = D; break;	                                            // LD IYH,D
                                case 0x63: cycle += 8; IYH = E; break;	                                            // LD IYH,E
                                case 0x64: cycle += 8; break;                                                       // LD IYH,IYH
                                case 0x65: cycle += 8; IYH = IYL; break;                                            // LD IYH,IYL
                                case 0x66: cycle += 19; H = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break;              // LD H,(IY+d)
                                case 0x67: cycle += 8; IYH = A; break;                                              // LD IYH,A
                                case 0x68: cycle += 8; IYL = B; break;                                              // LD IYL,B
                                case 0x69: cycle += 8; IYL = C; break;                                              // LD IYL,C
                                case 0x6a: cycle += 8; IYL = D; break;                                              // LD IYL,D
                                case 0x6b: cycle += 8; IYL = E; break;                                              // LD IYL,E
                                case 0x6c: cycle += 8; IYL = IYH; break;                                            // LD IYL,IYH
                                case 0x6d: cycle += 8; break;                                                       // LD IYL,IYL
                                case 0x6e: cycle += 19; L = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break;              // LD L,(IY+d)                
                                case 0x6f: cycle += 8; IYL = A; break;                                              // LD IYL,A
                                case 0x70: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = B; break;               // LD (IY+d),B
                                case 0x71: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = C; break;               // LD (IY+d),C
                                case 0x72: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = D; break;               // LD (IY+d),D
                                case 0x73: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = E; break;               // LD (IY+d),E
                                case 0x74: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = H; break;               // LD (IY+d),H
                                case 0x75: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = L; break;               // LD (IY+d),L
                                case 0x77: cycle += 19; memory[(UInt16)(IY + (sbyte)memory[PC++])] = A; break;               // LD (IY+d),A
                                case 0x7c: cycle += 8; A = IYH; break; 	                                            // LD A,IYH
                                case 0x7d: cycle += 8; A = IYL; break; 	                                            // LD A,IYL
                                case 0x7e: cycle += 19; A = memory[(UInt16)(IY + (sbyte)memory[PC++])]; break; 	            // LD A,(IY+d)
                                case 0x84: cycle += 8; addA(IYH); break; 	                                        // ADD A,IYH
                                case 0x85: cycle += 8; addA(IYL); break; 	                                        // ADD A,IYL
                                case 0x86: cycle += 19; addA(memory[(UInt16)(IY + (sbyte)memory[PC++])]); break; 	        // ADD A,(IY+d)
                                case 0x8c: cycle += 8; adcA(IYH); break;	                                        // ADC A,IYH
                                case 0x8d: cycle += 8; adcA(IYL); break;	                                        // ADC A,IYL
                                case 0x8e: cycle += 19; adcA(memory[(UInt16)(IY + (sbyte)memory[PC++])]); break; 	        // ADC A,(IY+d)
                                case 0x94: cycle += 8; subA(IYH); break; 	                                        // SUB A,IYH
                                case 0x95: cycle += 8; subA(IYL); break; 	                                        // SUB A,IYL
                                case 0x96: cycle += 19; subA(memory[(UInt16)(IY + (sbyte)memory[PC++])]); break; 	        // SUB A,(IY+d)
                                case 0x9c: cycle += 8; sbcA(IYH); break; 	                                        // SBC A,IYH
                                case 0x9d: cycle += 8; sbcA(IYL); break; 	                                        // SBC A,IYL
                                case 0x9e: cycle += 19; sbcA(memory[(UInt16)(IY + (sbyte)memory[PC++])]); break; 	        // SBC A,(IY+d)
                                case 0xa4: cycle += 8; A &= IYH; F = (byte)(SZP[A] | HF); break; 	                                    // AND IYH
                                case 0xa5: cycle += 8; A &= IYL; F = (byte)(SZP[A] | HF); break; 	                                    // AND IYL
                                case 0xa6: cycle += 19; A &= memory[(UInt16)(IY + (sbyte)memory[PC++])]; F = (byte)(SZP[A] | HF); break;        // AND (IY+d)
                                case 0xac: cycle += 8; A ^= IYH; F = SZP[A]; break; 	                                    // XOR IYH
                                case 0xad: cycle += 8; A ^= IYL; F = SZP[A]; break; 	                                    // XOR IYL
                                case 0xae: cycle += 19; A ^= memory[(UInt16)(IY + (sbyte)memory[PC++])]; F = SZP[A]; break; 	    // XOR (IY+d)
                                case 0xb4: cycle += 8; A |= IYH; F = SZP[A]; break; 	                        // OR IYH
                                case 0xb5: cycle += 8; A |= IYL; F = SZP[A]; break; 	                        // OR IYL
                                case 0xb6: cycle += 19; A |= memory[(UInt16)(IY + (sbyte)memory[PC++])]; F = SZP[A]; break; 	    // OR (IY+d)
                                case 0xbc: cycle += 8; result = A - IYH; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ IYH) & HF)); if (IYH > A) F |= CVPF; break;	// CP IYH
                                case 0xbd: cycle += 8; result = A - IYL; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ IYL) & HF)); if (IYL > A) F |= CVPF; break;	// CP IYL
                                case 0xbe: cycle += 19; cpA(memory[(UInt16)(IY + (sbyte)memory[PC++])]); break;  	        // CP (IY+d)
                                //////////////////////////////////
                                // Opcodes with $CB prefix
                                case 0xcb:
                                    {
                                        //increase_R();
                                        UInt16 IYd = (UInt16)((IYH << 8 | IYL) + (sbyte)memory[PC++]);
                                        switch (memory[PC++])
                                        {
                                            /////////////////////////////
                                            /// RLC instructions
                                            ///
                                            case 0x00: cycle += 23; B = memory[IYd]; B = rlc(B); memory[IYd] = B; break; // RLC (IYd),B
                                            case 0x01: cycle += 23; C = memory[IYd]; C = rlc(C); memory[IYd] = C; break; // RLC (IYd),C
                                            case 0x02: cycle += 23; D = memory[IYd]; D = rlc(D); memory[IYd] = D; break; // RLC (IYd),D
                                            case 0x03: cycle += 23; E = memory[IYd]; E = rlc(E); memory[IYd] = E; break; // RLC (IYd),E
                                            case 0x04: cycle += 23; H = memory[IYd]; H = rlc(H); memory[IYd] = H; break; // RLC (IYd),H
                                            case 0x05: cycle += 23; L = memory[IYd]; L = rlc(L); memory[IYd] = L; break; // RLC (IYd),L
                                            case 0x06: cycle += 23; memory[IYd] = rlc(memory[IYd]); break;               // RLC (IYd)
                                            case 0x07: cycle += 23; A = memory[IYd]; A = rlc(A); memory[IYd] = A; break; // RLC (IYd),A
                                            /////////////////////////////
                                            /// RRC instructions
                                            ///
                                            case 0x08: cycle += 23; B = memory[IYd]; B = rrc(B); memory[IYd] = B; break; // RRC (IYd),B
                                            case 0x09: cycle += 23; C = memory[IYd]; C = rrc(C); memory[IYd] = C; break; // RRC (IYd),C
                                            case 0x0A: cycle += 23; D = memory[IYd]; D = rrc(D); memory[IYd] = D; break; // RRC (IYd),D
                                            case 0x0B: cycle += 23; E = memory[IYd]; E = rrc(E); memory[IYd] = E; break; // RRC (IYd),E
                                            case 0x0C: cycle += 23; H = memory[IYd]; H = rrc(H); memory[IYd] = H; break; // RRC (IYd),H
                                            case 0x0D: cycle += 23; L = memory[IYd]; L = rrc(L); memory[IYd] = L; break; // RRC (IYd),L
                                            case 0x0E: cycle += 23; memory[IYd] = rrc(memory[IYd]); break;               // RRC (IYd)
                                            case 0x0F: cycle += 23; A = memory[IYd]; A = rrc(A); memory[IYd] = A; break; // RRC (IYd),A
                                            /////////////////////////////
                                            /// RL instructions
                                            ///
                                            case 0x10: cycle += 23; B = memory[IYd]; B = rl(B); memory[IYd] = B; break; // RL (IYd),B
                                            case 0x11: cycle += 23; C = memory[IYd]; C = rl(C); memory[IYd] = C; break; // RL (IYd),C
                                            case 0x12: cycle += 23; D = memory[IYd]; D = rl(D); memory[IYd] = D; break; // RL (IYd),D
                                            case 0x13: cycle += 23; E = memory[IYd]; E = rl(E); memory[IYd] = E; break; // RL (IYd),E
                                            case 0x14: cycle += 23; H = memory[IYd]; H = rl(H); memory[IYd] = H; break; // RL (IYd),H
                                            case 0x15: cycle += 23; L = memory[IYd]; L = rl(L); memory[IYd] = L; break; // RL (IYd),L
                                            case 0x16: cycle += 23; memory[IYd] = rl(memory[IYd]); break;               // RL (IYd)
                                            case 0x17: cycle += 23; A = memory[IYd]; A = rl(A); memory[IYd] = A; break; // RL (IYd),A
                                            /////////////////////////////
                                            /// RR instructions
                                            ///
                                            case 0x18: cycle += 23; B = memory[IYd]; B = rr(B); memory[IYd] = B; break; // RR (IYd),B
                                            case 0x19: cycle += 23; C = memory[IYd]; C = rr(C); memory[IYd] = C; break; // RR (IYd),C
                                            case 0x1A: cycle += 23; D = memory[IYd]; D = rr(D); memory[IYd] = D; break; // RR (IYd),D
                                            case 0x1B: cycle += 23; E = memory[IYd]; E = rr(E); memory[IYd] = E; break; // RR (IYd),E
                                            case 0x1C: cycle += 23; H = memory[IYd]; H = rr(H); memory[IYd] = H; break; // RR (IYd),H
                                            case 0x1D: cycle += 23; L = memory[IYd]; L = rr(L); memory[IYd] = L; break; // RR (IYd),L
                                            case 0x1E: cycle += 23; memory[IYd] = rr(memory[IYd]); break;               // RR (IYd)
                                            case 0x1F: cycle += 23; A = memory[IYd]; A = rr(A); memory[IYd] = A; break; // RR (IYd),A
                                            /////////////////////////////
                                            /// SLA instructions
                                            ///
                                            case 0x20: cycle += 23; B = memory[IYd]; B = sla(B); memory[IYd] = B; break; // SLA (IYd),B
                                            case 0x21: cycle += 23; C = memory[IYd]; C = sla(C); memory[IYd] = C; break; // SLA (IYd),C
                                            case 0x22: cycle += 23; D = memory[IYd]; D = sla(D); memory[IYd] = D; break; // SLA (IYd),D
                                            case 0x23: cycle += 23; E = memory[IYd]; E = sla(E); memory[IYd] = E; break; // SLA (IYd),E
                                            case 0x24: cycle += 23; H = memory[IYd]; H = sla(H); memory[IYd] = H; break; // SLA (IYd),H
                                            case 0x25: cycle += 23; L = memory[IYd]; L = sla(L); memory[IYd] = L; break; // SLA (IYd),L
                                            case 0x26: cycle += 23; memory[IYd] = sla(memory[IYd]); break;               // SLA (IYd)
                                            case 0x27: cycle += 23; A = memory[IYd]; A = sla(A); memory[IYd] = A; break; // SLA (IYd),A
                                            /////////////////////////////
                                            /// SRA instructions
                                            ///
                                            case 0x28: cycle += 23; B = memory[IYd]; B = sra(B); memory[IYd] = B; break; // SRA (IYd),B
                                            case 0x29: cycle += 23; C = memory[IYd]; C = sra(C); memory[IYd] = C; break; // SRA (IYd),C
                                            case 0x2A: cycle += 23; D = memory[IYd]; D = sra(D); memory[IYd] = D; break; // SRA (IYd),D
                                            case 0x2B: cycle += 23; E = memory[IYd]; E = sra(E); memory[IYd] = E; break; // SRA (IYd),E
                                            case 0x2C: cycle += 23; H = memory[IYd]; H = sra(H); memory[IYd] = H; break; // SRA (IYd),H
                                            case 0x2D: cycle += 23; L = memory[IYd]; L = sra(L); memory[IYd] = L; break; // SRA (IYd),L
                                            case 0x2E: cycle += 23; memory[IYd] = sra(memory[IYd]); break;               // SRA (IYd)
                                            case 0x2F: cycle += 23; A = memory[IYd]; A = sra(A); memory[IYd] = A; break; // SRA (IYd),A
                                            /////////////////////////////
                                            /// SLL instructions
                                            ///
                                            case 0x30: cycle += 23; B = memory[IYd]; B = sll(B); memory[IYd] = B; break; // SLL (IYd),B
                                            case 0x31: cycle += 23; C = memory[IYd]; C = sll(C); memory[IYd] = C; break; // SLL (IYd),C
                                            case 0x32: cycle += 23; D = memory[IYd]; D = sll(D); memory[IYd] = D; break; // SLL (IYd),D
                                            case 0x33: cycle += 23; E = memory[IYd]; E = sll(E); memory[IYd] = E; break; // SLL (IYd),E
                                            case 0x34: cycle += 23; H = memory[IYd]; H = sll(H); memory[IYd] = H; break; // SLL (IYd),H
                                            case 0x35: cycle += 23; L = memory[IYd]; L = sll(L); memory[IYd] = L; break; // SLL (IYd),L
                                            case 0x36: cycle += 23; memory[IYd] = sll(memory[IYd]); break;               // SLL (IYd)
                                            case 0x37: cycle += 23; A = memory[IYd]; A = sll(A); memory[IYd] = A; break; // SLL (IYd),A
                                            /////////////////////////////
                                            /// SRL instructions
                                            ///
                                            case 0x38: cycle += 23; B = memory[IYd]; tmp = (byte)(B & CF); B >>= 1; F = (byte)(SZP[B] | tmp); memory[IYd] = B; break; // SRL (IYd),B
                                            case 0x39: cycle += 23; C = memory[IYd]; tmp = (byte)(C & CF); C >>= 1; F = (byte)(SZP[C] | tmp); memory[IYd] = C; break; // SRL (IYd),C
                                            case 0x3A: cycle += 23; D = memory[IYd]; tmp = (byte)(D & CF); D >>= 1; F = (byte)(SZP[D] | tmp); memory[IYd] = D; break; // SRL (IYd),D
                                            case 0x3B: cycle += 23; E = memory[IYd]; tmp = (byte)(E & CF); E >>= 1; F = (byte)(SZP[E] | tmp); memory[IYd] = E; break; // SRL (IYd),E
                                            case 0x3C: cycle += 23; H = memory[IYd]; tmp = (byte)(H & CF); H >>= 1; F = (byte)(SZP[H] | tmp); memory[IYd] = H; break; // SRL (IYd),H
                                            case 0x3D: cycle += 23; L = memory[IYd]; tmp = (byte)(L & CF); L >>= 1; F = (byte)(SZP[L] | tmp); break; // SRL (IXd),L
                                            case 0x3E: cycle += 23; tmp = (byte)(memory[IYd] & CF); memory[IYd] >>= 1; F = (byte)(SZP[memory[IYd]] | tmp); break;               // SRL (IYd)
                                            case 0x3F: cycle += 23; A = memory[IYd]; tmp = (byte)(A & CF); A >>= 1; F = (byte)(SZP[A] | tmp); memory[IYd] = A; break; // SRL (IYd),A
                                            /////////////////////////////
                                            /// BIT instructions
                                            ///
                                            case 0x40:
                                            case 0x41:
                                            case 0x42:
                                            case 0x43:
                                            case 0x44:
                                            case 0x45:
                                            case 0x46:
                                            case 0x47: cycle += 20; bit(0x01, memory[IYd]); break;         //BIT 0,IYd
                                            case 0x48:
                                            case 0x49:
                                            case 0x4a:
                                            case 0x4b:
                                            case 0x4c:
                                            case 0x4d:
                                            case 0x4e:
                                            case 0x4f: cycle += 20; bit(0x02, memory[IYd]); break;         //BIT 1,IYd
                                            case 0x50:
                                            case 0x51:
                                            case 0x52:
                                            case 0x53:
                                            case 0x54:
                                            case 0x55:
                                            case 0x56:
                                            case 0x57: cycle += 20; bit(0x04, memory[IYd]); break;         //BIT 2,IYd
                                            case 0x58:
                                            case 0x59:
                                            case 0x5a:
                                            case 0x5b:
                                            case 0x5c:
                                            case 0x5d:
                                            case 0x5e:
                                            case 0x5f: cycle += 20; bit(0x08, memory[IYd]); break;         //BIT 3,IYd
                                            case 0x60:
                                            case 0x61:
                                            case 0x62:
                                            case 0x63:
                                            case 0x64:
                                            case 0x65:
                                            case 0x66:
                                            case 0x67: cycle += 20; bit(0x10, memory[IYd]); break;         //BIT 4,IYd
                                            case 0x68:
                                            case 0x69:
                                            case 0x6a:
                                            case 0x6b:
                                            case 0x6c:
                                            case 0x6d:
                                            case 0x6e:
                                            case 0x6f: cycle += 20; bit(0x20, memory[IYd]); break;         //BIT 5,IYd
                                            case 0x70:
                                            case 0x71:
                                            case 0x72:
                                            case 0x73:
                                            case 0x74:
                                            case 0x75:
                                            case 0x76:
                                            case 0x77: cycle += 20; bit(0x40, memory[IYd]); break;         //BIT 6,IYd
                                            case 0x78:
                                            case 0x79:
                                            case 0x7a:
                                            case 0x7b:
                                            case 0x7c:
                                            case 0x7d:
                                            case 0x7e:
                                            case 0x7f: cycle += 20; bit(0x80, memory[IYd]); break;         //BIT 7,IYd
                                            /////////////////////////////
                                            /// RES instructions
                                            ///
                                            case 0x80: cycle += 23; B = (byte)(memory[IYd] & 0xFE); memory[IYd] = B; break;      // RES 0,(XYd),B
                                            case 0x81: cycle += 23; C = (byte)(memory[IYd] & 0xFE); memory[IYd] = C; break;      // RES 0,(XYd),C
                                            case 0x82: cycle += 23; D = (byte)(memory[IYd] & 0xFE); memory[IYd] = D; break;      // RES 0,(XYd),D
                                            case 0x83: cycle += 23; E = (byte)(memory[IYd] & 0xFE); memory[IYd] = E; break;      // RES 0,(XYd),E
                                            case 0x84: cycle += 23; H = (byte)(memory[IYd] & 0xFE); memory[IYd] = H; break;      // RES 0,(XYd),H
                                            case 0x85: cycle += 23; L = (byte)(memory[IYd] & 0xFE); memory[IYd] = L; break;      // RES 0,(XYd),L
                                            case 0x86: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xFE); break;             // RES 0,(XYd)
                                            case 0x87: cycle += 23; A = (byte)(memory[IYd] & 0xFE); memory[IYd] = A; break;      // RES 0,(XYd),A                
                                            case 0x88: cycle += 23; B = (byte)(memory[IYd] & 0xFD); memory[IYd] = B; break;      // RES 1,(XYd),B
                                            case 0x89: cycle += 23; C = (byte)(memory[IYd] & 0xFD); memory[IYd] = C; break;      // RES 1,(XYd),C
                                            case 0x8A: cycle += 23; D = (byte)(memory[IYd] & 0xFD); memory[IYd] = D; break;      // RES 1,(XYd),D
                                            case 0x8B: cycle += 23; E = (byte)(memory[IYd] & 0xFD); memory[IYd] = E; break;      // RES 1,(XYd),E
                                            case 0x8C: cycle += 23; H = (byte)(memory[IYd] & 0xFD); memory[IYd] = H; break;      // RES 1,(XYd),H
                                            case 0x8D: cycle += 23; L = (byte)(memory[IYd] & 0xFD); memory[IYd] = L; break;      // RES 1,(XYd),L
                                            case 0x8E: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xFD); break;             // RES 1,(XYd)
                                            case 0x8F: cycle += 23; A = (byte)(memory[IYd] & 0xFD); memory[IYd] = A; break;      // RES 1,(XYd),A                
                                            case 0x90: cycle += 23; B = (byte)(memory[IYd] & 0xFB); memory[IYd] = B; break;      // RES 2,(XYd),B
                                            case 0x91: cycle += 23; C = (byte)(memory[IYd] & 0xFB); memory[IYd] = C; break;      // RES 2,(XYd),C
                                            case 0x92: cycle += 23; D = (byte)(memory[IYd] & 0xFB); memory[IYd] = D; break;      // RES 2,(XYd),D
                                            case 0x93: cycle += 23; E = (byte)(memory[IYd] & 0xFB); memory[IYd] = E; break;      // RES 2,(XYd),E
                                            case 0x94: cycle += 23; H = (byte)(memory[IYd] & 0xFB); memory[IYd] = H; break;      // RES 2,(XYd),H
                                            case 0x95: cycle += 23; L = (byte)(memory[IYd] & 0xFB); memory[IYd] = L; break;      // RES 2,(XYd),L
                                            case 0x96: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xFB); break;             // RES 2,(XYd)
                                            case 0x97: cycle += 23; A = (byte)(memory[IYd] & 0xFB); memory[IYd] = A; break;      // RES 2,(XYd),A                
                                            case 0x98: cycle += 23; B = (byte)(memory[IYd] & 0xF7); memory[IYd] = B; break;      // RES 3,(XYd),B
                                            case 0x99: cycle += 23; C = (byte)(memory[IYd] & 0xF7); memory[IYd] = C; break;      // RES 3,(XYd),C
                                            case 0x9A: cycle += 23; D = (byte)(memory[IYd] & 0xF7); memory[IYd] = D; break;      // RES 3,(XYd),D
                                            case 0x9B: cycle += 23; E = (byte)(memory[IYd] & 0xF7); memory[IYd] = E; break;      // RES 3,(XYd),E
                                            case 0x9C: cycle += 23; H = (byte)(memory[IYd] & 0xF7); memory[IYd] = H; break;      // RES 3,(XYd),H
                                            case 0x9D: cycle += 23; L = (byte)(memory[IYd] & 0xF7); memory[IYd] = L; break;      // RES 3,(XYd),L
                                            case 0x9E: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xF7); break;             // RES 3,(XYd)
                                            case 0x9F: cycle += 23; A = (byte)(memory[IYd] & 0xF7); memory[IYd] = A; break;      // RES 3,(XYd),A                
                                            case 0xA0: cycle += 23; B = (byte)(memory[IYd] & 0xEF); memory[IYd] = B; break;      // RES 4,(XYd),B
                                            case 0xA1: cycle += 23; C = (byte)(memory[IYd] & 0xEF); memory[IYd] = C; break;      // RES 4,(XYd),C
                                            case 0xA2: cycle += 23; D = (byte)(memory[IYd] & 0xEF); memory[IYd] = D; break;      // RES 4,(XYd),D
                                            case 0xA3: cycle += 23; E = (byte)(memory[IYd] & 0xEF); memory[IYd] = E; break;      // RES 4,(XYd),E
                                            case 0xA4: cycle += 23; H = (byte)(memory[IYd] & 0xEF); memory[IYd] = H; break;      // RES 4,(XYd),H
                                            case 0xA5: cycle += 23; L = (byte)(memory[IYd] & 0xEF); memory[IYd] = L; break;      // RES 4,(XYd),L
                                            case 0xA6: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xEF); break;             // RES 4,(XYd)
                                            case 0xA7: cycle += 23; A = (byte)(memory[IYd] & 0xEF); memory[IYd] = A; break;      // RES 4,(XYd),A                
                                            case 0xA8: cycle += 23; B = (byte)(memory[IYd] & 0xDF); memory[IYd] = B; break;      // RES 5,(XYd),B
                                            case 0xA9: cycle += 23; C = (byte)(memory[IYd] & 0xDF); memory[IYd] = C; break;      // RES 5,(XYd),C
                                            case 0xAA: cycle += 23; D = (byte)(memory[IYd] & 0xDF); memory[IYd] = D; break;      // RES 5,(XYd),D
                                            case 0xAB: cycle += 23; E = (byte)(memory[IYd] & 0xDF); memory[IYd] = E; break;      // RES 5,(XYd),E
                                            case 0xAC: cycle += 23; H = (byte)(memory[IYd] & 0xDF); memory[IYd] = H; break;      // RES 5,(XYd),H
                                            case 0xAD: cycle += 23; L = (byte)(memory[IYd] & 0xDF); memory[IYd] = L; break;      // RES 5,(XYd),L
                                            case 0xAE: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xDF); break;             // RES 5,(XYd)
                                            case 0xAF: cycle += 23; A = (byte)(memory[IYd] & 0xDF); memory[IYd] = A; break;      // RES 5,(XYd),A                
                                            case 0xB0: cycle += 23; B = (byte)(memory[IYd] & 0xBF); memory[IYd] = B; break;      // RES 6,(XYd),B
                                            case 0xB1: cycle += 23; C = (byte)(memory[IYd] & 0xBF); memory[IYd] = C; break;      // RES 6,(XYd),C
                                            case 0xB2: cycle += 23; D = (byte)(memory[IYd] & 0xBF); memory[IYd] = D; break;      // RES 6,(XYd),D
                                            case 0xB3: cycle += 23; E = (byte)(memory[IYd] & 0xBF); memory[IYd] = E; break;      // RES 6,(XYd),E
                                            case 0xB4: cycle += 23; H = (byte)(memory[IYd] & 0xBF); memory[IYd] = H; break;      // RES 6,(XYd),H
                                            case 0xB5: cycle += 23; L = (byte)(memory[IYd] & 0xBF); memory[IYd] = L; break;      // RES 6,(XYd),L
                                            case 0xB6: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0xBF); break;             // RES 6,(XYd)
                                            case 0xB7: cycle += 23; A = (byte)(memory[IYd] & 0xBF); memory[IYd] = A; break;      // RES 6,(XYd),A                
                                            case 0xB8: cycle += 23; B = (byte)(memory[IYd] & 0x7F); memory[IYd] = B; break;      // RES 7,(XYd),B
                                            case 0xB9: cycle += 23; C = (byte)(memory[IYd] & 0x7F); memory[IYd] = C; break;      // RES 7,(XYd),C
                                            case 0xBA: cycle += 23; D = (byte)(memory[IYd] & 0x7F); memory[IYd] = D; break;      // RES 7,(XYd),D
                                            case 0xBB: cycle += 23; E = (byte)(memory[IYd] & 0x7F); memory[IYd] = E; break;      // RES 7,(XYd),E
                                            case 0xBC: cycle += 23; H = (byte)(memory[IYd] & 0x7F); memory[IYd] = H; break;      // RES 7,(XYd),H
                                            case 0xBD: cycle += 23; L = (byte)(memory[IYd] & 0x7F); memory[IYd] = L; break;      // RES 7,(XYd),L
                                            case 0xBE: cycle += 23; memory[IYd] = (byte)(memory[IYd] & 0x7F); break;             // RES 7,(XYd)
                                            case 0xBF: cycle += 23; A = (byte)(memory[IYd] & 0x7F); memory[IYd] = A; break;      // RES 7,(XYd),A                
                                            /////////////////////////////
                                            /// SET instructions
                                            ///
                                            case 0xC0: cycle += 23; B = (byte)(memory[IYd] | 0x01); memory[IYd] = B; break;      // SET 0,(XYd),B
                                            case 0xC1: cycle += 23; C = (byte)(memory[IYd] | 0x01); memory[IYd] = C; break;      // SET 0,(XYd),C
                                            case 0xC2: cycle += 23; D = (byte)(memory[IYd] | 0x01); memory[IYd] = D; break;      // SET 0,(XYd),D
                                            case 0xC3: cycle += 23; E = (byte)(memory[IYd] | 0x01); memory[IYd] = E; break;      // SET 0,(XYd),E
                                            case 0xC4: cycle += 23; H = (byte)(memory[IYd] | 0x01); memory[IYd] = H; break;      // SET 0,(XYd),H
                                            case 0xC5: cycle += 23; L = (byte)(memory[IYd] | 0x01); memory[IYd] = L; break;      // SET 0,(XYd),L
                                            case 0xC6: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x01); break;             // SET 0,(XYd)
                                            case 0xC7: cycle += 23; A = (byte)(memory[IYd] | 0x01); memory[IYd] = A; break;      // SET 0,(XYd),A
                                            case 0xC8: cycle += 23; B = (byte)(memory[IYd] | 0x02); memory[IYd] = B; break;      // SET 1,(XYd),B
                                            case 0xC9: cycle += 23; C = (byte)(memory[IYd] | 0x02); memory[IYd] = C; break;      // SET 1,(XYd),C
                                            case 0xCA: cycle += 23; D = (byte)(memory[IYd] | 0x02); memory[IYd] = D; break;      // SET 1,(XYd),D
                                            case 0xCB: cycle += 23; E = (byte)(memory[IYd] | 0x02); memory[IYd] = E; break;      // SET 1,(XYd),E
                                            case 0xCC: cycle += 23; H = (byte)(memory[IYd] | 0x02); memory[IYd] = H; break;      // SET 1,(XYd),H
                                            case 0xCD: cycle += 23; L = (byte)(memory[IYd] | 0x02); memory[IYd] = L; break;      // SET 1,(XYd),L
                                            case 0xCE: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x02); break;             // SET 1,(XYd)
                                            case 0xCF: cycle += 23; A = (byte)(memory[IYd] | 0x02); memory[IYd] = A; break;      // SET 1,(XYd),A
                                            case 0xD0: cycle += 23; B = (byte)(memory[IYd] | 0x04); memory[IYd] = B; break;      // SET 2,(XYd),B
                                            case 0xD1: cycle += 23; C = (byte)(memory[IYd] | 0x04); memory[IYd] = C; break;      // SET 2,(XYd),C
                                            case 0xD2: cycle += 23; D = (byte)(memory[IYd] | 0x04); memory[IYd] = D; break;      // SET 2,(XYd),D
                                            case 0xD3: cycle += 23; E = (byte)(memory[IYd] | 0x04); memory[IYd] = E; break;      // SET 2,(XYd),E
                                            case 0xD4: cycle += 23; H = (byte)(memory[IYd] | 0x04); memory[IYd] = H; break;      // SET 2,(XYd),H
                                            case 0xD5: cycle += 23; L = (byte)(memory[IYd] | 0x04); memory[IYd] = L; break;      // SET 2,(XYd),L
                                            case 0xD6: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x04); break;             // SET 2,(XYd)
                                            case 0xD7: cycle += 23; A = (byte)(memory[IYd] | 0x04); memory[IYd] = A; break;      // SET 2,(XYd),A
                                            case 0xD8: cycle += 23; B = (byte)(memory[IYd] | 0x08); memory[IYd] = B; break;      // SET 3,(XYd),B
                                            case 0xD9: cycle += 23; C = (byte)(memory[IYd] | 0x08); memory[IYd] = C; break;      // SET 3,(XYd),C
                                            case 0xDA: cycle += 23; D = (byte)(memory[IYd] | 0x08); memory[IYd] = D; break;      // SET 3,(XYd),D
                                            case 0xDB: cycle += 23; E = (byte)(memory[IYd] | 0x08); memory[IYd] = E; break;      // SET 3,(XYd),E
                                            case 0xDC: cycle += 23; H = (byte)(memory[IYd] | 0x08); memory[IYd] = H; break;      // SET 3,(XYd),H
                                            case 0xDD: cycle += 23; L = (byte)(memory[IYd] | 0x08); memory[IYd] = L; break;      // SET 3,(XYd),L
                                            case 0xDE: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x08); break;             // SET 3,(XYd)
                                            case 0xDF: cycle += 23; A = (byte)(memory[IYd] | 0x08); memory[IYd] = A; break;      // SET 3,(XYd),A
                                            case 0xE0: cycle += 23; B = (byte)(memory[IYd] | 0x10); memory[IYd] = B; break;      // SET 4,(XYd),B
                                            case 0xE1: cycle += 23; C = (byte)(memory[IYd] | 0x10); memory[IYd] = C; break;      // SET 4,(XYd),C
                                            case 0xE2: cycle += 23; D = (byte)(memory[IYd] | 0x10); memory[IYd] = D; break;      // SET 4,(XYd),D
                                            case 0xE3: cycle += 23; E = (byte)(memory[IYd] | 0x10); memory[IYd] = E; break;      // SET 4,(XYd),E
                                            case 0xE4: cycle += 23; H = (byte)(memory[IYd] | 0x10); memory[IYd] = H; break;      // SET 4,(XYd),H
                                            case 0xE5: cycle += 23; L = (byte)(memory[IYd] | 0x10); memory[IYd] = L; break;      // SET 4,(XYd),L
                                            case 0xE6: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x10); break;             // SET 4,(XYd)
                                            case 0xE7: cycle += 23; A = (byte)(memory[IYd] | 0x10); memory[IYd] = A; break;      // SET 4,(XYd),A
                                            case 0xE8: cycle += 23; B = (byte)(memory[IYd] | 0x20); memory[IYd] = B; break;      // SET 5,(XYd),B
                                            case 0xE9: cycle += 23; C = (byte)(memory[IYd] | 0x20); memory[IYd] = C; break;      // SET 5,(XYd),C
                                            case 0xEA: cycle += 23; D = (byte)(memory[IYd] | 0x20); memory[IYd] = D; break;      // SET 5,(XYd),D
                                            case 0xEB: cycle += 23; E = (byte)(memory[IYd] | 0x20); memory[IYd] = E; break;      // SET 5,(XYd),E
                                            case 0xEC: cycle += 23; H = (byte)(memory[IYd] | 0x20); memory[IYd] = H; break;      // SET 5,(XYd),H
                                            case 0xED: cycle += 23; L = (byte)(memory[IYd] | 0x20); memory[IYd] = L; break;      // SET 5,(XYd),L
                                            case 0xEE: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x20); break;             // SET 5,(XYd)
                                            case 0xEF: cycle += 23; A = (byte)(memory[IYd] | 0x20); memory[IYd] = A; break;      // SET 5,(XYd),A
                                            case 0xF0: cycle += 23; B = (byte)(memory[IYd] | 0x40); memory[IYd] = B; break;      // SET 6,(XYd),B
                                            case 0xF1: cycle += 23; C = (byte)(memory[IYd] | 0x40); memory[IYd] = C; break;      // SET 6,(XYd),C
                                            case 0xF2: cycle += 23; D = (byte)(memory[IYd] | 0x40); memory[IYd] = D; break;      // SET 6,(XYd),D
                                            case 0xF3: cycle += 23; E = (byte)(memory[IYd] | 0x40); memory[IYd] = E; break;      // SET 6,(XYd),E
                                            case 0xF4: cycle += 23; H = (byte)(memory[IYd] | 0x40); memory[IYd] = H; break;      // SET 6,(XYd),H
                                            case 0xF5: cycle += 23; L = (byte)(memory[IYd] | 0x40); memory[IYd] = L; break;      // SET 6,(XYd),L
                                            case 0xF6: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x40); break;             // SET 6,(XYd)
                                            case 0xF7: cycle += 23; A = (byte)(memory[IYd] | 0x40); memory[IYd] = A; break;      // SET 6,(XYd),A
                                            case 0xF8: cycle += 23; B = (byte)(memory[IYd] | 0x80); memory[IYd] = B; break;      // SET 7,(XYd),B
                                            case 0xF9: cycle += 23; C = (byte)(memory[IYd] | 0x80); memory[IYd] = C; break;      // SET 7,(XYd),C
                                            case 0xFA: cycle += 23; D = (byte)(memory[IYd] | 0x80); memory[IYd] = D; break;      // SET 7,(XYd),D
                                            case 0xFB: cycle += 23; E = (byte)(memory[IYd] | 0x80); memory[IYd] = E; break;      // SET 7,(XYd),E
                                            case 0xFC: cycle += 23; H = (byte)(memory[IYd] | 0x80); memory[IYd] = H; break;      // SET 7,(XYd),H
                                            case 0xFD: cycle += 23; L = (byte)(memory[IYd] | 0x80); memory[IYd] = L; break;      // SET 7,(XYd),L
                                            case 0xFE: cycle += 23; memory[IYd] = (byte)(memory[IYd] | 0x80); break;             // SET 7,(XYd)
                                            case 0xFF: cycle += 23; A = (byte)(memory[IYd] | 0x80); memory[IYd] = A; break;      // SET 7,(XYd),A
                                        }
                                    }
                                    break;
                                //////////////////////////////////
                                case 0xe1: cycle += 14; IYL = memory[SP++]; IYH = memory[SP++]; break; 	                        // POP IY
                                case 0xe3: cycle += 23; tmp1 = IY; IY = peekw(SP); pokew(SP, tmp1); break;          // EX (SP),IY
                                case 0xe5: cycle += 15; memory[--SP] = IYH; memory[--SP] = IYL; break;	                            // PUSH IY
                                case 0xe9: cycle += 8; PC = IY; break;// ABSOLUTELY CORRECT! 	                    // JP (IY)
                                case 0xf9: cycle += 10; SP = IY; break;	                                            // LD SP,IY
                                //////////////////////////////////
                                case 0xdd: cycle += 4; INTblocked = true; break; // just return, not increase R
                                case 0xfd: cycle += 4; INTblocked = true; break; // just return, not increase R
                                default: cycle += 4; INTblocked = true; break; // just remove prefix and start as normal opcode
                            }
                            //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                        }
                        break;
                    //////////////////////////////////
                    case 0xfe: cycle += 7; cpA(memory[PC++]); break; 		                    // CP A,n
                    case 0xff: cycle += 11; SP -= 2; pokew(SP, PC); PC = 0x38; break;	        // RST $38
                }

                cycleCounter += cycle;
                totalCounter += cycle;
            }
        }	// end void exec()



        /// DAA
        private void daa()
        {            
            byte tmp1 = A;
            byte tmp2 = 0;
            int tmp3 = (F & 1);
            int tmp = tmp3;
            if (((F & HF) != 0) || ((tmp1 & 0x0f) > 0x09)) tmp2 |= 0x06;
            if ((tmp3 == 1) || (tmp1 > 0x9f) || ((tmp1 > 0x8f) && ((tmp1 & 0x0f) > 0x09))) { tmp2 |= 0x60; tmp = 1; }
            if (tmp1 > 0x99) tmp = 1;
            if ((F & NF) != 0) subA(tmp2); else addA(tmp2);
            F = (byte)((F & NCF & NVPF) | tmp);
            if((SZP[A] & VPF) != 0) F |= VPF;
        }



        //////////////////////////////////////////////////////////////////////////////////////////


        /////////////////////////////////////// ALU //////////////////////////////////////////////

        /// Compare value with Accu (correct!)
        private void cpA(byte value)
        {
            result = A - value; F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ value) & HF)); if (value > A) F |= CVPF;
        }

        /// Add value to Accu and set flags accordingly
        private void addA(byte value) // (correct!)
        {
            int result = A + value;
            F = (byte)(SZ[(byte)result] | ((A ^ result ^ value) & HF));
            if ((result & 0x100) != 0) F |= CVPF;
            A = (byte)result;
        }

        /// Add value with carry to Accu and set flags accordingly
        private void adcA(byte value) //(correct!)
        {
            int c = F & CF;
            int result = A + value + c;
            F = (byte)(SZ[(byte)result] | ((A ^ result ^ value) & HF));
            if ((result & 0x100) != 0) F |= CVPF;
            A = (byte)result;
        }

        /// Subtract value from Accu and set flags accordingly
        private void subA(byte value) //(correct!)
        {
            int result = A - value;
            F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ value) & HF));
            if (value > A) F |= CVPF;
            A = (byte)result;
        }

        /// Subtract value with carry from Accu and set flags accordingly
        private void sbcA(byte value) // (correct!)
        {
            int c = F & CF;
            int result = A - value - c;
            F = (byte)(SZ[(byte)result] | NF | ((A ^ result ^ value) & HF));
            if (result < 0) F |= CVPF;
            A = (byte)result;
        }

        /// Negate
        private void neg() // (correct!)
        {
            int result = -A;
            F = (byte)(SZ[(byte)result] | NF | ((result ^ A) & HF));
            if (result < 0) F |= CVPF;
            A = (byte)result;
        }



        /// 16bit Add
        private void addIX(UInt16 value) // (correct!!!)
        {
            result = IX + value;
            F = (byte)((F & SZVPF) | (((IX ^ result ^ value) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF));
            IX = (UInt16)result;
        }
        /// 16bit Add
        private void addIY(UInt16 value) // (correct!!!)
        {
            result = IY + value;
            F = (byte)((F & SZVPF) | (((IY ^ result ^ value) >> 8) & HF) | ((result >> 16) & CF) | ((result >> 8) & XYF));
            IY = (UInt16)result;
        }

        /// SBC HL,nn
        private void sbcHL(UInt16 value) //(correct!)
        {
            tmp1 = (UInt16)((H << 8) | L);
            result = tmp1 - value - (F & CF);
            tmp = (byte)(result >> 8);
            F = (byte)((((tmp1 ^ result ^ value) >> 8) & HF) | NF | ((result >> 16) & CF) | (tmp & SF) | (((result & 0xffff) != 0) ? 0 : ZF) | (((value ^ HL) & (HL ^ result) & 0x8000) >> 13) | (tmp & XYF));
            HL = (UInt16)result;
        }

        /// ADC HL,nn (correct!!!)
        private void adcHL(UInt16 value)
        {
            tmp1 = (UInt16)((H << 8) | L); int result = tmp1 + value + (F & CF); tmp = (byte)(result >> 8); F = (byte)((((tmp1 ^ result ^ value) >> 8) & HF) | ((result >> 16) & CF) | (tmp & SF) | (((result & 0xffff) != 0) ? 0 : ZF) | (((value ^ HL ^ 0x8000) & (value ^ result) & 0x8000) >> 13) | (tmp & XYF)); HL = (UInt16)result;
        }

        /////////////////////////////////////////////////////////////////////////////////////////////


        /// Shift left (logical) and reset bit 0 (correct!)
        private byte sla(byte value)
        {
            tmp = ((value & 0x80) != 0) ? CF : (byte)0;
            value <<= 1;
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// Shift left (logical) and set bit 0 (correct!)
        private byte sll(byte value)
        {
            tmp = ((value & 0x80) != 0) ? CF : (byte)0;
            value = (byte)((value << 1) | 1);
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// Arythmetic shift right (keeping the correct sign) (correct!)        
        private byte sra(byte value)
        {
            tmp = (byte)(value & CF);
            value = (byte)((value >> 1) | (value & SF));
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// cyclic left rotate (correct!)
        private byte rl(byte value)
        {
            tmp = ((value & 0x80) != 0) ? CF : (byte)0;
            value = (byte)((value << 1) | (F & CF));
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// cyclic right rotate      (correct!)
        private byte rr(byte value)
        {
            tmp = (byte)(value & CF);
            value = (byte)((value >> 1) | ((F & CF) << 7));
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// left rotate	(correct!)
        private byte rlc(byte value)
        {
            tmp = ((value & 0x80) != 0) ? CF : (byte)0;
            value = (byte)((value << 1) | tmp);
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// right rotate	 (correct!)
        private byte rrc(byte value)
        {
            tmp = (byte)(value & CF);
            value = (byte)((value >> 1) | (tmp << 7));
            F = (byte)(SZP[value] | tmp);
            return value;
        }

        /// RLD (correct!)
        private void rld()
        {
            byte q, t;
            q = t = memory[(H << 8) | L];
            t = (byte)((t << 4) | (A & 0x0F));
            A = (byte)((A & 0xF0) | (q >> 4));
            memory[(H << 8) | L] = t;
            F = (byte)((F & CF) | SZP[A]);
        }

        /// RRD  (correct!)
        private void rrd()
        {
            byte q, t;
            q = t = memory[(H << 8) | L];
            t = (byte)((t >> 4) | (A << 4));
            A = (byte)((A & 0xF0) | (q & 0x0F));
            memory[(H << 8) | L] = t;
            F = (byte)((F & CF) | SZP[A]);
        }

        /// test specified bit
        private void bit(byte bitval, byte value) // (correct!)
        {
            tmp = (byte)(value & bitval);
            F = (byte)((F & CF) | HF | (tmp & (SF | XYF)));
            if (tmp == 0) F |= ZVPF;
        }


        /////////////////////////////////////////////////////////////////////////////////////////////////

        /// LDDR (correct!!!)
        private void lddr()
        {
            memory[DE--] = memory[HL--]; F &= SZCF; if (C == 0) B--; C--; if (B !=0 || C != 0) F |= VPF; /* F |= (byte)(((byte)(tmp + A) << 2) & XYF); */
            if (BC != 0) { PC -= 2; cycle -= 5; }
        }


        /// CPI (correct!!!)
        private void cpi()
        {
            byte tmp1 = (byte)(F & CF);
            byte outval = memory[HL++];
            cpA(outval); if (C == 0) B--; C--;
            F = (byte)((F & NSZHF & NCF) | tmp1);
            if (B != 0 || C != 0) F |= VPF; else F &= NVPF;
            F |= (byte)(((byte)(A - outval - ((F & HF) >> 4)) << 2) & XYF);
        }

        /// CPIR (correct!!!)
        private void cpir()
        {
            cpi(); if (B !=0 || C != 0) { PC -= 2; cycle -= 5; }
        }

        /// CPD (correct!!!)
        private void cpd()
        {
            byte tmp1 = (byte)(F & CF);
            byte outval = memory[HL--];
            cpA(outval); if (C == 0) B--; C--;
            F = (byte)((F & NSZHF & NCF) | tmp1);
            if (B != 0 || C != 0) F |= VPF; else F &= NVPF;
            F |= (byte)(((byte)(A - outval - ((F & HF) >> 4)) << 2) & XYF);
        }

        /// CPDR (correct!!!)
        private void cpdr()
        {
            cpd(); if (B != 0 || C != 0) { PC -= 2; cycle -= 5; }
        }



        /// INI (correct!!!)
        private void ini()
        {
            byte outval = portIn(B, C);
            F = (byte)((F & CF) | SZHV_dec[--B]);
            memory[HL++]=outval;
            F = (byte)((F & NCHNF & NVPF) | ((outval & SF) >> 6)); // set NF
            if (outval + ((C + 1) & 255) > 255) F |= (byte)(CF | HF);
            if ((SZP[((outval + ((C + 1) & 255)) & 7) ^ B] & VPF) != 0) F |= VPF;
            F |= (byte)(((byte)(outval + A) << 2) & XYF);
        }

        /// INIR (correct!!!)
        private void inir()
        {
            byte outval = portIn(B--, C);
            memory[HL++]=outval;
            F = (B == 0) ? ZF : (byte)0;
            if (outval + ((C + 1) & 255) > 255) F |= CHF;
            F |= (byte)(SZP[((outval + ((C + 1) & 255)) & 7) ^ B] & VPF);
            if ((outval & 128) != 0) F |= NF;
            F |= (byte)(((byte)(outval + A) << 2) & XYF);
            if (B != 0) { PC -= 2; cycle -= 5; }
        }

        /// IND (correct!!!)
        private void ind()
        {
            byte outval = portIn(B, C);
            F = (byte)((F & CF) | SZHV_dec[--B]);
            memory[HL--] = outval;
            F = (byte)((F & NCHNF & NVPF) | ((outval & SF) >> 6)); // set NF
            if (outval + ((C - 1) & 255) > 255) F |= (byte)(CF | HF);
            if ((SZP[((outval + ((C - 1) & 255)) & 7) ^ B] & VPF) != 0) F |= VPF;
        }

        /// INDR (correct!!!)
        private void indr()
        {
            byte outval = portIn(B--, C);
            memory[HL--]=outval;
            F = (B == 0) ? ZF : (byte)0;
            if (outval + ((C - 1) & 255) > 255) F |= CHF;
            F |= (byte)(SZP[((outval + ((C - 1) & 255)) & 7) ^ B] & VPF);
            if ((outval & 128) != 0) F |= NF;
            F |= (byte)(((byte)(outval + A) << 2) & XYF);
            if (B != 0) { PC -= 2; cycle -= 5; }
        }

        /// OTIR (correct!!!)
        private void otir()
        {
            byte outval = memory[HL++];
            portOut(--B, C, outval);
            F = (B == 0) ? ZF : (byte)0;
            if (outval + L > 255) F |= CHF;
            F |= (byte)(SZP[((((outval) + L) & 7) ^ B)] & VPF);
            if ((outval & 128) != 0) F |= NF;
            if (B != 0) { PC -= 2; cycle -= 5; }
        }

        /// OTDR (correct!!!)
        private void otdr()
        {
            byte outval = memory[HL--];
            portOut(--B, C, outval);
            F = (B == 0) ? ZF : (byte)0;
            if (outval + L > 255) F |= CHF;
            if ((outval & 128) != 0) F |= NF;
            F |= (byte)(SZP[((((outval) + L) & 7) ^ B)] & VPF);
            if (B != 0) { PC -= 2; cycle -= 5; }
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////

        /// OUT
        byte CPCswitch = 0;
        byte CPCdata = 0;
        private void portOut(byte porth, byte portl, byte value)
        {
            // converting from CPC ports
            if (porth == 0xF4) { CPCdata = value; return; }
            else if (porth == 0xF6)
            {
                byte b = (byte)(value & 0xC0);
                if (CPCswitch == 0) CPCswitch = b;
                else
                if (b == 0)
                {
                    if(CPCswitch==0xC0)
                    {
                        porth = 0xff;
                        portl = 0xfd;
                        outb(porth, portl, CPCdata);
                    }
                    else if (CPCswitch == 0x80)
                    {
                        porth = 0xbf;
                        portl = 0xfd;
                        outb(porth, portl, CPCdata);
                    }
                    CPCswitch = 0;
                }
                return;
            }

            // ZX out values
            outb(porth, portl, value);
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////

        /// IN
        private byte portIn(byte porth, byte portl) // (correct!)
        {
            byte val = inb(porth, portl);
            F = (byte)((F & CF) | SZP[val]);
            return val;
        }

        private byte portIn_A(byte port) // (correct!)
        {
            return inb(A, port);
        }


        /////////////////////////////////////////////////////////////////////////////////////////////////

        /// Memory Related
        public byte peekb(UInt16 add)
        {
            return memory[add];
        }

        public void pokeb(UInt16 add, byte value)
        {
            memory[add] = value;
        }

        public UInt16 peekw(UInt16 add)
        {
            return (UInt16)(memory[add++] | (memory[add] << 8));
        }

        public void pokew(UInt16 add, UInt16 value)
        {
            memory[add++] = (byte)value;
            memory[add] = (byte)(value >> 8);
        }

        /////////////////////////////////////////////////////////////////////////////////////////////////

        /// check interrupt
        private void checkInterrupt()
        {
            if (NMI || (IFF0 && IRQ))
            {
                if (memory[PC] == 0x76) { PC++; halted = false; }

                if (NMI)
                { // Take NMI
                    //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                    IFF1 = IFF0;
                    NMI = IFF0 = false;
                    SP -= 2; pokew(SP, PC);
                    PC = 0x0066;
                    cycleCounter += 11;
                }

                if (IFF0 && IRQ)
                {	// Take interrupt if enabled
                    //R = (byte)((R & 0x80) | ((R + 1) & 0x7F));
                    IFF0 = IFF1 = false;
                    IRQ = false;
                    switch (IM)
                    {
                        case 0:	// IM0  --> exec 1-byte instruction (RST or CALL ZX has 255 on the bus, it means RST 0x38)
                        case 1:	// IM1	--> RST 0x38
                        case 3:	// IM 0/1 --> Undocumented
                            SP -= 2; pokew(SP, PC);
                            PC = 0x0038;
                            cycleCounter += 13; // RST 0x38 = 11 cycles + 2 cycles push
                            break;
                        case 2:	// IM2  --> Call I:Vector
                            SP -= 2; pokew(SP, PC);
                            PC = peekw((UInt16)((I << 8) | 0xFF));
                            //PC = peekw((UInt16)(I << 8));
                            cycleCounter += 19; // Call = 17 cycles  + 2 cycles push
                            break;
                        default:
                            break;
                    }
                }
            }
        }

    } // end class declaration
}
