library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.all;
use IEEE.STD_LOGIC_unsigned.all;

entity fdd is
	port
	(
		clk: in std_logic;
		reset: in std_logic;

		-- Avalon bus interface
		address: in std_logic_vector(0 downto 0);
		writedata: in std_logic_vector(15 downto 0);
		chipselect: in std_logic;
		read: in std_logic;
		write: in std_logic;
		readdata: out std_logic_vector(15 downto 0);

		-- Floppy disk interface
		fdd_ds0: out std_logic;  -- pin 10  0= Drive Select 0
		fdd_ds1: out std_logic;  -- pin 12  0= Drive Select 1
		fdd_ds2: out std_logic;
		fdd_ds3: out std_logic;
		fdd_mon: out std_logic;  -- pin 16  0= Motor Enable Drive 0
		fdd_dir: out std_logic;  -- pin 18  0= Direction Select
		fdd_st:  out std_logic;  -- pin 20  0= Head Step
		fdd_wd:  out std_logic;  -- pin 22  Write Data
		fdd_wen: out std_logic;  -- pin 24  Floppy Write Enable, 0=Write Gate
		fdd_sd:  out std_logic;  -- pin 32  0= Head Select
		fdd_idx: in std_logic;   -- pin 8   1= Index
		fdd_tr0: in std_logic;   -- pin 26  1= Track 00
		fdd_wp:  in std_logic;   -- pin 28  1=Write Protect
		fdd_rd:  in std_logic;   -- pin 30  Read Data
		fdd_rdy: in std_logic;   -- pin 34  0=Disk Change/1=Ready
		fdd_rez: out std_logic;

		test: out std_logic
	);
end fdd;

architecture behaviour of fdd is
	constant STATE_REG_ADDRESS: std_logic_vector(0 downto 0) := "0";  -- (177130 - 177130)/ 2
	constant DATA_REG_ADDRESS:  std_logic_vector(0 downto 0) := "1";  -- (177132  - 177130)/ 2

	signal prescale_count: std_logic_vector(3 downto 0);
	signal bit_count: std_logic_vector(3 downto 0);
	signal read_data_reg: std_logic_vector(15 downto 0);
	signal write_data_reg: std_logic_vector(15 downto 0);
	signal shift_reg: std_logic_vector(15 downto 0);
	signal prev_output_reg: std_logic_vector(1 downto 0);
	signal crc_reg: std_logic_vector(15 downto 0);

	signal proc_data_rd: std_logic;   -- data read from processor bus
	signal read_data_rdy: std_logic;
	signal proc_data_wr: std_logic;   -- data write from processor bus
	signal write_reg_full: std_logic;
	signal write_shifter_full: std_logic;

	signal Crc: std_logic;
	signal Tr: std_logic;
	signal Rez: std_logic;
	signal Wm: std_logic;
	signal Grd: std_logic;
	signal St_cnt: std_logic_vector(1 downto 0);
	signal Dir: std_logic;
	signal Hs: std_logic;
	signal Msw: std_logic;
	signal Ds3: std_logic;
	signal Ds2: std_logic := '0'; --    ,     reset   
	signal Ds1: std_logic;
	signal Ds0: std_logic;

	signal shift_in: std_logic;
	signal serial_output: std_logic_vector(1 downto 0);
	signal precomp_sequence: std_logic_vector(3 downto 0);
	signal mfm_output: std_logic;
	signal early: std_logic:= '0';
	signal late: std_logic:= '0';

	signal rd_del1, rd_del2: std_logic;
	signal input_edge: std_logic;
	signal read_CRC: std_logic;
	signal read_data_value: std_logic_vector(15 downto 0);

	signal pll_count: std_logic_vector(3 downto 0);
	type   pll_state_type is (pll_st_cur0, pll_st_wait00, pll_st_wait01, pll_st_cur1, pll_st_wait11);
	signal pll_state: pll_state_type;
	signal read_window: std_logic;
	signal phase_reverse: std_logic;
	signal phase_violation: std_logic;
	signal shift_en: std_logic;
	signal latch_en: std_logic;

	type state_type is (st_idle, st_wait_marker, st_read, st_wait_rotation, st_write, st_write_crc);
	signal state: state_type;
	signal address_marker_flag: std_logic;
	signal address_marker_cnt: std_logic_vector(2 downto 0);
	signal wait_marker_flag: std_logic;
	signal read_flag: std_logic;
	signal wm_reg: std_logic;

	signal idx_del1, idx_del2: std_logic;
	signal rotate: std_logic;
	signal rotate_cnt: std_logic_vector(6 downto 0);
begin
	prescale_count <= pll_count;

	-- signal latch_en strobes write data from shift register
	--    to data register.
	latch_control: process (reset, clk)
		variable vdel: std_logic;
	begin
		if reset = '1' then
			latch_en <= '0';
			vdel := '0';
		elsif rising_edge(clk) then
			latch_en <= vdel;
			vdel := shift_en;
		end if;
	end process latch_control;

	bit_counter: process(reset, clk)
	begin
		if reset = '1' then
-- pragma translate_off
			bit_count <= (others => '0');
-- pragma translate_on
		elsif rising_edge(clk) then
			if state = st_wait_marker then
				bit_count <= "0000";
			else
				if latch_en = '1' then
					bit_count <= bit_count- 1;
				end if;
			end if;
		end if;
	end process bit_counter;

	read_data: process(reset, clk)
		variable vrdy: std_logic;
	begin
		if reset = '1' then
-- pragma translate_off
			read_data_reg <= (others => '0');
-- pragma translate_on
			read_data_rdy <= '0';
			read_CRC <= '0';
		elsif rising_edge(clk) then
			vrdy := read_data_rdy;
			if (state = st_idle or state = st_wait_marker) and latch_en = '1' then
--        if read = '0' then
					read_data_reg <= shift_reg;
--        end if;
			end if;
			if state = st_read then
				if latch_en = '1' then
					if bit_count = "0000" then
--            if read = '0' then
							read_data_reg <= shift_reg;
--            end if;
						if shift_reg(15 downto 8) = crc_reg(15 downto 8) and
							 shift_reg(7 downto 0) = crc_reg(7 downto 0)
						then
							read_CRC <= '1';
						else
							read_CRC <= '0';
						end if;
						vrdy := '1';
					end if;
				end if;
			end if;
			if Grd = '1' or proc_data_rd = '1' then
				vrdy := '0';
			end if;
			read_data_rdy <= vrdy;
		end if;
	end process read_data;

	write_data: process(reset, clk)
	begin
		if reset = '1' then
			write_reg_full <= '0';
			write_shifter_full <= '0';
		elsif rising_edge(clk) then
			if proc_data_wr = '1' then
				write_reg_full <= '1';
			elsif state = st_write and shift_en = '1' and bit_count = "0000" then
				write_shifter_full <= write_reg_full;
				write_reg_full <= '0';
			end if;
		end if;
	end process write_data;

	mfm_out: process(reset, clk)
	begin
		if reset = '1' then
			mfm_output <= '0';
		elsif rising_edge(clk) then
			if (early = '1' and prescale_count(2 downto 0) = "000") or
				 (late = '1' and  prescale_count(2 downto 0) = "010") or
				 (early= '0' and late = '0' and prescale_count(2 downto 0) = "001")
			then
				if (serial_output(1) = '1' and prescale_count(3) = '1') or
					 (serial_output = "00"   and prescale_count(3) = '0'
					 -- address marker, skip impulse between bits 3 and 2
					 and not (wm_reg = '1' and bit_count(2 downto 0) = "010" and shift_reg(15 downto 13) = "001"))
				then
					mfm_output <= '1';
				else
					mfm_output <= '0';
				end if;
			else
				mfm_output <= '0';
			end if;
		end if;
	end process mfm_out;

	write_precompensation_value: process(precomp_sequence, Rez)
		variable v_early, v_late: std_logic;
	begin
		v_early := '0';
		v_late := '0';
		if Rez = '1' then
			case precomp_sequence is
				-- values from patent US4334250
				-- also same values used in module WF_FDC1772_IP from Suska III
				when "0001" => v_early := '1';
				when "0110" => v_early := '1';
				when "1110" => v_early := '1';
				when "0011" => v_late := '1';
				when "1011" => v_late := '1';
				when "1000" => v_late := '1';
				when others => null;
			end case;
		end if;
		early <= v_early;
		late <= v_late;
	end process write_precompensation_value;

	shifter: process(reset, clk)
		variable next_shift: std_logic_vector(15 downto 0);
	begin
		if reset = '1' then
-- pragma translate_off
			shift_reg <= (others => '0');
			prev_output_reg <= (others => '0');
-- pragma translate_on
			wm_reg <= '0';
		elsif rising_edge(clk) then
			next_shift := shift_reg;
			if phase_reverse = '1' then
				next_shift := (others => '0');
			elsif state = st_write and bit_count = "0000" and write_reg_full = '0' and write_shifter_full= '0' then
				next_shift := crc_reg;
			elsif shift_en = '1' then
				if state = st_write and bit_count = "0000" then
					if write_reg_full = '1' then
						next_shift := write_data_reg(7 downto 0) & write_data_reg(15 downto 8);
						wm_reg <= Wm;
					end if;
				else
					next_shift := shift_reg(14 downto 0) & shift_in;
				end if;
				prev_output_reg <= prev_output_reg(0) & shift_reg(15);
			end if;
			shift_reg <= next_shift;
		end if;
	end process shifter;

	serial_output(1) <= shift_reg(15);
--  serial_output(0) <= write_data_reg(7) when bit_count = "0000" and write_reg_full = '1'
--    else crc_reg(15) when bit_count = "0000" and write_reg_full = '0'
--    else shift_reg(14);
	serial_output(0) <= prev_output_reg(0);
	precomp_sequence(3) <= shift_reg(14);
	precomp_sequence(2) <= shift_reg(15);
	precomp_sequence(1) <= prev_output_reg(0);
	precomp_sequence(0) <= prev_output_reg(1);

	-- G(X) = X^16 + X^12 + X^5 + 1;
	crc_generator: process(reset, clk)
		variable c0: std_logic;
	begin
		if reset = '1' then
-- pragma translate_off
			crc_reg <= (others => '1');
-- pragma translate_on
		elsif rising_edge(clk) then
			if (state = st_wait_marker) or
				 (state = st_write and Wm = '1' and wm_reg = '0') then
				crc_reg <= (others => '1');
			elsif shift_en = '1' then
				c0 := shift_reg(15) xor crc_reg(15);
				crc_reg(15 downto 13) <= crc_reg(14 downto 12);
				crc_reg(12) <= crc_reg(11) xor c0;
				crc_reg(11 downto 6) <= crc_reg(10 downto 5);
				crc_reg(5) <= crc_reg(4) xor c0;
				crc_reg(4 downto 1) <= crc_reg(3 downto 0);
				crc_reg(0) <= c0;
			end if;
		end if;
	end process crc_generator;

	edge_detect: process(reset, clk, rd_del1, rd_del2)
	begin
		if reset = '1' then
			rd_del1 <= '0';
			rd_del2 <= '0';
		elsif rising_edge(clk) then
			rd_del2 <= rd_del1;
			rd_del1 <= fdd_rd;
		end if;
		input_edge <= rd_del1 and not rd_del2;
	end process edge_detect;

	digital_PLL: process(reset, clk, pll_count, pll_state, input_edge)
		variable zero_window: std_logic;
		variable one_window: std_logic;
		variable zero_window_last: std_logic;
		variable one_window_last: std_logic;
		variable next_pll_count: std_logic_vector(3 downto 0);
	begin
		if pll_count = "0000" or pll_count = "0001" or
			 pll_count = "0010" or pll_count = "0011" or
			 pll_count = "1100" or pll_count = "1101" or
			 pll_count = "1110" or pll_count = "1111"
		then
			zero_window := '1';
		else
			zero_window := '0';
		end if;
		one_window := not zero_window;
		if pll_count = "0011" then
			zero_window_last := '1';
		else
			zero_window_last := '0';
		end if;
		if pll_count = "1011" then
			one_window_last := '1';
		else
			one_window_last := '0';
		end if;
		if reset = '1' then
-- pragma translate_off
			pll_count <= "0000";
-- pragma translate_on
			pll_state <= pll_st_cur0;
		elsif rising_edge(clk) then
--      if Grd = '1' then
--        pll_state <= pll_st_cur0;
--      else
				case pll_state is
					when pll_st_cur0 =>
						if one_window = '1' then
							if input_edge = '1' then
								pll_state <= pll_st_cur1;
							elsif one_window_last = '1' then
								pll_state <= pll_st_wait00;
							end if;
						end if;
					when pll_st_wait00 =>
						if input_edge = '1' then
							pll_state <= pll_st_cur0;
						elsif zero_window_last = '1' then
							pll_state <= pll_st_wait01;
						end if;
					when pll_st_wait01 =>
						if input_edge = '1' then
							pll_state <= pll_st_cur1;
						elsif one_window_last = '1' then
							-- phase violation
							pll_state <= pll_st_wait00;
						end if;
					when pll_st_cur1 =>
						if one_window_last = '1' then
							pll_state <= pll_st_wait11;
						end if;
					when pll_st_wait11 =>
						if phase_reverse = '1' then
							pll_state <= pll_st_cur0;
						else
							if one_window = '1' and input_edge = '1' then
								pll_state <= pll_st_cur1;
							elsif one_window_last = '1' then
								pll_state <= pll_st_wait00;
							end if;
						end if;
					when others =>  null;
				end case;
--        if Grd = '1' then
--          next_pll_count := "0000";
				if state = st_write  or state = st_write_crc  then
					next_pll_count := pll_count + 1;
				elsif input_edge = '1' then
					case pll_count(2 downto 0) is
						when "001" | "010" | "011" =>
							next_pll_count := pll_count;
						when "100" | "101" | "110" =>
							next_pll_count := pll_count + 2;
						when others =>
							next_pll_count := pll_count + 1;
					end case;
				else
					next_pll_count := pll_count+ 1;
				end if;
				if phase_reverse = '1' then
					pll_count <= (not next_pll_count(3)) & next_pll_count(2 downto 0);
				else
					pll_count <= next_pll_count;
				end if;
			end if;
--    end if;
		read_window <= one_window;
		if pll_state = pll_st_cur1 or pll_state = pll_st_wait11 then
			shift_in <= '1';
		else
			shift_in <= '0';
		end if;
		if pll_count = "1100" then
			shift_en <= '1';
		else
			shift_en <= '0';
		end if;
		if pll_state = pll_st_wait01 and one_window_last = '1' and input_edge = '0' then
			phase_violation <= '1';
		else
			phase_violation <= '0';
		end if;
	end process digital_PLL;

	address_mark: process (reset, clk, address_marker_cnt)
	begin
		if reset = '1' then
			address_marker_cnt <= (others => '0');
		elsif rising_edge(clk) then
			if state = st_wait_marker then
				if phase_violation = '1' then
					address_marker_cnt <= "100";
				elsif shift_en = '1' and address_marker_cnt /= 0 then
					address_marker_cnt <= address_marker_cnt - 1;
				end if;
			else
				address_marker_cnt <= (others => '0');
			end if;
		end if;
		if address_marker_cnt /= 0 then
			address_marker_flag <= '1';
		else
			address_marker_flag <= '0';
		end if;
	end process address_mark;

	main_state: process(reset, clk, state, shift_reg)
	begin
		if reset = '1' then
			state <= st_idle;
		elsif rising_edge(clk) then
			case state is
				when st_idle =>
					if Grd = '1' then
						state <= st_wait_marker;
					elsif proc_data_wr = '1' then
						state <= st_wait_rotation;
					end if;
				when st_wait_marker =>
					if address_marker_flag = '1' and shift_reg(15 downto 8) = x"A1"
							and shift_reg(7 downto 0) = x"A1" then  -- address marker
						state <= st_read;
					end if;
				when st_read =>
					if Grd = '1' then
						state <= st_wait_marker;
					elsif proc_data_wr = '1' then
						state <= st_wait_rotation;
					end if;
				when st_wait_rotation =>
					if rotate = '1' then
						state <= st_write;
					end if;
				when st_write =>
					if chipselect = '1' and read = '1' and address = DATA_REG_ADDRESS then
						state <= st_idle;
					elsif write_reg_full = '0' and write_shifter_full = '0' then
						state <= st_write_crc;
					end if;
				when st_write_crc =>
					if shift_en = '1' and bit_count = "0000" then
						state <= st_idle;
					elsif proc_data_wr = '1' then
						state <= st_write;
					end if;
				when others => null;
			end case;
		end if;
		if state = st_wait_marker and shift_reg = x"FFFF" then
			phase_reverse <= '1';
		else
			phase_reverse <= '0';
		end if;
	end process main_state;

	rotation_detector: process (reset, clk)
	begin
		if reset = '1' then
			idx_del1 <= '0';
			idx_del2 <= '0';
			rotate <= '0';
--      rotate_cnt <= (others => '0');
		elsif rising_edge(clk) then
--      if input_edge = '1' or (idx_del1 = '1' and idx_del2 = '0') then
--        rotate <= '1';
--        rotate_cnt <= (others => '1');
--      elsif rotate_cnt /= 0 then
--        rotate_cnt <= rotate_cnt - 1;
--      else
--        rotate <= '0';
--      end if;
			if Ds0 = '0' and Ds1 = '0' then
				rotate <= '0';
			elsif (idx_del1 = '1' and idx_del2 = '0') then
				rotate <= '1';
			end if;

			idx_del2 <= idx_del1;
			idx_del1 <= fdd_idx;
		end if;
	end process rotation_detector;

	Tr <= read_data_rdy when state = st_read  else
				not write_reg_full when state = st_write or state = st_write_crc
			else '0';
	Crc <= read_CRC when state = st_read else
				 '1' when state = st_write_CRC
			else '0';
	wait_marker_flag <= '1' when state = st_wait_marker  else '0';
	read_flag <= '1' when state = st_read else '0';

	-- process generates one-clock proc_data_rd impulse after
	-- read signal becomes passive
	read_impulse: process (reset, clk)
		variable vdel1, vdel2: std_logic;
	begin
		if reset = '1' then
			vdel1 := '0';
			vdel2 := '0';
		elsif rising_edge(clk) then
			vdel2 := vdel1;
			if chipselect = '1' and read = '1' and address = DATA_REG_ADDRESS then
				vdel1 := '1';
			else
				vdel1 := '0';
			end if;
		end if;
		proc_data_rd <= not vdel1 and vdel2;
	end process read_impulse;

	external_bus: process(reset, clk, chipselect, read, write, address, read_data_reg,
		fdd_idx, fdd_wp, fdd_rdy, fdd_tr0, Crc, Tr)
	begin
		if reset = '1' then
-- pragma translate_off
			write_data_reg <= (others => '0');
-- pragma translate_on
			Rez <= '0';
			Wm <= '0';
			Grd <= '0';
			St_cnt <= (others => '0');
			Dir <= '0';
			Hs <= '0';
			Msw <= '0';
			Ds3 <= '0';
			Ds2 <= '0';
			Ds1 <= '0';
			Ds0 <= '0';
		elsif rising_edge(clk) then
			if chipselect= '1' and write = '1' then
				if address = STATE_REG_ADDRESS then
					Rez <= writedata(10);
					Wm <= writedata(9);
					Grd <= writedata(8);
					St_cnt(0) <= writedata(7);
					St_cnt(1) <= writedata(7);
					Dir <= writedata(6);
					Hs <= writedata(5);
					Msw <= writedata(4);
					Ds3 <= writedata(3);
					Ds2 <= writedata(2);
					Ds1 <= writedata(1);
					Ds0 <= writedata(0);
				end if;
				if address = DATA_REG_ADDRESS then
					write_data_reg <= writedata;
				end if;
			elsif St_cnt /= 0 then
				St_cnt <= St_cnt- 1;
			end if;
		end if;

		if address = DATA_REG_ADDRESS then
			read_data_value <= read_data_reg;
		else
			read_data_value <= fdd_idx & Crc & "000000" & Tr & "0000" & fdd_wp & fdd_rdy & fdd_tr0;
		end if;

		if chipselect = '1' and write = '1' and address = DATA_REG_ADDRESS then
			proc_data_wr <= '1';
		else
			proc_data_wr <= '0';
		end if;
	end process external_bus;

	data_latch_proc: process (chipselect, read_data_value)
	begin
		if chipselect = '0' then
			readdata <= read_data_value;
		end if;
	end process data_latch_proc;

--  test_out: process (reset, clk)
--    variable v_cnt: std_logic_vector(19 downto 0);
--    variable v_st1: std_logic;
--    variable v_st2: std_logic;
--  begin
--    if reset = '1' then
--      v_st1 := '0';
--      v_st2 := '0';
--      v_cnt := (others => '0');
--    elsif rising_edge(clk) then
--      if state = st_wait_marker then
--      if Grd = '1' then
--        v_st1 := '1';
--      end if;
--      if state = st_read then
--        v_st2 := '1';
--      end if;
--      v_cnt := v_cnt + 1;
--    end if;
--    if v_st1 = '1' then
--      test <= '0'; -- v_cnt(19);
--    elsif v_st2 = '1' then
--      test <= '1';
--    else
--      test <= v_cnt(18);
--    end if;
--  end process test_out;
	--test <= fdd_rdy;

	test <= '1' when (state = st_read and latch_en = '1' and bit_count = "0000") or
		((state = st_idle or state = st_wait_marker) and latch_en = '1')
		else '0';


	fdd_ds0 <= not Ds0;
	fdd_ds1 <= not Ds1;
	fdd_ds2 <= not Ds2;
	fdd_ds3 <= not Ds3;
	fdd_mon <= not Msw;
	fdd_dir <= not Dir;
	fdd_st <= '0' when St_cnt /= 0  else '1';
	fdd_wen <= '0' when state = st_write  or state = st_write_crc else '1';
	fdd_wd <= not mfm_output when state = st_write  or state = st_write_crc else '1';
	fdd_sd  <= not Hs;
	fdd_rez <= not Rez;
end behaviour;