# hfe.py
#
# HFEv1 to UDIv1 converter
#
# Written by Denis Dratov <volutar@gmail.com>
#
# This is free and unencumbered software released into the public domain.
# See the file COPYING for more details, or visit <http://unlicense.org>.

import sys, struct, argparse, time
from demfm import DeMFM

tracks=[]
syncs=[]
sectors=[]
cyls=0
sides=0
_CRC=0xffffffff

def udicrc(buf):
  global _CRC
  for i in range(len(buf)):
    _CRC ^= 0xffffffff ^ buf[i]
    for k in range(8):
      temp=_CRC&1
      _CRC=(_CRC&0x80000000)|_CRC>>1
      if temp: _CRC^=0xedb88320
    _CRC ^= 0xffffffff
  return _CRC


def importhfe(args):
  global tracks, syncs, sectors, cyls, sides
  with open(args.infile, "rb") as f:
    dat = f.read()

  header = struct.unpack("<8s4B2H2B1H6B", dat[0:26])
#  print (header)
  if header[0] != b"HXCPICFE":
    print ("Not HFEv1 format")
    return False
  cyls = dat[9]
  sides = dat[10]
  print ("Importing HFEv1 file \"%s\"..." %args.infile)
  print ("Cyls:", cyls," Sides:", sides)
  trk_offs = struct.unpack("<256H", dat[0x200:0x400])
#  print (trk_offs)
  if cyls>84:
    print ("Too many cylinders")
    return False

  track_range = range(cyls)

  demfm=DeMFM()
  diag=False

  curtime=time.time()
  tracks=[]
  syncs=[]
  sectors=[]

#full: 5D AA 3B 60
#  for tr in range (4):
#  for tr in range (78,82):
  for tr in track_range:
    trk_start = trk_offs[tr*2] * 512
    trk_size = trk_offs[tr*2+1]
#    print (tr,": ",trk_start,' ',trk_size)

    trk_dat1=bytearray()
    trk_dat2=bytearray()
    j=-256
    for i in range (trk_size//2): #reassemble hfe interleaved data
      if i%256==0: j+=256
      trk_dat1.append(dat[trk_start+i+j]) #side 0
      trk_dat2.append(dat[trk_start+256+i+j]) #side 1
#    print(trk_start,i,j)
#    print (trk_dat1)  
#    print(" ==== SIDE A")
    demfm.decode_mfm(trk_dat1)
    tracks.append(demfm.bytes)
    syncs.append(demfm.syncs)
    sectors.append(demfm.cat)
#    print(tracks[tr*2])

    if not diag:
      if demfm.cat[0][1]==1:
        print("Amiga disk format")
#    demfm.print_cat()

    demfm.print_cat_short()
    diag=True;

#    print(" ==== SIDE B")
    demfm.decode_mfm(trk_dat2)
    tracks.append(demfm.bytes)
    syncs.append(demfm.syncs)
    sectors.append(demfm.cat)

#    print(tracks[tr*2+1])

#    demfm.print_cat()
    demfm.print_cat_short()
  print("Import time:",time.time()-curtime)

def exportudi(args):
  global tracks, syncs, sectors, cyls, sides, _CRC

  print ("Exporting UDI file v1.0 \"%s\"..." %args.outfile)
  print ("Cyls:", cyls," Sides:", sides)


  data=bytearray()
  length=0
  track_range=len(tracks)
  for tr in range(track_range):
    tlen=len(tracks[tr])
    length+=(3+tlen+len(syncs[tr]))
    data+=struct.pack("<BH",0,tlen) #0=mfm
    data+=tracks[tr]
    data+=syncs[tr]
#    if len(sectors[tr])!=32: print(sectors[tr])
    print('.',end='')
#  print(length)

  header = struct.pack("<4sI4BI",
    b"UDI!",   # Signature
    length+16, # File length - last 4 bytes of CRC (head+tracks)
    0,         # Version
    cyls-1,    # Cylinders-1 (0x4f=80)
    sides-1,   # Sides-1 (0=1side, 1=2sides, 3..ff-reserved
    0,         # unused
    0)         # EXTHDL size


  _CRC=0xffffffff
  print("\nCalculating CRC...")
  udicrc(header)
  udicrc(data)
  print("Writing file...")
  with open(args.outfile, "wb") as f:
    f.write(header)
    f.write(data)
    f.write(struct.pack("<I",_CRC))
#  print('%04x'%_CRC)
  



def _main(argv):
  parser = argparse.ArgumentParser(
    formatter_class=argparse.ArgumentDefaultsHelpFormatter)
  parser.add_argument("infile", help="in filename")
  parser.add_argument("outfile", help="out filename")
  args = parser.parse_args(argv[1:])

  importhfe(args)
  exportudi(args)
  '''
  with open(args.file, "rb") as f:
    dat=f.read()
  global _CRC
  _CRC=0xffffffff
  udicrc(dat[0:-4])
  print("CALC=%08x"%_CRC)
  _CRC=struct.unpack("<I",dat[-4:])[0]
  print("FILE=%08x"%_CRC)
  '''

if __name__ == "__main__":
  _main(sys.argv)
