#include "stdinc.h"
#include "types.h"
#include "config.h"
#include "crc.h"

#define FILENAME_EEPROM_SIZE 125
#define TYPE_STRING 0
#define TYPE_BINARY 1

#if BETADSK_PRESENT

static int get_info(int item, word *offset, byte *type)
{
  int sz;

  *type = TYPE_STRING;
  sz = FILENAME_EEPROM_SIZE;

  switch (item) {
    case CFG_TAPE_FILENAME:
      *offset = 0;
      break;
    case CFG_DRVA_FILENAME:
      *offset = (1 + FILENAME_EEPROM_SIZE + 2) * 1;
      break;
    case CFG_DRVB_FILENAME:
      *offset = (1 + FILENAME_EEPROM_SIZE + 2) * 2;
      break;
    case CFG_DRVC_FILENAME:
      *offset = (1 + FILENAME_EEPROM_SIZE + 2) * 3;
      break;
    case CFG_DRVD_FILENAME:
      *offset = (1 + FILENAME_EEPROM_SIZE + 2) * 4;
      break;
    case CFG_FLAGS:
      *offset = (1 + FILENAME_EEPROM_SIZE + 2) * 5;  // size + data[] + crc
      *type = TYPE_BINARY;
      sz = sizeof(word);
    default:
      return 0;
  }
  return sz;
}

int read_config(int item, char *dst)
{
  int max_sz;
  word offset, crc;
  byte sz_in_epm, crc_epm, type, i, csreg;
  
  max_sz = get_info(item, &offset, &type);
  if (max_sz == 0) {
    return ERR_BAD_ITEM;
  }

  csreg = SREG;
  cli();
  sz_in_epm = eeprom_read_byte((uint8_t *)offset);
  if (sz_in_epm > max_sz) {
    SREG = csreg;
    return ERR_BAD_CONFIG;
  }
  eeprom_read_block(dst, (void *)(offset + 1), sz_in_epm);
  crc_epm = eeprom_read_byte((uint8_t *)(offset + max_sz));
  SREG = csreg;
  crc = crc_init();
  for (i = 0; i < sz_in_epm; i++) {
    crc = crc_add(crc, dst[i]);
  }
  if ((((byte)(crc >> 8)) ^ ((byte)(crc & 0xff)) ^ crc_epm) != 0) {
    return ERR_BAD_CRC;
  }
  if (type == TYPE_STRING) {
    dst[sz_in_epm] = 0;
  }
  return sz_in_epm;
}

int write_config(int item, const char *dst)
{
  int max_sz;
  word offset;
  byte sz_in_epm, type, csreg, i;
  word crc;

  max_sz = get_info(item, &offset, &type);
  if (max_sz == 0) {
    return ERR_BAD_ITEM;
  }
  if (type == TYPE_STRING) {
    sz_in_epm = strlen(dst);
  } else {
    sz_in_epm = max_sz;
  }
  if (sz_in_epm > max_sz) {
    return ERR_BAD_CONFIG;
  }
  crc = crc_init();
  for (i = 0; i < sz_in_epm; i++) {
    crc = crc_add(crc, dst[i]);
  }
  csreg = SREG;
  cli();
  eeprom_write_byte((uint8_t *)offset, sz_in_epm);
  eeprom_write_block(dst, (void *)(offset + 1), sz_in_epm);
  eeprom_write_byte((uint8_t *)(offset + max_sz), ((byte)(crc >> 8)) ^ ((byte)(crc & 0xff)));
  SREG = csreg;
  return sz_in_epm;
}

#endif
