/*
 * AVR keyboard firmware for ps2_cpld_kbd project
 * 
 * Designed to build on Arduino IDE.
 * 
 * @author Andy Karpov <andy.karpov@gmail.com>
 * Ukraine, 2019
 * 
 * AVR keyboard firmware mod for use with HC595 by Vladimir Gordeychik <vvgordeychik@gmail.com>
 *  
 */

#include "ps2.h"
#include "matrix.h"
#include "ps2_codes.h"
#include <SPI.h>

#define DEBUG_MODE 0

// ---- Pins for Atmega328
#define PIN_KBD_CLK 3 // PD3
#define PIN_KBD_DAT 2 // PD2

// hardware SPI
#define PIN_SS 9 // SPI slave select

// nes gamepad
#define PIN_JDAT  6
#define PIN_JLT  5
#define PIN_JCLK  4
#define JDELAY 10
#define JSER_COUNT 8

#define GP_K_A 0
#define GP_K_B 1
#define GP_K_SEL 2
#define GP_K_START 3
#define GP_K_U 4
#define GP_K_D 5
#define GP_K_L 6
#define GP_K_R 7

//service
#define LED_KBD A0
#define PIN_RST A1
#define PIN_MAGIC 7

#define MAGIC_HOLD_TIME 200
#define RST_HOLD_TIME 500


PS2KeyRaw kbd;

bool matrix[ZX_MATRIX_FULL_SIZE]; // matrix of pressed keys + special keys to be transmitted on CPLD side by SPI protocol
bool gamepad_matrix[JSER_COUNT];

bool blink_state = false;

unsigned long t = 0;  // current time
unsigned long tl = 0; // blink poll time
unsigned long te = 0; // eeprom store time

SPISettings settingsA(8000000, MSBFIRST, SPI_MODE0); // SPI transmission settings

// transform PS/2 scancodes into internal matrix of pressed keys
void fill_kbd_matrix(int sc)
{

  static bool is_up=false, is_e=false, is_e1=false;
  static bool is_ctrl=false, is_alt=false, is_del=false, is_bksp = false, is_shift = false, is_esc = false, is_ss_used = false, is_cs_used = false;

  // is extended scancode prefix
  if (sc == 0xE0) {
    is_e = 1;
    return;
  }

 if (sc == 0xE1) {
    is_e = 1;
    is_e1 = 1;
    return;
  }

  // is key released prefix
  if (sc == 0xF0 && !is_up) {
    is_up = 1;
    return;
  }

  int scancode = sc + ((is_e || is_e1) ? 0x100 : 0);

  is_ss_used = false;
  is_cs_used = false;

  switch (scancode) {
  
    // Shift -> CS for ZX
    case PS2_L_SHIFT: 
    case PS2_R_SHIFT:
      matrix[ZX_K_CS] = !is_up;
      is_shift = !is_up;
      break;

    // Ctrl -> SS for ZX
    case PS2_L_CTRL:
    case PS2_R_CTRL:
      matrix[ZX_K_SS] = !is_up;
      is_ctrl = !is_up;
      break;

    // Alt (L) -> SS+CS for ZX
    case PS2_L_ALT:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_CS] = !is_up;
      is_alt = !is_up;
      is_cs_used = !is_up;
      break;

    // Alt (R) -> SS+CS for ZX
    case PS2_R_ALT:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_CS] = !is_up;
      is_alt = !is_up;
      is_cs_used = !is_up;
      break;

    // Del -> SS+C for ZX
    case PS2_DELETE:
       matrix[ZX_K_SS] = !is_up;
       matrix[ZX_K_C] =  !is_up;
      is_del = !is_up;
    break;

    // Ins -> SS+A for ZX
    case PS2_INSERT:
       matrix[ZX_K_SS] = !is_up;
       matrix[ZX_K_A] =  !is_up;
    break;

    // Cursor -> CS + 5,6,7,8
    case PS2_UP:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_7] = !is_up;
      is_cs_used = !is_up;
      break;
    case PS2_DOWN:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_6] = !is_up;
      is_cs_used = !is_up;
      break;
    case PS2_LEFT:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_5] = !is_up;
      is_cs_used = !is_up;
      break;
    case PS2_RIGHT:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_8] = !is_up;
      is_cs_used = !is_up;
      break;

    // ESC -> CS+SPACE for ZX
    case PS2_ESC:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_SP] = !is_up;
      is_cs_used = !is_up;
      is_esc = !is_up;
      break;

    // Backspace -> CS+0
    case PS2_BACKSPACE:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_0] = !is_up;
      is_cs_used = !is_up;
      is_bksp = !is_up;
      break;

    // Enter
    case PS2_ENTER:
    case PS2_KP_ENTER:
      matrix[ZX_K_ENT] = !is_up;
      break;

    // Space
    case PS2_SPACE:
      matrix[ZX_K_SP] = !is_up;
      break;

    // Letters & numbers
    case PS2_A: matrix[ZX_K_A] = !is_up; break;
    case PS2_B: matrix[ZX_K_B] = !is_up; break;
    case PS2_C: matrix[ZX_K_C] = !is_up; break;
    case PS2_D: matrix[ZX_K_D] = !is_up; break;
    case PS2_E: matrix[ZX_K_E] = !is_up; break;
    case PS2_F: matrix[ZX_K_F] = !is_up; break;
    case PS2_G: matrix[ZX_K_G] = !is_up; break;
    case PS2_H: matrix[ZX_K_H] = !is_up; break;
    case PS2_I: matrix[ZX_K_I] = !is_up; break;
    case PS2_J: matrix[ZX_K_J] = !is_up; break;
    case PS2_K: matrix[ZX_K_K] = !is_up; break;
    case PS2_L: matrix[ZX_K_L] = !is_up; break;
    case PS2_M: matrix[ZX_K_M] = !is_up; break;
    case PS2_N: matrix[ZX_K_N] = !is_up; break;
    case PS2_O: matrix[ZX_K_O] = !is_up; break;
    case PS2_P: matrix[ZX_K_P] = !is_up; break;
    case PS2_Q: matrix[ZX_K_Q] = !is_up; break;
    case PS2_R: matrix[ZX_K_R] = !is_up; break;
    case PS2_S: matrix[ZX_K_S] = !is_up; break;
    case PS2_T: matrix[ZX_K_T] = !is_up; break;
    case PS2_U: matrix[ZX_K_U] = !is_up; break;
    case PS2_V: matrix[ZX_K_V] = !is_up; break;
    case PS2_W: matrix[ZX_K_W] = !is_up; break;
    case PS2_X: matrix[ZX_K_X] = !is_up; break;
    case PS2_Y: matrix[ZX_K_Y] = !is_up; break;
    case PS2_Z: matrix[ZX_K_Z] = !is_up; break;

    // digits
    case PS2_0: matrix[ZX_K_0] = !is_up; break;
    case PS2_1: matrix[ZX_K_1] = !is_up; break;
    case PS2_2: matrix[ZX_K_2] = !is_up; break;
    case PS2_3: matrix[ZX_K_3] = !is_up; break;
    case PS2_4: matrix[ZX_K_4] = !is_up; break;
    case PS2_5: matrix[ZX_K_5] = !is_up; break;
    case PS2_6: matrix[ZX_K_6] = !is_up; break;
    case PS2_7: matrix[ZX_K_7] = !is_up; break;
    case PS2_8: matrix[ZX_K_8] = !is_up; break;
    case PS2_9: matrix[ZX_K_9] = !is_up; break;

    // Keypad digits
    case PS2_KP_0: matrix[ZX_K_0] = !is_up; break;
    case PS2_KP_1: matrix[ZX_K_1] = !is_up; break;
    case PS2_KP_2: matrix[ZX_K_2] = !is_up; break;
    case PS2_KP_3: matrix[ZX_K_3] = !is_up; break;
    case PS2_KP_4: matrix[ZX_K_4] = !is_up; break;
    case PS2_KP_5: matrix[ZX_K_5] = !is_up; break;
    case PS2_KP_6: matrix[ZX_K_6] = !is_up; break;
    case PS2_KP_7: matrix[ZX_K_7] = !is_up; break;
    case PS2_KP_8: matrix[ZX_K_8] = !is_up; break;
    case PS2_KP_9: matrix[ZX_K_9] = !is_up; break;

    // '/" -> SS+P / SS+7
    case PS2_QUOTE:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_P : ZX_K_7] = !is_up;
      is_ss_used = is_shift;
      break;

    // ,/< -> SS+N / SS+R
    case PS2_COMMA:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_R : ZX_K_N] = !is_up;
      is_ss_used = is_shift;
      break;

    // ./> -> SS+M / SS+T
    case PS2_PERIOD:
    case PS2_KP_PERIOD:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_T : ZX_K_M] = !is_up;
      is_ss_used = is_shift;
      break;

    // ;/: -> SS+O / SS+Z
    case PS2_SEMICOLON:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_Z : ZX_K_O] = !is_up;
      is_ss_used = is_shift;
      break;

    // [,{ -> SS+Y / SS+F
    case PS2_L_BRACKET:
      if (!is_up) {
        send_macros(is_shift ? ZX_K_F : ZX_K_Y);
      }
      break;

    // ],} -> SS+U / SS+G
    case PS2_R_BRACKET:
      if (!is_up) {
        send_macros(is_shift ? ZX_K_G : ZX_K_U);
      }
      break;

    // /,? -> SS+V / SS+C
    case PS2_SLASH:
    case PS2_KP_SLASH:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_C : ZX_K_V] = !is_up;
      is_ss_used = is_shift;
      break;

    // \,| -> SS+D / SS+S
    case PS2_BACK_SLASH:
      if (!is_up) {
        send_macros(is_shift ? ZX_K_S : ZX_K_D);
      }
      break;

    // =,+ -> SS+L / SS+K
    case PS2_EQUALS:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_K : ZX_K_L] = !is_up;
      is_ss_used = is_shift;
      break;

    // -,_ -> SS+J / SS+0
    case PS2_MINUS:
      matrix[ZX_K_SS] = !is_up;
      matrix[is_shift ? ZX_K_0 : ZX_K_J] = !is_up;
      is_ss_used = is_shift;
      break;

    // `,~ -> SS+X / SS+A
    case PS2_ACCENT:
      if (is_shift and !is_up) {
        send_macros(is_shift ? ZX_K_A : ZX_K_X);
      }
      if (!is_shift) {
        matrix[ZX_K_SS] = !is_up;
        matrix[ZX_K_X] = !is_up;
        is_ss_used = is_shift;
      }
      break;

    // Keypad * -> SS+B
    case PS2_KP_STAR:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_B] = !is_up;
      break;

    // Keypad - -> SS+J
    case PS2_KP_MINUS:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_J] = !is_up;
      break;

    // Keypad + -> SS+K
    case PS2_KP_PLUS:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_K] = !is_up;
      break;

    // Tab
    case PS2_TAB:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_I] = !is_up;
      is_cs_used = !is_up;
      break;

    // CapsLock
    case PS2_CAPS:
      matrix[ZX_K_SS] = !is_up;
      matrix[ZX_K_CS] = !is_up;
      is_cs_used = !is_up;
      break;

    // PgUp -> CS+3 for ZX
    case PS2_PGUP:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_3] = !is_up;
      is_cs_used = !is_up;
      break;

    // PgDn -> CS+4 for ZX
    case PS2_PGDN:
      matrix[ZX_K_CS] = !is_up;
      matrix[ZX_K_4] = !is_up;
      is_cs_used = !is_up;
      break;

    // F5 -> Magick button
    case PS2_F5:
      if (is_up) {
        do_magick();
      }
    break;
  
  }

  if (is_ss_used and !is_cs_used) {
      matrix[ZX_K_CS] = false;
  }

  // Ctrl+Alt+Del -> RESET
  if (is_ctrl && is_alt && is_del) {
    is_ctrl = false;
    is_alt = false;
    is_del = false;
    is_shift = false;
    is_ss_used = false;
    is_cs_used = false;
    do_reset();
  }

  // Ctrl+Alt+Bksp -> REINIT controller
  if (is_ctrl && is_alt && is_bksp) {
      is_ctrl = false;
      is_alt = false;
      is_bksp = false;
      is_shift = false;
      is_ss_used = false;
      is_cs_used = false;
      clear_matrix(ZX_MATRIX_SIZE);
      matrix[ZX_K_RESET] = true;
      transmit_keyboard_matrix();
      matrix[ZX_K_S] = true;
      transmit_keyboard_matrix();
      delay(500);
      matrix[ZX_K_RESET] = false;
      transmit_keyboard_matrix();
      delay(500);
      matrix[ZX_K_S] = false;
  }

   // clear flags
   is_up = 0;
   if (is_e1) {
    is_e1 = 0;
   } else {
     is_e = 0;
   }
}

uint8_t get_matrix_byte(uint8_t pos)
{
  uint8_t result = 0;
  for (uint8_t i=0; i<8; i++) {
    uint8_t k = pos*8 + i;
    if (k < ZX_MATRIX_FULL_SIZE) {
      bitWrite(result, i, matrix[k]);
    }
  }
  return result;
}

uint8_t get_A11_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_1]);
  bitWrite(res, 1, matrix[ZX_K_2]);
  bitWrite(res, 2, matrix[ZX_K_3]);
  bitWrite(res, 3, matrix[ZX_K_4]);
  bitWrite(res, 4, matrix[ZX_K_5]);
  return res;
}

uint8_t get_A10_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_Q]);
  bitWrite(res, 1, matrix[ZX_K_W]);
  bitWrite(res, 2, matrix[ZX_K_E]);
  bitWrite(res, 3, matrix[ZX_K_R]);
  bitWrite(res, 4, matrix[ZX_K_T]);
  return res;
}

uint8_t get_A9_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_A]);
  bitWrite(res, 1, matrix[ZX_K_S]);
  bitWrite(res, 2, matrix[ZX_K_D]);
  bitWrite(res, 3, matrix[ZX_K_F]);
  bitWrite(res, 4, matrix[ZX_K_G]);
  return res;  
}

uint8_t get_A12_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_0] | gamepad_matrix[GP_K_A]);
  bitWrite(res, 1, matrix[ZX_K_9] | gamepad_matrix[GP_K_U]);
  bitWrite(res, 2, matrix[ZX_K_8] | gamepad_matrix[GP_K_D]);
  bitWrite(res, 3, matrix[ZX_K_7] | gamepad_matrix[GP_K_R]);
  bitWrite(res, 4, matrix[ZX_K_6] | gamepad_matrix[GP_K_L]);
  return res;
}

uint8_t get_A13_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_P]);
  bitWrite(res, 1, matrix[ZX_K_O]);
  bitWrite(res, 2, matrix[ZX_K_I]);
  bitWrite(res, 3, matrix[ZX_K_U]);
  bitWrite(res, 4, matrix[ZX_K_Y]);
  return res;
}

uint8_t get_A8_byte()
{
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_CS]);
  bitWrite(res, 1, matrix[ZX_K_Z]);
  bitWrite(res, 2, matrix[ZX_K_X]);
  bitWrite(res, 3, matrix[ZX_K_C]);
  bitWrite(res, 4, matrix[ZX_K_V]);
  return res;
}

uint8_t get_A14_byte()
{  
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_ENT] | gamepad_matrix[GP_K_START] );
  bitWrite(res, 1, matrix[ZX_K_L]  | gamepad_matrix[GP_K_SEL]);
  bitWrite(res, 2, matrix[ZX_K_K]);
  bitWrite(res, 3, matrix[ZX_K_J]);
  bitWrite(res, 4, matrix[ZX_K_H]);
  return res;
}

uint8_t get_A15_byte()
{  
  uint8_t res = 0;
  bitWrite(res, 0, matrix[ZX_K_SP]);
  bitWrite(res, 1, matrix[ZX_K_SS]);
  bitWrite(res, 2, matrix[ZX_K_M]  | gamepad_matrix[GP_K_B] );
  bitWrite(res, 3, matrix[ZX_K_N]);
  bitWrite(res, 4, matrix[ZX_K_B]);
  return res;
}

void spi_send() 
{
      SPI.beginTransaction(settingsA);
      digitalWrite(PIN_SS, LOW);
      
      SPI.transfer(~get_A11_byte());
      SPI.transfer(~get_A10_byte());
      SPI.transfer(~get_A9_byte());
      SPI.transfer(~get_A12_byte());
      SPI.transfer(~get_A13_byte());
      SPI.transfer(~get_A8_byte());
      SPI.transfer(~get_A14_byte());
      SPI.transfer(~get_A15_byte());
      
      digitalWrite(PIN_SS, HIGH);
      SPI.endTransaction();
}

// transmit keyboard matrix from AVR to HC595 side via SPI
void transmit_keyboard_matrix()
{    
  spi_send();    
}

// transmit keyboard macros (sequence of keyboard clicks) to emulate typing some special symbols [, ], {, }, ~, |, `
void send_macros(uint8_t pos)
{
  clear_matrix(ZX_MATRIX_SIZE);
  transmit_keyboard_matrix();
  delay(20);
  matrix[ZX_K_CS] = true;
  transmit_keyboard_matrix();
  delay(20);
  matrix[ZX_K_SS] = true;
  transmit_keyboard_matrix();
  delay(20);
  matrix[ZX_K_SS] = false;
  transmit_keyboard_matrix();
  delay(20);
  matrix[pos] = true;
  transmit_keyboard_matrix();
  delay(20);
  matrix[ZX_K_CS] = false;
  matrix[pos] = false;
  transmit_keyboard_matrix();
  delay(20);
}

void do_reset()
{
  digitalWrite(PIN_RST, LOW); 
  pinMode(PIN_RST, OUTPUT); //switch to GND
  delay(RST_HOLD_TIME);
  pinMode(PIN_RST, INPUT); //set Z-STATE
}

void do_magick()
{
  digitalWrite(PIN_RST, LOW); 
  pinMode(PIN_MAGIC, OUTPUT); //switch to GND
  delay(MAGIC_HOLD_TIME);
  pinMode(PIN_MAGIC, INPUT);  //set Z-STATE
}

void clear_matrix(int clear_size)
{
    // all keys up
  for (int i=0; i<clear_size; i++) {
      matrix[i] = false;
  }
}

void clear_gamepad_matrix(int clear_size)
{
  for (int i=0; i<clear_size; i++)
  {
    gamepad_matrix[i] = false;
  }
}

void init_gamepad(int pin_data, int pin_latch, int pin_clock)
{
    pinMode(pin_data, INPUT_PULLUP);
    pinMode(pin_clock, OUTPUT);
    pinMode(pin_latch, OUTPUT);

    digitalWrite(pin_clock, HIGH);
}

int get_gamepad_state(int pin_data, int pin_latch, int pin_clock)
{
    digitalWrite(pin_latch, HIGH);
    delayMicroseconds(JDELAY);
    digitalWrite(pin_latch, LOW);

    int keys_state = 0;

    for (int i = 0; i < JSER_COUNT; ++i) {
        delayMicroseconds(JDELAY);
        digitalWrite(pin_clock, LOW);

        keys_state <<= 1;
        keys_state |= digitalRead(pin_data);

        delayMicroseconds(JDELAY);
        digitalWrite(pin_clock, HIGH);
    }

    return keys_state;  
}

void fill_gamepad_matrix(int jstate)
{
    
  gamepad_matrix[GP_K_A] = (jstate & 0x80)>0 ? false : true;
  gamepad_matrix[GP_K_B] = (jstate & 0x40)>0 ? false : true;
  gamepad_matrix[GP_K_SEL] = (jstate & 0x20)>0 ? false : true;
  gamepad_matrix[GP_K_START] = (jstate & 0x10)>0 ? false : true;
  gamepad_matrix[GP_K_U] = (jstate & 0x08)>0 ? false : true;
  gamepad_matrix[GP_K_D] = (jstate & 0x04)>0 ? false : true;
  gamepad_matrix[GP_K_L] = (jstate & 0x02)>0 ? false : true;
  gamepad_matrix[GP_K_R] = (jstate & 0x01)>0 ? false : true;
   
}

// initial setup
void setup()
{
  Serial.begin(115200);
  Serial.flush();
  SPI.begin();

  pinMode(PIN_SS, OUTPUT);
  digitalWrite(PIN_SS, HIGH);
    
  pinMode(LED_KBD, OUTPUT);  
  digitalWrite(LED_KBD, HIGH);
  
  // ps/2

  pinMode(PIN_KBD_CLK, INPUT_PULLUP);
  pinMode(PIN_KBD_DAT, INPUT_PULLUP);

  // gamepad
  init_gamepad(PIN_JDAT, PIN_JLT, PIN_JCLK);
  clear_gamepad_matrix(JSER_COUNT);

  // magic
  digitalWrite(PIN_MAGIC, LOW); 
  pinMode(PIN_MAGIC, INPUT); //set Z-STATE
  

  // reset
  digitalWrite(PIN_RST, LOW);
  pinMode(PIN_RST, INPUT); //set Z-STATE
  

  // zx signals (output)

  // clear full matrix
  clear_matrix(ZX_MATRIX_FULL_SIZE);
  
  Serial.println(F("ZX PS/2 Keyboard controller v1.0"));

#if DEBUG_MODE
  Serial.println(F("Reset on boot..."));
#endif

  //do_reset();

#if DEBUG_MODE
  Serial.println(F("done"));
  Serial.println(F("Keyboard init..."));
#endif

  kbd.begin(PIN_KBD_DAT, PIN_KBD_CLK);

#if DEBUG_MODE
  Serial.println(F("done"));
#endif

  digitalWrite(LED_KBD, LOW);
  
}

static int jstate = 0;

// main loop
void loop()
{
  
  unsigned long n = millis();
  
  if (kbd.available()) {
    int c = kbd.read();
    blink_state = true;
    tl = n;
    digitalWrite(LED_KBD, HIGH);
#if DEBUG_MODE    
    Serial.print(F("Scancode: "));
    Serial.println(c, HEX);
#endif
    fill_kbd_matrix(c);
  }

    
  jstate = get_gamepad_state(PIN_JDAT, PIN_JLT, PIN_JCLK);
  fill_gamepad_matrix(jstate);
  
  if (jstate != 0xFF)
  {
    blink_state = true;
    tl = n;
    digitalWrite(LED_KBD, HIGH);

    jstate = 0xFF;
  }

  // transmit kbd always
  transmit_keyboard_matrix();


  // update leds
  if (n - tl >= 200) {
    digitalWrite(LED_KBD, LOW);
    blink_state = false;
  }
}
