#include <sprite.h>

Sprite::Sprite()
{
	bitmap = new Graphics::TBitmap;
    bitmap->Width = 8;
    bitmap->Height = 8;

    attr = 56;

    RefreshBitmap();
}

Sprite::~Sprite()
{
	delete bitmap;
}

bool Sprite::Empty()
{
	for (int i = 0; i < 8; i++)
    {
		byte a = data[i];
        if ( a != 0)
        	return false;
    }
    return	true;
}

void Sprite::RefreshBitmap()
{
	int brt = (attr & 64) == 0 ? 192 : 255;

	int R = ((attr >> 1) % 2) * brt;
    int G = ((attr >> 2) % 2) * brt << 8;
    int B = (attr % 2) * brt << 16;
	TColor ink = TColor(R+G+B);

	R = ((attr >> 4) % 2) * brt;
    G = ((attr >> 5) % 2) * brt << 8;
    B = ((attr >> 3) % 2) * brt << 16;
	TColor paper = TColor(R+G+B);


	for (int y = 0; y < 8; y++)
    for (int x = 0; x < 8; x++)
    	bitmap->Canvas->Pixels[x][y] = (data[y] & (128 >> x)) == 0 ? paper : ink;
}

AnsiString Sprite::GetDataText()
{
    AnsiString line;
	for (int b = 0; b < 7; b++)
    line += IntToStr(data[b]) + ",";
    line += IntToStr(data[7]);
    return line;
}

void Sprite::LoadFromFile(AnsiString fname)
{
	TImage * src = new TImage(NULL);
	src->Picture->LoadFromFile(fname);
    bitmap->Canvas->StretchDraw(bitmap->Canvas->ClipRect, src->Picture->Graphic);
    delete src;

    for (int y = 0; y < 8; y++)
    {
    	data[y] = 0;
    	for (int x = 0; x < 8; x++)
    	{
    		TColor c = bitmap->Canvas->Pixels[x][y];
        	if (((GetRValue(c) + GetGValue(c) + GetBValue(c)) / 3) < 127)
        		data[y] |= (128 >> x);
    	}
    }
    RefreshBitmap();
}

//-----------------------------------

void RibSprite::Ink(int c)
{
	attr = (attr & 248) | c;
}

void RibSprite::Paper(int c)
{
	attr = (attr & 199) | (c << 3);
}

void RibSprite::Flash(bool c)
{
	attr = (attr & 127) | (c << 7);
}

void RibSprite::Bright(bool c)
{
	attr = (attr & 191) | (c << 6);
}

//-----------------------------------
SpriteSet::SpriteSet()
{
	bitmap = new Graphics::TBitmap;
    bitmap->Width = S_Size * 16 + gap * 17;
    bitmap->Height = S_Size * 16 + gap * 17;
    bitmap->Canvas->FillRect(bitmap->Canvas->ClipRect);
	current = 0;
    for (int i = 0; i < 256; i++) DrawSprite(i);
}

SpriteSet::~SpriteSet()
{
	delete bitmap;
}

void SpriteSet::DrawSprite(int nom)
{
	int x = gap + (nom % 16) * (S_Size + gap);
    int y = gap + (nom / 16) * (S_Size + gap);
    {
        sprite[nom].attr = 56;
        sprite[nom].RefreshBitmap();
    	bitmap->Canvas->StretchDraw(TRect(x,y,x+S_Size,y+S_Size), sprite[nom].bitmap);
	}
}

void SpriteSet::LoadCurrentFromFile(AnsiString fname)
{
	sprite[current].LoadFromFile(fname);
    DrawSprite(current);
}

void SpriteSet::SetCurrent(int x, int y)
{
	x /= S_Size + gap;
    y /= S_Size + gap;
    if (x > 15) return;
    if (y > 15) return;

    int xx = (current % 16) * (S_Size + gap);
    int yy = (current / 16) * (S_Size + gap);
    bitmap->Canvas->Pen->Color = clWhite;
    bitmap->Canvas->Rectangle(xx, yy, xx+S_Size+gap*2, yy+S_Size+gap*2);
    DrawSprite(current);

    current = y * 16 + x;

    xx = (current % 16) * (S_Size + gap);
    yy = (current / 16) * (S_Size + gap);
    bitmap->Canvas->Pen->Color = clBlue;
    bitmap->Canvas->Rectangle(xx, yy, xx+S_Size+gap*2, yy+S_Size+gap*2);
    DrawSprite(current);
}


void SpriteSet::SaveSet(AnsiString fname)
{
    ofstream file;
    file.open(fname.c_str(), ios::out | ios::binary);
    for (unsigned int i = 0; i < 256; i++) file.write(sprite[i].data, 8);
    for (unsigned int i = 0; i < 256; i++) file.write(&sprite[i].attr, 1);
	file.close();
}

void SpriteSet::LoadSet(AnsiString fname)
{
    ifstream file;
    file.open(fname.c_str(), ios::in | ios::binary);
    for (unsigned int i = 0; i < 256; i++)
    {
    	for (unsigned int b = 0; b < 8; b++)
    		if (!file.eof()) file.read(&(sprite[i].data[b]), 1);
            else sprite[i].data[b] = 0;
    }
    for (unsigned int i = 0; i < 256; i++)
    {
    	if (!file.eof()) file.read(&sprite[i].attr, 1);
        sprite[i].RefreshBitmap();
        DrawSprite(i);
    }
    file.close();
    
}

void SpriteSet::SaveCode(AnsiString fname)
{
	ofstream file;
	file.open(fname.c_str(), ios::out);

    //file << "TILES" << endl;


    int count = 0;;
    for (int i = 255; i >= 0; i--)
    	if (!sprite[i].Empty())
        {
        	count = i;
            break;
        }

	AnsiString line;
	for (int i = 0; i <= count; i++)
    {
		line = "\tDEFB\t";

        for (int b = 0; b < 7; b++) line += IntToStr(sprite[i].data[b]) + ",";
        line += IntToStr(sprite[i].data[7]);
        line += "\t;" + IntToStr(i);

        file << line.c_str() << endl;
	}

	file.close();
}

//--------------------------------------------------------

Block::Block()
{
    rib_noms.clear();
	current = 0;
    name = "";
}

Block::Block(const Block & b)
{
    rib_noms.clear();
	for(unsigned int i = 0; i < b.rib_noms.size(); i++)
    	rib_noms.push_back(b.rib_noms[i]);
    current = b.current;
    name = b.name;
}



//--------------------------------------------------------

BlockSet::BlockSet()
{
    bitmap = new Graphics::TBitmap;
    current = 0;
}

BlockSet::~BlockSet()
{
	delete bitmap;
}


void BlockSet::EraseRecalc(int n)
{
    for (unsigned int i = 0; i < blocks.size(); i++)
    {
        unsigned int rns = blocks[i].rib_noms.size();
        if (rns == 0) continue;
        for (int rn = rns-1; rn >= 0; rn--)
        if (blocks[i].rib_noms[rn] == n)
            blocks[i].rib_noms.erase(blocks[i].rib_noms.begin()+rn);
        else
            if (blocks[i].rib_noms[rn] > n) blocks[i].rib_noms[rn]--;


    }
    RefreshBitmap();
}

void BlockSet::InsertRecalc(int n)
{
    for (unsigned int i = 0; i < blocks.size(); i++)
    {
        for (unsigned int rn = 0; rn < blocks[i].rib_noms.size(); rn++)
        if (blocks[i].rib_noms[rn] >= n) blocks[i].rib_noms[rn]++;
    }
    RefreshBitmap();

}

void BlockSet::InsertRibbon()
{
    if (blocks.size() == 0) return;
    unsigned int c = blocks[current].current;
    if (c >= blocks[current].rib_noms.size()) c = blocks[current].rib_noms.size();
    blocks[current].rib_noms.insert(blocks[current].rib_noms.begin()+c, r_set->current);
    RefreshBitmap();

}

void BlockSet::InsertBlock()
{
	unsigned int c = current;

    if (c >= blocks.size()) c = blocks.size();

	blocks.insert(blocks.begin()+c);
    RefreshBitmap();
}

void BlockSet::DeleteBlock()
{
    if (blocks.size() == 0) return;
	unsigned int c = current;

    if (c >= blocks.size()) c = blocks.size()-1;

	blocks.erase(blocks.begin()+c);
    RefreshBitmap();
}

void BlockSet::RefreshBitmap()
{
    if (blocks.size() == 0) return;
	UpdateBitmapSize();
    bitmap->Canvas->Brush->Color = clBtnShadow;
    bitmap->Canvas->FillRect(bitmap->Canvas->ClipRect);
    for (unsigned int i = 0; i < blocks[current].rib_noms.size(); i++)
        DrawRibbon(i, blocks[current].rib_noms[i]);
    DrawCursor();
}

void BlockSet::UpdateBitmapSize()
{
    //if (blocks.size() == 0) return;
	unsigned int max = 0;
    unsigned int rsize = blocks[current].rib_noms.size();
    for (unsigned int i = 0; i < rsize; i++)
    {
        int ribnom = blocks[current].rib_noms[i];
        if (r_set->ribbons[ribnom].sprite.size() > max) max = r_set->ribbons[ribnom].sprite.size();
    }
	bitmap->Width =  gap + (S_Size + gap) * max + 50;
    bitmap->Height = gap + (S_Size + gap) * rsize + 50;
}



void BlockSet::DrawRibbon(int ry, int rnom)
{
    for (unsigned int i = 0; i < r_set->ribbons[rnom].sprite.size(); i++)
    {
    	int x = gap + i * (S_Size + gap);
    	int y = gap + ry * (S_Size + gap);
        unsigned int snom = r_set->ribbons[rnom].sprite[i].num;
        r_set->sprite_set.sprite[snom].attr = r_set->ribbons[rnom].sprite[i].attr;
        r_set->sprite_set.sprite[snom].RefreshBitmap();
    	bitmap->Canvas->StretchDraw(TRect(x,y,x+S_Size,y+S_Size), r_set->sprite_set.sprite[snom].bitmap);
	}
}

void BlockSet::SetCurrent(unsigned int c)
{
    current = c;

    if (current > blocks.size()) current = blocks.size();


    RefreshBitmap();
}

void BlockSet::SetRibCurrent(unsigned int y)
{
    if (blocks.size() == 0) return;
    y /= S_Size + gap;


    if (y >= blocks[current].rib_noms.size())
		y = blocks[current].rib_noms.size();

    blocks[current].current = y;

    RefreshBitmap();
}

void BlockSet::DrawCursor()
{
    if (blocks[current].rib_noms.size() == 0) return;
    int xx = 0;
    if (blocks[current].current >= blocks[current].rib_noms.size())
        xx = S_Size + gap;
    else
    {
        int ribnom = blocks[current].rib_noms[blocks[current].current];
        xx = r_set->ribbons[ribnom].sprite.size() * (S_Size + gap);
    }

    int yy = blocks[current].current * (S_Size + gap);
    bitmap->Canvas->Brush->Style = bsClear;
    bitmap->Canvas->Pen->Color = clYellow;
    bitmap->Canvas->Rectangle(0, yy, xx+gap, yy+S_Size+gap*2);

}

//-----------------------------------

RibbonSet::RibbonSet()
{
	bitmap = new Graphics::TBitmap;
}

RibbonSet::~RibbonSet()
{
	delete bitmap;
}

void RibbonSet::SetCurrent(unsigned int x, unsigned int y)
{
        //RefreshBitmap();

	x /= S_Size + gap;
    y /= S_Size + gap;

    //int cx = 0;

    if (y >= ribbons.size())
		current = ribbons.size();
    else
    {
		if (x >= ribbons[y].sprite.size()) x = ribbons[y].sprite.size();
        ribbons[y].current = x;
		current = y;
	}

    RefreshBitmap();
}

void RibbonSet::DrawCursor()
{
	int xx = 0;
	if (current < ribbons.size())
		xx = ribbons[current].current * (S_Size + gap);
    int yy = current * (S_Size + gap);
    bitmap->Canvas->Brush->Style = bsClear;
    bitmap->Canvas->Pen->Color = clYellow;
    bitmap->Canvas->Rectangle(xx, yy, xx+S_Size+gap*2, yy+S_Size+gap*2);

}

void RibbonSet::DrawRibbon(int nom)
{
    for (unsigned int i = 0; i < ribbons[nom].sprite.size(); i++)
    {
    	int x = gap + i * (S_Size + gap);
    	int y = gap + nom * (S_Size + gap);

        sprite_set.sprite[ribbons[nom].sprite[i].num].attr = ribbons[nom].sprite[i].attr;
        sprite_set.sprite[ribbons[nom].sprite[i].num].RefreshBitmap();
    	bitmap->Canvas->StretchDraw(TRect(x,y,x+S_Size,y+S_Size), sprite_set.sprite[ribbons[nom].sprite[i].num].bitmap);
	}
}

void RibbonSet::RefreshBitmap()
{
	UpdateBitmapSize();
    bitmap->Canvas->Brush->Color = clBtnShadow;
    bitmap->Canvas->FillRect(bitmap->Canvas->ClipRect);
    for (unsigned int i = 0; i < ribbons.size(); i++)
    	DrawRibbon(i);
    DrawCursor();
}

void RibbonSet::UpdateBitmapSize()
{
	unsigned int max = 0;
    for (unsigned int i = 0; i < ribbons.size(); i++)
    if (ribbons[i].sprite.size() > max) max = ribbons[i].sprite.size();

	bitmap->Width =  gap + (S_Size + gap) * max + 50;
    bitmap->Height = gap + (S_Size + gap) * ribbons.size() + 50;
}

void RibbonSet::InsertRibbon()
{
	unsigned int c = current;

    if (c >= ribbons.size()) c = ribbons.size();

	ribbons.insert(ribbons.begin()+c);
    r_bitmap.insert(r_bitmap.begin()+c);
    r_bitmap[c] = new Graphics::TBitmap;

    current = c+1;
    RefreshBitmap();
}

void RibbonSet::DeleteRibbon()
{
	if (ribbons.size() == 0) return;

	ribbons.erase(ribbons.begin() + current);
    delete r_bitmap[current];
    r_bitmap.erase(r_bitmap.begin() + current);
    if (current != 0) current--;
    RefreshBitmap();
}

void RibbonSet::RefreshRibbonBitmap(int nom)
{
	r_bitmap[nom]->Height = S_Level;
	r_bitmap[nom]->Width = ribbons[nom].sprite.size() * S_Level;
    r_bitmap[nom]->Canvas->Brush->Color = clBtnShadow;
    r_bitmap[nom]->Canvas->FillRect(r_bitmap[nom]->Canvas->ClipRect);


    for (unsigned int i = 0; i < ribbons[nom].sprite.size(); i++)
    {
    	int x = i * S_Level;

        sprite_set.sprite[ribbons[nom].sprite[i].num].attr = ribbons[nom].sprite[i].attr;
        sprite_set.sprite[ribbons[nom].sprite[i].num].RefreshBitmap();
    	r_bitmap[nom]->Canvas->StretchDraw(TRect(x,0,x+S_Level-1,S_Level-1), sprite_set.sprite[ribbons[nom].sprite[i].num].bitmap);
        //r_bitmap[nom]->Canvas->TextOutA(x,0, IntToStr(ribbons[nom].sprite[i].num));

	}
}

void RibbonSet::InsertSprite()
{
	if (current >= ribbons.size())
    {
    	InsertRibbon();
        current--;
    }

	ribbons[current].InsertSprite(sprite_set.current);

    RefreshRibbonBitmap(current);
    RefreshBitmap();
}

void RibbonSet::DeleteSprite()
{
	if (ribbons.size() == 0) return;

    ribbons[current].DeleteSprite();
    RefreshRibbonBitmap(current);
    RefreshBitmap();
}

void RibbonSet::SaveSet(AnsiString fname)
{  /////////////////////////////////  DEL
	ofstream file;
    file.open(fname.c_str(), ios::out | ios::binary);
    for (unsigned int y = 0; y < ribbons.size(); y++)
    {
    	unsigned short i;
    	for (unsigned int x = 0; x < ribbons[y].sprite.size(); x++)
        {
        	i = ribbons[y].sprite[x].num;
    		file.write((char *)&i, sizeof(i));
            i = ribbons[y].sprite[x].attr;
    		file.write((char *)&i, sizeof(i));
        }
        i = 256;
        file.write((char *)&i, sizeof(i));
    }
    file.close();
}

void RibbonSet::SaveCode(AnsiString fname)
{
	ofstream file;
	file.open(fname.c_str(), ios::out);

    //file << "RIBBONS" << endl;

	AnsiString line;
	for (unsigned int i = 0; i < ribbons.size(); i++)
    {
		line = "\tDEFB\t";

        for (unsigned int b = 0; b < ribbons[i].sprite.size(); b++)
        {
        	line += IntToStr(ribbons[i].sprite[b].num) + ",";
            line += IntToStr(ribbons[i].sprite[b].attr) + ",";
        }
        line += IntToStr(255);

        file << line.c_str() << endl;
	}

	file.close();

}

void RibbonSet::LoadSet(AnsiString fname)
{  /////////////////////////   DEL

    for(unsigned int i = 0; i < ribbons.size(); i++)
    delete r_bitmap[i];
    r_bitmap.clear();
    ribbons.clear();

	ifstream file;
    file.open(fname.c_str(), ios::in | ios::binary);

    unsigned short i;

    for(file.read((char *)&i, sizeof(i)); !file.eof(); file.read((char *)&i, sizeof(i)))
    {
    	if (ribbons.size() == 0)
        {
        	ribbons.push_back();
            r_bitmap.push_back();
			r_bitmap[r_bitmap.size() - 1] = new Graphics::TBitmap;
        }

    	if (i == 256)
        {
        	ribbons.push_back();
            r_bitmap.push_back();
			r_bitmap[r_bitmap.size() - 1] = new Graphics::TBitmap;
        }
		else
        {
        	unsigned short attr;
            file.read((char *)&attr, sizeof(attr));
        	ribbons[ribbons.size() - 1].AddSprite((unsigned char) i, (unsigned char) attr);
		}
    }
    file.close();
    ribbons.erase(ribbons.end());
    delete r_bitmap[r_bitmap.size() - 1];
    r_bitmap.pop_back();

    for(unsigned int i = 0; i < ribbons.size(); i++)
    	RefreshRibbonBitmap(i);
    RefreshBitmap();


}


//-------------------------------------------

Ribbon::Ribbon()
{
	current = 0;
}

Ribbon::Ribbon(const Ribbon & r)
{
	for(unsigned int i = 0; i < r.sprite.size(); i++)
    	sprite.push_back(r.sprite[i]);
    current = r.current;
}

void Ribbon::InsertSprite(int nom)
{
	RibSprite rs;
    rs.num = nom;
    rs.attr = 56;
    
    sprite.insert(sprite.begin()+current, rs);
    current++;
}

void Ribbon::AddSprite(int nom, int attr)
{
	RibSprite rs;
    rs.num = nom;
    rs.attr = attr;
    sprite.push_back(rs);
}

void Ribbon::DeleteSprite()
{
	if (sprite.size() == 0) return;

    sprite.erase(sprite.begin() + current);
    if (current > sprite.size()) current--;
}

//-------------------------------------------

Level::Level()
{
	bitmap = new Graphics::TBitmap;
    b_set.r_set = &r_set;
}

Level::~Level()
{
	delete bitmap;
}

void Level::DrawCursor()
{
	int mx = 1;
	if (lines[current_line].size() != 0)
    if (lines[current_line].size() > current_rib)
    	mx = r_set.ribbons[lines[current_line][current_rib]].sprite.size();
	int xx = 0;
    int yy = current_line * S_Level;
    for (unsigned int r = 0; r < current_rib; r++)
    	xx += r_set.ribbons[lines[current_line][r]].sprite.size() * S_Level;

	bitmap->Canvas->Brush->Style = bsClear;
    bitmap->Canvas->Pen->Color = clYellow;
    bitmap->Canvas->Rectangle(xx, yy, xx+S_Level*mx, yy+S_Level);

}

void Level::EraseRecalc(int n)
{

    for (unsigned int i = 0; i < 24; i++)
    {
        unsigned int ls = lines[i].size();
        if (ls == 0) continue;

        for (int rn = ls-1; rn >= 0; rn--)
            if (lines[i][rn] == n)
                lines[i].erase(lines[i].begin()+rn);
            else
                if (lines[i][rn] > n) lines[i][rn]--;


    }
    RefreshBitmap();
}

void Level::InsertRecalc(int n)
{
    for (unsigned int i = 0; i < 24; i++)
    {
        for (unsigned int rn = 0; rn < lines[i].size(); rn++)
        if (lines[i][rn] >= n) lines[i][rn]++;
    }
    RefreshBitmap();

}

void Level::DeleteRibbon()
{
	if (lines[current_line].size() == 0) return;
    if (current_rib >= lines[current_line].size())
    current_rib = lines[current_line].size() - 1;
    lines[current_line].erase(lines[current_line].begin() + current_rib);
    RefreshBitmap();
}

void Level::AddBlock()
{
    unsigned int old_line = current_line;
    unsigned int old_rib = current_rib;

    unsigned int y = current_line * S_Level;
    unsigned int x = S_Level;
    for (unsigned int r = 0; r < current_rib; r++)
        x += r_set.ribbons[lines[current_line][r]].sprite.size() * S_Level;

    Block * b = &b_set.blocks[b_set.current];

    for (unsigned int i = 0; i < b->rib_noms.size(); i++)
    {
        if (old_line + i > 23) break;
        SetCurrent(x, y + i * S_Level);
        lines[current_line].insert(lines[current_line].begin()+current_rib, b->rib_noms[i]);
    }

    current_line = old_line;
    current_rib = old_rib+1;


	RefreshBitmap();
}

void Level::AddRibbon()
{

	if (r_set.ribbons.size() == 0) return;
	unsigned int r = current_rib;

    if (r >= lines[current_line].size()) r = lines[current_line].size();


	lines[current_line].insert(lines[current_line].begin()+r, r_set.current);
    current_rib = r+1;
	RefreshBitmap();
}

void Level::SetCurrent(unsigned int x, unsigned int y)
{
	y /= S_Level;
    x /= S_Level;
    if (y > 23) return;
    current_line = y;
    current_rib = 0;
    unsigned int cx = 0;

    for (unsigned int r = 0; r < lines[current_line].size(); r++)
    {

    	if (x >= cx) current_rib = r;
        cx += r_set.ribbons[lines[current_line][r]].sprite.size();
    }
    if (x > cx) current_rib = lines[current_line].size();
    
}

void Level::DrawLine(int nom)
{
	int x = 0;
    int y = nom * S_Level;

	for (unsigned int r = 0; r < lines[nom].size(); r++)
    {
    	unsigned int r_nom = lines[nom][r];
        if (r_nom >= r_set.ribbons.size()) continue;
        
        bitmap->Canvas->Draw(x, y, r_set.r_bitmap[r_nom]);
        x += S_Level * r_set.ribbons[r_nom].sprite.size();
    }

}

void Level::UpdateBitmapSize()
{
	int max = 0;


	for (unsigned int l = 0; l < 24; l++)
    {
    	int sprites = 0;

    	for (unsigned int r = 0; r < lines[l].size(); r++)
        sprites += r_set.ribbons[lines[l][r]].sprite.size();

        sprites *= S_Level;
        if (sprites > max) max = sprites;
    }

    bitmap->Width = max + 50;
    bitmap->Height = 24 * S_Level;
}

void Level::RefreshBitmap()
{
	UpdateBitmapSize();
    bitmap->Canvas->Brush->Color = clBtnShadow;
    bitmap->Canvas->FillRect(bitmap->Canvas->ClipRect);
    for (unsigned int i = 0; i < 24; i++)
    	DrawLine(i);

    bitmap->Canvas->Pen->Color = clBlack;
    unsigned int scr_w = S_Level * 32;
    for (unsigned int i = 1; i <= bitmap->Width / scr_w; i++)
    {
    	bitmap->Canvas->MoveTo(i*scr_w, 0);
        bitmap->Canvas->LineTo(i*scr_w, bitmap->Height);
    }

    DrawCursor();

}

void Level::SaveRBSet(AnsiString fname)
{
    ofstream file;
    file.open(fname.c_str(), ios::out | ios::binary);

    unsigned short rs = r_set.ribbons.size();
    file.write((char *)&rs, sizeof(rs));  // 

    for (unsigned short y = 0; y < rs; y++)
    {
        unsigned short i;
    	for (unsigned int x = 0; x < r_set.ribbons[y].sprite.size(); x++)
        {
        	i = r_set.ribbons[y].sprite[x].num;
    		file.write((char *)&i, sizeof(i));
            i = r_set.ribbons[y].sprite[x].attr;
    		file.write((char *)&i, sizeof(i));
        }
        i = 256;
        file.write((char *)&i, sizeof(i));
    }

    unsigned short bs = b_set.blocks.size();
    file.write((char *)&bs, sizeof(bs));  // 


    for (unsigned short b = 0; b < bs; b++)
    {
        unsigned short i;
    	for (unsigned int r = 0; r < b_set.blocks[b].rib_noms.size(); r++)
        {
        	i = b_set.blocks[b].rib_noms[r];
    		file.write((char *)&i, sizeof(i));
        }
        i = 256;
        file.write((char *)&i, sizeof(i));

        AnsiString name = b_set.blocks[b].name;
        char c;
        for (int n = 1; n <= name.Length(); n++)
        {
            c = name[n];
            file.write((char *)&c, sizeof(c));
        }
        c = 0;
        file.write((char *)&c, sizeof(c));
    }









    file.close();

}

void Level::LoadRBSet(AnsiString fname)
{
    for(unsigned int i = 0; i < r_set.ribbons.size(); i++)
    delete r_set.r_bitmap[i];

    r_set.r_bitmap.clear();
    r_set.ribbons.clear();

	ifstream file;
    file.open(fname.c_str(), ios::in | ios::binary);

    unsigned short rs;
    file.read((char *)&rs, sizeof(rs));



    for (unsigned short i = 0; i < rs; i++)
    {

        r_set.ribbons.push_back();
        r_set.r_bitmap.push_back();
        r_set.r_bitmap[r_set.r_bitmap.size() - 1] = new Graphics::TBitmap;

        unsigned short r;
        for(file.read((char *)&r, sizeof(r)); r != 256; file.read((char *)&r, sizeof(r)))
        {
            unsigned short attr;
            file.read((char *)&attr, sizeof(attr));
            r_set.ribbons[r_set.ribbons.size() - 1].AddSprite((unsigned char) r, (unsigned char) attr);
        }
    }

    for(unsigned int i = 0; i < r_set.ribbons.size(); i++)
    	r_set.RefreshRibbonBitmap(i);
    r_set.RefreshBitmap();

    b_set.blocks.clear();

    unsigned short bs;
    file.read((char *)&bs, sizeof(bs));

    for (unsigned short i = 0; i < bs; i++)
    {
        b_set.blocks.push_back();
        int cbs = b_set.blocks.size();
        
        unsigned short rn;
        for(file.read((char *)&rn, sizeof(rn)); rn != 256; file.read((char *)&rn, sizeof(rn)))
            b_set.blocks[cbs - 1].rib_noms.push_back(rn);


        char c;
        AnsiString name;
        for(file.read((char *)&c, sizeof(c)); c != 0; file.read((char *)&c, sizeof(c)))
            name += char(c);

        b_set.blocks[cbs - 1].name = name;
        //ShowMessage(b_set.blocks[0].name);
    }






}

void Level::Save(AnsiString fname)
{
	ofstream file;
    file.open(fname.c_str(), ios::out | ios::binary);
    for (unsigned int y = 0; y < 24; y++)
    {
    	for (unsigned int x = 0; x < lines[y].size(); x++)
        {
        	unsigned short i = lines[y][x];
			file.write((char *)&i, sizeof(i));
        }
        unsigned short i = 256;
        file.write((char *)&i, sizeof(i));
    }
    file.close();


}

void Level::SaveCode(AnsiString fname)
{
	ofstream file;
	file.open(fname.c_str(), ios::out);

    //file << "LEVEL" << endl;

	AnsiString line;
	for (unsigned int i = 0; i < 24; i++)
    {
		line = "\tDEFB\t";

        
    	unsigned short ribnom = 256;
    	unsigned short ribcnt = 0;


        for (unsigned int b = 0; b < lines[i].size(); b++)
        {
            unsigned short rib = lines[i][b];

        	if (ribnom == rib)
            {
            	if (ribcnt == 255)
                {
                	line += IntToStr(ribnom) + ",";
            		line += IntToStr(ribcnt) + ",";
                    ribcnt = 0;
                }
            	ribcnt++;
            }
            else
            {
            	if (ribnom != 256)
                {
					line += IntToStr(ribnom) + ",";
            		line += IntToStr(ribcnt) + ",";
                }
                ribnom = lines[i][b];
                ribcnt = 1;
            }
        }

        if (ribnom != 256)
        {
        	line += IntToStr(ribnom) + ",";
            line += IntToStr(ribcnt) + ",";
        }

        line += IntToStr(255);

        file << line.c_str() << endl;
	}

	file.close();


}

void Level::Load(AnsiString fname)
{
	ifstream file;
    file.open(fname.c_str(), ios::in | ios::binary);

    unsigned short i;
    unsigned short c;
    for (unsigned int y = 0; y < 24; y++)
    {
    	lines[y].clear();

    	for(file.read((char *)&i, sizeof(i)); i != 256; file.read((char *)&i, sizeof(i)))
			lines[y].push_back(i);

    }

    file.close();


    RefreshBitmap();



}
