;      
;   /  51, ASCII-    
;    /  
;    IM1   038!
;     SD Card
;==================================================================================
; Contents of this file are copyright Grant Searle
; HEX routines from Joel Owens.
;
; You have permission to use this for NON COMMERCIAL USE ONLY
; If you wish to use it elsewhere, please include an acknowledgement to myself.
;
; http://searle.hostei.com/grant/index.html
;
; eMail: home.micros01@btinternet.com
;
; If the above don't work, please perform an Internet search to see if I have
; updated the web page hosting service.
;
;==================================================================================

;------------------------------------------------------------------------------
;
; Z80 Monitor Rom
;
;------------------------------------------------------------------------------
; General Equates
;------------------------------------------------------------------------------

CR		.EQU	0DH
LF		.EQU	0AH
ESC		.EQU	1BH
CTRLC	.EQU	03H
CLS		.EQU	0CH

; SD registers/configs
SD_DATA		.EQU	02Ah
SD_CONFIG	.EQU	02Bh

SD_WAIT_WR	.EQU	3000 ; Wait cycles to write finish
SD_WAIT		.EQU	3000 ; Wait cycles to CMD answer
SD_WAIT_I	.EQU	200 ; Wait cycles to INIT finish

loadAddr	.EQU	0cf00h	; CP/M load address
numSecs		.EQU	25	; Number of 512 sectors to be loaded

;BASIC cold and warm entry points
BASCLD		.EQU	$2000
BASWRM		.EQU	$2003

SER_BUFSIZE		.EQU	40H
SER_FULLSIZE	.EQU	30H
SER_EMPTYSIZE	.EQU	5

RTS_HIGH	.EQU	05H
RTS_LOW		.EQU	025H ;  51

SER_D		.EQU	$08
SER_C		.EQU	$09

		.ORG	$4000

serBuf		.ds	SER_BUFSIZE
serInPtr	.ds	2
serRdPtr	.ds	2
serBufUsed	.ds	1

primaryIO	.ds	1
secNo		.ds	1
dmaAddr		.ds	2

lba0		.DS	1
lba1		.DS	1
lba2		.DS	1
lba3		.DS	1
crc			.DS 1
sdhcs		.DS 1

erflag:		.ds	1		;error reporting
ercode:		.ds 1		;error code


stackSpace	.ds	32

STACK   	.EQU    $	; Stack top



;------------------------------------------------------------------------------
;                         START OF MONITOR ROM
;------------------------------------------------------------------------------

MON		.ORG	$0000		; MONITOR ROM RESET VECTOR
;------------------------------------------------------------------------------
; Reset
;------------------------------------------------------------------------------
RST00:
		DI			;Disable INTerrupts
		JP	INIT		;Initialize Hardware and go
		NOP
		NOP
		NOP
		NOP
;------------------------------------------------------------------------------
; TX a character over RS232 wait for TXDONE first.
;------------------------------------------------------------------------------
RST08:
		JP	conout
		NOP
		NOP
		NOP
		NOP
		NOP
;------------------------------------------------------------------------------
; RX a character from buffer wait until char ready.
;------------------------------------------------------------------------------
RST10:
		JP	conin
		NOP
		NOP
		NOP
		NOP
		NOP
;------------------------------------------------------------------------------
; Check input buffer status
;------------------------------------------------------------------------------
RST18:
		JP	CKINCHAR

;------------------------------------------------------------------------------
; SIO Vector = 0x38
;------------------------------------------------------------------------------

		.ORG	$0038
		
;------------------------------------------------------------------------------
; Serial interrupt handlers
; Same interrupt called if either of the inputs receives a character
; so need to check the status of each SIO input.
;------------------------------------------------------------------------------

intVector:
		DI
		PUSH	AF
		PUSH	HL

		;    51

		IN   	A,(SER_C)			; -, D1=1   	
		AND 00000010B			;  	
		JR	NZ, serialInt		;  1,    51

otherInt:;    8251,   

		POP	HL
		POP	AF

		EI
		RETI


serialInt:
		LD	HL,(serInPtr)
		INC	HL
		LD	A,L
		CP	(serBuf+SER_BUFSIZE) & $FF
		JR	NZ, notWrap
		LD	HL,serBuf
notWrap:
		LD	(serInPtr),HL
		IN	A,(SER_D)
		LD	(HL),A

		LD	A,(serBufUsed)
		INC	A
		LD	(serBufUsed),A
		CP	SER_FULLSIZE
		JR	C, rts0
        LD	A,RTS_HIGH
		OUT	(SER_C),A
rts0:
		POP	HL
		POP	AF
		EI
		RETI

;------------------------------------------------------------------------------
; Console input routine
; Use the "primaryIO" flag to determine which input port to monitor.
;------------------------------------------------------------------------------
conin:
		PUSH	HL

waitForChar:
		LD	A,(serBufUsed)
		CP	$00
		JR	Z, waitForChar
		LD	HL,(serRdPtr)
		INC	HL
		LD	A,L
		CP	(serBuf+SER_BUFSIZE) & $FF
		JR	NZ, notRdWrap
		LD	HL,serBuf
notRdWrap:
		DI
		LD	(serRdPtr),HL

		LD	A,(serBufUsed)
		DEC	A
		LD	(serBufUsed),A

		CP	SER_EMPTYSIZE
		JR	NC,rts1
		LD	A,RTS_LOW
		OUT	(SER_C),A
rts1:
		LD	A,(HL)
		POP	HL
		EI
		RET	; Char ready

;------------------------------------------------------------------------------
; Console output routine
; Use the "primaryIO" flag to determine which output port to send a character.
;------------------------------------------------------------------------------
conout:
		PUSH	AF		; Store character

conout1:
		IN	A,(SER_C)	;   D2=1  	
		AND 00000001B			;  	
		JR	Z,conout1	; Loop until SIO flag signals ready

		POP	AF		; RETrieve character
		OUT	(SER_D),A	; OUTput the character
		RET


;------------------------------------------------------------------------------
; Check if there is a character in the input buffer
; Use the "primaryIO" flag to determine which port to check.
;------------------------------------------------------------------------------
CKINCHAR:
		LD	A,(serBufUsed)
		CP	$0
		RET

;------------------------------------------------------------------------------
; Filtered Character I/O
;------------------------------------------------------------------------------

RDCHR:
		RST	10H
		CP	LF
		JR	Z,RDCHR		; Ignore LF
		CP	ESC
		JR	NZ,RDCHR1
		LD	A,CTRLC		; Change ESC to CTRL-C
RDCHR1:
		RET

WRCHR:
		CP	CR
		JR	Z,WRCRLF	; When CR, write CRLF
		CP	CLS
		JR	Z,WR		; Allow write of "CLS"
		CP	' '		; Don't write out any other control codes
		JR	C,NOWR		; ie. < space
WR:
		RST	08H
NOWR:
		RET

WRCRLF:
		LD	A,CR
		RST	08H
		LD	A,LF
		RST	08H
		LD	A,CR
		RET


;------------------------------------------------------------------------------
; Initialise hardware and start main loop
;------------------------------------------------------------------------------
INIT:
		LD   SP,STACK		; Set the Stack Pointer

		LD	HL,serBuf
		LD	(serInPtr),HL
		LD	(serRdPtr),HL

		xor	a			;0 to accumulator
		LD	(serBufUsed),A

;	Initialise 8251:   >>> 8 , 1 -,   ,  1:16 >>>    ,  RTS_LOW    

		LD	HL,SINTTB
		LD	BC,$0006
INIT1:
		DEC B
		JR NZ,INIT1
		
		LD	A,(HL)
		INC	HL
		OUT	(SER_C),A
		DEC C
		JR	NZ,INIT1
		
		IN A,(SER_D) ; 
		
		IM	1
		EI

		LD   HL,SIGNON	; Print SIGNON message
		CALL PRINT

;------------------------------------------------------------------------------
; Monitor command loop
;------------------------------------------------------------------------------
MAIN:
  		LD   HL,MAIN	; Save entry point for Monitor	
		PUSH HL		; This is the return address
MAIN0:
		CALL TXCRLF	; Entry point for Monitor, Normal	
		LD   A,'>'	; Get a ">"	
		RST 08H		; print it

MAIN1:
		CALL RDCHR	; Get a character from the input port
		CP   ' '	; <spc> or less? 	
		JR   C,MAIN1	; Go back
	
		CP   ':'	; ":"?
		JP   Z,LOAD	; First character of a HEX load

		CALL WRCHR	; Print char on console

		CP   '?'
		JP   Z,HELP

		AND  $5F	; Make character uppercase

		CP   'R'
		JP   Z,RST00

		CP   'B'
		JP   Z,BASIC

		CP   'G'
		JP   Z,GOTO

		CP   'X'
		JP   Z,CPMLOAD

		LD   A,'?'	; Get a "?"	
		RST 08H		; Print it
		JR   MAIN0
	
;------------------------------------------------------------------------------
; Print string of characters to Serial A until byte=$00, WITH CR, LF
;------------------------------------------------------------------------------
PRINT:
		LD   A,(HL)	; Get character
		OR   A		; Is it $00 ?
		RET  Z		; Then RETurn on terminator
		RST  08H	; Print it
		INC  HL		; Next Character
		JR   PRINT	; Continue until $00


TXCRLF:
		LD   A,$0D	; 
		RST  08H	; Print character 
		LD   A,$0A	; 
		RST  08H	; Print character
		RET

;------------------------------------------------------------------------------
; Get a character from the console, must be $20-$7F to be valid (no control characters)
; <Ctrl-c> and <SPACE> breaks with the Zero Flag set
;------------------------------------------------------------------------------	
GETCHR:
		CALL RDCHR	; RX a Character
		CP   $03	; <ctrl-c> User break?
		RET  Z			
		CP   $20	; <space> or better?
		JR   C,GETCHR	; Do it again until we get something usable
		RET
;------------------------------------------------------------------------------
; Gets two ASCII characters from the console (assuming them to be HEX 0-9 A-F)
; Moves them into B and C, converts them into a byte value in A and updates a
; Checksum value in E
;------------------------------------------------------------------------------
GET2:
		CALL GETCHR	; Get us a valid character to work with
		LD   B,A	; Load it in B
		CALL GETCHR	; Get us another character
		LD   C,A	; load it in C
		CALL BCTOA	; Convert ASCII to byte
		LD   C,A	; Build the checksum
		LD   A,E
		SUB  C		; The checksum should always equal zero when checked
		LD   E,A	; Save the checksum back where it came from
		LD   A,C	; Retrieve the byte and go back
		RET
;------------------------------------------------------------------------------
; Gets four Hex characters from the console, converts them to values in HL
;------------------------------------------------------------------------------
GETHL:
		LD   HL,$0000	; Gets xxxx but sets Carry Flag on any Terminator
		CALL ECHO	; RX a Character
		CP   $0D	; <CR>?
		JR   NZ,GETX2	; other key		
SETCY:
		SCF		; Set Carry Flag
		RET             ; and Return to main program		
;------------------------------------------------------------------------------
; This routine converts last four hex characters (0-9 A-F) user types into a value in HL
; Rotates the old out and replaces with the new until the user hits a terminating character
;------------------------------------------------------------------------------
GETX:
		LD   HL,$0000	; CLEAR HL
GETX1:
		CALL ECHO	; RX a character from the console
		CP   $0D	; <CR>
		RET  Z		; quit
		CP   $2C	; <,> can be used to safely quit for multiple entries
		RET  Z		; (Like filling both DE and HL from the user)
GETX2:
		CP   $03	; Likewise, a <ctrl-C> will terminate clean, too, but
		JR   Z,SETCY	; It also sets the Carry Flag for testing later.
		ADD  HL,HL	; Otherwise, rotate the previous low nibble to high
		ADD  HL,HL	; rather slowly
		ADD  HL,HL	; until we get to the top
		ADD  HL,HL	; and then we can continue on.
		SUB  $30	; Convert ASCII to byte	value
		CP   $0A	; Are we in the 0-9 range?
		JR   C,GETX3	; Then we just need to sub $30, but if it is A-F
		SUB  $07	; We need to take off 7 more to get the value down to
GETX3:
		AND  $0F	; to the right hex value
		ADD  A,L	; Add the high nibble to the low
		LD   L,A	; Move the byte back to A
		JR   GETX1	; and go back for next character until he terminates
;------------------------------------------------------------------------------
; Convert ASCII characters in B C registers to a byte value in A
;------------------------------------------------------------------------------
BCTOA:
		LD   A,B	; Move the hi order byte to A
		SUB  $30	; Take it down from Ascii
		CP   $0A	; Are we in the 0-9 range here?
		JR   C,BCTOA1	; If so, get the next nybble
		SUB  $07	; But if A-F, take it down some more
BCTOA1:
		RLCA		; Rotate the nybble from low to high
		RLCA		; One bit at a time
		RLCA		; Until we
		RLCA		; Get there with it
		LD   B,A	; Save the converted high nybble
		LD   A,C	; Now get the low order byte
		SUB  $30	; Convert it down from Ascii
		CP   $0A	; 0-9 at this point?
		JR   C,BCTOA2	; Good enough then, but
		SUB  $07	; Take off 7 more if it's A-F
BCTOA2:
		ADD  A,B	; Add in the high order nybble
		RET

;------------------------------------------------------------------------------
; Get a character and echo it back to the user
;------------------------------------------------------------------------------
ECHO:
		CALL	RDCHR
		CALL	WRCHR
		RET

;------------------------------------------------------------------------------
; GOTO command
;------------------------------------------------------------------------------
GOTO:
		CALL GETHL		; ENTRY POINT FOR <G>oto addr. Get XXXX from user.
		RET  C			; Return if invalid       	
		PUSH HL
		RET			; Jump to HL address value

;------------------------------------------------------------------------------
; LOAD Intel Hex format file from the console.
; [Intel Hex Format is:
; 1) Colon (Frame 0)
; 2) Record Length Field (Frames 1 and 2)
; 3) Load Address Field (Frames 3,4,5,6)
; 4) Record Type Field (Frames 7 and 8)
; 5) Data Field (Frames 9 to 9+2*(Record Length)-1
; 6) Checksum Field - Sum of all byte values from Record Length to and 
;   including Checksum Field = 0 ]
;------------------------------------------------------------------------------	
LOAD:
		LD   E,0	; First two Characters is the Record Length Field
		CALL GET2	; Get us two characters into BC, convert it to a byte <A>
		LD   D,A	; Load Record Length count into D
		CALL GET2	; Get next two characters, Memory Load Address <H>
		LD   H,A	; put value in H register.
		CALL GET2	; Get next two characters, Memory Load Address <L>
		LD   L,A	; put value in L register.
		CALL GET2	; Get next two characters, Record Field Type
		CP   $01	; Record Field Type 00 is Data, 01 is End of File
		JR   NZ,LOAD2	; Must be the end of that file
		CALL GET2	; Get next two characters, assemble into byte
		LD   A,E	; Recall the Checksum byte
		AND  A		; Is it Zero?
		JR   Z,LOAD00	; Print footer reached message
		JR   LOADERR	; Checksums don't add up, Error out
		
LOAD2:
		LD   A,D	; Retrieve line character counter	
		AND  A		; Are we done with this line?
		JR   Z,LOAD3	; Get two more ascii characters, build a byte and checksum
		CALL GET2	; Get next two chars, convert to byte in A, checksum it
		LD   (HL),A	; Move converted byte in A to memory location
		INC  HL		; Increment pointer to next memory location	
		LD   A,'.'	; Print out a "." for every byte loaded
		RST  08H	;
		DEC  D		; Decrement line character counter
		JR   LOAD2	; and keep loading into memory until line is complete
		
LOAD3:
		CALL GET2	; Get two chars, build byte and checksum
		LD   A,E	; Check the checksum value
		AND  A		; Is it zero?
		RET  Z

LOADERR:
		LD   HL,CKSUMERR  ; Get "Checksum Error" message
		CALL PRINT	; Print Message from (HL) and terminate the load
		RET

LOAD00:
		LD   HL,LDETXT	; Print load complete message
		CALL PRINT
		RET

;------------------------------------------------------------------------------
; Start BASIC command
;------------------------------------------------------------------------------
BASIC:
    	LD HL,BASTXT
		CALL PRINT
		CALL GETCHR
		RET Z	; Cancel if CTRL-C
		AND  $5F ; uppercase
		CP 'C'
		JP  Z,BASCLD
		CP 'W'
		JP  Z,BASWRM
		RET

;------------------------------------------------------------------------------
; Display Help command
;------------------------------------------------------------------------------
HELP:
   	 	LD   HL,HLPTXT	; Print Help message
		CALL PRINT
		RET
	
;------------------------------------------------------------------------------
; CP/M load command
;------------------------------------------------------------------------------
CPMLOAD:
    	LD HL,CPMTXT
		CALL PRINT
		CALL GETCHR
		RET Z	; Cancel if CTRL-C
		AND  $5F ; uppercase
		CP 'Y'
		JP  Z,CPMLOAD2
		RET

CPMTXT:
		.BYTE	$0D,$0A
		.TEXT	"Boot CP/M?"
		.BYTE	$00

CPMTXT2:
		.BYTE	$0D,$0A
		.TEXT	"Loading CP/M..."
		.BYTE	$0D,$0A
		.TEXT	"Try to init SD Card..."
		.DB 0
CPMTXT3:
		.TEXT "Error - "
		.DB 0

CPMTXT4:
		.TEXT "Ok"
		.DB CR, LF
		.TEXT "Rading CPM"
		.DB 0
CPMTXT5:
		.TEXT "Ok"
		.DB CR, LF, 0

CPMLOAD2:
    	LD HL,CPMTXT2
		CALL PRINT

		CALL	sdInit
		
		LD	A, (erflag)
		CP	0
		JR Z, CPMl1
		
    	LD HL,CPMTXT3
		CALL PRINT
		
		LD	A, (erflag)
		CALL printHEX

		LD	A, 020H
		RST 08H
		
		LD	A, (ercode)
		CALL printHEX

		CALL TXCRLF

		RET
CPMl1:
    	LD HL,CPMTXT4
		CALL PRINT

		LD	B, numSecs

		LD	A, 0
		LD	(secNo),A
		LD	HL, loadAddr
		LD	(dmaAddr), HL

processSectors:

		LD	A, (sdhcs)
		AND	040H
		JR NZ, rdS1
		
		LD	A, (secNo)
		RLC	A
		LD	(lba1), A

		LD	A,0
		LD	(lba0), A
		LD	(lba2), A
		LD	(lba3), A

		JP	rdS2
rdS1:
		LD	A,(secNo)
		LD	(lba0), A
		LD	A,0
		LD	(lba1), A
		LD	(lba2), A
		LD	(lba3), A
rdS2:
		LD	A, '.'
		RST	08H

		call	readLBA

		LD	A, (erflag)
		CP	0
		JR NZ, rdSectorsEr

		LD	DE,0200H
		LD	HL,(dmaAddr)
		ADD	HL,DE
		LD	(dmaAddr),HL
		LD	A,(secNo)
		INC	A
		LD	(secNo),A

		djnz	processSectors

    	LD HL,CPMTXT5
		CALL PRINT

; Start CP/M using entry at top of BIOS
; The current active console stream ID is pushed onto the stack
; to allow the CBIOS to pick it up
; 0 = SIO A, 1 = SIO B
		
		ld	A,1
		PUSH	AF
		ld	HL,($FFFE)
		jp	(HL)

rdSectorsEr:

    	LD HL,CPMTXT3
		CALL PRINT

		LD	A, (erflag)
		CALL printHEX

		LD	A, 020H
		RST 08H
		
		LD	A, (ercode)
		CALL printHEX

		CALL TXCRLF

		RET

;================================================================================================
; Read LBA sector from host to (HL)
;================================================================================================

readLBA:
		PUSH 	AF
		PUSH 	BC

		LD		A, 1
		OUT		(SD_CONFIG), A ; Set SD Fast mode, CS Low

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this

		LD		A, 0
		LD		(crc), A
		LD		A, 040H + 17
		CALL	sdSendCmd ; Send CMD17 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		0
		LD		A, 01H ; Set Error Code to 1 (Stop at CMD17)
		JP NZ,	rdLBAExit ; Exit with error

		CALL	sdWait
		LD		(ercode), A
		
		CP		0FEh
		LD		A, 2 ; Set Error Code to 2 (Timeout waiting data)
		JP NZ,	rdLBAExit
		
		LD		C, 4

rdLBA4secs:

		LD		B, 128

rdLBAByte:
		LD		A, 0FFH
		OUT		(SD_DATA), A

		IN	 	A,(SD_DATA)
		LD		(HL), A
		INC 	HL
		DEC 	B
		JR 	NZ, rdLBAByte
		DEC 	C
		JR 	NZ,	rdLBA4secs

		LD		A, 0FFH
		OUT		(SD_DATA), A
		IN		A, (SD_DATA) ; Read CRC1

		LD		A, 0FFH
		OUT		(SD_DATA), A
		IN		A, (SD_DATA) ; Read CRC1

		XOR 	A
rdLBAExit:
		ld		(erflag),A

		LD		A, 2
		OUT		(SD_CONFIG), A ; Set SD Slow mode, CS Low

		POP 	BC
		POP 	AF

		RET

;================================================================================================
; SD Card send command (CMD = A, Op1 = lba3, Op2 = lba2, Op3 - lba1, Op4 = lba0, CRC = crc
;================================================================================================

sdSendCmd:
;		CALL	printHEX

		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba3)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba2)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba1)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba0)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (crc)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		RET

;================================================================================================
; SD Card wait for response (Return A = response)
;================================================================================================

sdWait:
		PUSH	BC
		
		LD		BC, SD_WAIT ; Try 1000 attempt
sdWait1:
		LD		A, 0FFH
		OUT		(SD_DATA), A
		CALL	SPIready
		
		IN		A, (SD_DATA)
		
;		call printHEX
		
		CP		0FFH
		JR NZ,	sdWait2
		
		DEC		C
		JR NZ,	sdWait1

		DEC		B
		JR NZ,	sdWait1
		
sdWait2:		
		POP		BC
		RET

;================================================================================================
; SD Card Init procedure
;================================================================================================

sdInit:
		PUSH 	AF
		PUSH 	BC
		PUSH 	HL

		LD		A, 02h
		OUT		(SD_CONFIG), A ; Set Slow mode, CS High
		
		LD		B, 10
sdInit1:
		LD		A, 0FFh
		OUT		(SD_DATA), A
		CALL	SPIready
		
		DEC		B
		JR NZ,	sdInit1

		LD		A, 0
		OUT		(SD_CONFIG), A ; Set Slow mode, CS Low
		CALL	SPIready

		LD		B, 4 ; Try 4 times to send CMD0
sdInit0:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		A, 095H
		LD		(crc), A
		LD		A, 040H + 0
		CALL	sdSendCmd ; Send CMD0 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		01H
		JP Z,	sdInit8 ; CMD 0 applied
		
		DEC		B
		JR NZ,	sdInit0

		LD		A, 01H ; Set Error Code to 1 (Stop at CMD0)
		JP		sdInitX
sdInit8:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready
		
		LD		A, 0AAh
		LD		(lba0), A
		LD		A, 01H
		LD		(lba1), A
		LD		A, 0
		LD		(lba2), A
		LD		A, 0
		LD		(lba3), A
		LD		A, 086H
		LD		(crc), A
		LD		A, 040H + 8
		CALL	sdSendCmd ; Send CMD8 command (Argument = 0x000001AA)
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; CMD8 command not supported
		JR Z,	sdInitOldSD
		
		CP		01H
		LD		A, 02H ; Set Error Code to 2 (Stop at CMD8)
		JP NZ,	sdInitX ; Exit with error
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		(ercode), A
;		CALL	printHEX

		CP		01H
		JR Z,	sdInit2

		LD		A, 03H ; Set Error Code to 3 (Card voltage out of range)
		JP		sdInitX ; Exit with error
sdInit2:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		(ercode), A
;		CALL	printHEX

		CP		0AAH
		JR Z,	sdInit3
		
		LD		A, 03H ; Set Error Code to 3 (Card voltage out of range)
		JP		sdInitX ; Exit with error
sdInit3:
		LD		A, 040H
		LD		(sdhcs), A
		
sdInitOldSD:

		LD		B, SD_WAIT_I ; Try to init with ACMD41 100 times
sdInit4:		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 55
		CALL	sdSendCmd ; Send CMD55 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; CMD55 command not supported
		JR Z,	sdInit5

		CP		01H
		LD		A, 04H ; Set Error Code to 4 (Stop at CMD55)
		JP NZ,	sdInitX ; Exit with error
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready
		
		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		A, (sdhcs)
		LD		(lba3), A
		LD		A, 0
		LD		(crc), A
		LD		A, 040H + 41
		CALL	sdSendCmd ; Send ACMD41 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; ACMD41 command not supported
		JR Z,	sdInit5

		CP		00H
		JP Z,	sdInitCpl ; Init complete
		
		CP		01H
		LD		A, 05H ; Set Error Code to 5 (Stop at ACMD41)
		JP NZ,	sdInitX ; Exit with error
		
		DEC		B
		JR NZ,	sdInit4
		
		LD		A, 06H ; Set Error Code to 6 (Init timeout)
		JP		sdInitX ; Exit with error
sdInit5:
		LD		B, SD_WAIT_I ; Try to init with CMD1 100 times
sdInit6:		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 1
		CALL	sdSendCmd ; Send CMD1 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		00H
		JP Z,	sdInitCpl ; Init complete
		
		CP		01H
		LD		A, 07H ; Set Error Code to 7 (Stop at CMD1)
		JP NZ,	sdInitX ; Exit with error
		
		DEC		B
		JR NZ,	sdInit6
		
		LD		A, 06H ; Set Error Code to 6 (Init timeout)
		JP		sdInitX ; Exit with error

sdInitCpl:

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 58
		CALL	sdSendCmd ; Send CMD58 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		0
		LD		A, 08H ; Set Error Code to 8 (Stop at CMD58)
		JP NZ,	sdInitX ; Exit with error

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		B, A
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX
		
		LD		A, B
		LD		(sdhcs), A
		AND		040H
		JR NZ,	sdInit7 ; SD Card is high capacity, no need to set 512 large blocks
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		A, 02H ; Set block size = 512 bytes
		LD		(lba1), A
		LD		A, 0
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 16
		CALL	sdSendCmd ; Send CMD16 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		0
		LD		A, 09H ; Set Error Code to 9 (Stop at CMD16)
		JP NZ,	sdInitX ; Exit with error
sdInit7:
		LD		A, 0
sdInitX:
		LD		(erflag), A

		LD		A, 2
		OUT		(SD_CONFIG), A ; Set SD Slow mode, CS High

		POP 	HL
		POP 	BC
		POP 	AF

		RET

SPIready:
		PUSH	AF
SPIr1:
		IN		A, (SD_CONFIG)
		AND		01H
		JR NZ,	SPIr1

		POP		AF
		
		RET

printHEX:
		PUSH AF
		PUSH BC

		LD	B, A

		RRC	A
		RRC	A
		RRC	A
		RRC	A
		
		AND 0FH
		
		CP	10
		JR	C, pHEX1
		
		ADD A, 7
pHEX1:
		ADD	A, 30H
		RST 08H

		LD	A, B

		AND 0FH
		
		CP	10
		JR	C, pHEX2
		
		ADD A, 7
pHEX2:
		ADD	A, 30H
		RST 08H
		
		POP BC
		POP AF

		RET

;------------------------------------------------------------------------------

SINTTB:	.BYTE	$0,$0,$0,$40,$4F,$35

SIGNON:
		.BYTE	"Z80MINI Boot ROM"
		.BYTE	$0D,$0A
		.BYTE	"Type ? for options"
		.BYTE	$0D,$0A,$00

BASTXT:
		.BYTE	$0D,$0A
		.TEXT	"Cold or Warm ?"
		.BYTE	$0D,$0A,$00

CKSUMERR:
		.BYTE	"Checksum error"
		.BYTE	$0D,$0A,$00

LDETXT:
		.TEXT	"Load complete."
		.BYTE	$0D,$0A, $00


HLPTXT:
		.BYTE	$0D,$0A
		.TEXT	"R           - Reset"
		.BYTE	$0D,$0A
		.TEXT	"BC or BW    - ROM NASCOM BASIC Cold or Warm"
		.BYTE	$0D,$0A
		.TEXT	"X           - Boot CP/M (load $CF00-$FFFF from disk)"
		.BYTE	$0D,$0A
		.TEXT	":nnnnnn...  - Load Intel-Hex file record"
		.BYTE	$0D,$0A
       	.BYTE   $00

;------------------------------------------------------------------------------

FINISH	.END	

