module sd_card_manager(

	input wire clock50m,
	output wire cs,
	output wire mosi,
	input wire miso,
	output wire clk,
	input wire [7:0] data_in,
	output wire [7:0] data_out,
	input wire [7:0] mode,
	input wire out_flag,
	input wire n_res,
	output wire busy

);

// SD Card interface
//
// data_in = 8 bit data to send
// data_out = 8 bit data readed
// out_flag = when falling from HIGH to LOW start TX/RX cycle
// mode = MODE register
// busy = busy flag, set to HIGH when TX/RX cycle executing
//

// SD Card MODE register:
//
// Bit 0: 0 = Slow mode (~200kHz Clock), 1 = Fast mode (12.5Mhz Clock)
// Bit 1: 0 = CS set to LOW, 1 = CS set to HIGH (for init purpose)
// Bit 2: Reserved
// Bit 3: Reserved
// Bit 4: Reserved
// Bit 5: Reserved
// Bit 6: Reserved
// Bit 7: Reserved
//

	localparam [1:0] IDLE = 2'h0;
	localparam [1:0] START = 2'h1;
	localparam [1:0] CLR_CLOCK = 2'h2;
	localparam [1:0] SET_CLOCK = 2'h3;
	
	reg [1:0] state;
	reg [6:0] clock_divider;
	reg [2:0] bit_counter;
	reg [7:0] data_latch_in;
	reg [7:0] data_latch_out;
	reg [2:0] out_flag_latch = 3'b0;
	reg clk_latch;
	reg first_bit;
	reg busy_latch;
	
	always @(posedge clock50m) begin
		out_flag_latch <= {out_flag_latch[1:0], out_flag};
	end

	wire out_flag_negedge = out_flag_latch[2] & ~out_flag_latch[1];
	
	always @(posedge clock50m) begin
	
		if (!n_res) begin
		
			clk_latch <= 1;
			state <= IDLE;
			busy_latch <= 0;

		end else begin
	
			clock_divider <= clock_divider + 7'h1;

			if (out_flag_negedge) begin // found NEG edge at out_flag
				state <= START;
				clock_divider <= 0;
			end
	
			case (state)
				START:
							if ((mode[0] == 1'b1 && clock_divider[0] == 1'b1) || (mode[0] == 1'b0 && clock_divider == 7'h0))
							begin
								bit_counter <= 3'h7;
								data_latch_out <= data_in;
								state <= CLR_CLOCK;
								first_bit <= 1'b1;
								busy_latch <= 1;
							end
				CLR_CLOCK:
							if ((mode[0] == 1'b1 && clock_divider[0] == 1'b1) || (mode[0] == 1'b0 && clock_divider == 7'h0))
							begin
								clk_latch <= 1'b0;
								if (~first_bit) data_latch_out <= {data_latch_out[6:0], 1'b0};
								bit_counter <= bit_counter + 3'b1;
								state <= SET_CLOCK;
							end
				SET_CLOCK:	
							if ((mode[0] == 1'b1 && clock_divider[0] == 1'b1) || (mode[0] == 1'b0 && clock_divider == 7'h0))
							begin
								clk_latch <= 1'b1;
								data_latch_in <= {data_latch_in[6:0], miso};
								first_bit <= 1'b0;
								if (bit_counter == 3'h7) begin
									state <= IDLE;
									busy_latch <= 0;
								end else state <= CLR_CLOCK;
							end
			endcase
		end
	end
	
	assign mosi = data_latch_out[7];
	assign data_out = data_latch_in;
	assign clk = clk_latch;
	assign cs = mode[1];
	assign busy = busy_latch;
	
endmodule