;==================================================================================
; Test read from SD card example
;==================================================================================

loadAddr	.EQU	0D000h
numSecs		.EQU	24	; Number of 512 sectors to be loaded


LF		.EQU	0AH		;line feed
FF		.EQU	0CH		;form feed
CR		.EQU	0DH		;carriage RETurn

; SD registers/constants
SD_DATA		.EQU	02Ah
SD_CONFIG	.EQU	02Bh

SD_WAIT_WR	.EQU	3000 ; Wait cycles to write finish
SD_WAIT		.EQU	3000 ; Wait cycles to CMD answer
SD_WAIT_I	.EQU	200 ; Wait cycles to INIT finish

;================================================================================================

		.ORG	5000H		; Loader origin.

		CALL	printInline
		.TEXT "Test READ/Write (C)2023"
		.DB CR, LF
		.TEXT "Try to init SD card..."
		.DB 0

		CALL sdInit

		LD	A, (erflag)
		CP	0
		JR Z, main1
		
		CALL	printInline
		.TEXT "Error - "
		.DB 0
		
		CALL printHEX

		LD	A, 020H
		RST	08H
		
		LD	A, (ercode)
		CALL printHEX
		
		RET
		
main1:
		CALL	printInline
		.TEXT "Ok"
		.DB CR, LF, 0

;		LD	A, (sdhcs)
;		CALL	printHEX

		LD	A, 1
		LD	(secNo), A
		LD	HL, buffer
		LD	(dmaAddr), HL

		LD	A, (sdhcs)
		AND	040H
		JR NZ, ps1
		
		LD	A, (secNo)
		RLC	A
		LD	(lba1), A
		
		CALL printHEX
		
		LD	A,0
		LD	(lba0), A
		LD	(lba2), A
		LD	(lba3), A
		JP	ps2

ps1:
		LD	A, (secNo)
		LD	(lba0), A
		CALL printHEX
		LD	A,0
		LD	(lba1), A
		LD	(lba2), A
		LD	(lba3), A
ps2:
		call	readLBA
		
		LD	A, (erflag)
		CP	0
		JR NZ, psX

		CALL	printInline
		.DB CR,LF
		.TEXT "Read complete."
		.DB CR,LF,0

		RET
psX:
		CALL	printInline
		.DB CR,LF
		.TEXT "Read failure - "
		.DB 0
		
		CALL printHEX

		LD	A, 020H
		RST	08H
		
		LD	A, (ercode)
		CALL printHEX
		
		RET

;================================================================================================
; Write physical sector to host
;================================================================================================

write:
		PUSH 	AF
		PUSH 	BC
		PUSH 	HL

		LD		A, 1
		OUT		(SD_CONFIG), A ; Set SD Fast mode, CS Low

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this

		LD		A, 0
		LD		(crc), A
		LD		A, 040H + 24
		CALL	sdSendCmd ; Send CMD24 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		0
		LD		A, 01H ; Set Error Code to 1 (Stop at CMD24)
		JP NZ,	wrExit ; Exit with error

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this

		LD		A, 0FEH
		OUT		(SD_DATA), A ; Send Data token to SD Card

		LD		C, 4
wr4secs:
		LD		B, 128
wrByte:	
		LD		A, (HL)
		OUT 	(SD_DATA),A

		INC 	HL
		DEC 	B
		JR 	NZ,	wrByte

		DEC 	C
		JR 	NZ,	wr4secs

		LD		A, 0H
		OUT		(SD_DATA), A ; Send fake CRC byte 1
		
		OUT		(SD_DATA), A ; Send fake CRC byte 2

		CALL	sdWait
		LD		(ercode), A
		
		AND		01FH
		
		CP		05H ; Check that data accepted
		LD		A, 2 ; Set error to 2 (Data not accepted by SD Card)
		JP NZ,	wrExit
		
		LD		BC, SD_WAIT_WR ; 3000 cycles to write complete
wrWait1:
		CALL	sdWait
		LD		(ercode), A

		CP		0
		JR NZ,	wrWait2
		
		DEC		C
		JR NZ,	wrWait1
		
		DEC		B
		JR NZ,	wrWait1
		
		LD		A, 3 ; Set error to 3 (Card timeout to write)
		JP		wrExit
wrWait2:
		XOR 	a
wrExit:
		ld	(erflag),a

		LD		A, 2
		OUT		(SD_CONFIG), A ; Set SD Fast mode, CS High

		POP 	HL
		POP 	BC
		POP 	AF

		RET

;================================================================================================
; Read LBA sector from host to (HL)
;================================================================================================

readLBA:
		PUSH 	AF
		PUSH 	BC

		LD		A, 1
		OUT		(SD_CONFIG), A ; Set SD Fast mode, CS Low

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this

		LD		A, 0
		LD		(crc), A
		LD		A, 040H + 17
		CALL	sdSendCmd ; Send CMD17 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		0
		LD		A, 01H ; Set Error Code to 1 (Stop at CMD17)
		JP NZ,	rdLBAExit ; Exit with error

		CALL	sdWait
		LD		(ercode), A
		
		CP		0FEh
		LD		A, 2 ; Set Error Code to 2 (Timeout waiting data)
		JP NZ,	rdLBAExit
		
		LD		C, 4

rdLBA4secs:

		LD		B, 128

rdLBAByte:

		LD		A, 0FFH
		OUT		(SD_DATA), A

		IN	 	A,(SD_DATA)
		LD		(HL), A
		
		call printHEX
		
		INC 	HL
		DEC 	B
		JR 	NZ, rdLBAByte
		DEC 	C
		JR 	NZ,	rdLBA4secs

		LD		A, 0FFH
		OUT		(SD_DATA), A
		IN		A, (SD_DATA) ; Read CRC1

		LD		A, 0FFH
		OUT		(SD_DATA), A
		IN		A, (SD_DATA) ; Read CRC1

		XOR 	A
rdLBAExit:
		ld		(erflag),A

		LD		A, 2
		OUT		(SD_CONFIG), A ; Set SD Fast mode, CS Low

		POP 	BC
		POP 	AF

		RET

;================================================================================================
; SD Card send command (CMD = A, Op1 = lba3, Op2 = lba2, Op3 - lba1, Op4 = lba0, CRC = crc
;================================================================================================

sdSendCmd:
;		CALL	printHEX

		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba3)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba2)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba1)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (lba0)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		LD		A, (crc)
		OUT		(SD_DATA), A
		CALL	SPIready
		
		RET
		

;================================================================================================
; SD Card wait for response (Return A = response)
;================================================================================================

sdWait:
		PUSH	BC
		
		LD		BC, SD_WAIT ; Try 1000 attempt
sdWait1:
		LD		A, 0FFH
		OUT		(SD_DATA), A
		CALL	SPIready
		
		IN		A, (SD_DATA)
		
;		call printHEX
		
		CP		0FFH
		JR NZ,	sdWait2
		
		DEC		C
		JR NZ,	sdWait1

		DEC		B
		JR NZ,	sdWait1
		
sdWait2:		
		POP		BC
		RET

;================================================================================================
; SD Card Init procedure
;================================================================================================

sdInit:
		PUSH 	AF
		PUSH 	BC
		PUSH 	HL

		LD		A, 02h
		OUT		(SD_CONFIG), A ; Set Slow mode, CS High
		
		LD		B, 10
sdInit1:
		LD		A, 0FFh
		OUT		(SD_DATA), A
		CALL	SPIready
		
		DEC		B
		JR NZ,	sdInit1

		LD		A, 0
		OUT		(SD_CONFIG), A ; Set Slow mode, CS Low
		CALL	SPIready

		LD		B, 4 ; Try 4 times to send CMD0
sdInit0:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		A, 095H
		LD		(crc), A
		LD		A, 040H + 0
		CALL	sdSendCmd ; Send CMD0 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		01H
		JP Z,	sdInit8 ; CMD 0 applied
		
		DEC		B
		JR NZ,	sdInit0

		LD		A, 01H ; Set Error Code to 1 (Stop at CMD0)
		JP		sdInitX
sdInit8:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready
		
		LD		A, 0AAh
		LD		(lba0), A
		LD		A, 01H
		LD		(lba1), A
		LD		A, 0
		LD		(lba2), A
		LD		A, 0
		LD		(lba3), A
		LD		A, 086H
		LD		(crc), A
		LD		A, 040H + 8
		CALL	sdSendCmd ; Send CMD8 command (Argument = 0x000001AA)
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; CMD8 command not supported
		JR Z,	sdInitOldSD
		
		CP		01H
		LD		A, 02H ; Set Error Code to 2 (Stop at CMD8)
		JP NZ,	sdInitX ; Exit with error
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		(ercode), A
;		CALL	printHEX

		CP		01H
		JR Z,	sdInit2

		LD		A, 03H ; Set Error Code to 3 (Card voltage out of range)
		JP		sdInitX ; Exit with error
sdInit2:
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		(ercode), A
;		CALL	printHEX

		CP		0AAH
		JR Z,	sdInit3
		
		LD		A, 03H ; Set Error Code to 3 (Card voltage out of range)
		JP		sdInitX ; Exit with error
sdInit3:
		LD		A, 040H
		LD		(sdhcs), A
		
sdInitOldSD:

		LD		B, SD_WAIT_I ; Try to init with ACMD41 100 times
sdInit4:		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 55
		CALL	sdSendCmd ; Send CMD55 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; CMD55 command not supported
		JR Z,	sdInit5

		CP		01H
		LD		A, 04H ; Set Error Code to 4 (Stop at CMD55)
		JP NZ,	sdInitX ; Exit with error
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready
		
		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		A, (sdhcs)
		LD		(lba3), A
		LD		A, 0
		LD		(crc), A
		LD		A, 040H + 41
		CALL	sdSendCmd ; Send ACMD41 command
		
		CALL	sdWait
		LD		(ercode), A
		
		CP		05H	; ACMD41 command not supported
		JR Z,	sdInit5

		CP		00H
		JP Z,	sdInitCpl ; Init complete
		
		CP		01H
		LD		A, 05H ; Set Error Code to 5 (Stop at ACMD41)
		JP NZ,	sdInitX ; Exit with error
		
		DEC		B
		JR NZ,	sdInit4
		
		LD		A, 06H ; Set Error Code to 6 (Init timeout)
		JP		sdInitX ; Exit with error
sdInit5:
		LD		B, SD_WAIT_I ; Try to init with CMD1 100 times
sdInit6:		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 1
		CALL	sdSendCmd ; Send CMD1 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		00H
		JP Z,	sdInitCpl ; Init complete
		
		CP		01H
		LD		A, 07H ; Set Error Code to 7 (Stop at CMD1)
		JP NZ,	sdInitX ; Exit with error
		
		DEC		B
		JR NZ,	sdInit6
		
		LD		A, 06H ; Set Error Code to 6 (Init timeout)
		JP		sdInitX ; Exit with error

sdInitCpl:

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		(lba1), A
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 58
		CALL	sdSendCmd ; Send CMD58 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		0
		LD		A, 08H ; Set Error Code to 8 (Stop at CMD58)
		JP NZ,	sdInitX ; Exit with error

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
		LD		B, A
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX

		LD		A, 0FFH
		OUT		(SD_DATA), A ; Send FF to read a byte
		CALL	SPIready
		IN		A, (SD_DATA)
;		CALL	printHEX
		
		LD		A, B
		LD		(sdhcs), A
		AND		040H
		JR NZ,	sdInit7 ; SD Card is high capacity, no need to set 512 large blocks
		
		LD		A, 0FFH
		OUT		(SD_DATA), A ; Skip a 8 clock, some card required this
		CALL	SPIready

		LD		A, 0
		LD		(lba0), A
		LD		A, 02H ; Set block size = 512 bytes
		LD		(lba1), A
		LD		A, 0
		LD		(lba2), A
		LD		(lba3), A
		LD		(crc), A
		LD		A, 040H + 16
		CALL	sdSendCmd ; Send CMD16 command
		
		CALL	sdWait
		LD		(ercode), A

		CP		0
		LD		A, 09H ; Set Error Code to 9 (Stop at CMD16)
		JP NZ,	sdInitX ; Exit with error
sdInit7:
		LD		A, 0
sdInitX:
		LD		(erflag), A

		LD		A, 2
		OUT		(SD_CONFIG), A ; Set SD Slow mode, CS High

		POP 	HL
		POP 	BC
		POP 	AF

		RET

;================================================================================================
; Utilities
;================================================================================================

SPIready:
		PUSH	AF
SPIr1:
		IN		A, (SD_CONFIG)
		AND		01H
		JR NZ,	SPIr1

		POP		AF
		
		RET

printInline:
		EX 	(SP),HL 	; PUSH HL and put RET ADDress into HL
		PUSH 	AF
		PUSH 	BC
nextILChar:	LD 	A,(HL)
		CP	0
		JR	Z,endOfPrint
		RST 	08H
		INC 	HL
		JR	nextILChar
endOfPrint:	INC 	HL 		; Get past "null" terminator
		POP 	BC
		POP 	AF
		EX 	(SP),HL 	; PUSH new RET ADDress on stack and restore HL
		RET

printHEX:
		PUSH AF
		PUSH BC

		LD	B, A

		RRC	A
		RRC	A
		RRC	A
		RRC	A
		
		AND 0FH
		
		CP	10
		JR	C, pHEX1
		
		ADD A, 7
pHEX1:
		ADD	A, 30H
		RST 08H

		LD	A, B

		AND 0FH
		
		CP	10
		JR	C, pHEX2
		
		ADD A, 7
pHEX2:
		ADD	A, 30H
		RST 08H
		
		POP BC
		POP AF

		RET

dmaAddr		.dw	0
secNo		.db	0

erflag		.ds	1		;error reporting
ercode		.db 0
lba0		.DB	00h
lba1		.DB	00h
lba2		.DB	00h
lba3		.DB	00h
crc			.DB 00h
sdhcs		.DB 00h

buffer		.ds 512

	.END
