; 8039 Nixie Clock (c) Tronix 2023
;
; v1.0 - initial release
; v1.1 - fix hour/min setup
;      - fix burn-in delays
; v1.2 - fix burn-in update
; v1.3 - stop timer before 1-wire in/out
;
; XTAL_FREQ = 12 MHz
; CYCLE_TIME = 1 / (XTAL_FREQ / 15) = 1.25us
; TIMER_COUNTER = XTAL_FREQ^6 / 480 = 25 KHz
; COUNTER_UPDATE = 32 * CYCLE_TIME = 40us
; TIMER_OFERFLOW = 40us * 256 = 10240us = 10.24ms

;GLOBAL CODE DEFINES
LCD       equ 1			; use 1602 i2c display
LCD_DELAY equ 1		; use delay when i2c LCD transactions
;DEBUG     equ 1		; debug timings for proteus

; RAM layout
digits12 equ 20h		; 1-2 digits
digits34 equ 21h		; 3-4 digits

scrach 	 equ 22h		; scrach are for DS1820 read
				; 9 bytes long
local_temp_l equ 2Bh
local_temp_h equ 2Ch

bcd_buf1 equ 2dh
bcd_buf2 equ 2eh

rtc_time equ 2fh		; 3 bytes

mseconds  equ 32h			; 10.24 mseconds count
seconds   equ 33h			; seconds counter
minutes	  equ 34h			; minutes counter

old_i2c_sec equ 35h			; i2c update second
old_ow_sec  equ 36h			; 1-wire update second
old_upd_min equ 37h			; burn-in update minutes

ind_mask    equ 38h			; indicators mask

temp_reg    equ 39h			; temp register

; DEFINES
CR          equ 0DH
LF          equ 0AH

 ifdef DEBUG
i2c_upd     equ 5			; update i2c every N sec
ow_upd	    equ 11			; update 1-wire every N sec
upd_upd     equ 2			; burn-in every N min
 else
i2c_upd     equ 10			; update i2c every N sec
ow_upd	    equ 51			; update 1-wire every N sec
upd_upd     equ 2			; burn-in every N min
 endif
;------------------------------------------------------------------------            
; compare the value in REGISTER to the value in the Accumulator.
; zero flag is set if the values are equal.
; carry flag is set if the value in REGISTER is equal to or greater than the value in the Accumulator.
; carry flag is cleared if the value in REGISTER is less than the value in the Accumulator.
; modifies the Accumulator.
;------------------------------------------------------------------------
compare     MACRO REGISTER
            cpl A
            inc A
            add A,REGISTER
            ENDM

; P2.6 on
DBG_ON	    MACRO
	orl p2,#040h
	    ENDM

; P2.6 off
DBG_OFF     MACRO
	anl p2,#0bfh
	    ENDM

; reset vector
 org  0
start:
	dis   i      		; disable interrupts
  	jmp  main

; external interrupt vector
 org  3
intr:
  	retr

; timer overflow interrupt
 org  7
time_int:
	sel rb1
	mov r4,a		; save a to RB1:R4

	mov r0,#mseconds
	inc @r0
	mov a,#97		; 97*10.24 ~= 1000ms = 1 sec
	compare @r0
	jnz skip_sec
	clr a
	mov @r0,a		; clear ms counter

	inc r0			; r0 = #seconds
	inc @r0			; inc seconds
	mov a,#60		
	compare @r0             ; seconds = 60?
	jnz skip_sec

	clr a			
	mov @r0,a		; clear seconds counter
	inc r0			; r0 = #minutes
	inc @r0			; inc minutes

	mov r0,#old_i2c_sec	; clear old_i2c_sec
	mov @r0,a
	inc r0			; 
	mov @r0,a		; clear old_ow_sec
	inc r0
	mov @r0,a		; clear old_upd_min

skip_sec:
	mov a,r4		; restore a from RB1:R4
	sel rb0
  	retr

; STARTUP
main:
	dis tcnti			; disable timer overflow interrupt

	call i2c_init

	;mov r0,#digits12		; fill digit buffer with 0x1234
	;mov @r0,#12h
	;inc r0
	;mov @r0,#34h

	;mov r0,#scrach
	;mov @r0,#0AAh
	;inc r0
	;mov @r0,#055h

	; setup i8255
	mov a,#10001000b	;Port A,B,C Mode 1, all output, except PC4-PC7
	mov r0,#3		;control word (A1=1,A0=1)
	movx @r0,a

	; turn on all segments
	mov a,#00001111b
	mov r0,#ind_mask
	mov @r0,a
	mov r0,#2
	movx @r0,a

        mov A,#titletxt & 255   ; display sign-on message
        call txtout

	call do_ow_stuff
	call do_i2c_stuff

	clr a
	mov t,a			; clear timer
	mov r1,#mseconds	; clear 10.24msec var
	mov @r1,a
	inc r1			; r1 = seconds
	mov @r1,a		; clear seconds var
	inc r1			; r1 = minutes
	mov @r1,a		; clear minutes var

	mov r1,#old_i2c_sec	; clear old_i2c_sec
	mov @r1,a
	inc r1			; 
	mov @r1,a		; clear old_ow_sec
	inc r1
	mov @r1,a		; clear old_upd_min

	en tcnti		; enable timer interrupt
	strt t			; start timer

 IFDEF LCD
	call lcd_init
	call lcd_display_time
	call lcd_display_temp
	call lcd_display_copyr
 ENDIF
main_loop:
	call display_digits
	call proceed_delimiter

 IFDEF LCD
	call lcd_display_sec	; ":" or " "
 ENDIF

	;if (seconds) % i2c_upd
	mov r2,#0
	mov r0,#seconds
	mov a,@r0
	jz skip_zero_sec
	mov r7,a		; save current second
	mov r1,#i2c_upd
	call div16
	xch a,r2		; reminder = 0?
	jnz skip_i2c

	mov a,r7		; restore seconds
	mov r1,#old_i2c_sec
	compare @r1		; current second = old_i2c_sec?
	jz skip_i2c		; if so, skip

 IFDEF LCD
 ENDIF
	mov a,r7
	mov r0,#old_i2c_sec
	mov @r0,a		; update old_i2c_sec

	call draw_tmr_msg

	mov r0,#old_i2c_sec
	mov a,@r0
	call draw_hex_sec

	call do_i2c_stuff	; DO i2c stuff (read time)
 IFDEF LCD
	call lcd_display_time
 ENDIF

	; turn on all segments
	;mov a,#00001111b
	mov r0,#ind_mask
	mov a,@r0
	orl a,#00000111b
	mov @r0,a
	mov r0,#2
	movx @r0,a
skip_i2c:
	
	;if (seconds) % ow_upd
	mov r2,#0
	mov r0,#seconds
	mov a,@r0
	mov r7,a		; save current second
	mov r1,#ow_upd
	call div16
	xch a,r2		; reminder = 0?
	jnz skip_ow

	mov a,r7		; restore seconds
	mov r1,#old_ow_sec
	compare @r1		; current second = old_ow_sec?
	jz skip_ow		; if so, skip

	mov a,r7
	mov r0,#old_ow_sec
	mov @r0,a		; update old_ow_sec

	mov r0,#old_i2c_sec
	mov @r0,a		; update old_i2c_sec, we don't need i2c now

	call draw_tmr_msg

	mov r0,#old_ow_sec
	mov a,@r0
	call draw_hex_sec

	call do_ow_stuff
 IFDEF LCD
	call lcd_display_temp
 ENDIF

	; turn on XX:xX segments
	;mov a,#00001101b
	mov r0,#ind_mask
	mov a,@r0
	anl a,#11111011b
	mov @r0,a
	mov r0,#2
	movx @r0,a
skip_zero_sec:
skip_ow:
	;if (minutes) % upd_upd
	mov r2,#0
	mov r0,#minutes
	mov a,@r0
	mov r7,a		; save current minutes
	mov r1,#upd_upd
	call div16
	xch a,r2		; reminder = 0?
	jnz skip_upd

	mov a,r7		; restore seconds
	mov r1,#old_upd_min
	compare @r1		; current minutes = old_upd_min?
	jz skip_upd		; if so, skip

	mov a,r7
	mov r0,#old_upd_min
	mov @r0,a		; update old_i2c_sec

	call draw_tmr_msg

	mov r0,#old_upd_min
	mov a,@r0
	call printhex
	mov a,#"m"
	call putch
	call newline

	; turn on all segments
	;mov a,#00001111b
	mov r0,#ind_mask
	mov a,@r0
	orl a,#00000111b
	mov @r0,a
	mov r0,#2
	movx @r0,a

	call do_update		; burn-in indicators
	call do_i2c_stuff	; DO i2c stuff

skip_upd:

	call proceed_buttons

forever:jmp main_loop

;SUB-FUNCTION
 org 100h

; draw tmr msg
draw_tmr_msg:
	call newline
	mov r2,#20
	mov a,#"-"
draw_line:
	call putch
	djnz r2,draw_line
	mov a,#msg_timer & 255
	call txtout
	ret

; Display digits12 and digits13
display_digits:
	mov r0,#digits12
	mov a,@r0
	swap a
	mov r0,#0
	movx @r0,a		; PortA 8255
	mov r0,#digits34
	mov a,@r0
	swap a
	mov r0,#1
	movx @r0,a		; PortB 8255
	ret

;READ TEMP FROM DS18B20
do_ow_stuff:
	mov a,#ow_init_msg & 255; dispaly "ow_reset = "
	call txtout

	call ow_reset		; 1-wire reset, return A = 0 ok
	mov r0,a		; save A

	call printhex
	call space

	xch a,r0		; restore A
	jz ow_ok
	mov a,#msg_err & 255	; display "ERR!"
	call txtout
	jmp read_fam
ow_ok:
	mov a,#msg_ok & 255	; display "OK"
	call txtout
read_fam:

	mov a,#ow_code_msg & 255; dispaly "ow_fam_code = "
	call txtout

	call read_famcode
	mov r0,#28h
	compare r0
	jz fam_ok
	mov a,#msg_err & 255	; display "ERR!"
	call txtout
	jmp read_pad
fam_ok:
	mov a,#msg_ok & 255	; display "OK"
	call txtout
read_pad:
	mov a,#ow_pad_msg & 255 ; diaplay "pad = "
	call txtout

	call read_scrachpad
	call space
	mov a,#msg_crc & 255	; display "CRC "
	call txtout

	call crc8dallas
	xch a,r2

	mov r0,#scrach
	mov a,#8
	add a,r0
	xch a,r0

	mov a,@r0		; 9-th byte from scrach
	compare r2
	jz crc_ok
	xch a,r2
	call printhex
	call space
	mov a,#msg_err & 255
	call txtout
	jmp do_not_extract	; crc error = exit
crc_ok:
	mov a,#msg_ok & 255
	call txtout
do_convert:	
	
	mov a,#msg_l_temp & 255 ; display "loc_temp = "
	call txtout

	call get_temperature_from_scratchpad

do_not_extract:
	mov r0,#bcd_buf1
	mov r2,#0
	mov r1,#local_temp_h
	mov a,@r1
	call bin2bcd

	mov r0,#local_temp_l
	mov a,@r0
	anl a,#11110000b
	swap a
	mov r0,#bcd_buf2
	mov @r0,a
	
	;mov r0,#bcd_buf2
	;mov r2,#0
	;mov r1,#local_temp_l
	;mov a,@r1
	;call bin2bcd

	mov a,#msg_t_bcd & 255 ; dispaly "temp bcd = "
	call txtout

	mov r3,#2
	mov r1,#bcd_buf1
print_buf:
	mov a,@r1
	call printhex
        inc r1
        djnz r3,print_buf
	call newline

	mov r0,#bcd_buf1
	mov r1,#digits12
	mov a,@r0
	mov @r1,a
	inc r0
	inc r1
	mov a,@r0
	mov @r1,a

	ret

;;; READ TIME FROM PCF8563
do_i2c_stuff:
	mov a,#msg_rd_time & 255
	call txtout

	call rtc_read_time
	mov r3,#3
	mov r1,#rtc_time
print_buf_time:
	mov a,@r1
	call printhex
        inc r1
        djnz r3,print_buf_time
	call space

	jf0 time_err
	mov a,#msg_ok & 255
	call txtout
	jmp vl_chk
time_err:
	mov a,#msg_err & 255
	call txtout
	jmp done_rd_time
vl_chk:
	mov a,#msg_time_vl & 255
	call txtout
	mov r0,#rtc_time
	mov a,@r0
	anl a,#080h
	jz vl_ok
	mov a,#msg_err & 255
	call txtout
	jmp rd_time_with_vl
vl_ok:
	mov a,#msg_ok & 255
	call txtout

rd_time_with_vl:
	;refresh digits
	mov r0,#digits12
	inc r0
	mov r1,#rtc_time
	inc r1
	mov a,@r1
	anl a,#07Fh
	mov @r0,a
	inc r1
	dec r0
	mov a,@r1
	anl a,#03Fh
	mov @r0,a

done_rd_time:
	ret

;FUNCTIONS 1-WIRE
 org 200h
;------------------------------------------------------------------------
; 1-wire reset; manipulate P1.0
; Input : None
; Output: A = 0 when 1-wire presense; A = 1 if no device found
; uses A, R1
;------------------------------------------------------------------------
ow_reset:
	stop tcnt
	orl p1,#1	; P1.0 = 1
	anl p1,#0FEh	; P1.0 = 0
	mov r1,#190;#197	; 480us
	call delay
	orl p1,#1	; P1.0 = 1
	mov r1,#26;#33	; 70us
	call delay
	in a,p1
	mov r1,#162;#169	; 410us
	call delay
        anl a,#1
	strt t
	ret

;------------------------------------------------------------------------
; 1-wire write bit
; Input : A bit (1 or 0)
; Output: None
; uses A, R1
;------------------------------------------------------------------------
ow_write_bit:
	anl p1,#0FEh	; P1.0 = 0
	jz zero_bit
	;mov r1,#4;#7	; 6us
	;call delay
	nop
	nop
	nop
	nop
	orl p1,#1	; P1.0 = 1
	mov r1,#24;31	; 64us
	call delay
	ret
zero_bit:
	mov r1,#22 ;#26;29	; 60us
	call delay
	orl p1,#1	; P1.0 = 1
	mov r1,#2;#9	; 10us
	call delay
	ret

;------------------------------------------------------------------------
; 1-wire read bit
; Input : None
; Output: A - bit (1 or 0)
; uses A, R1
;------------------------------------------------------------------------
ow_read_bit:
	anl p1,#0FEh	; P1.0 = 0
	;mov r1,#4;#6	; 6us
	;call delay
	nop
	nop
	nop
	nop
	orl p1,#1	; P1.0 = 1
	mov r1,#2;#9	; 9us
	call delay
	in a,p1
	mov r1,#20;#27	; 55us
	call delay
	anl a,#1
	ret

;------------------------------------------------------------------------
; 1-wire read byte
; Input : None
; Output: A - byte (8 bit)
; uses A, R1, R2, R6
;------------------------------------------------------------------------
ow_read_byte:
	mov r6,#8		; bit counter
	mov r2,#0		; result byte
	stop tcnt
shift_cycle:
	mov a,r2
	;clr c
	rr a
	;mov r2, a
	xch a,r2
	call ow_read_bit
	anl a,#1
	jz skip_shift
	;mov a,r2
	xch a,r2
	orl a,#080h
	;mov r2,a
	xch a,r2
skip_shift:
	djnz r6,shift_cycle
	;mov a,r2
	xch a,r2
	strt t
	ret

;------------------------------------------------------------------------
; 1-wire write byte
; Input : A - byte (8 bit)
; Output: None
; uses A, R0, R1, R2
;------------------------------------------------------------------------
ow_write_byte:
	mov r0,#8		; bit counter
	;mov r2,a		; byte
	xch a,r2
	stop tcnt
do_bits:
	mov a,r2
	anl a,#1
	call ow_write_bit

	;mov a,r2
	xch a,r2
	rr a
	;mov r2,a
	xch a,r2
	djnz r0,do_bits
	strt t
	ret

;------------------------------------------------------------------------
; 1-wire read family code
; Input : None
; Output: A - DS18B20 family code, default = 0x28h
; uses A, R0, R1, R2, R6, R7
;------------------------------------------------------------------------
read_famcode:
	call ow_reset
	mov a,#033h
	call ow_write_byte
	call ow_read_byte
	mov r0,a
	call printhex
	call space
	xch a,r0
	ret

;------------------------------------------------------------------------
; 1-wire read scrachpad
; Input : None
; Output: A - byte (8 bit)
; uses A, R0, R1, R2, R6, R7
;------------------------------------------------------------------------
read_scrachpad:

	call ow_reset
	mov a,#0CCh
	call ow_write_byte
	mov a,#044h
	call ow_write_byte
	mov r1,#255
	call delay
	call ow_reset
	mov a,#0CCh
	call ow_write_byte
	mov a,#0BEh
	call ow_write_byte

	mov r3,#9
	mov r0,#scrach
do_read_ds:
	call ow_read_byte
	mov @r0,a
	inc r0
	djnz r3,do_read_ds

	mov r3,#9
	mov r1,#scrach
print_scrach:
	mov a,@r1
	call printhex
        inc r1
        djnz r3,print_scrach
	ret

;------------------------------------------------------------------------
; read temperature from scrachpad
; Input : #scrach
; Output: #local_temp_l, #local_temp_h
; uses A, R0, R1, R2
;------------------------------------------------------------------------
get_temperature_from_scratchpad:
	mov r1,#scrach
	; DS18B20 - T MSB|LSB ma 4 bity po przecinku
	mov A, @R1	; T LSB
	anl A, #00001111b
;	swap A

		;shl ax, 1  ; *2    // 2
		;mov bx, ax ; *2    // 4
		;shl ax, 1  ; *4    // 6
		;shl ax, 1  ; *8    // 8
		;add ax, bx ; *10   // 10
	rl a
	mov r2,a
	rl a
	rl a
	add a,r2

 if 0
	mov r2,#0
	rlc a			; * 2
	xch a,r2
	rlc a
	xch a,r2

	mov r3,a
	mov a,r2
	mov r4,a
	mov a,r3

	rlc a			; * 4
	xch a,r2
	rlc a
	xch a,r2

	rlc a			; * 8
	xch a,r2
	rlc a
	xch a,r2

	add a,r2		; * 10
	xch a,r2
	addc a,r4
;	xch a,r2
 endif

	mov r0,#local_temp_l
	mov @r0, A

	call printhex
	mov A, @R1	; T LSB
	swap A
	anl A, #00001111b
	mov r2, A
	inc R1
	mov A, @R1	; T MSB
	swap A
	anl A, #11110000b
	orl A, r2
	inc r0
	mov @r0, A	; dla DS18S20: l_temp_h + B - C
	call printhex
	call newline

	ret


;------------------------------------------------------------------------
; CRC8 Dallas
; Input : #scrach
; Output: A
; uses A, R0, R2, R6, R7
;------------------------------------------------------------------------
crc8dallas:
	mov r2,#0
	mov r7,#8		; scrachpad len cntr
	mov r0,#scrach
rr_rot:
	xch a,r2
	xrl a,@r0
	xch a,r2
	mov r6,#8		; for (i = 0; i < 8; i++)
rr_inner:
	mov a,r2
	anl a,#1
	jz rr_shr
	xch a,r2
	clr c
	rrc a
	xrl a,#08Ch
	xch a,r2
	jmp rr_next
rr_shr:	
	xch a,r2
	clr c
	rrc a
	xch a,r2
rr_next:
	djnz r6,rr_inner
	inc r0
	djnz r7,rr_rot

	xch a,r2
	ret

; FUNCTIONS I2C
 org 400h

SDA_SET macro
	orl p1,#2	; SDA=1
	endm

SCL_SET macro
	orl p1,#4	; SCL=1
	endm

SDA_CLR macro
	anl p1,#0FDh	; SDA=0
	endm

SCL_CLR macro	
	anl p1,#0FBh    ; SCL=0
	endm

; INIT i2c
i2c_init:
	SDA_SET
	SCL_SET
	ret

; START condition
i2c_start:
	SDA_SET
	SCL_SET
	SDA_CLR
	SCL_CLR
	ret

; Issuing a STOP condition
i2c_stop:
	SDA_CLR
	SCL_SET
	SDA_SET
	ret

; Clock pulse generation. The function returns data or acknowledgment bit
; Return A = 0
i2c_clock:
	SCL_SET
wait_scl_hi:		; WARNING THERE! INFINITE LOOP MAY OCCUR
	in a,p1
	anl a,#4
	jz wait_scl_hi	; if a pulse was stretched

	in a,p1		; read SDA
	anl a,#2
	SCL_CLR
	ret

;  Input A - byte
;  Writing a byte to a slave, with most significant bit first. The
;  function returns acknowledgment bit. 
i2c_write:
	xch a,r3
	mov r2,#080h
i2c_wr_cyc:
	mov a,r2
	jz done_wr
	mov a,r3
	anl a,r2
	jz set_zero
	SDA_SET
	jmp do_clk
set_zero:
	SDA_CLR
do_clk:
	call i2c_clock
	xch a,r2
	clr c
	rrc a
	xch a,r2
	jmp i2c_wr_cyc
done_wr:
	SDA_SET
	call i2c_clock
	ret

;bit i2c_write(uchar byte)
;{
;	uchar mask = 0x80;
;
;	while (mask) {
;		if (byte & mask)
;			SDA = 1;
;		else
;			SDA = 0;
;		i2c_clock();
;		mask >>= 1;
;	}
;	SDA = 1;
;	return (i2c_clock());
;}

;  Input: A - ack
;  Reading byte from a slave, with most significant bit
;   first. The parameter indicates, whether to acknowledge
;   (1) or not (0)
i2c_read:
	mov r3,a
	mov r2,#080h
	mov r1,#0
i2c_rd_cyc:
	mov a,r2
	jz i2c_rd_cy_done
	call i2c_clock
	jz i2c_mask
	xch a,r1
	orl a,r2
	xch a,r1
i2c_mask:
	xch a,r2
	clr c
	rrc a
	xch a,r2
	jmp i2c_rd_cyc
i2c_rd_cy_done:	
	mov a,r3
	jz not_ack
	SDA_CLR
	call i2c_clock
	SDA_SET
	jmp i2c_rd_done
not_ack:
	SDA_SET
	call i2c_clock
i2c_rd_done:
	mov a,r1
	ret

;uchar i2c_read(bit acknowledgment)
;	uchar mask = 0x80, byte = 0x00;
;
;	while (mask) {
;		if (i2c_clock())
;			byte |= mask;
;		mask >>= 1;	/* next bit to receive */
;	}
;	if (acknowledgment) {
;		SDA = 0;
;		i2c_clock();
;		SDA = 1;
;	} else {
;		SDA = 1;
;		i2c_clock();
;	}
;	return (byte);


; PCF8563 RTC functions 
; Return: rtc_time[3] = time SS:MM:HH pcf8563 reg 02:03:04
;         F0 flag = 1 error, 0 ok
rtc_read_time:
	cpl f0		; set error flag
	call i2c_start

	mov a,#0A2h
	call i2c_write
	jnz rtc_err1

	mov a,#2
	call i2c_write
	jnz rtc_err1

	call i2c_start
rtc_err1:
	mov a,#0A3h
	call i2c_write
	jnz rtc_err2

	mov r0,#rtc_time
	mov a,#1
	call i2c_read
	mov @r0,a
	inc r0
	mov a,#1
	call i2c_read
	mov @r0,a
	inc r0
	clr a
	call i2c_read
	mov @r0,a
	clr f0		; clear error flag
rtc_err2:
	call i2c_stop
	ret
;{
;	i2c_start();
;	if (!i2c_write(PCF8563))
;		if (!i2c_write(0x02))
;			i2c_start();
;	if (!i2c_write(PCF8563 | 0x01)) {
;		time[0] = i2c_read(1);
;		time[1] = i2c_read(1);
;		time[2] = i2c_read(0);
;	}
;	i2c_stop();
;}


; Input: r4 = addr, r5 = value
; Return: F0 flag = 1 error, 0 ok
rtc_write:
	cpl f0
	call i2c_start
	mov a,#0A2h
	call i2c_write
	jnz rtc_wr_err
	mov a,r4
	call i2c_write
	jnz rtc_wr_err
	mov a,r5
	call i2c_write
	jnz rtc_wr_err
	clr f0
rtc_wr_err:
	call i2c_stop
	ret

rtc_write_time:
	mov r4,#2		; set seconds to zero
	mov r5,#0
	call rtc_write
	mov r0,#digits12
	mov r4,#4		; set hours
	mov a,@r0
	xch a,r5
	call rtc_write
	jf0 wr_time_err
	mov r4,#3		; set minutes
	inc r0
	mov a,@r0
	xch a,r5
	call rtc_write
wr_time_err:
	ret
;void rtc_write_time(const char *time)
;{
;	/* set time */
;	if (!rtc_write(0x02, time[0]))	/* set seconds */
;		goto FAIL;
;
;	if (!rtc_write(0x03, time[1]))	/* set minutes */
;		goto FAIL;
;
;	if (!rtc_write(0x04, time[2])) 	/* set hours */
;		goto FAIL;
;
;	return;
;
; FAIL:
;	return;
;}


; SUB FUNCTION
 org 500h
;------------------------------------------------------------------------
; Delay tCYC
; Input: R1 = us (delay_us-5)/ 2.5us
; for ex. needed delay 480us, so R1 = 190
; Output: None
; uses R1
;------------------------------------------------------------------------
delay:
	;call = 2 cycle
	djnz r1,delay		; 2 cycle (1,25*2=2,5us)
	;ret = 2 cycle
	ret 


;------------------------------------------------------------------------
; print carriage return and line feed
;------------------------------------------------------------------------            
newline:    mov A,#CR
            call putch
            mov A,#LF
            jmp putch
            
;------------------------------------------------------------------------
; print a space
;------------------------------------------------------------------------            
space:      mov A,#' '
            jmp putch
            

;------------------------------------------------------------------------            
; prints the contents of the accumulator as two hex digits
; uses A, R5, R6, R7
;------------------------------------------------------------------------
printhex:   mov R5,A            ; save the value on A in R0
            rr A
            rr A
            rr A
            rr A
            call hex2ascii
            call putch          ; print the most significant digit
            mov A,R5            ; recall the value from R0
            call hex2ascii
            call putch          ; print the least significant digit
            ret

; returns the ASCII value for the hex nibble in A
; the table of hex nibbles starts at 0300H
hex2ascii:  anl A,#0FH
            movp3 A,@A
            ret

; org 0200H
;------------------------------------------------------------------------
; sends the character in A out from the serial output (P2.7)
; uses A, R6 and R7.
;------------------------------------------------------------------------
putch:      
	    stop tcnt
            anl P2,#7FH             ; make serial output low to send the start bit
            mov R6,#8               ; load R6 with the number of bits to send
            mov R7,#38 ;#36              ; load the number of cycles to delay into R7
            djnz R7,$               ; delay 60 cycles

            ;send bits 0-7
putch1:     jb0 putch2              ; jump if the bit to send is "1"
            anl P2,#7FH             ; else, send "0"
            jmp putch3              ; skip the next part
putch2:     orl P2,#80H             ; send "1"
            jmp putch3              ; makes the timing equal for both "0" and "1" bits
putch3:     rr A                    ; rotate the next bit into position
            mov R7,#36 ;#35              ; load the number of cycles to delay into R7
            djnz R7,$               ; delay 58 cycles
	nop
            djnz R6,putch1          ; loop to send all 8 bits

            ;send the stop bit
            nop
            nop
            orl P2,#80H             ; make serial output high to send the stop bit
            mov R7,#21; ;#22              ; load the number of cycles to delay into R7
            djnz R7,$               ; delay 36 cycles (1/2 bit time)
            strt t
            ret

;------------------------------------------------------------------------
; print the string in page 3 of program memory pointed to by A.
; the string must be terminated by zero.
; in addition to A, uses R5, R6 and R7 in register bank 1.
;------------------------------------------------------------------------
txtout:     sel RB1
            mov R5,A
txtout1:    mov A,R5
            movp3 A,@A          ; move to A from page 3 of program memory
            anl A,#07FH
            jz txtdone
            call putch
            sel RB1
            inc R5
            jmp txtout1
txtdone:    sel RB0
            ret

;------------------------------------------------------------------------
; 16 bit binary to BCD
; A  = lower eighth bits of binary value
; R2 = upper eighth bits of binary value
; R0 = pointer to a packed BCD string
; used r0,r1,r2,r3,r4,r5
;------------------------------------------------------------------------            
bin2bcd:
xa    equ r2
count equ r3
icnt  equ r4
digpr equ 1

temp1 set r5

	xch a,r0
	mov r1,a
	xch a,r0
	mov icnt,#digpr
bcdcoa: mov @r1,#00
	inc r1
	djnz icnt,bcdcoa

	mov count,#16
bcdcob: ; BIN = BIN*2
	clr c
	rlc a
	xch a,xa
	rlc a
	xch a,xa
	; BCD = BCD*2+CARRY
	xch a,r0
	mov r1,a
	xch a,r0
	mov icnt,#digpr
	mov temp1,a
bcdoc:  mov a,@r1
	addc a,@r1
	da a
	mov @r1,a
	inc r1
	djnz icnt,bcdoc
	mov a,temp1
	; if carry from bcdacc goto error
	jc bcdcod
	djnz count,bcdcob
	clr c
bcdcod: ret

;------------------------------------------------------------------------
; DIV16
; Input:
;        A  = LOWER 8 bit of dest operand
;        XA = UPPER 8 bit of divident
;        R1 = diviser
; Output:
;        A  = lower 8 bit result
;        XA = remainder
;------------------------------------------------------------------------
div16:

	xch a,xa
	mov COUNT,#8
	cpl a
	add a,r1
	cpl a
	jc DIVIA
	cpl c
	jmp DIVIB
DIVIA:
	add a,r1
DIVILP:
	clr c
	xch a,xa
	rlc a
	xch a,xa
	rlc a
	jnc DIVIE
	cpl a
	add a,r1
	cpl a
	jmp DIVIC
DIVIE:
	cpl a
	add a,r1
	cpl a
	jnc DIVIC
	add a,r1
	jmp DIVID
DIVIC:
	inc xa
DIVID:
	djnz COUNT,DIVILP
	clr c
DIVIB:
	xch a,xa
	ret


; DO BURN-IN INDICATORS
do_update:
	mov a,#msg_burnin & 255
	call txtout
	
	; turn on all segments
	mov r0,#ind_mask
	mov a,@r0
	orl a,#00000111b
	mov @r0,a
	mov r0,#2
	movx @r0,a

	mov r0,#digits12
	clr a
	mov @r0,a
	inc r0
	mov @r0,a

	mov r7,#10
count_up:
	call display_digits

	mov r0,#digits12
	mov a,@r0
	add a,#011h
	mov @r0,a
	inc r0
	mov @r0,a

	call big_delay
	djnz r7,count_up

	mov r0,#digits12
	mov a,#088h
	mov @r0,a
	inc r0
	mov @r0,a

	mov r7,#9
count_down:
	call display_digits

	mov r0,#digits12
	mov a,@r0
	jz skip_dec
	;dec a,#011h
	mov r5,#011h
	cpl a
	add a,r5
	cpl a
	mov @r0,a
	inc r0
	mov @r0,a
skip_dec:
	call big_delay
	djnz r7,count_down
	ret

; Increment BCD number
; Input A - BCD num
; Output A
bcdinc:
	mov r2,a		;save A
	anl a,#0fh
	mov r1,#9
	compare r1
	jnz simple_inc
	mov a,r2
	add a,#7
	ret
simple_inc:
	mov a,r2
	inc a
	ret

; draw hex then "s"
draw_hex_sec:
	call printhex
	mov a,#"s"
	call putch
	call newline
	ret

 org 600h

;LCD 16x2
lcd_slave equ 04Eh

time_msg: db "TIME ",0
lcd_dbg:  db "DBG_MODE",0

 IFDEF LCD_DELAY
; R6 = ms count
delay_ms:
	mov r1,#100
	call delay
	;mov r1,#143
	;call delay
	djnz r6,delay_ms
	ret
 ENDIF

; send cmd to lcd
; r1 = cmd
lcd_send_cmd:
	call i2c_start
	mov a,#lcd_slave
	call i2c_write
	mov a,r1
	;clr c
	rl a
	rl a
	rl a
	rl a
	anl a,#0f0h
	mov r4,a	; cmd_l
	mov a,r1
	anl a,#0f0h
	mov r5,a	; cmd_u
	orl a,#0ch
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#1
	call delay_ms	; 1ms
 ENDIF
	mov a,r5
	orl a,#08h
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#10
	call delay_ms	; 10ms
 ENDIF
	mov a,r4
	orl a,#0ch
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#1
	call delay_ms	; 1ms
 ENDIF
	mov a,r4
	orl a,#08h
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#10
	call delay_ms	; 10ms
 ENDIF
	call i2c_stop
	ret

; send data to lcd
; r1 = data
lcd_send_data:
	call i2c_start
	mov a,#lcd_slave
	call i2c_write
	mov a,r1
	;clr c
	rl a
	rl a
	rl a
	rl a
	anl a,#0f0h
	mov r4,a	; data_l
	mov a,r1
	anl a,#0f0h
	mov r5,a	; data_u
	orl a,#0dh
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#1
	call delay_ms	; 1ms
 ENDIF
	mov a,r5
	orl a,#09h
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#10
	call delay_ms	; 10ms
 ENDIF
	mov a,r4
	orl a,#0dh
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#1
	call delay_ms	; 1ms
 ENDIF
	mov a,r4
	orl a,#09h
	call i2c_write
 IFDEF LCD_DELAY
	mov r6,#10
	call delay_ms	; 10ms
 ENDIF
	call i2c_stop
	ret

lcd_init:
	mov r1,#02h		; return home
	call lcd_send_cmd
	mov r1,#028h		; 4-bit mode
	call lcd_send_cmd
	mov r1,#0ch		; Display On, Cursor off
	call lcd_send_cmd
	mov r1,#06h		; Increment cursor (shift cursor to right)
	call lcd_send_cmd
	mov r1,#01h		; Clear Display
	call lcd_send_cmd
	ret

;------------------------------------------------------------------------
; print the string in current page of program memory pointed to by A.
; the string must be terminated by zero.
; in addition to A, uses R5, R6 and R7 in register bank 1.
;------------------------------------------------------------------------
lcdout:     sel RB1
            mov R5,A
lcdout1:    mov A,R5
            movp A,@A          ; move to A from current page of program memory
            ;anl A,#07FH
            jz lcddone
	    sel rb0
	    mov r1,a
            call lcd_send_data
            sel RB1
            inc R5
            jmp lcdout1
lcddone:    sel RB0
            ret


; UPDATE LCD delimiter
lcd_display_sec:
	mov r0,#seconds
	mov a,@r0
	jb0 n_even_sec
	mov r1,#087h		; 1st string, 7 row
	call lcd_send_cmd
	mov r1,#" "
	call lcd_send_data
	ret
n_even_sec:	
	mov r1,#087h		; 1st string, 7 row
	call lcd_send_cmd
	mov r1,#":"
	call lcd_send_data
	ret

; DISPLAY TIME
lcd_display_time:
	mov r1,#080h
	call lcd_send_cmd
	mov a,#time_msg & 255
	call lcdout

	mov r0,#digits12
	mov a,@r0
	anl a,#0F0h
	rr a
	rr a
	rr a
	rr a
	add a,#030h
	mov r1,a
	call lcd_send_data
	
	mov a,@r0
	anl a,#0Fh
	add a,#030h
	mov r1,a
	call lcd_send_data

	mov r1,#":"
	call lcd_send_data
	inc r0

	mov a,@r0
	anl a,#0F0h
	rr a
	rr a
	rr a
	rr a
	add a,#030h
	mov r1,a
	call lcd_send_data
	
	mov a,@r0
	anl a,#0Fh
	add a,#030h
	mov r1,a
	call lcd_send_data
	ret


 org 700h

; DISPLAY TIME
lcd_display_temp:
	mov r1,#08Bh		; 1st string, 11 row
	call lcd_send_cmd

	mov r0,#bcd_buf1
	mov a,@r0
	anl a,#0F0h
	rr a
	rr a
	rr a
	rr a
	add a,#030h
	mov r1,a
	call lcd_send_data
	
	mov a,@r0
	anl a,#0Fh
	add a,#030h
	mov r1,a
	call lcd_send_data

	mov r1,#"."
	call lcd_send_data
	inc r0
	
	mov a,@r0
	anl a,#0Fh
	add a,#030h
	mov r1,a
	call lcd_send_data
	mov r1,#"C"
	call lcd_send_data
	ret

lcd_display_copyr:
	mov r1,#0C4h
	call lcd_send_cmd
	mov a,#lcd_dbg & 255
	call lcdout
	ret


; UPDATE delimiter
proceed_delimiter:
	mov r0,#seconds
	mov a,@r0
	jb0 nt_even_sec
	mov r0,#ind_mask
	mov a,@r0
	orl a,#00001000b
	mov @r0,a
	mov r0,#2
	movx @r0,a
	ret
nt_even_sec:	
	mov r0,#ind_mask
	mov a,@r0
	anl a,#11110111b
	mov @r0,a
	mov r0,#2
	movx @r0,a
	ret

; big delay
big_delay:
	mov r4,#20
big_delay_lp1:
	mov r0,#temp_reg
	mov a,r4
	mov @r0,a
;	mov r6,#19; #255
;big_delay_lp2:
;	mov r6,#1
;big_delay_lp3:
 IFDEF LCD
	call lcd_display_sec	; ":" or " "
 ENDIF
	call proceed_delimiter
;	djnz r6,big_delay_lp3
;	djnz r6,big_delay_lp2
	mov r0,#temp_reg
	mov a,@r0
	mov r4,a
	djnz r4,big_delay_lp1
	ret

;------------------------------------------------------------------------            
; prints the contents of the accumulator as two hex digits
; uses A, R5, R6, R7
;------------------------------------------------------------------------
printlcdhex:   mov R7,A            ; save the value on A in R0
            rr A
            rr A
            rr A
            rr A
            call hex2ascii
 		mov r1,a
            call lcd_send_data          ; print the most significant digit
            mov A,R7            ; recall the value from R0
            call hex2ascii
		mov r1,a
            call lcd_send_data          ; print the least significant digit
            ret


; T0 T1 = buttons
proceed_buttons:
	mov r2,#255		; anti-noise delay preload
	jnt0 hours_btn
	jnt1 min_btn
	ret
btn_done:
	call rtc_write_time
 IFDEF LCD
	call lcd_display_time
 ENDIF
	ret
hours_btn:
	djnz r2,$		; anti-noise
	jt0 btn_done

	call do_i2c_stuff	; DO i2c stuff (read time)
	mov r0,#digits12
	mov a,@r0
	inc a
	mov r2,#024h
	compare r2
	jnc clr_hours
	xch a,@r0
	call bcdinc
	xch a,@r0

wait_hr_unpress:
	jt0 btn_done		; wait when key unpressed
	jmp wait_hr_unpress

clr_hours:
	clr a
	mov @r0,a
	jmp wait_hr_unpress

min_btn:
	djnz r2,$		; anti-noise
	jt1 btn_done


	call do_i2c_stuff	; DO i2c stuff (read time)
	mov r0,#digits34
	mov a,@r0
	inc a
	mov r2,#059h
	compare r2
	jnc clr_mins
	xch a,@r0
	call bcdinc
	xch a,@r0

wait_min_unpress:
	jt1 btn_done		; wait when key unpressed
	jmp wait_min_unpress

clr_mins:
	clr a
	mov @r0,a
	jmp wait_min_unpress

 org 300h

 db  30H,31H,32H,33H,34H,35H,36H,37H,38H,39H,41H,42H,43H,44H,45H,46H  ; hex digits 0-F in ASCII

titletxt:   db  CR,LF
	    db  "8039 Nixie clock v1.4 by Tronix (C)2023",CR,LF
	    db  "CLK=12MHz;CYC_TIME=1.25us;TMR_CNT=25KHz",CR,LF
            db  "Assembled on ",DATE," at ",TIME,CR,LF,CR,LF,0

ow_init_msg:db  "ow_reset = ",0
ow_code_msg:db  "ow_fam_code = ",0
ow_pad_msg: db  "pad = ",0
msg_ok:	    db  "OK",CR,LF,0
msg_err:    db  "ERR!",CR,LF,0
msg_crc:    db  "CRC ",0
msg_l_temp: db  "loc_temp = ",0
msg_t_bcd:  db  "temp bcd = ",0
msg_rd_time:db  "read_pc8563 = ",0
msg_time_vl:db  "VLbit = ",0
msg_burnin: db  "BURNIN",CR,LF,0
msg_timer:  db  CR,LF,"TMR = ",0
end