/*
		(c) Perlin Production Co. Ltd, 1991-2023
			All rights reserved
		
	Pseudographic window I/O system ver.5.0 for Vector-06c
	(modified ver 3.1 from 08.05.1993, previous millenium)
 			SuperSoft C 1.2.3 compiler
 			
Console I/O
 
			LIST OF FUNCTIONS:

	bputch	(char)	character directly to console using BDOS
	cputch	(char)	character directly to console using BDOS
char	getch	()	gets console character no echo
char	getche	()	gets console character with echo
char	getkey	()	waits for keyboard input console character no echo
char	kbhit	()	determines if keyboard was pressed
*/
#include "bdos.h"

#define DUMMY	0

/* determines if keyboard was pressed */
char	kbhit()		{ return bdos(_CONST,DUMMY); /* bios(CONST) */}

/* gets console character no echo */
char	getch()		{ return bdos(_CONIO,0xff); /* bios(CONIN) */}

/* waits for keyboard input console character no echo */
char getkey()
{
	register char key;

	while (!(key=getch()));
	return key;
}

/* gets console character with echo */
char	getche()	{ return bdos(_CONIN,DUMMY); /* n/a in BIOS */}

/* character directly to console using BDOS
 * cputch(c) char c;	104 cycles
 * 	faster tnan bdos(_CONOUT,c,DUMMY) or 128 cycles cconout(c)
 *	does not translate '\n' to '\r' contrary to putchar
 *	unable redirecting console to stdout
 *
 *	For info: SuperSoft C console redirection to stdout
 *	#if REDIRECT	putchar()->cput() { stdout ? putc(buf) : cconout }
 *	#else		putchar()->cconout()
 */
#asm
CCBDOS	EQU	5
CCONOUT	EQU	2
public	cputch;
cputch:	push	b		; 16 save register variable
	lxi	h,4		; 12
	dad	sp		; 12
	mov	e,m		;  8 character from <E> to console
	mvi	c,CCONOUT	;  8 BDOS console out function
	CALL	CCBDOS		; 24 BDOS call
	POP	B		; 12 restore register variable
	ret			; 12
#endasm

/* character directly to console using BIOS
 * bputch(c) char c;
 */
#ifdef ROMABLE
/* bputch(c)
 *	slow
 *	ROMable
 */
#define BCONOUT	4	/* BIOS CONOUT entry (jump number) */
bputch	(c) char c; { bios(BCONOUT,c,DUMMY); }

#else
/* bputch(c) char c;	96 cycles
 *	non-ROMable (self-modifies to preset BIOS CONOUT entry)
 *	does not translate '\n' to '\r' contrary to putchar
 *	unable redirecting console to stdout
 */
#asm
BIOSENT	EQU	1	; BIOS table vector
BCONOUT	EQU	10	; (4 * 3 - 3 + 1) output to CON:
OP_CALL	EQU	205	; Manx assembler accepts 0xCD
public	bputch;
bputch:	push	b	; 16 save register variable
	lxi	h,4	; 12
	dad	sp	; 12
	mov	c,m	;  8 character from <C> to console
;	push	b	; written character is returned on success
	DB	OP_CALL
bconjmp:DW	init	; 24 address used in BIOS jmp to console out handler
;	pop	h	; written character is returned on success
	pop	b	; 12 restore register variable
	ret		; 12

init:
;	push	d
	lhld	BIOSENT
	lxi	d,BCONOUT
	dad	d
	mov	e,m
	inx	h
	mov	d,m
	xchg
	shld	bconjmp
;	pop	d
	pchl
#endasm
#endif	/*ROMABLE*/
