;/* C Define for CH376        */

;/* ********************************************************************************************************************* */
;/* Hardware features */

#define	CH376_DAT_BLOCK_LEN	$40			/* USB single data packet, maximum length of data block, default buffer length */

;/* ********************************************************************************************************************* */
;/* command code */
;/* Some commands are compatible with the CH375 chip, but the input data or output data may be partially different) */
;/* A command operation sequence includes:
;           A command code (for serial port mode, two synchronization codes are required before the command code),
;           Several input data (can be 0),
;           Generate an interrupt notification or several output data (can be 0), choose one of the two, if there is an interrupt notification, there must be no output data, if there is output data, there must be no interrupt.
;        Only the CMD01_WR_REQ_DATA command is an exception, the sequence includes: a command code, an output data, several input data
;    Command code naming rules: CMDxy_NAME
;        Where x and y are both numbers, x indicates the minimum number of input data (number of bytes), y indicates the minimum number of output data (number of bytes), and if y is H, it indicates that an interrupt notification is generated.
;        Some commands can read and write data blocks of 0 to multiple bytes. The number of bytes of the data block itself is not included in the above x or y */
;/* By default, this file will also provide a command code format compatible with the CH375 chip command code (that is, after removing x and y). If not, you can define _NO_CH375_COMPATIBLE_forbidden */
;/* ********************************************************************************************************************* */
;/* Main commands (Manual 1), commonly used */

#define CMD01_GET_IC_VER $01 ;/* Get the chip and firmware version */
;/* Output: version number (bit 7 is 0, bit 6 is 1, bit 5~bit 0 are version numbers) */
;/* The value of the version number returned by CH376 is 041H, that is, the version number is 01H */

#define CMD21_SET_BAUDRATE $02 ;/* Serial port mode: Set the serial port communication baud rate (the default baud rate after power-on or reset is 9600bps, selected by the D4/D5/D6 pin) */
;/* Input: baud rate division coefficient, baud rate division constant */
;/* Output: Operation status (CMD_RET_SUCCESS or CMD_RET_ABORT, other values indicate that the operation is not completed) */

#define CMD00_ENTER_SLEEP $03 ;/* Enter sleep state */

#define CMD00_RESET_ALL $05 ;/* Perform hardware reset */

#define CMD11_CHECK_EXIST $06 ;/* Test communication interface and working status */
;/* Input: any data */
;/* Output: bitwise inversion of input data */

#define CMD20_CHK_SUSPEND $0B ;/* Device mode: Set the method to check the USB bus suspend status */
;/* Input: data 10H, check mode */
;/* 00H=Do not check USB hang, 04H=Check USB hang at 50mS intervals, 05H=Check USB hang at 10mS intervals */

#define CMD20_SET_SDO_INT $0B ;/* SPI interface mode: Set the interrupt mode of SPI SDO pin */
;/* Input: data 16H, interrupt mode */
;/* 10H=disable SDO pin for interrupt output, tri-state output is disabled when SCS chip select is invalid, 90H=SDO pin also serves as interrupt request output when SCS chip select is invalid */

#define CMD14_GET_FILE_SIZE $0C ;/* Host file mode: Get the current file length */
;/* Input: data 68H */
;/* Output: Current file length (total length 32 bits, low byte first) */

#define CMD50_SET_FILE_SIZE $0D ;/* Host file mode: Set the current file length */
;/* Input: data 68H, current file length (total length 32 bits, low byte first) */

#define CMD11_SET_USB_MODE $15 ;/* Set USB working mode */
;/* Input: pattern code 
;00H=Unenabled device mode, 
;01H=Enabled device mode and uses external firmware mode (serial port does not support),
;02H=Enabled device mode and uses built-in firmware mode 
;03H=SD card host mode/unenabled host mode, used to manage and access files in the SD card 
;04H=Disabled host mode, 
;05H=Enabled host mode, 
;06H=Enabled host mode and automatically generates SOF packet, 
;07H=Enabled host mode and resets the USB bus */
;/* Output: Operation status (CMD_RET_SUCCESS or CMD_RET_ABORT, other values indicate that the operation is not completed) */

#define CMD01_GET_STATUS $22 ;/* Get the interrupt status and cancel the interrupt request */
;/* Output: interrupt status */

#define CMD00_UNLOCK_USB $23 ;/* Device mode: Release the current USB buffer */

#define CMD01_RD_USB_DATA0 $27 ;/* Read the data block from the endpoint buffer of the current USB interrupt or the receive buffer of the host endpoint */
;/* Output: length, data stream */

#define CMD01_RD_USB_DATA $28 ;/* Device mode: Read the data block from the current USB interrupt endpoint buffer and release the buffer, equivalent to CMD01_RD_USB_DATA0 + CMD00_UNLOCK_USB */
;/* Output: length, data stream */

#define CMD10_WR_USB_DATA7 $2B ;/* Device mode: Write data block to the send buffer of USB endpoint 2 */
;/* Input: length, data stream */

#define CMD10_WR_HOST_DATA $2C ;/* Write data block to the send buffer of the USB host endpoint */
;/* Input: length, data stream */

#define CMD01_WR_REQ_DATA $2D; /* Write the requested data block to the internal specified buffer */
;/* Output: length */
;/* Input: data stream */

#define CMD20_WR_OFS_DATA $2E ;/* Write the data block to the specified offset address in the internal buffer */
;/* Input: offset, length, data stream */

#define CMD10_SET_FILE_NAME $2F ;/* Host file mode: Set the file name of the file to be operated */
;/* Input: A string terminated by 0 (the length including the terminator 0 shall not exceed 14 characters) */

;/* ********************************************************************************************************************** */
;/* Main commands (Manual 1), commonly used, the following commands always generate an interrupt notification at the end of the operation, and always output no data */

#define CMD0H_DISK_CONNECT $30 ;/* Host file mode/SD card not supported: Check whether the disk is connected */
;/* Output interrupt */

#define CMD0H_DISK_MOUNT $31 ;/* Host file mode: initialize the disk and test whether the disk is ready */
;/* Output interrupt */

#define CMD0H_FILE_OPEN $32 ;/* Host file mode: open a file or directory (folder), or enumerate files and directories (folder) */
;/* Output interrupt */

#define CMD0H_FILE_ENUM_GO $33 ;/* Host file mode: continue to enumerate files and directories (folders) */
;/* Output interrupt */

#define CMD0H_FILE_CREATE $34 ;/* Host file mode: create a new file, if the file already exists, delete it first */
;/* Output interrupt */

#define CMD0H_FILE_ERASE $35 ;/* Host file mode: Delete the file. If it is already open, delete it directly. Otherwise, the file will be opened first and then deleted. The subdirectory must be opened first */
;/* Output interrupt */

#define CMD1H_FILE_CLOSE $36 ;/* Host file mode: close the currently opened file or directory (folder) */
;/* Input: whether to allow updating the file length */
;/* 00H=Disable update length, 01H=Allow update length */
;/* Output interrupt */

#define CMD1H_DIR_INFO_READ $37; /* Host file mode: Read the directory information of the file */
;/* Input: Specify the index number within the sector of the directory information structure to be read */
;/* The index number range is 00H~0FH, and the index number 0FFH is the currently opened file */
;/* Output interrupt */

#define CMD0H_DIR_INFO_SAVE $38 ;/* Host file mode: save the directory information of the file */
;/* Output interrupt */

#define CMD4H_BYTE_LOCATE $39 ;/* Host file mode: move the current file pointer in bytes */
;/* Input: offset bytes (total length 32 bits, low byte first) */
;/* Output interrupt */

#define CMD2H_BYTE_READ $3A ;/* Host file mode: Read data block from the current position in bytes */
;/* Input: The number of bytes requested to be read (total length 16 bits, low byte first) */
;/* Output interrupt */

#define CMD0H_BYTE_RD_GO $3B ;/* Host file mode: continue byte reading */
;/* Output interrupt */

#define CMD2H_BYTE_WRITE $3C ;/* Host file mode: Write data block to the current location in bytes */
;/* Input: The number of bytes requested to be written (total length 16 bits, low byte first) */
;/* Output interrupt */

#define CMD0H_BYTE_WR_GO $3D ;/* Host file mode: continue byte writing */
;/* Output interrupt */

#define CMD0H_DISK_CAPACITY $3E ;/* Host file mode: Query disk physical capacity */
;/* Output interrupt */

#define CMD0H_DISK_QUERY $3F ;/* Host file mode: Query disk space information */
;/* Output interrupt */

#define CMD0H_DIR_CREATE $40 ;/* Host file mode: Create a new directory (folder) and open it. If the directory already exists, open it directly */
;/* Output interrupt */

#define CMD4H_SEC_LOCATE $4A ;/* Host file mode: move the current file pointer in units of sectors */
;/* Input: offset sector number (total length 32 bits, low byte first) */
;/* Output interrupt */

#define CMD1H_SEC_READ $4B ;/* Host file mode/SD card not supported: Read data blocks from the current location in units of sectors */
;/* Input: Number of sectors requested to be read */
;/* Output interrupt */

#define CMD1H_SEC_WRITE $4C ;/* Host file mode/SD card not supported: Write data block at the current location in units of sectors */
;/* Input: Number of sectors requested to be written */
;/* Output interrupt */

#define CMD0H_DISK_BOC_CMD $50 ;/* Host mode/Does not support SD card: Execute BulkOnly transfer protocol command on USB storage */
;/* Output interrupt */

#define CMD5H_DISK_READ $54 ;/* Host mode/SD card not supported: read physical sectors from USB memory */
;/* Input: LBA physical sector address (total length 32 bits, low byte first), sector number (01H~FFH) */
;/* Output interrupt */
;
#define CMD0H_DISK_RD_GO $55 ;/* Host mode/SD card not supported: Continue to perform physical sector read operation of USB memory */
;/* Output interrupt */

#define CMD5H_DISK_WRITE $56 ;/* Host mode/SD card not supported: write physical sectors to USB memory */
;/* Input: LBA physical sector address (total length 32 bits, low byte first), sector number (01H~FFH) */
;/* Output interrupt */

#define CMD0H_DISK_WR_GO $57 ;/* Host mode/SD card not supported: Continue to perform physical sector write operation of USB memory */
;/* Output interrupt */

;/* ************************************************ *************************************************** ******************* */
;/* Auxiliary commands (Manual 2), less commonly used or for compatibility with CH375 and CH372 */

#define CMD10_SET_USB_SPEED $04 ;/* Set the USB bus speed, which will automatically return to 12Mbps full speed each time CMD11_SET_USB_MODE sets the USB working mode */
;/* Input: bus speed code */
;/* 00H=12Mbps full speed FullSpeed (default value), 01H=1.5Mbps (only modify frequency), 02H=1.5Mbps low speed LowSpeed */

#define CMD11_GET_DEV_RATE $0A ;/* Host mode: Get the data rate type of the currently connected USB device */
;/* Input: data 07H */
;/* Output: data rate type */
;/* If bit 4 is 1, it is a 1.5Mbps low-speed USB device, otherwise it is a 12Mbps full-speed USB device */

#define CMD11_GET_TOGGLE $0A ;/* Get the synchronization status of OUT transaction */
;/* Input: data 1AH */
;/* Output: synchronization status */
;/* If bit 4 is 1, the OUT transaction is synchronized, otherwise the OUT transaction is not synchronized */

#define CMD11_READ_VAR8 $0A ;/* Read the specified 8-bit file system variable */
;/* Input: variable address */
;/* Output Data */

;/*#define CMD11_GET_MAX_LUN = CMD11_READ_VAR8( VAR_UDISK_LUN )*/ /* Host mode: Get the maximum and current logical unit number of USB memory */

#define CMD20_SET_RETRY $0B ;/* Host mode: Set the number of retries for USB transaction operations */
;/* Input: data 25H, number of retries */
;/* If bit 7 is 0, it will not retry when NAK is received. If bit 7 is 1, bit 6 is 0, it will retry infinitely when NAK is received. If bit 7 is 1 and bit 6 is 1, it will retry for up to 3 seconds when NAK is received. , bit 5~bit 0 are the number of retries after timeout*/

#define CMD20_WRITE_VAR8 $0B ;/* Set the specified 8-bit file system variable */
;/* Input: variable address, data */

;/*#define CMD20_SET_DISK_LUN = CMD20_WRITE_VAR8( VAR_UDISK_LUN )*/ /* Host mode: Set the current logical unit number of the USB memory */

#define CMD14_READ_VAR32 $0C ;/* Read the specified 32-bit file system variable */
;/* Input: variable address */
;/* Output: data (total length 32 bits, low byte first) */

#define CMD50_WRITE_VAR32 $0D ;/* Set the specified 32-bit file system variable */
;/* Input: variable address, data (total length 32 bits, low byte first) */

#define CMD01_DELAY_100US $0F ;/* Delay 100uS (not supported by serial port) */
;/* Output: 0 during delay, non-0 at end of delay */

#define CMD40_SET_USB_ID $12 ;/* Device mode: Set USB manufacturer VID and product PID */
;/* Input: manufacturer ID low byte, manufacturer ID high byte, product ID low byte, product ID high byte */

#define CMD10_SET_USB_ADDR $13 ;/* Set USB address */
;/* Input: address value */

#define CMD01_TEST_CONNECT $16 ;/* Host mode/SD card not supported: Check USB device connection status */
;/* Output: status (USB_INT_CONNECT or USB_INT_DISCONNECT or USB_INT_USB_READY, other values indicate that the operation is not completed) */

#define CMD00_ABORT_NAK $17 ;/* Host mode: give up the current NAK retry */

#define CMD10_SET_ENDP2 $18 ;/* Device mode (not supported by serial port): Set the receiver of USB endpoint 0 */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000-Ready ACK, 1110-Busy NAK, 1111-Error STALL */

#define CMD10_SET_ENDP3 $19 ;/* Device mode (not supported by serial port): Set the transmitter of USB endpoint 0 */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000~1000-Ready ACK, 1110-Busy NAK, 1111-Error STALL */

#define CMD10_SET_ENDP4 $1A ;/* Device mode (not supported by serial port): Set the receiver of USB endpoint 1 */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000-Ready ACK, 1110-Busy NAK, 1111-Error STALL */

#define CMD10_SET_ENDP5 $1B ;/* Device mode (not supported by serial port): Set the transmitter of USB endpoint 1 */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000~1000-Ready ACK, 1110-Busy NAK, 1111-Error STALL */

#define CMD10_SET_ENDP6 $1C ;/* Set the receiver of USB endpoint 2/host endpoint */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000-Ready ACK, 1101-Ready but does not return ACK, 1110-Busy NAK, 1111-Error STALL */

#define CMD10_SET_ENDP7 $1D ;/* Set the transmitter of USB endpoint 2/host endpoint */
;/* Input: working mode */
;/* If bit 7 is 1, bit 6 is the synchronization trigger bit, otherwise the synchronization trigger bit remains unchanged */
;/* Bit 3~Bit 0 is the transaction response mode: 0000-Ready ACK, 1101-Ready but no response required, 1110-Busy NAK, 1111-Error STALL */

#define CMD00_DIRTY_BUFFER $25; /* Host file mode: clear internal disk and file buffers */

#define CMD10_WR_USB_DATA3 $29 ;/* Device mode (not supported by serial port): Write data block to the send buffer of USB endpoint 0 */
;/* Input: length, data stream */

#define CMD10_WR_USB_DATA5 $2A ;/* Device mode (not supported by serial port): Write data block to the send buffer of USB endpoint 1 */
;/* Input: length, data stream */

;/* ************************************************ *************************************************** ******************* */
;/* Auxiliary commands (Manual 2), less commonly used or for compatibility with CH375 and CH372, the following commands always generate an interrupt notification at the end of the operation, and always output no data */

#define CMD1H_CLR_STALL $41 ;/* Host mode: control transfer-clear endpoint error */
;/* Input: endpoint number */
;/* Output interrupt */

#define CMD1H_SET_ADDRESS $45 ;/* Host mode: control transmission-set USB address */
;/* Input: address value */
;/* Output interrupt */

#define CMD1H_GET_DESCR $46; /* Host mode: control transfer-get descriptor */
;/* Input: descriptor type */
;/* Output interrupt */

#define CMD1H_SET_CONFIG $49 ;/* Host mode: control transmission-set USB configuration */
;/* Input: configuration value */
;/* Output interrupt */

#define CMD0H_AUTO_SETUP $4D ;/* Host mode/Does not support SD card: automatically configure USB device */
;/* Output interrupt */

#define CMD2H_ISSUE_TKN_X $4E ;/* Host mode: issue synchronization token, execute transaction, this command can replace CMD10_SET_ENDP6/CMD10_SET_ENDP7 + CMD1H_ISSUE_TOKEN */
;/* Input: synchronization flag, transaction attributes */
;/* Bit 7 of the synchronization flag is the synchronization trigger bit of the host endpoint IN, bit 6 is the synchronization trigger bit of the host endpoint OUT, bits 5~bit 0 must be 0 */
;/* The lower 4 bits of the transaction attribute are the token, and the higher 4 bits are the endpoint number */
;/* Output interrupt */

#define CMD1H_ISSUE_TOKEN $4F ;/* Host mode: issue token, execute transaction, it is recommended to use CMD2H_ISSUE_TKN_X command */
;/* Input: transaction attributes */
;/* The lower 4 bits are the token, the higher 4 bits are the endpoint number */
;/* Output interrupt */

#define CMD0H_DISK_INIT $51 ;/* Host mode/SD card not supported: Initialize USB memory */
;/* Output interrupt */

#define CMD0H_DISK_RESET $52 ;/* Host mode/SD card not supported: control transfer-reset USB memory */
;/* Output interrupt */

#define CMD0H_DISK_SIZE $53 ;/* Host mode/SD card not supported: Get the capacity of USB memory */
;/* Output interrupt */

#define CMD0H_DISK_INQUIRY $58 ;/* Host mode/SD card not supported: Query USB memory characteristics */
;/* Output interrupt */

#define CMD0H_DISK_READY $59 ;/* Host mode/SD card not supported: Check USB storage readiness */
;/* Output interrupt */

#define CMD0H_DISK_R_SENSE $5A; /* Host mode/SD card not supported: Check USB memory error */
;/* Output interrupt */

#define CMD0H_RD_DISK_SEC $5B ;/* Host file mode: Read one sector of data from disk to internal buffer */
;/* Output interrupt */

#define CMD0H_WR_DISK_SEC $5C ;/* Host file mode: write one sector of data in the internal buffer to disk */
;/* Output interrupt */

#define CMD0H_DISK_MAX_LUN $5D ;/* Host mode: control transmission-get the maximum logical unit number of USB memory */
;/* Output interrupt */

;/* ************************************************ *************************************************** ******************* */
;/* The following definitions are only for compatibility with the command name format in the INCLUDE file of CH375 */

#ifndef	_NO_CH375_COMPATIBLE_
#define	CMD_GET_IC_VER		CMD01_GET_IC_VER
#define	CMD_SET_BAUDRATE	CMD21_SET_BAUDRATE
#define	CMD_ENTER_SLEEP		CMD00_ENTER_SLEEP
#define	CMD_RESET_ALL		CMD00_RESET_ALL
#define	CMD_CHECK_EXIST		CMD11_CHECK_EXIST
#define	CMD_CHK_SUSPEND		CMD20_CHK_SUSPEND
#define	CMD_SET_SDO_INT		CMD20_SET_SDO_INT
#define	CMD_GET_FILE_SIZE	CMD14_GET_FILE_SIZE
#define	CMD_SET_FILE_SIZE	CMD50_SET_FILE_SIZE
#define	CMD_SET_USB_MODE	CMD11_SET_USB_MODE
#define	CMD_GET_STATUS		CMD01_GET_STATUS
#define	CMD_UNLOCK_USB		CMD00_UNLOCK_USB
#define	CMD_RD_USB_DATA0	CMD01_RD_USB_DATA0
#define	CMD_RD_USB_DATA		CMD01_RD_USB_DATA
#define	CMD_WR_USB_DATA7	CMD10_WR_USB_DATA7
#define	CMD_WR_HOST_DATA	CMD10_WR_HOST_DATA
#define	CMD_WR_REQ_DATA		CMD01_WR_REQ_DATA
#define	CMD_WR_OFS_DATA		CMD20_WR_OFS_DATA
#define	CMD_SET_FILE_NAME	CMD10_SET_FILE_NAME
#define	CMD_DISK_CONNECT	CMD0H_DISK_CONNECT
#define	CMD_DISK_MOUNT		CMD0H_DISK_MOUNT
#define	CMD_FILE_OPEN		CMD0H_FILE_OPEN
#define	CMD_FILE_ENUM_GO	CMD0H_FILE_ENUM_GO
#define	CMD_FILE_CREATE		CMD0H_FILE_CREATE
#define	CMD_FILE_ERASE		CMD0H_FILE_ERASE
#define	CMD_FILE_CLOSE		CMD1H_FILE_CLOSE
#define	CMD_DIR_INFO_READ	CMD1H_DIR_INFO_READ
#define	CMD_DIR_INFO_SAVE	CMD0H_DIR_INFO_SAVE
#define	CMD_BYTE_LOCATE		CMD4H_BYTE_LOCATE
#define	CMD_BYTE_READ		CMD2H_BYTE_READ
#define	CMD_BYTE_RD_GO		CMD0H_BYTE_RD_GO
#define	CMD_BYTE_WRITE		CMD2H_BYTE_WRITE
#define	CMD_BYTE_WR_GO		CMD0H_BYTE_WR_GO
#define	CMD_DISK_CAPACITY	CMD0H_DISK_CAPACITY
#define	CMD_DISK_QUERY		CMD0H_DISK_QUERY
#define	CMD_DIR_CREATE		CMD0H_DIR_CREATE
#define	CMD_SEC_LOCATE		CMD4H_SEC_LOCATE
#define	CMD_SEC_READ		CMD1H_SEC_READ
#define	CMD_SEC_WRITE		CMD1H_SEC_WRITE
#define	CMD_DISK_BOC_CMD	CMD0H_DISK_BOC_CMD
#define	CMD_DISK_READ		CMD5H_DISK_READ
#define	CMD_DISK_RD_GO		CMD0H_DISK_RD_GO
#define	CMD_DISK_WRITE		CMD5H_DISK_WRITE
#define	CMD_DISK_WR_GO		CMD0H_DISK_WR_GO
#define	CMD_SET_USB_SPEED	CMD10_SET_USB_SPEED
#define	CMD_GET_DEV_RATE	CMD11_GET_DEV_RATE
#define	CMD_GET_TOGGLE		CMD11_GET_TOGGLE
#define	CMD_READ_VAR8		CMD11_READ_VAR8
#define	CMD_SET_RETRY		CMD20_SET_RETRY
#define	CMD_WRITE_VAR8		CMD20_WRITE_VAR8
#define	CMD_READ_VAR32		CMD14_READ_VAR32
#define	CMD_WRITE_VAR32		CMD50_WRITE_VAR32
#define	CMD_DELAY_100US		CMD01_DELAY_100US
#define	CMD_SET_USB_ID		CMD40_SET_USB_ID
#define	CMD_SET_USB_ADDR	CMD10_SET_USB_ADDR
#define	CMD_TEST_CONNECT	CMD01_TEST_CONNECT
#define	CMD_ABORT_NAK		CMD00_ABORT_NAK
#define	CMD_SET_ENDP2		CMD10_SET_ENDP2
#define	CMD_SET_ENDP3		CMD10_SET_ENDP3
#define	CMD_SET_ENDP4		CMD10_SET_ENDP4
#define	CMD_SET_ENDP5		CMD10_SET_ENDP5
#define	CMD_SET_ENDP6		CMD10_SET_ENDP6
#define	CMD_SET_ENDP7		CMD10_SET_ENDP7
#define	CMD_DIRTY_BUFFER	CMD00_DIRTY_BUFFER
#define	CMD_WR_USB_DATA3	CMD10_WR_USB_DATA3
#define	CMD_WR_USB_DATA5	CMD10_WR_USB_DATA5
#define	CMD_CLR_STALL		CMD1H_CLR_STALL
#define	CMD_SET_ADDRESS		CMD1H_SET_ADDRESS
#define	CMD_GET_DESCR		CMD1H_GET_DESCR
#define	CMD_SET_CONFIG		CMD1H_SET_CONFIG
#define	CMD_AUTO_SETUP		CMD0H_AUTO_SETUP
#define	CMD_ISSUE_TKN_X		CMD2H_ISSUE_TKN_X
#define	CMD_ISSUE_TOKEN		CMD1H_ISSUE_TOKEN
#define	CMD_DISK_INIT		CMD0H_DISK_INIT
#define	CMD_DISK_RESET		CMD0H_DISK_RESET
#define	CMD_DISK_SIZE		CMD0H_DISK_SIZE
#define	CMD_DISK_INQUIRY	CMD0H_DISK_INQUIRY
#define	CMD_DISK_READY		CMD0H_DISK_READY
#define	CMD_DISK_R_SENSE	CMD0H_DISK_R_SENSE
#define	CMD_RD_DISK_SEC		CMD0H_RD_DISK_SEC
#define	CMD_WR_DISK_SEC		CMD0H_WR_DISK_SEC
#define	CMD_DISK_MAX_LUN	CMD0H_DISK_MAX_LUN
#endif

;/* ********************************************************************************************************************* */
;/* Parallel port mode, bit definition of status port (read command port) */
#ifndef PARA_STATE_INTB
#define PARA_STATE_INTB $80 ;/* Bit 7 of parallel port status port: interrupt flag, low effective */
#define PARA_STATE_BUSY $10 ;/* Bit 4 of parallel port status port: busy flag, high valid */
#endif

;/* ************************************************ *************************************************** ******************* */
;/* Serial port mode, boot synchronization code before operation command */
#ifndef SER_CMD_TIMEOUT
#define SER_CMD_TIMEOUT 32 ;/* Serial port command timeout, unit is mS. The interval between synchronization codes and between synchronization codes and command codes should be as short as possible. The processing method after timeout is to discard */
#define SER_SYNC_CODE1 $57 ;/* The first serial port synchronization code to start the operation */
#define SER_SYNC_CODE2 $AB ;/* The second serial port synchronization code to start the operation */
#endif

;/* ************************************************ *************************************************** ******************* */
;/* Operation status */

#ifndef CMD_RET_SUCCESS
#define CMD_RET_SUCCESS $51 ;/* Command operation successful */
#define CMD_RET_ABORT $5F ;/* Command operation failed */
#endif

;/* ************************************************ *************************************************** ******************* */
;/* USB interrupt status */

#ifndef USB_INT_EP0_SETUP

;/* The following status codes are special event interrupts. If the USB bus hang check is enabled through CMD20_CHK_SUSPEND, the interrupt status of USB bus hang and sleep wake-up must be processed */
#define USB_INT_USB_SUSPEND $05; /* USB bus suspend event */
#define USB_INT_WAKE_UP $06; /* Wake up from sleep event */

;/* The following status code $H is used in USB device mode */
;/* In built-in firmware mode, only need to process: USB_INT_EP1_OUT, USB_INT_EP1_IN, USB_INT_EP2_OUT, USB_INT_EP2_IN */
;/* Bit 7-bit 4 is 0000 */
;/* Bit 3-Bit 2 indicates the current transaction, 00=OUT, 10=IN, 11=SETUP */
;/* Bit 1-Bit 0 indicates the current endpoint, 00=endpoint 0, 01=endpoint 1, 10=endpoint 2, 11=USB bus reset */
#define USB_INT_EP0_SETUP $0C ;/* SETUP of USB endpoint 0 */
#define USB_INT_EP0_OUT $00 ;/* OUT of USB endpoint 0 */
#define USB_INT_EP0_IN $08 ;/* IN of USB endpoint 0 */
#define USB_INT_EP1_OUT $01 ;/* OUT of USB endpoint 1 */
#define USB_INT_EP1_IN $09 ;/* IN of USB endpoint 1 */
#define USB_INT_EP2_OUT $02 ;/* OUT of USB endpoint 2 */
#define USB_INT_EP2_IN $0A ;/* IN of USB endpoint 2 */
;/* USB_INT_BUS_RESET $000$X11B */ /* USB bus reset */
#define USB_INT_BUS_RESET1 $03 ;/* USB bus reset */
#define USB_INT_BUS_RESET2 $07 ;/* USB bus reset */
#define USB_INT_BUS_RESET3 $0B ;/* USB bus reset */
#define USB_INT_BUS_RESET4 $0F ;/* USB bus reset */

#endif

;/* The following status codes 2XH-3XH are used for communication failure codes in USB host mode */
;/* Bit 7-bit 6 is 00 */
;/* Bit 5 is 1 */
;/* Bit 4 indicates whether the currently received data packet is synchronized */
;/* Bit 3-Bit 0 indicates the response of the USB device when communication fails: 0010=ACK, 1010=NAK, 1110=STALL, 0011=DATA0, 1011=DATA1, XX00=Timeout */
;/* USB_INT_RET_ACK $001X0010B */ /* Error: Return ACK for IN transaction */
;/* USB_INT_RET_NAK $001X1010B */ /* Error: Return NAK */
;/* USB_INT_RET_STALL $001X1110B */ /* Error: Return to STALL */
;/* USB_INT_RET_DATA0 $001X0011B */ /* Error: DATA0 returned for OUT/SETUP transaction */
;/* USB_INT_RET_DATA1 $001X1011B */ /* Error: Return DATA1 for OUT/SETUP transaction */
;/* USB_INT_RET_TOUT $001XXX00B */ /* Error: Return timeout */
;/* USB_INT_RET_TOGX $001$011B */ /* Error: Data returned for IN transaction is out of sync */
;/* USB_INT_RET_PID $001XXXXXB */ /* Error: Undefined */

;/* The following status code 1XH is used for USB host mode operation status code */
#ifndef USB_INT_SUCCESS
#define USB_INT_SUCCESS $14 ;/* USB transaction or transfer operation successful */
#define USB_INT_CONNECT $15 ;/* USB device connection event detected, may be a new connection or reconnection after disconnection */
#define USB_INT_DISCONNECT $16 ;/* USB device disconnect event detected */
#define USB_INT_BUF_OVER $17 ;/* The data transmitted by USB is incorrect or the data is too much and the buffer overflows */
#define USB_INT_USB_READY $18 ;/* USB device has been initialized (USB address has been assigned) */
#define USB_INT_DISK_READ $1D ;/* USB memory request data read */
#define USB_INT_DISK_WRITE $1E ;/* USB memory request data writing */
#define USB_INT_DISK_ERR $1F ;/* USB memory operation failed */
#endif

;/* The following status codes are used for file system error codes in host file mode */
#ifndef ERR_DISK_DISCON
#define ERR_DISK_DISCON $82 ;/* The disk is not connected yet, maybe the disk has been disconnected */
#define ERR_LARGE_SECTOR $84 ;/* The disk sector is too large and only supports 512 bytes per sector */
#define ERR_TYPE_ERROR $92 ;/* The disk partition type is not supported, only FAT12/FAT16/BigDOS/FAT32 is supported, and needs to be repartitioned by the disk management tool */
#define ERR_BPB_ERROR $A1 ;/* The disk has not been formatted, or the parameters are wrong, and it needs to be reformatted by WINDOWS using the default parameters */
#define ERR_DISK_FULL $B1 ;/* The disk file is too full, the remaining space is too little or no more, disk defragmentation is required */
#define ERR_FDT_OVER $B2 ;/* There are too many files in the directory (folder) and there are no free directory entries. The number of files in the FAT12/FAT16 root directory should be less than 512 and disk defragmentation is required */
#define ERR_FILE_CLOSE $B4 ;/* The file has been closed. If you need to use it, you should reopen the file */
#define ERR_OPEN_DIR $41 ;/* The directory (folder) with the specified path is opened */
#define ERR_MISS_FILE $42 ;/* The file in the specified path was not found, maybe the file name is wrong */
#define ERR_FOUND_NAME $43 ;/* A matching file name was searched, or a directory (folder) was requested to be opened but the file was actually opened */
;/* The following file system error codes are used for file system subroutines */
#define ERR_MISS_DIR $B3 ;/* A certain subdirectory (folder) of the specified path was not found, maybe the directory name is wrong */
#define ERR_LONG_BUF_OVER $48 ;/* Long file buffer overflow */
#define ERR_LONG_NAME_ERR $49 ;/* The short file name does not have a corresponding long file name or the long file name is wrong */
#define ERR_NAME_EXIST $4A ;/* The short file with the same name already exists, it is recommended to regenerate another short file name */
#endif

;/************************************************* *************************************************** ******************* */
;/* The following status codes are used for disk and file status in host file mode, VAR_DISK_STATUS */
#ifndef DEF_DISK_UNKNOWN
#define DEF_DISK_UNKNOWN $00 ;/* Not initialized yet, unknown status */
#define DEF_DISK_DISCONN $01 ;/* The disk is not connected or has been disconnected */
#define DEF_DISK_CONNECT $02 ;/* The disk is connected, but it has not been initialized or the disk cannot be recognized */
#define DEF_DISK_MOUNTED $03 ;/* The disk has been initialized successfully, but the file system has not been analyzed or the file system does not support it */
#define DEF_DISK_READY $10 ;/* The file system of the disk has been analyzed and can support */
#define DEF_DISK_OPEN_ROOT $12 ;/* The root directory has been opened and must be closed after use. Note that the FAT12/FAT16 root directory has a fixed length */
#define DEF_DISK_OPEN_DIR $13 ;/* Subdirectory (folder) has been opened */
#define DEF_DISK_OPEN_FILE $14 ;/* The file has been opened */
#endif

;/************************************************* *************************************************** ******************* */
;/* Common definitions of file systems */

#ifndef DEF_SECTOR_SIZE
#define DEF_SECTOR_SIZE 512 ;/* The default physical sector size of U disk or SD card */
#endif

#ifndef DEF_WILDCARD_CHAR
#define DEF_WILDCARD_CHAR $2A ;/* Wildcard character '*' for path name */
#define DEF_SEPAR_CHAR1 $5C ;/* Path name separator '\' */
#define DEF_SEPAR_CHAR2 $2F ;/* Path name separator '/' */
#define DEF_FILE_YEAR 2004 ;/* Default file date: 2004 */
#define DEF_FILE_MONTH 1; /*Default file date: January */
#define DEF_FILE_DATE 1 ;/* Default file date: 1st */
#endif

#ifndef ATTR_DIRECTORY

;/* File directory information in the FAT data area */
;typedef struct _FAT_DIR_INFO {
;UINT8 DIR_Name[11]; ;/* 00H, file name, 11 bytes in total, fill in the blanks */
;UINT8 DIR_Attr; /* 0BH, file attribute, refer to the following instructions */
;UINT8 DIR_NTRes; /* 0CH */
;UINT8 DIR_CrtTimeTenth; /* 0DH, file creation time, counted in 0.1 second units */
;UINT16 DIR_CrtTime; /* 0EH, file creation time */
;UINT16 DIR_CrtDate; /* 10H, date of file creation */
;UINT16 DIR_LstAccDate; /* 12H, the date of the latest access operation */
;UINT16 DIR_FstClusHI; /* 14H */
;UINT16 DIR_WrtTime; /* 16H, file modification time, refer to the previous macro MAKE_FILE_TIME */
;UINT16 DIR_WrtDate; /* 18H, file modification date, refer to the previous macro MAKE_FILE_DATE */
;UINT16 DIR_FstClusLO; /* 1AH */
;UINT32 DIR_FileSize; /* 1CH, file length */
;} FAT_DIR_INFO, *P_FAT_DIR_INFO; /* 20H */

;/* File properties */
#define ATTR_READ_ONLY $01 ;/* The file is read-only */
#define ATTR_HIDDEN $02 ;/* The file is a hidden attribute */
#define ATTR_SYSTEM $04 ;/* The file is a system attribute */
#define ATTR_VOLUME_ID $08 ;/* volume label */
#define ATTR_DIRECTORY $10 ;/* Subdirectory (folder) */
#define ATTR_ARCHIVE $20 ;/* File is archive attribute */
#define ATTR_LONG_NAME ( ATTR_READ_ONLY | ATTR_HIDDEN | ATTR_SYSTEM | ATTR_VOLUME_ID ) ;/* Long file name attribute */
#define ATTR_LONG_NAME_MASK ( ATTR_LONG_NAME | ATTR_DIRECTORY | ATTR_ARCHIVE )
;/* File attributes UINT8 */
;/* bit0 bit1 bit2 bit3 bit4 bit5 bit6 bit7 */
;/* Only the hidden system volume directory is stored undefined */
;/* Read the Tibetan system standard file */
;/* File time UINT16 */
;/* Time = (Hour<<11) + (Minute<<5) + (Second>>1) */
#define MAKE_FILE_TIME( h, m, s ) ( (h<<11) + (m<<5) + (s>>1) ); /* Generate file time data of specified hours, minutes and seconds */
;;/* File date UINT16 */
;/* Date = ((Year-1980)<<9) + (Month<<5) + Day */
#define MAKE_FILE_DATE( y, m, d ) ( ((y-1980)<<9) + (m<<5) + d ) ;/* Generate file date data of the specified year, month and day */

#define LONE_NAME_MAX_CHAR (255*2) ;/* Maximum number of characters/bytes in a long file name */
#define LONG_NAME_PER_DIR (13*2) ;/* The number of characters/bytes of the long file name in each file directory information structure */

#endif

;/************************************************* *************************************************** ******************* */
;/* SCSI command and data input and output structures */

#ifndef SPC_CMD_INQUIRY

;/* SCSI command code */
#define SPC_CMD_INQUIRY $12
#define SPC_CMD_READ_CAPACITY $25
#define SPC_CMD_READ10 $28
#define SPC_CMD_WRITE10 $2A
#define SPC_CMD_TEST_READY $00
#define SPC_CMD_REQUEST_SENSE $03
#define SPC_CMD_MODESENSE6 $1A
#define SPC_CMD_MODESENSE10 $5A
#define SPC_CMD_START_STOP $1B

;/*Command block of BulkOnly protocol */
;typedef struct _BULK_ONLY_CBW {
;UINT32 CBW_Sig;
;UINT32 CBW_Tag;
;UINT8 CBW_DataLen0; /* 08H, input: data transmission length, the valid value for input data is 0 to 48, and the valid value for output data is 0 to 33 */
;UINT8 CBW_DataLen1;
;UINT16 CBW_DataLen2;
;UINT8 CBW_Flag; /* 0CH, input: transmission direction and other flags, if bit 7 is 1, data is input, if bit 0 is 0, data is output or no data */
;UINT8 CBW_LUN;
;UINT8 CBW_CB_Len; /* 0EH, input: the length of the command block, valid values are 1 to 16 */
;UINT8 CBW_CB_Buf[16]; /* 0FH, input: command block, the buffer is up to 16 bytes */
;} BULK_ONLY_CBW, *P_BULK_ONLY_CBW; /*Command block of BulkOnly protocol, input CBW structure */

;/* Return data of INQUIRY command */
;typedef struct _INQUIRY_DATA {
;UINT8 DeviceType; /* 00H, device type */
;UINT8 RemovableMedia; /* 01H, bit 7 is 1, indicating that it is removable storage */
;UINT8 Versions; /* 02H, protocol version */
;UINT8 DataFormatAndEtc; /* 03H, specify the return data format */
;UINT8 AdditionalLength; /* 04H, length of subsequent data */
;UINT8 Reserved1;
;UINT8 Reserved2;
;UINT8 MiscFlag; /* 07H, some control flags */
;UINT8 VendorIdStr[8]; /* 08H, vendor information */
;UINT8 ProductIdStr[16]; /* 10H, product information */
;UINT8 ProductRevStr[4]; /* 20H, product version */
;} INQUIRY_DATA, *P_INQUIRY_DATA; /* 24H */

;/* Return data of REQUEST SENSE command */
;typedef struct _SENSE_DATA {
;UINT8 ErrorCode; /* 00H, error code and valid bits */
;UINT8 SegmentNumber;
;UINT8 SenseKeyAndEtc; /* 02H, primary key code */
;UINT8 Information0;
;UINT8 Information1;
;UINT8 Information2;
;UINT8 Information3;
;UINT8 AdditSenseLen; /* 07H, length of subsequent data */
;UINT8 CmdSpecInfo[4];
;UINT8 AdditSenseCode; /* 0CH, additional key code */
;UINT8 AddSenCodeQual; /* 0DH, detailed additional key code */
;UINT8 FieldReplaUnit;
;UINT8 SenseKeySpec[3];
;} SENSE_DATA, *P_SENSE_DATA; /* 12H */

#endif

;/************************************************* *************************************************** ******************* */
;/* Data input and output structures in host file mode */

;#ifndef MAX_FILE_NAME_LEN

;#define MAX_FILE_NAME_LEN (13+1) /* The maximum length of the file name, the maximum length is 1 root directory character + 8 main file names + 1 decimal point + 3 type names + terminator = 14 */

;/* Input data and output data of command */
;typedef union _CH376_CMD_DATA {
;struct {
;UINT8 mBuffer[MAX_FILE_NAME_LEN];
;} Default;

;INQUIRY_DATA DiskMountInq; /* Return: return data of INQUIRY command */
;/* CMD0H_DISK_MOUNT: Initialize the disk and test whether the disk is ready, when executing for the first time */

;FAT_DIR_INFO OpenDirInfo; /* Return: enumerated file directory information */
;/* CMD0H_FILE_OPEN: Enumerate files and directories (folders) */

;FAT_DIR_INFO EnumDirInfo; /* Return: enumerated file directory information */
;/* CMD0H_FILE_ENUM_GO: Continue to enumerate files and directories (folders) */

;struct {
;UINT8 mUpdateFileSz; /* Input parameters: whether to allow the file length to be updated, 0 disables the length update */
;} FileCLose; /* CMD1H_FILE_CLOSE: Close the currently open file */

;struct {
;UINT8 mDirInfoIndex; /* Input parameters: Specify the index number in the sector of the directory information structure to be read, 0FFH is the currently opened file */
;} DirInfoRead; /* CMD1H_DIR_INFO_READ: Read the directory information of the file */

;union {
;UINT32 mByteOffset; /* Input parameters: offset bytes, offset in bytes (total length 32 bits, low byte first) */
;UINT32 mSectorLba; /* Returns: the absolute linear sector number corresponding to the current file pointer, 0FFFFFFFFH has reached the end of the file (total length 32 bits, low byte first) */
;} ByteLocate; /* CMD4H_BYTE_LOCATE: Move the current file pointer in bytes */

;struct {
;UINT16 mByteCount; /* Input parameters: The number of bytes requested to be read (total length 16 bits, low byte first) */
;} ByteRead; /* CMD2H_BYTE_READ: Read the data block from the current position in bytes */

;struct {
;UINT16 mByteCount; /* Input parameters: The number of bytes requested to be written (total length 16 bits, low byte first) */
;} ByteWrite; /* CMD2H_BYTE_WRITE: Write the data block to the current location in bytes */

;union {
;UINT32 mSectorOffset; /* Input parameters: Offset sector number, offset in sectors (total length 32 bits, low byte first) */
;UINT32 mSectorLba; /* Returns: the absolute linear sector number corresponding to the current file pointer, 0FFFFFFFFH has reached the end of the file (total length 32 bits, low byte first) */
;} SectorLocate; /* CMD4H_SEC_LOCATE: Move the current file pointer in units of sectors */

;struct {
;UINT8 mSectorCount; /* Input parameter: Number of sectors requested to be read */
;/* Return: Number of sectors allowed to be read */
;UINT8 mReserved1;
;UINT8 mReserved2;
;UINT8 mReserved3;
;UINT32 mStartSector; /* Returns: The starting absolute linear sector number of the sector block allowed to be read (total length 32 bits, low byte first) */
;} SectorRead; /* CMD1H_SEC_READ: Read the data block from the current position in units of sectors */

;struct {
;UINT8 mSectorCount; /* Input parameter: Number of sectors requested to be written */
;/* Return: Number of sectors allowed to be written */
;UINT8 mReserved1;
;UINT8 mReserved2;
;UINT8 mReserved3;
;UINT32 mStartSector; /* Returns: The starting absolute linear sector number of the sector block that is allowed to be written (total length 32 bits, low byte first) */
;} SectorWrite; /* CMD1H_SEC_WRITE: Write the data block at the current position in units of sectors */

;struct {
;UINT32 mDiskSizeSec; /* Return: the total number of sectors of the entire physical disk (total length 32 bits, low byte first) */
;} DiskCapacity; /* CMD0H_DISK_CAPACITY: Query the physical capacity of the disk */

;struct {
;UINT32 mTotalSector; /* Return: the total number of sectors of the current logical disk (total length 32 bits, low byte first) */
;UINT32 mFreeSector; /* Returns: The number of remaining sectors of the current logical disk (total length 32 bits, low byte first) */
;UINT8 mDiskFat; /* Return: FAT type of current logical disk, 1-FAT12, 2-FAT16, 3-FAT32 */
;} DiskQuery; /* CMD_DiskQuery, query disk information */

;BULK_ONLY_CBW DiskBocCbw; /* Input parameters: CBW command structure */
;/* CMD0H_DISK_BOC_CMD: Execute BulkOnly transfer protocol command on USB memory */

;struct {
;UINT8 mMaxLogicUnit; /* Returns: the maximum logical unit number of the USB memory */
;} DiskMaxLun; /* CMD0H_DISK_MAX_LUN: Control transfer-get the maximum logical unit number of USB memory */

;INQUIRY_DATA DiskInitInq; /* Return: return data of INQUIRY command */
;/* CMD0H_DISK_INIT: Initialize USB memory */

;INQUIRY_DATA DiskInqData; /* Return: return data of INQUIRY command */
;/* CMD0H_DISK_INQUIRY: Query USB memory characteristics */

;SENSE_DATA ReqSenseData; /* Return: return data of REQUEST SENSE command */
;/* CMD0H_DISK_R_SENSE: Check USB memory errors */

;struct {
;UINT32 mDiskSizeSec; /* Return: the total number of sectors of the entire physical disk (total length 32 bits, high byte first) */
;} DiskSize; /* CMD0H_DISK_SIZE: Get the capacity of USB memory */

;struct {
;UINT32 mStartSector; /* Input parameter: LBA sector address (total length 32 bits, low byte first) */
;UINT8 mSectorCount; /* Input parameter: Number of sectors requested to be read */
;} DiskRead; /* CMD5H_DISK_READ: Read data blocks (in sectors) from USB memory */

;struct {
;UINT32 mStartSector; /* Input parameter: LBA sector address (total length 32 bits, low byte first) */
;UINT8 mSectorCount; /* Input parameter: Number of sectors requested to be written */
;} DiskWrite; /* CMD5H_DISK_WRITE: Write data blocks (in sectors) to USB memory */
;} CH376_CMD_DATA, *P_CH376_CMD_DATA;

;#endif

;/* ************************************************ *************************************************** ******************* */
;/* The address of the file system variable in host file mode */

#ifndef VAR_FILE_SIZE

;/* 8-bit/single-byte variable */
#define VAR_SYS_BASE_INFO $20 ;/* Basic information of the current system */
;/* Bit 6 is used to indicate the SubClass-Code of the USB storage device. Bit 6 is 0, which means the subclass is 6, and bit 6 is 1, which means the subclass is a value other than 6 */
;/* Bit 5 is used to indicate the USB configuration status in USB device mode and the USB device connection status in USB host mode */
;/* In USB device mode, if bit 5 is 1, the USB configuration is completed, if bit 5 is 0, it has not been configured yet */
;/* In USB host mode, if bit 5 is 1, there is a USB device in the USB port, and if bit 5 is 0, there is no USB device in the USB port */
;/* Bit 4 is used to indicate the buffer lock status in USB device mode. Bit 4 is 1, which means the USB buffer is in the locked state, and bit 6 is 1, which means it has been released */
;/* Other bits, reserved, do not modify */
#define VAR_RETRY_TIMES $25 ;/* Number of retries for USB transaction operations */
;/* If bit 7 is 0, there will be no retry when NAK is received. If bit 7 is 1 and bit 6 is 0, it will retry infinitely when NAK is received (you can use the CMD_ABORT_NAK command to give up the retry). If bit 7 is 1, bit 6 is 1. Retry for up to 3 seconds when receiving NAK*/
;/* Bit 5 ~ Bit 0 are the number of retries after timeout */
#define VAR_FILE_BIT_FLAG $26 ;/* Bit flag in host file mode */
;/* Bit 1 and 0, FAT file system flag of the logical disk, 00-FAT12, 01-FAT16, 10-FAT32, 11-illegal */
;/* Bit 2, flag of whether the FAT table data in the current buffer has been modified, 0-not modified, 1-modified */
;/* Bit 3, the file length needs to be modified flag, the current file is appended with data, 0-not appended and does not need to be modified, 1-has been appended and needs to be modified */
;/* Other bits, reserved, do not modify */
#define VAR_DISK_STATUS $2B ;/* Disk and file status in host file mode */
#define VAR_SD_BIT_FLAG $30 ;/* Bit flag of SD card in host file mode */
;/* Bit 0, SD card version, 0-only supports SD first version, 1-supports SD second version */
;/* Bit 1, automatic identification, 0-SD card, 1-MMC card */
;/* Bit 2, automatic recognition, 0-standard capacity SD card, 1-high capacity SD card (HC-SD) */
;/* Bit 4, ACMD41 command timeout */
;/* Bit 5, CMD1 command timeout */
;/* Bit 6, CMD58 command timeout */
;/* Other bits, reserved, do not modify */
#define VAR_UDISK_TOGGLE $31 ;/* Synchronization flag of BULK-IN/BULK-OUT endpoint of USB storage device */
;/* Bit 7, synchronization flag of Bulk-In endpoint */
;/* Bit 6, synchronization flag of Bulk-In endpoint */
;/* Bit 5~Bit 0, must be 0 */
#define VAR_UDISK_LUN $34 ;/* Logical unit number of USB storage device */
;/* Bit 7~Bit 4, the current logical unit number of the USB storage device. After CH376 initializes the USB storage device, the default is to access the 0# logical unit */
;/* Bit 3~Bit 0, the maximum logical unit number of the USB storage device, plus 1 equals the number of logical units */
#define VAR_SEC_PER_CLUS $38 ;/* Number of sectors per cluster of logical disk */
#define VAR_FILE_DIR_INDEX $3B ;/* The index number of the current file directory information in the sector */
#define VAR_CLUS_SEC_OFS $3C ;/* The sector offset of the current file pointer within the cluster. If it is $FF, it points to the end of the file and the end of the cluster */

;/* 32-bit/4-byte variable */
#define VAR_DISK_ROOT $44 ;/* For FAT16 disk, it is the number of sectors occupied by the root directory, and for FAT32 disk, it is the starting cluster number of the root directory (total length 32 bits, low byte first) */
#define VAR_DSK_TOTAL_CLUS $48 ;/* The total number of clusters in the logical disk (total length 32 bits, low byte first) */
#define VAR_DSK_START_LBA $4C ;/* The starting absolute sector number LBA of the logical disk (total length 32 bits, low byte first) */
#define VAR_DSK_DAT_START $50 ;/* Starting LBA of the data area of the logical disk (total length 32 bits, low byte first) */
#define VAR_LBA_BUFFER $54 ;/* LBA corresponding to the data in the current disk data buffer (total length 32 bits, low byte first) */
#define VAR_LBA_CURRENT $58 ;/* Current disk starting LBA address for reading and writing (total length 32 bits, low byte first) */
#define VAR_FAT_DIR_LBA $5C ;/* Sector LBA address where the current file directory information is located (total length 32 bits, low byte first) */
#define VAR_START_CLUSTER $60 ;/* The starting cluster number of the current file or directory (folder) (total length 32 bits, low byte first) */
#define VAR_CURRENT_CLUUST $64 ;/* The current cluster number of the current file (total length 32 bits, low byte first) */
#define VAR_FILE_SIZE $68 ;/* The length of the current file (total length 32 bits, low byte first) */
#define VAR_CURRENT_OFFSET $6C ;/* Current file pointer, byte offset of current read and write position (total length 32 bits, low byte first) */

#endif

;/* ************************************************ *************************************************** ******************* */
;/* Common USB definitions */

;/* USB packet identification PID, which may be used in host mode */
#ifndef DEF_USB_PID_SETUP
#define DEF_USB_PID_NULL $00 ;/* Reserved PID, undefined */
#define DEF_USB_PID_SOF $05
#define DEF_USB_PID_SETUP $0D
#define DEF_USB_PID_IN $09
#define DEF_USB_PID_OUT $01
#define DEF_USB_PID_ACK $02
#define DEF_USB_PID_NAK $0A
#define DEF_USB_PID_STALL $0E
#define DEF_USB_PID_DATA0 $03
#define DEF_USB_PID_DATA1 $0B
#define DEF_USB_PID_PRE $0C
#endif

;/* USB request type, may be used in external firmware mode */
#ifndef DEF_USB_REQ_TYPE
#define DEF_USB_REQ_READ $80 ;/* Control read operation */
#define DEF_USB_REQ_WRITE $00 ;/* Control write operation */
#define DEF_USB_REQ_TYPE $60 ;/* Control request type */
#define DEF_USB_REQ_STAND $00; /* Standard request */
#define DEF_USB_REQ_CLASS $20 ;;/* Device class request */
#define DEF_USB_REQ_VENDOR $40 ;/* Vendor request */
#define DEF_USB_REQ_RESERVE $60 ;/* Reserve request */
#endif

;/* USB standard device request, bit 6 and bit 5 of RequestType=00 (Standard), may be used in external firmware mode */
#ifndef DEF_USB_GET_DESCR
#define DEF_USB_CLR_FEATURE $01
#define DEF_USB_SET_FEATURE $03
#define DEF_USB_GET_STATUS $00
#define DEF_USB_SET_ADDRESS $05
#define DEF_USB_GET_DESCR $06
#define DEF_USB_SET_DESCR $07
#define DEF_USB_GET_CONFIG $08
#define DEF_USB_SET_CONFIG $09
#define DEF_USB_GET_INTERF $0A
#define DEF_USB_SET_INTERF $0B
#define DEF_USB_SYNC_FRAME $0C
#endif

;/* ************************************************ *************************************************** ******************* */

;;#ifdef __cplusplus
;}
;#endif

;#endif