                ; i8080 assembler code
                .project microbios.rom
                .tape rk-bin


                ORIGIN          equ $E000               ; Начало ПЗУ

                ;###############
                ;## КОНСТАНТЫ ##
                ;###############

                KR580VG75       equ $C000
                KR580VT57       equ $E000
                KR580VV55_KBD   equ $8000
                KR580VV55_PPI   equ $A000
                KR580VI53       equ $B000

                COLS		equ 78
                ROWS		equ 30
                DMA_SIZE        equ COLS * ROWS - 1

                ; Конфигурация PEACE.OS

                SCREEN          equ $76D0
                VARS            equ SCREEN - $D0
                STACK           equ SCREEN - 1
                ROM             equ $F800
                RAM             equ $0000

                MARGIN_TOP      equ 3
                MARGIN_LEFT     equ 8

                ; Светодиод РУС/ЛАТ
                LED_MASK        equ 00001000b

                ; Видеорежим
                DISPLAY_78X30   equ (ROWS - 1) << 8 | (COLS - 1)
                GRAPH_78X30     equ $9399
                DMA_SIZE_78X30  equ 78 * 30 - 1
                CG              equ $84
                
                ; Монитор
                CURSOR_ADDR     equ $7600
                CURSOR_XY       equ $7602

                
                ;################
                ;## ПЕРЕМЕННЫЕ ##
                ;################

                ; Системные переменные
                SCREEN_ADDR     equ VARS               
                CARRIAGE        equ SCREEN_ADDR         + 2
                CARRIAGE_ADDR   equ CARRIAGE            + 2
                MEMORY_HEAP     equ CARRIAGE_ADDR       + 2
                TMP             equ MEMORY_HEAP         + 2

                org ORIGIN

                JMP START
                
                ; Константы
                dw KR580VG75
                dw KR580VT57
                dw KR580VV55_KBD
                dw KR580VV55_PPI
                dw KR580VI53
                dw $0000
                dw $0000
                dw $0000
                dw $0000
                dw $0000

                ; Вектора подпрограмм
                JMP VG75_INIT           ; Инициализация видеоконтроллера
                JMP CARRIAGE_SET        ; Установка каретки
		JMP PRINT_TEXT          ; Вывад текста
		JMP SCROLL_UP           ; Скролл на одну строку вверх


        SET_DISPLAY: ; Настройка ВГ75
        
                ; DE - строки и столбцы
                ; BC - размер знакоместа и курсор

		; Останов КР580ВГ75
		LXI H, KR580VG75 + 1


		; Скрываем курсор

        	MVI M, $80
		DCR L
		MVI M, $FF
		MVI M, $FF

		; Останов КР580ВГ75

		INR L
		MVI M, 0
		DCR L

        	MOV M, E	; Столбцы
        	MOV M, D	; Строки
        	MOV M, C	; Позиция курсора и кол-во линий в знакоместе
        	MOV M, B	; 0101.0011
        	
 		; Старт КР580ВГ75

		INR L
        	MVI M, $25

		RET

		
        SET_DMA: ; Настройка ПДП

                ; DE - адрес экрана
                ; BC - размер экрана
                
                LXI H, KR580VT57 + 8
		MVI M, $80
		MVI L, $04
		MOV M, E
		MOV M, D

		INR L
		MOV M, C
		MOV M, B
		MVI L, 8
		MVI M, $A4

                RET


        VG75_READY: ; Ждем обратный ход

                LXI H, KR580VG75 + 1
		MOV A, M
		MOV A, M \\ ANI $20 \\ JZ . - 3
                RET            


        VG75_INIT: ; Инициализация видеоконтроллера

                ; Очистка экрана
                LXI D, $0000
                LXI B, COLS * ROWS / 2
                CALL FILL_SCREEN

                ; Настройка знакогенератора
                LXI D, DISPLAY_78X30
                LXI B, GRAPH_78X30
                CALL SET_DISPLAY
                
                CALL VG75_READY
                
                ; Настройка ПДП
                LHLD SCREEN_ADDR
                XCHG
		MVI C, DMA_SIZE_78X30 & $00FF
		MVI B, ($4000 + DMA_SIZE_78X30) >> 8
                CALL SET_DMA

                LXI H, $0000
                CALL CARRIAGE_SET

                RET            


        FILL_SCREEN: ; Очистка экрана стеком
        
                ; DE - чем заполнять
                ; BC - сколько заполнять / 2
                
                ; Сохраняем стек                
                LXI H, $0000
                DAD SP
                
                LXI SP, SCREEN + COLS * ROWS
                
                PUSH D \\ DCX B
                MOV A, C \\ ORA B \\ JNZ . - 4

                SPHL

                RET


        VI53_RST: ; Сброс ВИ53
        
                LXI H, KR580VI53 + 3
                MVI M, $36 \\ MVI M, $66 \\ MVI M, $B6
                RET
                
                
        LED_ON:

                LXI H, KR580VV55_KBD + 2
                MOV A, M
                ORI LED_MASK
                MOV M, A
                RET
                
                
        LED_OFF:

                LXI H, KR580VV55_KBD  + 2
                MOV A, M
                CMA
                ANI LED_MASK
                MOV M, A
                RET

                
        BEEP:
        
                MOV A, B
                
                BEEP_LOP:
                
                EI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                DI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                
                DCR A \\ JNZ BEEP_LOP

                RET


                ;##################################
                ;## Печать с управляющими кодами ##
                ;##################################


        PRINT_TEXT:

                ; HL - адрес строки

                XCHG

                PRINT_TEXT_LOOP:

		LDAX D

                ; Конец строки ($00)
		CODE_00:
		ANA A \\ RZ

                ; Табуляция ($09)
		CODE_09:
		CPI $09 \\ JNZ CODE_0A
		LHLD CARRIAGE
		MOV A, L
		ORI 00000111B
		INR A
		MOV L, A
		SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Перевод строки ($0A)
		CODE_0A:
		CPI $0A \\ JNZ CODE_0D
		LHLD CARRIAGE \\ INR H \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Возврат каретки ($0D)
		CODE_0D:
		CPI $0D \\ JNZ CODE_80
		LHLD CARRIAGE \\ MVI L, $00 \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Вывод в HEX ($80)
		CODE_80:
		CPI $80 \\ JNZ PRINT_TEXT_CHAR
		INX D \\ LDAX D
		LHLD CARRIAGE_ADDR \\ PUSH D \\ CALL PRINT_HEX \\ POP D
		LHLD CARRIAGE \\ INR L \\ INR L \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP		
		
                PRINT_TEXT_CHAR:

		LHLD CARRIAGE_ADDR
		
		; Экранирование спецсимволов
		ADI 0 \\ JP . + 5 \\ MVI A, '.'

		; Вывод символа
		MOV M, A
		
        	LXI H, CARRIAGE \\ INR M

		PRINT_TEXT_SKIP:
		
		LHLD CARRIAGE
		PUSH D \\ CALL CARRIAGE_SET \\ POP D
		INX D

		JMP PRINT_TEXT_LOOP
		
		RET


        PRINT_HEX:
        
                ; А - ччисло
                ; HL - адрес
                
		XCHG
		LXI H, tableHEX
		MVI B, $00
		
                PUSH PSW
                
                ANI 11110000b
                RLC \\ RLC \\ RLC \\ RLC
                MOV C, A
                DAD B
                MOV A, M
		XCHG
                MOV M, A
                
		XCHG
		LXI H, tableHEX
		
		POP PSW
                PUSH PSW

                ANI 00001111b
                MOV C, A
                DAD B
                MOV A, M
		XCHG
		INX H
                MOV M, A
                
		POP PSW

                RET


        PRINT_HEX_WORD:
        
                ; HL - слово
                
                SHLD TMP \\ LHLD CARRIAGE_ADDR
                PUSH H \\ LDA TMP + 1 \\ CALL PRINT_HEX
                POP H \\ INR L \\ INR L \\ LDA TMP \\ CALL PRINT_HEX
                LXI H, TXT_EOL \\ CALL PRINT_TEXT

                RET


        PRINT_BIN:
        
                ; А - ччисло
                ; HL - адрес
                
                MVI C, $08
                
                PRINT_BIN_LOOP:
                
                RAL
                JC . + 5 \\ MVI M, '0'
                JNC . + 5 \\ MVI M, '1'
                
                INX H \\ DCR C
                
                JNZ PRINT_BIN_LOOP

                RET


                ;####################№##
                ;## Установка каретки ##
                ;#######################
                

        CARRIAGE_SET:
        
                ; H - строка
                ; L - позиция в строке
                
                MOV A, L \\ CPI 64 \\ JC . + 06 \\ ANI 00111111B \\ INR H \\ MOV L, A
                MOV A, H \\ ANI 00011111B \\ MOV H, A

                SHLD CARRIAGE

                XCHG
                LHLD SCREEN_ADDR \\ LXI B, COLS * MARGIN_TOP + MARGIN_LEFT \\ DAD B
                LXI B, COLS
                
                XRA A
                
                GET_ADDR_LOOP:
                
                CMP D \\ JZ GET_ADDR_SKIP
                DAD B \\ DCR D
                JMP GET_ADDR_LOOP
                
                GET_ADDR_SKIP:
                
                MOV C, E \\ DAD B
                
                SHLD CARRIAGE_ADDR

                RET


        SCROLL_UP:
        
                ; Скролл на одну строку вверх
                
                LXI H, $0000
                DAD SP
                SHLD TMP
                
                LXI SP, SCREEN + 7 * COLS + MARGIN_LEFT
                LXI H, SCREEN + 6 * COLS + MARGIN_LEFT

 		MVI C, $16
 		
                SCROLL_STRING:
        
                MVI A, $20

 		POP D
 		MOV M, E \\ INX H
 		MOV M, D \\ INX H
 		DCR A
 		JNZ . - 6
 		
 		LXI D, $000E
                DAD D
 		XCHG
                LXI H, $000E
 		DAD SP
                SPHL
 		XCHG
 		
 		DCR C
 		JNZ SCROLL_STRING

 		LHLD TMP
                SPHL

                RET


        TEST_RAM_4K:
        
                ; HL - начало памяти для проверки
                ; DE - где рисовать прогрессбар

                TEST_RAM_4K_LOOP:

                MOV A, M \\ MVI M, $00 \\ MOV M, A
                CMP M \\ JNZ TEST_RAM_4K_ERROR

                INX H
                
                XRA A
                CMP L
                JNZ TEST_RAM_4K_LOOP
                
                MOV A, H
                ANI 00001111B
                ANA A
                JNZ TEST_RAM_4K_LOOP

                ; Проверяем клавишу АР2                
                LDA KR580VV55_KBD + 1
                CPI 11111011b
                JZ TEST_RAM_4K_NO_ERROR

                MVI A, $7F \\ STAX D \\ INX D

                MVI A, $80
                CMP H
                JNZ TEST_RAM_4K_LOOP
                
                TEST_RAM_4K_NO_ERROR:

                ORA A \\ RET
                
                TEST_RAM_4K_ERROR:

                STC
                
                RET


        START: 

                ;###########################
                ;## Сброс и инициализация ##
                ;###########################
        
                ; Устанавливаем стек
                LXI SP, STACK

                ; Сброс генератора звука
                CALL VI53_RST

                ; Настройка порта клавиатуры        
                LXI H, KR580VV55_KBD + 3
                MVI M, $8A ; КР580ВВ55: A - вывод, B - ввод
                
                ; Настройка ППА
                MVI A, $90 \\ STA KR580VV55_PPI + 3
                MVI A, $80 \\ STA KR580VV55_PPI + 2

                ; Включаем светодиод РУС/ЛАТ
                CALL LED_ON            

                ; Гасим экран
                MVI A, $20 \\ STA KR580VG75 + 1
                MVI A, $80 \\ STA KR580VT57

                ; Звуковой сигнал
                LXI B, $8040 \\ CALL BEEP

                ; Выключаем светодиод РУС/ЛАТ
                CALL LED_OFF             

                ; Устанавливакем адрес экрана
                LXI H, SCREEN \\ SHLD SCREEN_ADDR

                ; Настройка видеоконтроллера
                CALL VG75_INIT

                ; Вывод названия системы
                LXI H, TXT_TITLE \\ CALL PRINT_TEXT
                
                ; Проверка ОЗУ
                LHLD CARRIAGE_ADDR
                LXI D, $0000 - COLS * 2 + 20
                DAD D \\ XCHG \\ LXI H, RAM
                
                CALL TEST_RAM_4K
                
                ; Ошибка памяти?
                JC ERROR

                RAM_OK:
                
                LXI H, TXT_TABLE \\ CALL PRINT_TEXT
                LXI H, TXT_KEYS \\ CALL PRINT_TEXT

                
        MENU:
		; Опрос клавиавтуры
                LXI H, KR580VV55_KBD
                MVI M, 11111101b
                
                LDA KR580VV55_KBD + 1
                CPI 11111011b \\ JZ ROM
                
		; Опрос клавиавтуры
                LXI H, KR580VV55_KBD
                MVI M, 11111110b
                
                ; Проверяем клавиши F1 и F2
                LDA KR580VV55_KBD + 1
                CPI 11110111b \\ JZ PPI_FF
                CPI 11101111b \\ JZ PPI_7FFF
                CPI 11011111b \\ JZ PPI_DEBUG
                CPI 10111111b \\ JZ DUMP_HEX
                CPI 01111111b \\ JZ AT28C64
                
                JMP MENU


        PPI_FF:

                ; Попытка загрузиться с SD-привода
                LXI H, $8000 \\ SHLD KR580VV55_PPI + 1
                LXI H, KR580VV55_PPI + 1
                LXI D, RAM
                MOV C, E

                LOOP_PPI_LOAD_FF:
        
                MOV M, C
                LDA KR580VV55_PPI
                STAX D
                INX D
                INR C
                XRA A
                CMP C
                JNZ LOOP_PPI_LOAD_FF
                
                ; Выставляем адрес курсора в координатах Монитора
                LHLD CARRIAGE_ADDR
                LXI D, $0000 - MARGIN_LEFT
                DAD D
                SHLD CURSOR_ADDR
                
                RST 0

                
        PPI_7FFF:
                
                LXI H, TXT_PPI \\ CALL PRINT_TEXT
        
                ; Загрузка через ППА 32к памяти от носа до хвоста
        
                LXI H, $0000
                LXI D, RAM

                LOOP_PPI_LOAD_7FFF:
        
                SHLD KR580VV55_PPI + 1
                LDA KR580VV55_PPI
                STAX D
                INX H
                INX D
                
                ; Задержка для совместимости с WI-FI девайсом
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP

                MVI A, $80
                CMP D
                JNZ LOOP_PPI_LOAD_7FFF 
                
                LXI H, $8000
                SHLD KR580VV55_PPI + 1

                RST 0

                
        PPI_DEBUG:
                
                LXI H, TXT_PPI_DEBUG \\ CALL PRINT_TEXT
                
                ; Установка режима ввода
                MVI A, 10011011B
                STA KR580VV55_PPI + 3

                ; Чтение и вывод ППА
                LOOP_PPI_DEBUG:

                LXI H, RAM \\ SHLD TMP
                LDA KR580VV55_PPI \\ CALL PPI_DEBUG_STR
                LDA KR580VV55_PPI + 1 \\ CALL PPI_DEBUG_STR
                LDA KR580VV55_PPI + 2 \\ CALL PPI_DEBUG_STR
                
                MVI A, $60 \\ STA KR580VG75 + 1
                LDA KR580VG75 \\ CALL PRINT_HEX
                
                INX H \\ MVI M, 'X' \\ INX H
                LDA KR580VG75 \\ CALL PRINT_HEX

                INX H \\ MVI M, $00

                LXI H, $1800 \\ CALL CARRIAGE_SET
                LXI H, RAM \\ CALL PRINT_TEXT
                
                CALL SCROLL_UP

                JMP LOOP_PPI_DEBUG

                
        PPI_DEBUG_STR:
        
                PUSH PSW
                LHLD TMP \\ CALL PRINT_HEX
                MVI C, $20 \\ INX H \\ MOV M, C \\ INX H \\ MOV M, C
                POP PSW
                
                CALL PRINT_BIN
                
                MVI C, $20 \\ MVI A, $04
                MOV M, C \\ INX H \\ DCR A \\ JNZ . - 4
                
                SHLD TMP
                
                RET

                
        DUMP_HEX:

                
                LXI H, KR580VV55_KBD
                MVI M, 01111111B

                LXI H, $1800 \\ CALL CARRIAGE_SET
                LXI D, $0000

                ; Чтение и вывод ППА
                LOOP_DUMP_HEX:
                
                LHLD CARRIAGE_ADDR

                PUSH D
                PUSH D
                MOV A, D \\ CALL PRINT_HEX \\ INX H
                POP D
                MOV A, E \\ CALL PRINT_HEX 
                POP D
                INX H \\ INX H
                
                MVI A, $10

                LOOP_DUMP_HEX_STR:
                
                PUSH PSW

                LDAX D
                PUSH D \\ CALL PRINT_HEX \\ POP D
                INX H \\ INX H \\ INX D
                
                POP PSW 
                DCR A \\ JNZ LOOP_DUMP_HEX_STR

                PUSH D \\ CALL SCROLL_UP \\ POP D
                
                MOV A, D \\ ANI 01111111B \\ MOV D, A

                DUMP_HEX_PAUSE:

                LDA KR580VV55_KBD + 1
                CPI 01111111b
                JZ DUMP_HEX_PAUSE

                JMP LOOP_DUMP_HEX


        AT28C64:

                ; Вывод текста
                MVI A, $04
                
                AT28C64_SCROLL:
                
                PUSH PSW 
                CALL SCROLL_UP
                LHLD CARRIAGE \\ DCR H
                CALL CARRIAGE_SET
                POP PSW
                
                DCR A \\ JNZ AT28C64_SCROLL
                
                LXI H, TXT_AT28C64 \\ CALL PRINT_TEXT
                
		; Опрос клавиавтуры
                LXI H, KR580VV55_KBD
                MVI M, 11111101b
                
                WAIT_ENTER:

		; Ждём ВК
                LDA KR580VV55_KBD + 1
                CPI 11111011b
                JNZ WAIT_ENTER
                
                CALL SCROLL_UP

                PPI_OUT:

                LXI H, $0000
                LXI D, RAM

                LOOP_PPI_OUT:
                
                LDAX D
                STA KR580VV55_PPI
                SHLD KR580VV55_PPI + 1
                
                PUSH H \\ PUSH D \\ PUSH PSW
                
                MOV A, L
                ANI $0F
                MOV C, A \\ ADD C \\ ADD C
                MOV L, A
                MVI H, $18
                
                ANA A
                JNZ SKIP_SCROLL
                
                ; Скролл вверх и вывод адреса
                PUSH H
                XCHG \\ SHLD TMP + 2
                CALL SCROLL_UP
                LXI H, $1800 \\ CALL CARRIAGE_SET
                LDA TMP + 3 \\ CALL PRINT_HEX
                INX H \\ LDA TMP + 2 \\ CALL PRINT_HEX
                POP H
                
                SKIP_SCROLL:
                
                MVI A, $06 \\ ADD L \\ MOV L, A
                CALL CARRIAGE_SET
                
                POP PSW \\ CALL PRINT_HEX

                POP D \\ POP H
                INX H \\ INX D

                MVI A, $FF \\ STA KR580VV55_PPI + 2

                ; Задержка для формирования сигнала записи
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP

                MVI A, $20 \\ CMP D \\ JNZ LOOP_PPI_OUT 

                ; Вывод текста
                CALL SCROLL_UP
                CALL SCROLL_UP
                LXI H, $1800 \\ CALL CARRIAGE_SET
                LXI H, TXT_NOTES \\ CALL PRINT_TEXT
                
                JMP . + 0

                
        ERROR:
                
                ; Ошибка
                LXI H, TXT_ERROR \\ CALL PRINT_TEXT

                ; Звуковой сигнал
                LXI B, $8020 \\ CALL BEEP
                
                ERROR_AP2:
                
                ; Проверяем клавишу АР2                
                LDA KR580VV55_KBD + 1
                CPI 11111011b
                JZ ROM

                JMP ERROR_AP2
                

tableHEX        db '0123456789ABCDEF'

TXT_TITLE:      db $20, $04, $11, "radio-86rk kompx`ter radiol`bitelq", $0A, $0D
                db $04, $17, $11, "bsww <mir> wersiq 3.15 2024 - 2026", $0A, $0A, $0D
                db "prowerka ozu (ar2): ******** 0000 - 7FFF", $0A, $0A, $0D, $00

TXT_TABLE:      
                db $80, KR580VT57 >> 8, $80, KR580VT57 & $FF, $09, ": kr580wt57", $09, $09
                db $80, ORIGIN >> 8, $80, ORIGIN & $FF, $09, ": pzu", $0A, $0D
                
                db $80, KR580VG75 >> 8, $80, KR580VG75 & $FF, $09, ": kr580wg75", $09, $09
                db $80, SCREEN >> 8, $80, SCREEN & $FF, $09, ": |kran (78*30)", $0A, $0D
                
                db $80, KR580VI53 >> 8, $80, KR580VI53 & $FF, $09, ": kr580wi53", $09, $09
                db $80, VARS >> 8, $80, VARS & $FF, $09, ": peremennye", $0A, $0D
                
                db $80, KR580VV55_PPI >> 8, $80, KR580VV55_PPI & $FF, $09, ": kr580ww55 (ppa)", $09
                db $80, STACK >> 8, $80, STACK & $FF, $09, ": stek", $0A, $0D

                db $80, KR580VV55_KBD >> 8, $80, KR580VV55_KBD & $FF, $09, ": kr580ww55 (k)", $09, $09
                db "80/", $80, CG, $09, ": perekl`~enie zg", $0A, $0A, $0D

                db $00
                
TXT_KEYS:       db "wk - wyhod w monitor", $0A, $0A, $0D
                db "F1 - zagruzka ~erez ppa 16 bajt (R,FF)", $0A, $0D
                db "F2 - zagruzka ~erez ppa 32 kbajt (R,7FFF)", $0A, $0D
                db "F3 - otladka ppa i swetowogo pera", $0A, $0D
                db "F4 - damp HEX 0000-7FFF", $0A, $0D
                db "F5 - programmator (ppa) 8k", $0A, $0A, $0D
                db $00

TXT_PPI:        db "zagruzka ~erez ppa 32k...", $00
TXT_PPI_DEBUG:  db "otladka ppa:", $0A, $0A, $0D, $00

TXT_AT28C64:    db "podkl`~enie AT28C64 k ppa:", $0A, $0A, $0D
                db "PA0 - PA7 ....... D0 - D7", $0A, $0D
                db "PB0 - PB7 ....... A0 - A7", $0A, $0D
                db "PC0 - PC4 ....... A8 - A12", $0A, $0D
                db "PC7 ............. /CE", $0A, $0D
                db "/OE ............. +5V", $0A, $0D
                db "/WE ............. GND", $0A, $0A, $0D
                db "navmite <wk> dlq starta pro{iwki 8kb...", $0A, $0A, $0D, $00

TXT_NOTES:      db "pro{iwka zawer{ena!", $0A, $0D
                db "izwlekite mikroshemu iz programmatora", $0A, $0D, $00

TXT_ERROR:      db $0A, "obru{enie sistemy", $0A, $0D, "my ne znaem ~to delatx", $0A, $0D, "ar2 - wyhod w monitor", $0A, $0D, $00

TXT_SPC:        db $20, $00
TXT_TAB:        db $09, $00
TXT_EOL:        db $0A, $0D, $00

                db $00
                
                