                ; https://svofski.github.io/pretty-8080-assembler/
                ; i8080 assembler code
                .project bios-m.rom
                .tape rk-bin


                ORIGIN          equ $E000               ; Начало ПЗУ

                ;###############
                ;## КОНСТАНТЫ ##
                ;###############

                KR580VG75       equ $C000
                KR580VT57       equ $E000
                KR580VV55_KBD   equ $8000
                KR580VV55_PPI   equ $A000
                KR580VI53       equ $B000

                COLS		equ 78
                ROWS		equ 30
                DMA_SIZE        equ COLS * ROWS - 1

                ; Конфигурация PEACE.OS

                SCREEN          equ $76D0
                VARS            equ SCREEN - $D0
                STACK           equ SCREEN - 1
                ROM             equ $F800
                RAM             equ $0000

                MARGIN_TOP      equ 3
                MARGIN_LEFT     equ 8

                ; Светодиод РУС/ЛАТ
                LED_MASK        equ 00001000b

                ; Видеорежим
                DISPLAY_78X30   equ (ROWS - 1) << 8 | (COLS - 1)
                GRAPH_78X30     equ $9399
                DMA_SIZE_78X30  equ 78 * 30 - 1
                CG              equ $84

                
                ;################
                ;## ПЕРЕМЕННЫЕ ##
                ;################

                ; Системные переменные
                CARRIAGE_ADDR   equ VARS
                CARRIAGE        equ CARRIAGE_ADDR       + 2
                COMMAND         equ CARRIAGE            + 2
                KEYS            equ COMMAND             + 2
                TMP             equ KEYS                + 1

                org ORIGIN

                JMP START
                
                ; Константы
                dw KR580VG75
                dw KR580VT57
                dw KR580VV55_KBD
                dw KR580VV55_PPI
                dw KR580VI53
                dw $0000
                dw $0000
                dw $0000
                dw $0000
                dw $0000

                ; Вектора подпрограмм
                JMP VG75_INIT           ; Инициализация видеоконтроллера
                JMP CARRIAGE_SET        ; Установка каретки
		JMP PRINT_TEXT          ; Вывад текста
		JMP SCROLL_UP           ; Скролл на одну строку вверх
		JMP LED_ON              ; Включить светодиод РУС/ЛАТ
		JMP LED_OFF             ; Выключить светодиод РУС/ЛАТ
		JMP BEEP                ; Звуковой сигнал
		JMP BEEP_DECAY          ; Звуковой сигнал с падением частоты
		JMP FILL_SCREEN         ; Очистка экрана стеком
		JMP PRINT_HEX           ; Печать шестнадцатиричного значения байта
		JMP PRINT_DEC           ; Печать десятичного числа в переменную TMP
		JMP LEAD_ZERO           ; Удаление ведущих нулей десятичного числа в переменной TMP
		JMP EXEC                ; Командный процессор


        SET_DISPLAY: ; Настройка ВГ75
        
                ; DE - строки и столбцы
                ; BC - размер знакоместа и курсор

		; Останов КР580ВГ75
		LXI H, KR580VG75 + 1


		; Скрываем курсор

        	MVI M, $80
		DCR L
		MVI M, $FF
		MVI M, $FF

		; Останов КР580ВГ75

		INR L
		MVI M, 0
		DCR L

        	MOV M, E	; Столбцы
        	MOV M, D	; Строки
        	MOV M, C	; Позиция курсора и кол-во линий в знакоместе
        	MOV M, B	; 0101.0011
        	
 		; Старт КР580ВГ75

		INR L
        	MVI M, $25

		RET

		
        SET_DMA: ; Настройка ПДП

                ; DE - адрес экрана
                ; BC - размер экрана
                
                LXI H, KR580VT57 + 8
		MVI M, $80
		MVI L, $04
		MOV M, E
		MOV M, D

		INR L
		MOV M, C
		MOV M, B
		MVI L, 8
		MVI M, $A4

                RET


        VG75_READY: ; Ждем обратный ход

                LXI H, KR580VG75 + 1
		MOV A, M
		MOV A, M \\ ANI $20 \\ JZ . - 3
                RET            


        VG75_INIT: ; Инициализация видеоконтроллера

                ; Очистка экрана
                LXI D, $0000
                LXI B, COLS * ROWS / 2
                CALL FILL_SCREEN

                ; Настройка знакогенератора
                LXI D, DISPLAY_78X30
                LXI B, GRAPH_78X30
                CALL SET_DISPLAY
                
                CALL VG75_READY
                
                ; Настройка ПДП
                LXI H, SCREEN
                XCHG
		MVI C, DMA_SIZE_78X30 & $00FF
		MVI B, ($4000 + DMA_SIZE_78X30) >> 8
                CALL SET_DMA

                LXI H, $0000
                CALL CARRIAGE_SET

                RET            


        FILL_SCREEN: ; Очистка экрана стеком
        
                ; DE - чем заполнять
                ; BC - сколько заполнять / 2
                
                ; Сохраняем стек                
                LXI H, $0000
                DAD SP
                
                LXI SP, SCREEN + COLS * ROWS
                
                PUSH D \\ DCX B
                MOV A, C \\ ORA B \\ JNZ . - 4

                SPHL

                RET


        VI53_RST: ; Сброс ВИ53
        
                LXI H, KR580VI53 + 3
                MVI M, $36 \\ MVI M, $66 \\ MVI M, $B6
                RET
                
                
        LED_ON:

                LXI H, KR580VV55_KBD + 2
                MOV A, M
                ORI LED_MASK
                MOV M, A
                RET
                
                
        LED_OFF:

                LXI H, KR580VV55_KBD  + 2
                MOV A, M
                CMA
                ANI LED_MASK
                MOV M, A
                RET

                
        BEEP:
        
                MOV A, B
                
                BEEP_LOP:
                
                EI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                DI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                
                DCR A \\ JNZ BEEP_LOP

                RET


        BEEP_DECAY:

                EI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                DI \\ MOV B, C \\ DCR B \\ JNZ . - 1
                
                INR C \\ INR C \\ INR C \\ INR C
                
                DCR A \\ JNZ BEEP_DECAY

                RET


                ;##################################
                ;## Печать с управляющими кодами ##
                ;##################################


        PRINT_TEXT:

                ; HL - адрес строки

                XCHG

                PRINT_TEXT_LOOP:

		LDAX D

                ; Конец строки ($00)
		CODE_00:
		ANA A \\ RZ

                ; Табуляция ($09)
		CODE_09:
		CPI $09 \\ JNZ CODE_0A
		LHLD CARRIAGE
		MOV A, L
		ORI 00000111B
		INR A
		MOV L, A
		SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Перевод строки ($0A)
		CODE_0A:
		CPI $0A \\ JNZ CODE_0D
		LHLD CARRIAGE \\ INR H \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Возврат каретки ($0D)
		CODE_0D:
		CPI $0D \\ JNZ CODE_80
		LHLD CARRIAGE \\ MVI L, $00 \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP

                ; Вывод в HEX ($80)
		CODE_80:
		CPI $80 \\ JNZ CODE_81
		INX D \\ LDAX D
		LHLD CARRIAGE_ADDR \\ PUSH D \\ CALL PRINT_HEX \\ POP D
		LHLD CARRIAGE \\ INR L \\ INR L \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP		

                ; Вывод в DEC ($81)
		CODE_81:
		CPI $81 \\ JNZ PRINT_TEXT_CHAR
		XCHG \\ INX H
		MOV E, M \\ INX H \\ MOV D, M
		XCHG
		PUSH D
		CALL PRINT_DEC
		CALL LEAD_ZERO
		LHLD CARRIAGE_ADDR
		LXI D, TMP
		LXI B, $0000
		
		DEC_BEGIN: 
		
		LDAX D
		ANA A \\ JZ DEC_END
		MOV M, A
		INX D \\ INX H \\ INX B
		JMP DEC_BEGIN
		
		DEC_END:
		
		POP D

		LHLD CARRIAGE \\ DAD B \\ SHLD CARRIAGE
		JMP PRINT_TEXT_SKIP		
		
                PRINT_TEXT_CHAR:

		LHLD CARRIAGE_ADDR
		
		; Экранирование спецсимволов
		ADI 0 \\ JP . + 5 \\ MVI A, '.'

		; Вывод символа
		MOV M, A
		
        	LXI H, CARRIAGE \\ INR M

		PRINT_TEXT_SKIP:
		
		LHLD CARRIAGE
		PUSH D \\ CALL CARRIAGE_SET \\ POP D
		INX D

		JMP PRINT_TEXT_LOOP
		
		RET


        PRINT_HEX:
        
                ; А - ччисло
                ; HL - адрес
                
		XCHG
		LXI H, tableHEX
		MVI B, $00
		
                PUSH PSW
                
                ANI 11110000b
                RLC \\ RLC \\ RLC \\ RLC
                MOV C, A
                DAD B
                MOV A, M
		XCHG
                MOV M, A
                
		XCHG
		LXI H, tableHEX
		
		POP PSW
                PUSH PSW

                ANI 00001111b
                MOV C, A
                DAD B
                MOV A, M
		XCHG
		INX H
                MOV M, A
                
		POP PSW

                RET


        PRINT_BIN:
        
                ; А - ччисло
                ; HL - адрес
                
                MVI C, $08
                
                PRINT_BIN_LOOP:
                
                RAL
                JC . + 5 \\ MVI M, '0'
                JNC . + 5 \\ MVI M, '1'
                
                INX H \\ DCR C
                
                JNZ PRINT_BIN_LOOP

                RET


        PRINT_DEC:
        
                ; HL - число для вывода
                ; TMP - десятичное число с ведущими нулями

                LXI B, TMP
                
                MVI A, $2F
                
                PRINT_DEC_11:	
                
                XCHG
        	INR A
        	LXI H, $D8F0
        	DAD D
        	JC PRINT_DEC_11
        	
        	XCHG
        	STAX B
        	INX B
        	CALL PRINT_DEC_12
        	
                PRINT_DEC_12:
                
                CALL PRINT_DEC_13
                PRINT_DEC_13: 
                MVI A, $2F
        
                PRINT_DEC_14:	
                XCHG
        	INR	A
        	LXI	H, $FC18
        	DAD	D
        	JC	PRINT_DEC_14
        	STAX	B
        	INX	B
        	MOV	L,E
        	MOV	H,D
        	DAD	H
        	DAD	H
        	DAD	D
        	DAD	H
        	
        	RET      
                
                
        LEAD_ZERO:
        
                ; Удаляем ведущие нули у десятичного числа по адресу TMP
        
                MVI A, $30
                
                LEAD_ZERO_LOOP:
                
                LXI H, TMP \\ CMP M \\ RNZ
                
                LHLD TMP + 1 \\ SHLD TMP
                LHLD TMP + 3 \\ SHLD TMP + 2
                LHLD TMP + 5 \\ SHLD TMP + 4
                LXI H, TMP + 5 \\ MVI M, $00

                JMP LEAD_ZERO_LOOP

                RET


                ;####################№##
                ;## Установка каретки ##
                ;#######################
                

        CARRIAGE_SET:
        
                ; H - строка
                ; L - позиция в строке
                
                MOV A, L \\ CPI 64 \\ JC . + 06 \\ ANI 00111111B \\ INR H \\ MOV L, A
                MOV A, H \\ ANI 00011111B \\ MOV H, A

                SHLD CARRIAGE

                XCHG
                LXI H, SCREEN \\ LXI B, COLS * MARGIN_TOP + MARGIN_LEFT \\ DAD B
                LXI B, COLS
                
                XRA A
                
                GET_ADDR_LOOP:
                
                CMP D \\ JZ GET_ADDR_SKIP
                DAD B \\ DCR D
                JMP GET_ADDR_LOOP
                
                GET_ADDR_SKIP:
                
                MOV C, E \\ DAD B
                
                SHLD CARRIAGE_ADDR

                RET


        SCROLL_UP:
        
                ; Скролл на одну строку вверх
                
                LXI H, $0000
                DAD SP
                SHLD TMP
                
                LXI SP, SCREEN + 7 * COLS + MARGIN_LEFT
                LXI H, SCREEN + 6 * COLS + MARGIN_LEFT

 		MVI C, $16
 		
                SCROLL_STRING:
        
                MVI A, $20

 		POP D
 		MOV M, E \\ INX H
 		MOV M, D \\ INX H
 		DCR A
 		JNZ . - 6
 		
 		LXI D, COLS - 64
                DAD D
 		XCHG
                LXI H, COLS - 64
 		DAD SP
                SPHL
 		XCHG
 		
 		DCR C
 		JNZ SCROLL_STRING

 		LHLD TMP
                SPHL

                RET


        TEST_RAM_4K:
        
                ; HL - начало памяти для проверки
                ; DE - где рисовать прогрессбар

                TEST_RAM_4K_LOOP:

                MOV A, M \\ MVI M, $00 \\ MOV M, A
                CMP M \\ JNZ TEST_RAM_4K_ERROR

                INX H
                
                XRA A
                CMP L
                JNZ TEST_RAM_4K_LOOP
                
                MOV A, H
                ANI 00001111B
                ANA A
                JNZ TEST_RAM_4K_LOOP

                ; Проверяем клавишу АР2                
                LDA KR580VV55_KBD + 1
                CPI 11111011b
                JZ TEST_RAM_4K_NO_ERROR

                MVI A, $7F \\ STAX D \\ INX D

                MVI A, $80
                CMP H
                JNZ TEST_RAM_4K_LOOP
                
                TEST_RAM_4K_NO_ERROR:

                ORA A \\ RET
                
                TEST_RAM_4K_ERROR:

                STC
                
                RET


        FUNCTION_STRLEN:
        
                ; HL - строка
                ; E - длина
                
                XRA A \\ MOV E, A
                CMP M \\ RZ
                INR E \\ INX H \\ JMP . - 4

                RET

                
        FUNCTION_STRCMP:
                
                ; HL - переменная
                ; DE - константа
                ; Z - результат

		LDAX D
		CPI $00 \\ RZ
		CMP M \\ RNZ
		INX H \\ INX D
		JMP FUNCTION_STRCMP

                RET


        FUNCTION_SKIP_SPACE:
        
                ; HL - строка

                MVI A, $20
                CMP M \\ RNZ \\ INX H \\ JMP . - 3

                RET

                
        FUNCTION_DECODE_HEX:

                ; A - HEX символ
                ; B - код символа
                ; Z - результат
                
                LXI H, tableHEX
                LXI B, $0010
                
                READ_HEX_LOOP:
                
                CMP M \\ RZ
                INX H \\ INR B
                DCR C \\ JNZ READ_HEX_LOOP
                
                CPI $FF ; Сброс флага Z

                RET

                
        FUNCTION_SCANF_HEX:
        
                ; HL - строка
                ; DE, A - результат
                ; Z - результат

                XCHG \\ DCX D

                ; Ищем первый HEX символ                
                SCANF_FIND_CHAR:
                
                INX D
                LDAX D \\ CPI $00 \\ JNZ . + 6 
                CPI $FF \\ RET ; Конец строки
                
                CALL FUNCTION_DECODE_HEX
                JNZ SCANF_FIND_CHAR

                LXI H, $0000 \\ MOV L, B \\ PUSH H

                SCANF_HEX_LOOP:

                INX D \\ LDAX D \\ CALL FUNCTION_DECODE_HEX \\ JNZ SCANF_HEX_RESULT
                
                POP H \\ MVI C, $04
                
                SCANF_HEX_SHIFT_LOOP:
                
                MOV A, L \\ ORA A \\ RAL \\ MOV L, A
                MOV A, H \\ RAL \\ MOV H, A
                DCR C
                JNZ SCANF_HEX_SHIFT_LOOP
                
                MOV A, B \\ ORA L \\ MOV L, A
                PUSH H
                
                JMP SCANF_HEX_LOOP
                
                SCANF_HEX_RESULT:

                POP H \\ XCHG \\ MOV A, E \\ CMP A ; Выставляем флаг Z

                RET

                
                ;########################
                ;## Обработчики команд ##
                ;########################

                ; Команда вывода таблицы с информацией о системе
                EXEC_SYSINFO:
                
                LXI H, TXT_TABLE \\ CALL PRINT_TEXT
                RET
                

                ; Очистка экрана
                EXEC_CLR: 

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                LXI B, COLS * ROWS / 2
                CALL FILL_SCREEN
                LXI H, $0000
                CALL CARRIAGE_SET
                RET
                
                
                ; Команда скролла экрана вверх
                EXEC_FLUSH: 

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                CPI $00 \\ RZ
                
                EXEC_FLUSH_LOOP:
                
                PUSH PSW
                CALL SCROLL_UP
                POP PSW
                DCR A
                JNZ EXEC_FLUSH_LOOP
                RET
                

                ; Команда установки каретки
                EXEC_CARRIAGE: 

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                XCHG \\ CALL CARRIAGE_SET
                RET


                ; Команда вывода строки на экран
                EXEC_PRINT:
                
                XCHG \\ CALL FUNCTION_SKIP_SPACE \\ CALL PRINT_TEXT
                RET
                
                ; Перевод каретки
                EXEC_NL: 
                
                LXI H, TXT_EOL \\ CALL PRINT_TEXT
                RET

                
                ; Звуковой сигнал
                EXEC_BEEP:

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                PUSH D
                POP B
                CALL BEEP
                RET
                
                
                ; Звуковой сигнал
                EXEC_PEW:

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                PUSH D
                POP B
                CALL BEEP_DECAY
                RET
                
                
                ; Команда перехода
                EXEC_JUMP: 

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                XCHG \\ PCHL
                RET


                ; Команда чтения из памяти
                EXEC_READ:

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                LDAX D
                PUSH PSW
                LHLD CARRIAGE_ADDR
                CALL PRINT_HEX
                POP PSW
                INX H \\ INX H
                CALL PRINT_BIN
                JMP EXEC_NL
                
                
                ; Команда записи в ячейку памяти
                EXEC_WRITE:
                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                PUSH D
                CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                POP H
                MOV M, A
                
                                
                RET
                
                
                ; Команда чтения восьми клавиш
                EXEC_KEYS:
                
		; Опрос ПРОБЕЛ
                LXI H, KR580VV55_KBD
                MVI M, 01111111b
                
                INX H \\ MOV A, M
                CMA \\ ANI 10000000B \\ RLC \\ MOV B, A

		; Опрос ВК
                DCX H \\ MVI M, 11111101b
                
                INX H \\ MOV A, M
                CMA \\ ANI 00000100B \\ RRC \\ ORA B \\ MOV B, A
                
                ; Опрос клавиш F1 - F5 и АР2
                DCX H \\ MVI M, 11111110b
                
                INX H \\ MOV A, M
                CMA \\ ANI 11111100B \\ ORA B \\ MOV B, A
                
                XRA A
                CMP B
                JZ EXEC_KEYS
                
                MOV A, B \\ STA KEYS
                DCX H \\ MVI M, $FF
                

                RET
                
                
                ; Выполнение команды при условии нажатия клавиши
                EXEC_KEY:

                XCHG \\ CALL FUNCTION_SCANF_HEX \\ JNZ EXEC_ERROR
                LDA KEYS \\ CMP E \\ RNZ
                CALL FUNCTION_SKIP_SPACE \\ SHLD COMMAND
                JMP EXEC


        EXEC_PPI_FF:

                ; Попытка загрузиться с SD-привода
                LXI H, $8000 \\ SHLD KR580VV55_PPI + 1
                LXI H, KR580VV55_PPI + 1
                LXI D, RAM
                MOV C, E

                LOOP_PPI_LOAD_FF:
        
                MOV M, C
                LDA KR580VV55_PPI
                STAX D
                INX D
                INR C
                XRA A
                CMP C
                JNZ LOOP_PPI_LOAD_FF
                
                ; Выставляем адрес курсора в координатах Монитора
                LHLD CARRIAGE_ADDR
                LXI D, $0000 - MARGIN_LEFT
                DAD D
                SHLD CARRIAGE_ADDR
                XRA A \\ STA KEYS - 1
                
                RST 0

                
        EXEC_PPI_7FFF:
                
                LXI H, TXT_PPI \\ CALL PRINT_TEXT
        
                ; Загрузка через ППА 32к памяти от носа до хвоста
        
                LXI H, $0000
                LXI D, RAM

                LOOP_PPI_LOAD_7FFF:
        
                SHLD KR580VV55_PPI + 1
                LDA KR580VV55_PPI
                STAX D
                INX H
                INX D
                
                ; Задержка для совместимости с WI-FI девайсом
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP
                NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP \\ NOP

                MVI A, $80
                CMP D
                JNZ LOOP_PPI_LOAD_7FFF 
                
                LXI H, $8000
                SHLD KR580VV55_PPI + 1

                RST 0

                
                ; Команда записи в ячейку памяти
                EXEC_ERROR:
                
                LXI H, TXT_EXEC_EROOR
                CALL PRINT_TEXT
                
                RET


        EXEC:

                ;#########################
                ;## Командный процессор ##
                ;#########################
                
                LXI D, T_EXEC_COMMANDS
                
                PARSE_CMD:
                
                LHLD COMMAND
                
                CALL FUNCTION_STRCMP \\ JNZ NEXT_CMD
                
                PUSH H
                XCHG
                INX H \\ MOV E, M
                INX H \\ MOV D, M
                XCHG
                POP D
                
                LXI B, EXEC_END
                PUSH B
                PCHL
                
                NEXT_CMD:
                
                ; Поиск конца строки команды
                INX D \\ LDAX D \\ CPI $00 \\ JNZ . - 4
                INX D \\ INX D \\ INX D
                LDAX D \\ CPI $FF \\ JNZ PARSE_CMD
                
                ; Очищаем командную строку, скролл, курсор
                EXEC_END:


                RET


        START: 

                ;###########################
                ;## Сброс и инициализация ##
                ;###########################
        
                ; Устанавливаем стек
                LXI SP, STACK

                ; Сброс генератора звука
                CALL VI53_RST

                ; Настройка порта клавиатуры        
                LXI H, KR580VV55_KBD + 3
                MVI M, $8A ; КР580ВВ55: A - вывод, B - ввод
                
                ; Настройка ППА
                MVI A, $90 \\ STA KR580VV55_PPI + 3
                MVI A, $80 \\ STA KR580VV55_PPI + 2

                ; Включаем светодиод РУС/ЛАТ
                CALL LED_ON            

                ; Гасим экран
                MVI A, $20 \\ STA KR580VG75 + 1
                MVI A, $80 \\ STA KR580VT57

                ; Звуковой сигнал
                LXI B, $8040 \\ CALL BEEP

                ; Выключаем светодиод РУС/ЛАТ
                CALL LED_OFF             

                ; Настройка видеоконтроллера
                CALL VG75_INIT

                ; Вывод названия системы
                LXI H, TXT_TITLE \\ CALL PRINT_TEXT
                
                ; Проверка ОЗУ
                LHLD CARRIAGE_ADDR
                LXI D, $0000 - COLS * 2 + 20
                DAD D \\ XCHG \\ LXI H, RAM
                
                CALL TEST_RAM_4K
                
                ; Ошибка памяти?
                JC ERROR

                RAM_OK:
                
                ;Исполняем командный скрипт
                LXI H, AUTOEXEC
                
                LOOP_AUTOEXEC:
                
                ; Проверяем конец файла
                MVI A, $FF \\ CMP M \\ JZ STOP
                
                ; Пропускаем все специальные символы
                MOV A, M \\ CPI $20 \\ INX H \\ JC . - 4 \\ DCX H

                PUSH H
                SHLD COMMAND
                CALL EXEC
                POP H

                ; Пропускаем все не специальные символы
                MOV A, M \\ CPI $20 \\ INX H \\ JNC . - 4

                JMP LOOP_AUTOEXEC
                
        STOP:

                JMP STOP

        ERROR:
                
                ; Ошибка
                LXI H, TXT_ERROR \\ CALL PRINT_TEXT

                ; Звуковой сигнал
                LXI B, $8020 \\ CALL BEEP
                
                ERROR_AP2:
                
                ; Проверяем клавишу АР2                
                LDA KR580VV55_KBD + 1
                CPI 11111011b
                JZ ROM

                JMP ERROR_AP2
                

tableHEX        db '0123456789ABCDEF'

TXT_TITLE:      db $20, $04, $11, "radio-86rk kompx`ter radiol`bitelq", $0A, $0D
                db $04, $17, $11, "bsww <mir> wersiq 3.15/M 2024 - 2026", $0A, $0A, $0D
                db "prowerka ozu (ar2): ******** 0000 - 7FFF", $0A, $0A, $0D, $00

TXT_TABLE:      
                db $80, KR580VT57 >> 8, $80, KR580VT57 & $FF, $09, ": kr580wt57", $09, $09
                db $80, ORIGIN >> 8, $80, ORIGIN & $FF, $09, ": pzu", $0A, $0D
                
                db $80, KR580VG75 >> 8, $80, KR580VG75 & $FF, $09, ": kr580wg75", $09, $09
                db $80, SCREEN >> 8, $80, SCREEN & $FF, $09, ": |kran (", $81, COLS, $00, "*", $81, ROWS, $00,")", $0A, $0D
                
                db $80, KR580VI53 >> 8, $80, KR580VI53 & $FF, $09, ": kr580wi53", $09, $09
                db $80, VARS >> 8, $80, VARS & $FF, $09, ": peremennye", $0A, $0D
                
                db $80, KR580VV55_PPI >> 8, $80, KR580VV55_PPI & $FF, $09, ": kr580ww55 (ppa)", $09
                db $80, STACK >> 8, $80, STACK & $FF, $09, ": stek", $0A, $0D

                db $80, KR580VV55_KBD >> 8, $80, KR580VV55_KBD & $FF, $09, ": kr580ww55 (k)", $09, $09
                db "80/", $80, CG, $09, ": wybor {rifta", $0A, $0A, $0D

                db $00

TXT_PPI:        db "zagruzka ~erez ppa 32k...", $00

TXT_ERROR:      db $0A, "obru{enie sistemy", $0A, $0D, "ar2 - wyhod w monitor", $00
TXT_EXEC_EROOR  db "* o{ibka interpretatora *", $0A, $0D, $00

TXT_SPC:        db $20, $00
TXT_TAB:        db $09, $00
TXT_EOL:        db $0A, $0D, $00

TIMEOUT:        db $05

T_EXEC_COMMANDS:

                db "SYSINFO", $00       \\      dw EXEC_SYSINFO         ; Вывод таблицы системных настроек
                
                db "CLR", $00           \\      dw EXEC_CLR             ; CLR ABCD - очистка экрана, ABCD - чем заполнять
                db "FLUSH", $00         \\      dw EXEC_FLUSH           ; FLUSH N - скролл текста вверх на N строк
                db "CARRIAGE", $00      \\      dw EXEC_CARRIAGE        ; CARRIAGE YYXX - установка каретки
                db "PRINT", $00         \\      dw EXEC_PRINT           ; PRINT XXXXX... - печать строки в текущую позицию экрана
                db "NL", $00            \\      dw EXEC_NL              ; Перевод каретки

                db "BEEP", $00          \\      dw EXEC_BEEP            ; BEEP NNMM - звуковой сигнал, где NN - длительность, MM - тон
                db "PEW", $00           \\      dw EXEC_PEW             ; PEW NNMM - звуковой сигнал, где NN - длительность, MM - тон

                db "JUMP", $00          \\      dw EXEC_JUMP            ; JUMP ХХХХ - безусловный переход на адрес ХХХХ
                db "READ", $00          \\      dw EXEC_READ            ; READ ХХХХ - чтение байта из ячейки памяти ХХХХ и вывод на экран
                db "WRITE", $00         \\      dw EXEC_WRITE           ; WRITE ХХХХ AA - запись байта АА в ячейку памяти ХХХХ
                
                db "SDBIOS", $00        \\      dw EXEC_PPI_FF          ; Загрузка с SD-привода Алексея Морозова
                db "WIFI", $00          \\      dw EXEC_PPI_7FFF        ; Загрузка 32к через ППА, оптимизировано для WiFi устройства
                
                db "KEYS", $00          \\      dw EXEC_KEYS            ; Ожидание нажатия одной из клавиш F1-F5, АР2, ВК, пробел
                db "KEY", $00           \\      dw EXEC_KEY             ; KEY А ... - выполнение команды при условии нажатия клавиши побитно F5-F1, АР2, ВК, пробел

                db $FF

AUTOEXEC:       
                db "SYSINFO", $00
                db "PRINT wk - wyhod w monitor", $00
                db "NL", $00
                db "NL", $00
                db "PRINT F1 - zagruzka ~erez ppa 16 bajt (R,FF)", $00
                db "NL", $00
                db "PRINT F2 - zagruzka ~erez ppa 32 kbajt (R,7FFF)", $00
                db "NL", $00
                db "NL", $00
                db "KEYS", $00
                db "PRINT kod: ", $00
                db "READ 7606", $00
                db "KEY 02 JUMP F800", $00
                db "KEY 08 SDBIOS", $00
                db "NL", $00
                db "KEY 10 WIFI", $00
                db $FF


MONITOR:
                
                org $F800

                db $C3, $36, $F8, $C3, $63, $FE, $C3, $98, $FB, $C3, $BA, $FC, $C3, $46, $FC, $C3
                db $BA, $FC, $C3, $01, $FE, $C3, $A5, $FC, $C3, $22, $F9, $C3, $72, $FE, $C3, $7B
                db $FA, $C3, $7F, $FA, $C3, $B6, $FA, $C3, $49, $FB, $C3, $16, $FB, $C3, $CE, $FA
                db $C3, $52, $FF, $C3, $56, $FF, $3E, $8A, $32, $03, $80, $31, $CF, $76, $CD, $CE
                db $FA, $21, $00, $76, $11, $5F, $76, $0E, $00, $CD, $ED, $F9, $21, $CF, $76, $22
                db $1C, $76, $21, $5A, $FF, $CD, $22, $F9, $CD, $CE, $FA, $21, $FF, $75, $22, $31
                db $76, $21, $2A, $1D, $22, $2F, $76, $3E, $C3, $32, $26, $76, $31, $CF, $76, $21
                db $66, $FF, $CD, $22, $F9, $32, $02, $80, $3D, $32, $02, $A0, $CD, $EE, $F8, $21
                db $6C, $F8, $E5, $21, $33, $76, $7E, $FE, $58, $CA, $D3, $FF, $FE, $55, $CA, $00
                db $E0, $F5, $CD, $2C, $F9, $2A, $2B, $76, $4D, $44, $2A, $29, $76, $EB, $2A, $27
                db $76, $F1, $FE, $44, $CA, $C5, $F9, $FE, $43, $CA, $D7, $F9, $FE, $46, $CA, $ED
                db $F9, $FE, $53, $CA, $F4, $F9, $FE, $54, $CA, $FF, $F9, $FE, $4D, $CA, $26, $FA
                db $FE, $47, $CA, $3F, $FA, $FE, $49, $CA, $86, $FA, $FE, $4F, $CA, $2D, $FB, $FE
                db $4C, $CA, $08, $FA, $FE, $52, $CA, $68, $FA, $C3, $00, $F0, $3E, $33, $BD, $CA
                db $F1, $F8, $E5, $21, $9E, $FF, $CD, $22, $F9, $E1, $2B, $C3, $F3, $F8, $21, $33
                db $76, $06, $00, $CD, $63, $FE, $FE, $08, $CA, $DC, $F8, $FE, $7F, $CA, $DC, $F8
                db $C4, $B9, $FC, $77, $FE, $0D, $CA, $1A, $F9, $FE, $2E, $CA, $6C, $F8, $06, $FF
                db $3E, $52, $BD, $CA, $AE, $FA, $23, $C3, $F3, $F8, $78, $17, $11, $33, $76, $06
                db $00, $C9, $7E, $A7, $C8, $CD, $B9, $FC, $23, $C3, $22, $F9, $21, $27, $76, $11
                db $2D, $76, $0E, $00, $CD, $ED, $F9, $11, $34, $76, $CD, $5A, $F9, $22, $27, $76
                db $22, $29, $76, $D8, $3E, $FF, $32, $2D, $76, $CD, $5A, $F9, $22, $29, $76, $D8
                db $CD, $5A, $F9, $22, $2B, $76, $D8, $C3, $AE, $FA, $21, $00, $00, $1A, $13, $FE
                db $0D, $CA, $8E, $F9, $FE, $2C, $C8, $FE, $20, $CA, $5D, $F9, $D6, $30, $FA, $AE
                db $FA, $FE, $0A, $FA, $82, $F9, $FE, $11, $FA, $AE, $FA, $FE, $17, $F2, $AE, $FA
                db $D6, $07, $4F, $29, $29, $29, $29, $DA, $AE, $FA, $09, $C3, $5D, $F9, $37, $C9
                db $7C, $BA, $C0, $7D, $BB, $C9, $CD, $A4, $F9, $CD, $90, $F9, $C2, $A2, $F9, $33
                db $33, $C9, $23, $C9, $CD, $72, $FE, $FE, $03, $C0, $CD, $CE, $FA, $C3, $AE, $FA
                db $E5, $21, $6C, $FF, $CD, $22, $F9, $E1, $C9, $7E, $C5, $CD, $A5, $FC, $3E, $20
                db $CD, $B9, $FC, $C1, $C9, $CD, $78, $FB, $CD, $B9, $F9, $CD, $96, $F9, $7D, $E6
                db $0F, $CA, $C5, $F9, $C3, $C8, $F9, $0A, $BE, $CA, $E6, $F9, $CD, $78, $FB, $CD
                db $B9, $F9, $0A, $CD, $BA, $F9, $03, $CD, $96, $F9, $C3, $D7, $F9, $71, $CD, $99
                db $F9, $C3, $ED, $F9, $79, $BE, $CC, $78, $FB, $CD, $96, $F9, $C3, $F4, $F9, $7E
                db $02, $03, $CD, $99, $F9, $C3, $FF, $F9, $CD, $78, $FB, $7E, $B7, $FA, $15, $FA
                db $FE, $20, $D2, $17, $FA, $3E, $2E, $CD, $B9, $FC, $CD, $96, $F9, $7D, $E6, $0F
                db $CA, $08, $FA, $C3, $0B, $FA, $CD, $78, $FB, $CD, $B9, $F9, $E5, $CD, $EE, $F8
                db $E1, $D2, $3B, $FA, $E5, $CD, $5A, $F9, $7D, $E1, $77, $23, $C3, $26, $FA, $CD
                db $90, $F9, $CA, $5A, $FA, $EB, $22, $23, $76, $7E, $32, $25, $76, $36, $F7, $3E
                db $C3, $32, $30, $00, $21, $A2, $FF, $22, $31, $00, $31, $18, $76, $C1, $D1, $E1
                db $F1, $F9, $2A, $16, $76, $C3, $26, $76, $3E, $90, $32, $03, $A0, $22, $01, $A0
                db $3A, $00, $A0, $02, $03, $CD, $99, $F9, $C3, $6D, $FA, $2A, $02, $76, $C9, $E5
                db $2A, $00, $76, $7E, $E1, $C9, $3A, $2D, $76, $B7, $CA, $91, $FA, $7B, $32, $2F
                db $76, $CD, $B6, $FA, $CD, $78, $FB, $EB, $CD, $78, $FB, $EB, $C5, $CD, $16, $FB
                db $60, $69, $CD, $78, $FB, $D1, $CD, $90, $F9, $C8, $EB, $CD, $78, $FB, $3E, $3F
                db $CD, $B9, $FC, $C3, $6C, $F8, $3E, $FF, $CD, $FF, $FA, $E5, $09, $EB, $CD, $FD
                db $FA, $E1, $09, $EB, $E5, $CD, $0A, $FB, $3E, $FF, $CD, $FF, $FA, $E1, $E5, $21
                db $01, $C0, $36, $00, $2B, $36, $4D, $36, $1D, $36, $99, $36, $93, $23, $36, $27
                db $7E, $7E, $E6, $20, $CA, $E1, $FA, $21, $08, $E0, $36, $80, $2E, $04, $36, $D0
                db $36, $76, $2C, $36, $23, $36, $49, $2E, $08, $36, $A4, $E1, $C9, $3E, $08, $CD
                db $98, $FB, $47, $3E, $08, $CD, $98, $FB, $4F, $C9, $3E, $08, $CD, $98, $FB, $77
                db $CD, $99, $F9, $C3, $0A, $FB, $01, $00, $00, $7E, $81, $4F, $F5, $CD, $90, $F9
                db $CA, $9F, $F9, $F1, $78, $8E, $47, $CD, $99, $F9, $C3, $19, $FB, $79, $B7, $CA
                db $35, $FB, $32, $30, $76, $E5, $CD, $16, $FB, $E1, $CD, $78, $FB, $EB, $CD, $78
                db $FB, $EB, $E5, $60, $69, $CD, $78, $FB, $E1, $C5, $01, $00, $00, $CD, $46, $FC
                db $05, $E3, $E3, $C2, $4D, $FB, $0E, $E6, $CD, $46, $FC, $CD, $90, $FB, $EB, $CD
                db $90, $FB, $EB, $CD, $86, $FB, $21, $00, $00, $CD, $90, $FB, $0E, $E6, $CD, $46
                db $FC, $E1, $CD, $90, $FB, $C3, $CE, $FA, $C5, $CD, $B0, $F9, $7C, $CD, $A5, $FC
                db $7D, $CD, $BA, $F9, $C1, $C9, $4E, $CD, $46, $FC, $CD, $99, $F9, $C3, $86, $FB
                db $4C, $CD, $46, $FC, $4D, $C3, $46, $FC, $E5, $C5, $D5, $57, $3E, $80, $32, $08
                db $E0, $21, $00, $00, $39, $31, $00, $00, $22, $0D, $76, $0E, $00, $3A, $02, $80
                db $0F, $0F, $0F, $0F, $E6, $01, $5F, $F1, $79, $E6, $7F, $07, $4F, $26, $00, $25
                db $CA, $34, $FC, $F1, $3A, $02, $80, $0F, $0F, $0F, $0F, $E6, $01, $BB, $CA, $BF
                db $FB, $B1, $4F, $15, $3A, $2F, $76, $C2, $DC, $FB, $D6, $12, $47, $F1, $05, $C2
                db $DD, $FB, $14, $3A, $02, $80, $0F, $0F, $0F, $0F, $E6, $01, $5F, $7A, $B7, $F2
                db $0B, $FC, $79, $FE, $E6, $C2, $FF, $FB, $AF, $32, $2E, $76, $C3, $09, $FC, $FE
                db $19, $C2, $B7, $FB, $3E, $FF, $32, $2E, $76, $16, $09, $15, $C2, $B7, $FB, $21
                db $04, $E0, $36, $D0, $36, $76, $23, $36, $23, $36, $49, $3E, $27, $32, $01, $C0
                db $3E, $E0, $32, $01, $C0, $2E, $08, $36, $A4, $2A, $0D, $76, $F9, $3A, $2E, $76
                db $A9, $C3, $A1, $FC, $2A, $0D, $76, $F9, $CD, $CE, $FA, $7A, $B7, $F2, $AE, $FA
                db $CD, $A4, $F9, $C3, $9C, $FB, $E5, $C5, $D5, $F5, $3E, $80, $32, $08, $E0, $21
                db $00, $00, $39, $31, $00, $00, $16, $08, $F1, $79, $07, $4F, $3E, $01, $A9, $32
                db $02, $80, $3A, $30, $76, $47, $F1, $05, $C2, $66, $FC, $3E, $00, $A9, $32, $02
                db $80, $15, $3A, $30, $76, $C2, $7A, $FC, $D6, $0E, $47, $F1, $05, $C2, $7B, $FC
                db $14, $15, $C2, $58, $FC, $F9, $21, $04, $E0, $36, $D0, $36, $76, $23, $36, $23
                db $36, $49, $3E, $27, $32, $01, $C0, $3E, $E0, $32, $01, $C0, $2E, $08, $36, $A4
                db $F1, $D1, $C1, $E1, $C9, $F5, $0F, $0F, $0F, $0F, $CD, $AE, $FC, $F1, $E6, $0F
                db $FE, $0A, $FA, $B7, $FC, $C6, $07, $C6, $30, $4F, $F5, $C5, $D5, $E5, $CD, $01
                db $FE, $21, $85, $FD, $E5, $2A, $02, $76, $EB, $2A, $00, $76, $3A, $04, $76, $3D
                db $FA, $EE, $FC, $CA, $65, $FD, $E2, $73, $FD, $79, $D6, $20, $4F, $0D, $FA, $E9
                db $FC, $C5, $CD, $B9, $FD, $C1, $C3, $DD, $FC, $AF, $32, $04, $76, $C9, $79, $E6
                db $7F, $4F, $FE, $1F, $CA, $A3, $FD, $FE, $0C, $CA, $B2, $FD, $FE, $0D, $CA, $F3
                db $FD, $FE, $0A, $CA, $47, $FD, $FE, $08, $CA, $D6, $FD, $FE, $18, $CA, $B9, $FD
                db $FE, $19, $CA, $E2, $FD, $FE, $1A, $CA, $C5, $FD, $FE, $1B, $CA, $9E, $FD, $FE
                db $07, $C2, $38, $FD, $01, $F0, $05, $78, $FB, $3D, $C2, $28, $FD, $78, $F3, $3D
                db $C2, $2E, $FD, $0D, $C2, $27, $FD, $C9, $71, $CD, $B9, $FD, $7A, $FE, $03, $C0
                db $7B, $FE, $08, $C0, $CD, $E2, $FD, $7A, $FE, $1B, $C2, $C5, $FD, $E5, $D5, $21
                db $C2, $77, $11, $10, $78, $01, $9E, $07, $1A, $77, $23, $13, $0B, $79, $B0, $C2
                db $58, $FD, $D1, $E1, $C9, $79, $FE, $59, $C2, $E9, $FC, $CD, $B2, $FD, $3E, $02
                db $C3, $EA, $FC, $79, $D6, $20, $4F, $0D, $3E, $04, $FA, $EA, $FC, $C5, $CD, $C5
                db $FD, $C1, $C3, $77, $FD, $22, $00, $76, $EB, $22, $02, $76, $3E, $80, $32, $01
                db $C0, $7D, $32, $00, $C0, $7C, $32, $00, $C0, $E1, $D1, $C1, $F1, $C9, $3E, $01
                db $C3, $EA, $FC, $21, $F4, $7F, $11, $25, $09, $AF, $77, $2B, $1B, $7B, $B2, $C2
                db $A9, $FD, $11, $08, $03, $21, $C2, $77, $C9, $7B, $23, $1C, $FE, $47, $C0, $1E
                db $08, $01, $C0, $FF, $09, $7A, $FE, $1B, $01, $4E, $00, $C2, $D3, $FD, $16, $02
                db $01, $B0, $F8, $14, $09, $C9, $7B, $2B, $1D, $FE, $08, $C0, $1E, $47, $01, $40
                db $00, $09, $7A, $FE, $03, $01, $B2, $FF, $C2, $F0, $FD, $16, $1C, $01, $50, $07
                db $15, $09, $C9, $7D, $93, $D2, $F9, $FD, $25, $6F, $1E, $08, $01, $08, $00, $09
                db $C9, $3A, $02, $80, $E6, $80, $CA, $0E, $FE, $3A, $05, $76, $B7, $C0, $E5, $2A
                db $09, $76, $CD, $72, $FE, $BD, $6F, $CA, $2A, $FE, $3E, $01, $32, $0B, $76, $26
                db $15, $AF, $22, $09, $76, $E1, $32, $05, $76, $C9, $25, $C2, $21, $FE, $3C, $CA
                db $22, $FE, $3C, $CA, $51, $FE, $C5, $01, $03, $50, $CD, $27, $FD, $C1, $3A, $0B
                db $76, $26, $E0, $3D, $32, $0B, $76, $CA, $4C, $FE, $26, $40, $3E, $FF, $C3, $22
                db $FE, $3A, $02, $80, $E6, $80, $CA, $51, $FE, $3A, $06, $76, $2F, $32, $06, $76
                db $C3, $1A, $FE, $CD, $01, $FE, $B7, $CA, $63, $FE, $AF, $32, $05, $76, $3A, $09
                db $76, $C9, $3A, $02, $80, $E6, $80, $C2, $7D, $FE, $3E, $FE, $C9, $AF, $32, $00
                db $80, $32, $02, $80, $3A, $06, $76, $E6, $01, $F6, $06, $32, $03, $80, $3A, $01
                db $80, $3C, $C2, $97, $FE, $3D, $C9, $E5, $2E, $01, $26, $07, $7D, $0F, $6F, $2F
                db $32, $00, $80, $3A, $01, $80, $2F, $B7, $C2, $B3, $FE, $25, $F2, $9C, $FE, $3E
                db $FF, $E1, $C9, $2E, $20, $3A, $01, $80, $2F, $B7, $CA, $AF, $FE, $2D, $C2, $B5
                db $FE, $2E, $08, $2D, $07, $D2, $C3, $FE, $7C, $65, $6F, $FE, $01, $CA, $FA, $FE
                db $DA, $F3, $FE, $07, $07, $07, $C6, $20, $B4, $FE, $5F, $C2, $06, $FF, $3E, $20
                db $E1, $C9, $09, $0A, $0D, $7F, $08, $19, $18, $1A, $0C, $1F, $1B, $00, $01, $02
                db $03, $04, $05, $7C, $21, $EA, $FE, $C3, $FE, $FE, $7C, $21, $E2, $FE, $85, $6F
                db $7E, $FE, $40, $E1, $D8, $E5, $6F, $3A, $02, $80, $67, $E6, $40, $C2, $1A, $FF
                db $7D, $FE, $40, $FA, $3F, $FF, $E6, $1F, $E1, $C9, $3A, $06, $76, $B7, $CA, $2A
                db $FF, $7D, $FE, $40, $FA, $2A, $FF, $F6, $20, $6F, $7C, $E6, $20, $C2, $3F, $FF
                db $7D, $FE, $40, $FA, $3B, $FF, $7D, $EE, $20, $E1, $C9, $7D, $E6, $2F, $6F, $7D
                db $FE, $40, $E1, $F0, $E5, $6F, $E6, $0F, $FE, $0C, $7D, $FA, $50, $FF, $EE, $10
                db $E1, $C9, $2A, $31, $76, $C9, $22, $31, $76, $C9, $1F, $72, $61, $64, $69, $6F
                db $2D, $38, $36, $72, $6B, $00, $0D, $0A, $2D, $2D, $3E, $00, $0D, $0A, $18, $18
                db $18, $18, $00, $0D, $0A, $20, $50, $43, $2D, $0D, $0A, $20, $48, $4C, $2D, $0D
                db $0A, $20, $42, $43, $2D, $0D, $0A, $20, $44, $45, $2D, $0D, $0A, $20, $53, $50
                db $2D, $0D, $0A, $20, $41, $46, $2D, $19, $19, $19, $19, $19, $19, $00, $08, $20
                db $08, $00, $22, $16, $76, $F5, $E1, $22, $1E, $76, $E1, $2B, $22, $14, $76, $21
                db $00, $00, $39, $31, $1E, $76, $E5, $D5, $C5, $2A, $14, $76, $31, $CF, $76, $CD
                db $78, $FB, $EB, $2A, $23, $76, $CD, $90, $F9, $C2, $6C, $F8, $3A, $25, $76, $77
                db $C3, $6C, $F8, $21, $73, $FF, $CD, $22, $F9, $21, $14, $76, $06, $06, $5E, $23
                db $56, $C5, $E5, $EB, $CD, $78, $FB, $CD, $EE, $F8, $D2, $F6, $FF, $CD, $5A, $F9
                db $D1, $D5, $EB, $72, $2B, $73, $E1, $C1, $05, $23, $C2, $DE, $FF, $C9, $FF, $FF
