; ---------------------------------------------------------------------------
; zx.com — CP/M loader for Argo ZX mode
;
; This program runs under CP/M and prepares the machine to execute the
; standard ZX Spectrum ROM (ROM.SNC) on Argo hardware.
;
; Responsibilities:
;   • Initialize basic system state (sysctl, DMA reset, RAM clearing)
;   • Load KBD.SNC into RAM (keyboard + ZX hardware service layer)
;   • Load ROM.SNC into RAM (unaltered ZX Spectrum ROM image)
;   • Prepare screen / attribute memory regions
;   • Copy a stage-2 hardware-init stub to high memory and execute it
;
; After loading, the program switches memory configuration and transfers
; control away from CP/M. From this point on, ZX ROM code runs directly and
; CP/M is no longer active.
;
; KBD.SNC and the stage-2 stub provide the missing ZX ULA functionality
; (keyboard scanning, CRT timing, DMA setup, memory paging) required for
; the original ZX ROM to operate correctly on Argo hardware.
; ---------------------------------------------------------------------------

; zx.com — annotated disassembly for Argo (К580* based) hardware
; CP/M .COM (ORG 0100h)
;
; This version re-annotates I/O and memory behavior using the provided hardware spec:
;   - port 80h: sysctl  (crt.border[0..3], beep.input[4], etc.)
;   - port 81h: IN  -> amm.nmi.code ; OUT -> casctl (tape control)
;   - port A1h: IN  -> keyboard data ; OUT -> memcfg (offset 0)
;   - port A9h: OUT -> memcfg (offset 1)
;   - port B1h: OUT -> memcfg (offset 2)
;   - port B9h: OUT -> memcfg (offset 3)
;   - port C0h: crt.data (offset 0)
;   - port C4h: crt.data (offset 1)
;   - port E0h–EFh: dma.data (offset = low nibble), so E4/E5/E8 are DMA registers 4/5/8
;
; The program:
;   1) Initializes sysctl and clears F800h–FFFFh with 20h
;   2) Loads KBD.SNC into 4000h (via BDOS OPEN + READ SEQ, DMA=0080h)
;   3) Loads ROM.SNC into 5800h
;   4) Draws/fills some RAM patterns around 4800h/58xxh (likely display-related)
;   5) Copies stage-2 stub (at 01C5h) to 99C5h, sets SP, patches vectors, jumps to 99C5h
;   6) Stage-2 stub programs CRT/DMA and switches memory configuration (memcfg), then WBOOT
;
; Notes on CP/M calls:
;   CALL 0005h is BDOS entry:
;     C=0Fh OPEN (DE=FCB)
;     C=14h READ SEQUENTIAL (DE=FCB) -> reads 128 bytes into current DMA
;     C=1Ah SET DMA ADDRESS (DE=addr)
;     C=02h CONOUT (E=char)
;   JP 0000h is BIOS WBOOT (warm boot)
;
        ORG 0100h

start:
0100:   F3              DI
0101:   31 FF 0F        LD  SP,0FFFh

; sysctl: write[0-3]=crt.border, write[4]=beep.input
0104:   3E A0           LD  A,0A0h
0106:   D3 80           OUT (080h),A          ; sysctl <= A0h (border/beep baseline)

; Clear top RAM page F800h–FFFFh with 20h
0108:   21 00 F8        LD  HL,0F800h
010B:   06 20           LD  B,020h            ; fill byte = 20h (' ')
010D:   AF              XOR A
fill_f800_ffff:
010E:   70              LD  (HL),B
010F:   23              INC HL
0110:   BC              CP  H
0111:   C2 0E 01        JP  NZ,fill_f800_ffff

; Print banner (zero-terminated) at 02EBh
0114:   21 EB 02        LD  HL,msg_banner
0117:   CD 9C 02        CALL print_zstring

; DMA register programming (E4h is dma.data[4])
011A:   3E 1A           LD  A,01Ah
011C:   D3 E4           OUT (0E4h),A          ; dma.data[4] <= 1Ah
011E:   3E 4B           LD  A,04Bh
0120:   D3 E4           OUT (0E4h),A          ; dma.data[4] <= 4Bh

; Build FCB1 (005Ch) name/ext = "KBD     " / "SNC"
; CP/M default FCB is at 005Ch; name starts at 005Dh; ext at 0065h.
0122:   21 5D 00        LD  HL,005Dh
0125:   0E 08           LD  C,08h
0127:   3E 20           LD  A,020h
fcb_pad1:
0129:   77              LD  (HL),A
012A:   23              INC HL
012B:   0D              DEC C
012C:   C2 27 01        JP  NZ,fcb_pad1

012F:   21 5D 00        LD  HL,005Dh
0132:   36 4B           LD  (HL),'K'
0134:   23              INC HL
0135:   36 42           LD  (HL),'B'
0137:   23              INC HL
0138:   36 44           LD  (HL),'D'

013A:   21 65 00        LD  HL,0065h
013D:   36 53           LD  (HL),'S'
013F:   23              INC HL
0140:   36 4E           LD  (HL),'N'
0142:   23              INC HL
0143:   36 43           LD  (HL),'C'

; OPEN FCB1
0145:   CD 55 02        CALL bdos_open_fcb
0148:   FE FF           CP  0FFh
014A:   CC 93 02        CALL Z,fatal_wboot     ; if OPEN failed => warm boot

; Load file sequentially to 4000h (max 3FFFh bytes)
014D:   21 00 40        LD  HL,04000h
0150:   CD 62 02        CALL load_file_to_HL_max3FFF

; Print second message
0153:   21 89 03        LD  HL,msg_after_kbd
0156:   CD 9C 02        CALL print_zstring

; Build FCB for "ROM     " / "SNC"
0159:   21 5D 00        LD  HL,005Dh
015C:   0E 08           LD  C,08h
015E:   3E 20           LD  A,020h
fcb_pad2:
0160:   77              LD  (HL),A
0161:   23              INC HL
0162:   0D              DEC C
0163:   C2 5E 01        JP  NZ,fcb_pad2

0166:   21 5D 00        LD  HL,005Dh
0169:   36 52           LD  (HL),'R'
016B:   23              INC HL
016C:   36 4F           LD  (HL),'O'
016E:   23              INC HL
016F:   36 4D           LD  (HL),'M'

0171:   21 65 00        LD  HL,0065h
0174:   36 53           LD  (HL),'S'
0176:   23              INC HL
0177:   36 4E           LD  (HL),'N'
0179:   23              INC HL
017A:   36 43           LD  (HL),'C'

017C:   CD 55 02        CALL bdos_open_fcb
017F:   FE FF           CP  0FFh
0181:   CC 93 02        CALL Z,fatal_wboot

; Load ROM.SNC to 5800h (max 3FFFh bytes)
0184:   21 00 58        LD  HL,05800h
0187:   CD 62 02        CALL load_file_to_HL_max3FFF

; RAM decoration / fill patterns (display-related). Exact purpose is machine-specific.
018A:   21 00 48        LD  HL,04800h
018D:   1E 9F           LD  E,09Fh
018F:   36 00           LD  (HL),00h
0191:   23              INC HL
0192:   CD AD 02        CALL fill_zero_E

0195:   01 8A 58        LD  BC,0588Ah
0198:   CD B5 02        CALL write_attr_rows
019B:   04              INC B
019C:   0E 9A           LD  C,09Ah
019E:   CD B5 02        CALL write_attr_rows
01A1:   04              INC B
01A2:   0E 8E           LD  C,08Eh
01A4:   CD B5 02        CALL write_attr_rows

01A7:   1E A0           LD  E,0A0h
01A9:   CD AD 02        CALL fill_zero_E

; Copy stage-2 stub (01C5..01C5+02FF) to 99C5 and jump there
01AC:   21 C5 01        LD  HL,stage2_stub
01AF:   01 C5 99        LD  BC,099C5h
01B2:   11 FF 02        LD  DE,02FFh
01B5:   CD 84 02        CALL memcpy_hl_to_bc_de

; Switch SP, patch pointers, transfer control
01B8:   31 FF 98        LD  SP,098FFh
01BB:   21 84 9A        LD  HL,09A84h
01BE:   22 8C 9A        LD  (09A8Ch),HL
01C1:   22 90 9A        LD  (09A90h),HL
01C4:   C3 C5 99        JP  099C5h

; -----------------------------------------------------------------------------
; Stage-2 stub (copied to 99C5h and executed there)
; -----------------------------------------------------------------------------
stage2_stub:
01C5:   21 00 00        LD  HL,0000h
01C8:   01 20 4D        LD  BC,04D20h
01CB:   11 00 01        LD  DE,0100h
01CE:   CD 84 9A        CALL 09A84h           ; external routine (not in this COM)

; CRT programming
; C0h = crt.data[0], C4h = crt.data[1]
01D1:   3E 1D           LD  A,01Dh
01D3:   D3 C0           OUT (0C0h),A          ; crt.data[0] <= 1Dh
01D5:   3E E0           LD  A,0E0h
01D7:   D3 C4           OUT (0C4h),A          ; crt.data[1] <= E0h

; DMA programming: E4/E5/E8 are dma.data[4]/[5]/[8]
01D9:   3E 80           LD  A,080h
01DB:   D3 E8           OUT (0E8h),A          ; dma.reg8 <= 80h
01DD:   AF              XOR A
01DE:   D3 E4           OUT (0E4h),A          ; dma.reg4 <= 00h
01E0:   3E 48           LD  A,048h
01E2:   D3 E4           OUT (0E4h),A          ; dma.reg4 <= 48h
01E4:   3E C2           LD  A,0C2h
01E6:   D3 E5           OUT (0E5h),A          ; dma.reg5 <= C2h
01E8:   3E 84           LD  A,084h
01EA:   D3 E5           OUT (0E5h),A          ; dma.reg5 <= 84h
01EC:   3E 23           LD  A,023h
01EE:   D3 E8           OUT (0E8h),A          ; dma.reg8 <= 23h

; Similar CRT/DMA sequences repeat (kept verbatim, annotated as such)
01F0:   3E 1D           LD  A,01Dh
01F2:   D3 C0           OUT (0C0h),A          ; crt.data[0]
01F4:   3E E0           LD  A,0E0h
01F6:   D3 C4           OUT (0C4h),A          ; crt.data[1]
01F8:   3E 80           LD  A,080h
01FA:   D3 E8           OUT (0E8h),A          ; dma.reg8
01FC:   AF              XOR A
01FD:   D3 E4           OUT (0E4h),A          ; dma.reg4
01FF:   3E 48           LD  A,048h
0201:   D3 E4           OUT (0E4h),A          ; dma.reg4
0203:   3E C2           LD  A,0C2h
0205:   D3 E5           OUT (0E5h),A          ; dma.reg5
0207:   3E 84           LD  A,084h
0209:   D3 E5           OUT (0E5h),A          ; dma.reg5

020B:   3E 1D           LD  A,01Dh
020D:   D3 C0           OUT (0C0h),A          ; crt.data[0]
020F:   3E E0           LD  A,0E0h
0211:   D3 C4           OUT (0C4h),A          ; crt.data[1]
0213:   3E 80           LD  A,080h
0215:   D3 E8           OUT (0E8h),A          ; dma.reg8
0217:   AF              XOR A
0218:   D3 E4           OUT (0E4h),A          ; dma.reg4
021A:   3E 48           LD  A,048h
021C:   D3 E4           OUT (0E4h),A          ; dma.reg4
021E:   3E C2           LD  A,0C2h
0220:   D3 E5           OUT (0E5h),A          ; dma.reg5
0222:   3E 84           LD  A,084h
0224:   D3 E5           OUT (0E5h),A          ; dma.reg5

0226:   3E 23           LD  A,023h
0228:   D3 C4           OUT (0C4h),A          ; crt.data[1] <= 23h (note: uses C4 here)
022A:   3E E4           LD  A,0E4h
022C:   D3 E8           OUT (0E8h),A          ; dma.reg8 <= E4h

; Memory configuration / paging / zxmode/font/timer page controls
; OUT A1/A9/B1/B9 feed memcfg decoder (offset 0/1/2/3).
022E:   3E C1           LD  A,0C1h
0230:   D3 A1           OUT (0A1h),A          ; memcfg[0] <= C1h (also affects reg1 bit7 functions)
0232:   3E CB           LD  A,0CBh
0234:   D3 B1           OUT (0B1h),A          ; memcfg[2] <= CBh
0236:   3E D8           LD  A,0D8h
0238:   D3 B9           OUT (0B9h),A          ; memcfg[3] <= D8h

; Return to CP/M BIOS warm boot (may be intentional to re-enter CP/M with new map)
023A:   C3 00 00        JP  0000h

; -----------------------------------------------------------------------------
; BDOS helpers
; -----------------------------------------------------------------------------

; Read sequential record into DMA=0080h, returns BC=0080h (start of DMA buffer)
bdos_read_record:
023D:   D5              PUSH DE
023E:   E5              PUSH HL
023F:   11 80 00        LD  DE,0080h
0242:   0E 1A           LD  C,01Ah
0244:   CD 05 00        CALL 0005h            ; BDOS 1Ah: SET DMA ADDRESS
0247:   11 5C 00        LD  DE,005Ch
024A:   0E 14           LD  C,014h
024C:   CD 05 00        CALL 0005h            ; BDOS 14h: READ SEQUENTIAL
024F:   E1              POP HL
0250:   D1              POP DE
0251:   01 80 00        LD  BC,0080h
0254:   C9              RET

; Open default FCB1 at 005Ch; returns A=FFh on failure (per CP/M)
bdos_open_fcb:
0255:   AF              XOR A
0256:   32 7C 00        LD  (007Ch),A         ; clear some flag byte
0259:   11 5C 00        LD  DE,005Ch
025C:   0E 0F           LD  C,00Fh
025E:   CD 05 00        CALL 0005h            ; BDOS 0Fh: OPEN
0261:   C9              RET

; Load file sequentially to (HL), copying from DMA buffer;
; Stops after 3FFFh bytes copied OR when BDOS read yields non-zero A (EOF/error).
load_file_to_HL_max3FFF:
0262:   01 80 00        LD  BC,0080h
0265:   11 FF 3F        LD  DE,03FFFh         ; max bytes to copy
0268:   CD 3D 02        CALL bdos_read_record
copy_loop:
026B:   0A              LD  A,(BC)            ; from DMA
026C:   77              LD  (HL),A            ; to destination
026D:   23              INC HL
026E:   03              INC BC
026F:   1B              DEC DE
0270:   AF              XOR A
0271:   BA              CP  D
0272:   C2 7A 02        JP  NZ,need_more
0275:   BB              CP  E
0276:   C2 7A 02        JP  NZ,need_more
0279:   C9              RET                   ; reached max
need_more:
027A:   B9              CP  C
027B:   CC 3D 02        CALL Z,bdos_read_record ; if C wrapped to 00h, fetch next record
027E:   FE 00           CP  00h
0280:   C0              RET NZ                ; if BDOS returned non-zero previously (EOF/error), stop
0281:   C3 6B 02        JP  copy_loop

; memcpy: copy DE bytes from (HL) to (BC)
memcpy_hl_to_bc_de:
0284:   7E              LD  A,(HL)
0285:   02              LD  (BC),A
0286:   23              INC HL
0287:   03              INC BC
0288:   1B              DEC DE
0289:   AF              XOR A
028A:   BA              CP  D
028B:   C2 94 02        JP  NZ,memcpy_hl_to_bc_de
028E:   BB              CP  E
028F:   C2 94 02        JP  NZ,memcpy_hl_to_bc_de
0292:   C9              RET

fatal_wboot:
0293:   C3 00 00        JP  0000h             ; BIOS WBOOT

; Print 00-terminated string at HL via BDOS CONOUT
print_zstring:
029C:   AF              XOR A
next_ch:
029D:   5E              LD  E,(HL)
029E:   BB              CP  E
029F:   C8              RET Z
02A0:   0E 02           LD  C,002h
02A2:   E5              PUSH HL
02A3:   F5              PUSH AF
02A4:   CD 05 00        CALL 0005h            ; BDOS 02h: CONOUT (E=char)
02A7:   F1              POP AF
02A8:   E1              POP HL
02A9:   23              INC HL
02AA:   C3 9D 02        JP  next_ch

; Fill E bytes of zero starting at HL (note: function is actually "write 00h E times")
fill_zero_E:
02AD:   36 00           LD  (HL),00h
02AF:   23              INC HL
02B0:   1D              DEC E
02B1:   C8              RET Z
02B2:   C3 AD 02        JP  fill_zero_E

; Writes 4 small blocks using helper draw_block_at_HL_A and C as a parameter
write_attr_rows:
02B5:   00              NOP
02B6:   CD C9 02        CALL draw_block_at_HL_A
02B9:   3E 20           LD  A,020h
02BB:   CD C9 02        CALL draw_block_at_HL_A
02BE:   3E 40           LD  A,040h
02C0:   CD C9 02        CALL draw_block_at_HL_A
02C3:   3E 60           LD  A,060h
02C5:   CD C9 02        CALL draw_block_at_HL_A
02C8:   C9              RET

draw_block_at_HL_A:
02C9:   1E 08           LD  E,08h
02CB:   CD AD 02        CALL fill_zero_E
02CE:   71              LD  (HL),C
02CF:   23              INC HL
02D0:   1E 20           LD  E,020h
02D2:   CD DF 02        CALL ramp_and_B
02D5:   36 80           LD  (HL),080h
02D7:   23              INC HL
02D8:   1E 08           LD  E,08h
02DA:   AF              XOR A
02DB:   CD AD 02        CALL fill_zero_E
02DE:   C9              RET

; Writes pattern: (HL)=A; A++ ; (HL+1)=B; repeat E times
ramp_and_B:
02DF:   00              NOP
02E0:   00              NOP
02E1:   77              LD  (HL),A
02E2:   3C              INC A
02E3:   23              INC HL
02E4:   70              LD  (HL),B
02E5:   23              INC HL
02E6:   1D              DEC E
02E7:   C8              RET Z
02E8:   C3 E1 02        JP  ramp_and_B

; -----------------------------------------------------------------------------
; Data (messages are 00-terminated)
; -----------------------------------------------------------------------------
msg_banner:
02EB:   ; KOI8-R intro banner (00-terminated)
        ; 

		 	ПРОГРАММА ПЕРЕВОДА


ПЭВМ   "АРГО"  в режим "ZX SPECTRUM"


			ИК АН Украины   1991г.


 
        DB 0Ah,0Dh,09h,09h,20h,09h,F0h,F2h,EFh,E7h,F2h,E1h,EDh,EDh,E1h,20h
        DB F0h,E5h,F2h,E5h,F7h,EFh,E4h,E1h,0Ah,0Dh,0Ah,0Dh,F0h,FCh,F7h,EDh
        DB 20h,20h,20h,22h,E1h,F2h,E7h,EFh,22h,20h,20h,D7h,20h,D2h,C5h,D6h
        DB C9h,CDh,20h,22h,5Ah,58h,20h,53h,50h,45h,43h,54h,52h,55h,4Dh,22h
        DB 0Ah,0Dh,0Ah,0Dh,09h,09h,09h,E9h,EBh,20h,E1h,EEh,20h,F5h,CBh,D2h
        DB C1h,C9h,CEh,D9h,20h,20h,20h,31h,39h,39h,31h,C7h,2Eh,0Ah,0Dh,0Ah
        DB 0Dh,00h

msg_file_not_found:
0355:   ; KOI8-R: "Файл не найден" + CRLF + 00
        DB E6h,C1h,CAh,CCh,20h,CEh,C5h,20h,CEh,C1h,CAh,C4h,C5h,CEh,0Ah,0Dh
        DB 00h

msg_load_kbd:
036B:   ; KOI8-R: "Загрузка файла KBD.SNC" + CRLF + 00
        DB FAh,C1h,C7h,D2h,D5h,DAh,CBh,C1h,20h,C6h,C1h,CAh,CCh,C1h,20h,4Bh
        DB 42h,44h,2Eh,53h,4Eh,43h,0Ah,0Dh,00h

msg_load_rom:
0389:   ; KOI8-R: "Загрузка файла ROM.SNC" + CRLF CRLF + 00
        DB FAh,C1h,C7h,D2h,D5h,DAh,CBh,C1h,20h,C6h,C1h,CAh,CCh,C1h,20h,52h
        DB 4Fh,4Dh,2Eh,53h,4Eh,43h,0Ah,0Dh,0Ah,0Dh,00h

; End of file
