#include <string.h>
#include <ctype.h>
#include "dialog.h"
#include "font.h"
#include "dirwork.h"
#include "graphics.h"
#include "lib_wd1793/unreal_wd1793.h"
#include "lib_z80ex/z80ex_dasm.h"
#include "devs.h"
#include "tape/tape.h"

#include "images/zemu_ico.h"

#define MAX_FILES 4096
#define MAX_FNAME 256

extern C_Font font;
char oldFileName[4][MAX_PATH] = {"", "", "" ,""};
int currentDrive = 0;

int FileNameCmp(int f1, char *s1, int f2, char *s2)
{
	if (f1 == f2) return stricmp(s1, s2);
	else
	if (f1) return (-1);
	else return (1);
}

void DlgClearScreen(void)
{
	int i, j, cl;
	int *s, *p;

	if (SDL_MUSTLOCK(screen)) {if (SDL_LockSurface(screen) < 0) return;}
	s = (int *)screen->pixels;

	for (i = 0; i < HEIGHT; i++)
	{
		p = s;
		cl = (i & 1) ? DRGB(0,0,64) : DRGB(0,0,32);
		for (j = 0; j < WIDTH; j++) *(p++) = cl;
		s += PITCH;
	}

	if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
}

bool DlgConfirm(const char *message)
{
	int x, y, key;
	SDL_Event event;
	int wdt = font.StrLenPx(message) + 0x10;
	int hgt = font.Height() + 0x10;

	x = (WIDTH - wdt) / 2;
	y = (HEIGHT - hgt) / 2;

	DlgClearScreen();
	Bar(x, y, x+wdt-1, y+hgt-1, DRGB(0x80, 0x20, 0x20));
	font.PrintString(x+8, y+8, message);
	ScaleImage();
	UpdateScreen();

	for (;;)
	{
		for (key = 0; SDL_PollEvent(&event);)
		{
			if (event.type == SDL_QUIT) exit(0);
			if (event.type == SDL_KEYDOWN) key = event.key.keysym.sym;
		}

		if (key==SDLK_y || key==SDLK_RETURN) return true;
		if (key==SDLK_n || key==SDLK_ESCAPE) return false;

		SDL_Delay(10);
	}
}

#define MAX_INPUT_STRING 64

const char* DlgInputString(const char *message)
{
	static char buffer[MAX_INPUT_STRING + 1];
	char buf[0x100 + MAX_INPUT_STRING];

	buffer[0] = 0;
	int bufferSz = 0;

	int wdt, hgt;
	int x, y, key;
	SDL_Event event;

	for (;;)
	{
		sprintf(buf, "%s: %s", message, buffer);

		wdt = font.StrLenPx(buf) + 0x10;
		hgt = font.Height() + 0x10;

		x = (WIDTH - wdt) / 2;
		y = (HEIGHT - hgt) / 2;

		DlgClearScreen();
    		Bar(x, y, x+wdt-1, y+hgt-1, DRGB(0x80, 0x20, 0x20));
		font.PrintString(x+8, y+8, buf);
		ScaleImage();
		UpdateScreen();

		do
		{
			for (key = 0; SDL_PollEvent(&event);)
			{
				if (event.type == SDL_QUIT) exit(0);
				if (event.type == SDL_KEYDOWN) key = event.key.keysym.sym;
			}
		} while (key == 0);

		if (key == SDLK_ESCAPE)
		{
		    buffer[0] = 0;
		    return buffer;
		}
		else
		if (key == SDLK_RETURN)
		{
		    return buffer;
		}
		else
		if (key == SDLK_BACKSPACE)
		{
		    if (bufferSz > 0) {
			buffer[--bufferSz] = 0;
		    }
		}
		else
		if (key>=32 && key<=127)
		{
		    if (bufferSz < MAX_INPUT_STRING-1) {
			buffer[bufferSz++] = key;
			buffer[bufferSz] = 0;
		    }
		}

		SDL_Delay(10);
	}
}

char* SelectFile(char *oldFile)
{
	int key;
	SDL_Event event;
	C_DirWork dw;
	char buf[0x100];
	static char path[MAX_PATH], ofl[MAX_FNAME+1];
	char fnames[MAX_FILES][MAX_FNAME+1], temp[MAX_FNAME+1];
	int i, j, filesCnt, folders[MAX_FILES], pos, csr, mx, gl, h, x, tmp;
	int scrEnd, keyx;
	bool isRoot;

	scrEnd = HEIGHT - font.Height() - 8;

	strcpy(ofl, C_DirWork::ExtractFileName(oldFile));
	strcpy(path, C_DirWork::ExtractPath(oldFile));
	strcpy(path, C_DirWork::Normalize(path));

	h = font.Height();
	mx = scrEnd / h;
	x = font.StrLenPx("[]");

	do
	{
		isRoot = !strcmp(path, "/");
		strcat(path, "*.*");
		filesCnt = 0;

		if (dw.EnumFiles(path))
		{
			do
			{
				if (strcmp(dw.name, "."))
				{
					if (strcmp(dw.name, "..") || !isRoot)
					{
						strcpy(fnames[filesCnt], dw.name);
						folders[filesCnt] = (dw.attr == DW_FOLDER);
						filesCnt++;
					}
				}
			} while (dw.EnumNext());

			dw.EnumClose();
		}

		if (filesCnt > 0)
		{
			if (strcmp(fnames[0], "..")) gl = 0; else gl = 1;
		} else gl = 0;

		for (i = gl; i < filesCnt-1; i++)
			for (j = i+1; j < filesCnt; j++)
			{
				if (FileNameCmp(folders[i], fnames[i], folders[j], fnames[j]) > 0)
				{
					strcpy(temp, fnames[i]);
					strcpy(fnames[i], fnames[j]);
					strcpy(fnames[j], temp);

					tmp = folders[i];
					folders[i] = folders[j];
					folders[j] = tmp;
				}
			}

		pos = 0;
		csr = 0;

		for (i = gl; i < filesCnt; i++)
		{
			if (!strcmp(ofl, fnames[i]))
			{
				csr = i;

				if (filesCnt > mx)
				{
					pos = csr - mx / 2;
					if (pos < 0) pos = 0;
				}

				break;
			}
		}

		if (filesCnt > mx) gl = mx; else gl = filesCnt;

		do
		{
			do
			{
				for (key = 0; SDL_PollEvent(&event);)
				{
					if (event.type == SDL_QUIT) exit(0);
					if (event.type == SDL_KEYDOWN) key = event.key.keysym.sym;
				}

				DlgClearScreen();
				Bar(0, scrEnd, WIDTH-1, HEIGHT-1, DRGB(0x80, 0x20, 0x20));

				Bar(4+(currentDrive*16), scrEnd+4, 4+(currentDrive*16)+12, scrEnd+4+12, DRGB(0x40, 0x10, 0x10));
				font.PrintString(5, scrEnd+5, "A");
				font.PrintString(5+0x10+1, scrEnd+5, "B");
				font.PrintString(5+0x20+1, scrEnd+5, "C");
				font.PrintString(5+0x30+1, scrEnd+5, "D");

				Bar(4+0x50, scrEnd+4, 4+0x50+20, scrEnd+4+12, wd1793_is_disk_wprotected(currentDrive) ? DRGB(255,128,32) : DRGB(0x40,0x10,0x10));
				Bar(4+0x68, scrEnd+4, 4+0x68+20, scrEnd+4+12, wd1793_is_disk_loaded(currentDrive) ? DRGB(255,128,32) : DRGB(0x40,0x10,0x10));
				Bar(4+0x80, scrEnd+4, 4+0x80+20, scrEnd+4+12, wd1793_is_disk_changed(currentDrive) ? DRGB(255,128,32) : DRGB(0x40,0x10,0x10));

				font.PrintString(5+0x50, scrEnd+5, "WP");
				font.PrintString(5+0x68, scrEnd+5, "LD");
				font.PrintString(5+0x80, scrEnd+5, "CH");

				Bar(4+0xB0, scrEnd+4, 4+0xB0+72, scrEnd+4+12, C_Tape::IsActive() ? DRGB(255,128,32) : DRGB(0x40,0x10,0x10));

				if (C_Tape::IsLoaded()) sprintf(buf, "TAPE %d%%", C_Tape::GetPosPerc());
				else sprintf(buf, "TAPE NOP");

				font.PrintString(5+0xB0, scrEnd+5, buf);

				for (i = 0; i < gl; i++)
				{
					if (csr-pos == i) Bar(0, i*h, x+font.StrLenPx(fnames[i+pos])+x, i*h+h-1, DRGB(0x80, 0x20, 0x20));

					if (folders[i+pos]) font.PrintString(0, i*h, "[]");
					font.PrintString(x, i*h, fnames[i+pos]);
				}

				OutputGimpImage(WIDTH - img_zemuIco.width - 8, 8, (s_GimpImage *) &img_zemuIco);
				ScaleImage();
				UpdateScreen();
				SDL_Delay(10);
			} while (key == 0);

			if (key == SDLK_s)
			{
				if (DlgConfirm("Are you sure to save disk? (Y/N)")) {
					wd1793_save_dimage(oldFileName[currentDrive], currentDrive, imgTRD);
				}
			}
			else
			if (key == SDLK_e)
			{
				wd1793_eject_dimage(currentDrive);
			}
			else
			if (key == SDLK_d)
			{
				C_Tape::Eject();
			}
			else
			if (key == SDLK_r)
			{
				C_Tape::Rewind();
			}
			else
			if (key == SDLK_t)
			{
				if (C_Tape::IsActive()) C_Tape::Stop();
				else C_Tape::Start();
			}
			else
			if (key == SDLK_w)
			{
				wd1793_set_disk_wprotected(currentDrive, !wd1793_is_disk_wprotected(currentDrive));
			}
			else
			if (key == SDLK_UP)
			{
				csr--;

				if (csr-pos < 0)
				{
					if (pos > 0) pos--;
					else csr++;
				}
			}
			else
			if (key == SDLK_DOWN)
			{
				csr++;

				if (csr-pos >= gl)
				{
					if (pos+gl < filesCnt) pos++;
					else csr--;
				}
			}
			else
			if (key == SDLK_HOME)
			{
				csr = 0;
				pos = 0;
			}
			else
			if (key == SDLK_END)
			{
				csr = filesCnt - 1;

				if (filesCnt > mx)
				{
					pos = filesCnt - mx;
				}
				else pos = 0;
			}
			else
			if (key == SDLK_PAGEUP)
			{
				pos -= mx;
				csr -= mx;

				if (pos < 0)
				{
					pos = 0;
					csr = 0;
				}
			}
			else
			if (key == SDLK_PAGEDOWN)
			{
				pos += mx;
				csr += mx;

				if (pos+gl > filesCnt)
				{
					if (filesCnt > mx)
					{
						csr = filesCnt - 1;
						pos = filesCnt - mx;
					}
					else
					{
						pos = 0;
						csr = filesCnt - 1;
					}
				}
			}
			else
			if (key == SDLK_LEFT)
			{
				currentDrive--;
				if (currentDrive < 0) currentDrive = 0;
			}
			else
			if (key == SDLK_RIGHT)
			{
				currentDrive++;
				if (currentDrive > 3) currentDrive = 3;
			}

		} while (key!=SDLK_RETURN && key!=SDLK_ESCAPE && key!=SDLK_BACKSPACE);

		do
		{
			SDL_Delay(1);

			for (keyx = 0; SDL_PollEvent(&event);)
			{
				if (event.type == SDL_QUIT) exit(0);
				if (event.type == SDL_KEYUP) keyx = event.key.keysym.sym;
			}
		} while (key != keyx);

		if (key == SDLK_BACKSPACE)
		{
			strcpy(ofl, C_DirWork::LastDirName(path));
			strcpy(path, C_DirWork::LevelUp(path));
		}
		else
		if (key == SDLK_RETURN)
		{
			if (folders[csr])
			{
				strcpy(ofl, C_DirWork::LastDirName(path));

				if (!strcmp(fnames[csr], "..")) strcpy(path, C_DirWork::LevelUp(path));
				else
				{
					strcpy(path, C_DirWork::ExtractPath(path));
					strcat(path, fnames[csr]);
					strcat(path, "/");
				}
			}
			else
			{
				strcpy(path, dw.ExtractPath(path));
				strcat(path, fnames[csr]);
				return path;
			}
		}
	} while (key != SDLK_ESCAPE);

	return NULL;
}

void FileDialog(void)
{
	char *fname;

	disableSound = true;

	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	fname = SelectFile(oldFileName[currentDrive]);
	SDL_EnableKeyRepeat(0, SDL_DEFAULT_REPEAT_INTERVAL);

	if (fname != NULL)
	{
		strcpy(oldFileName[currentDrive], fname);
		TryNLoadFile(oldFileName[currentDrive], currentDrive);
	}

	disableSound = false;
}

void FileDialogInit(void)
{
	char *str;

	if (config.GetString("root/Drives/A", &str)) strcpy(oldFileName[0], str);
	else strcpy(oldFileName[0], "/");
}

void PrintByte(int x, int y, BYTE num)
{
	char buf[3];
	sprintf(buf, "%02X", num);
	font.PrintString(x, y, buf);
}

void PrintWord(int x, int y, WORD num)
{
	char buf[5];
	sprintf(buf, "%04X", num);
	font.PrintString(x, y, buf);
}

#define MAX_INSTR_SIZE 80

struct s_Instruction
{
	WORD addr;
	WORD size;
	char cmd[MAX_INSTR_SIZE];
};

void DebugIt(void)
{
	int key, keyx;
	SDL_Event event;

	int scrEnd = HEIGHT - font.Height() - 8;
	int h = font.Height();
	int mx = scrEnd / h;

	int t, t2;
	char buf[100];
	s_Instruction* dispBuf = new s_Instruction[mx];

	// lastCommandAddr

	WORD curAddr = z80ex_get_reg(cpu, regPC);
	WORD userAddr = curAddr;
	bool correctAddr = true;
	int userPos;

	bool hexMode = false;
	bool exactAddr = false;

	do
	{
		int pos = (mx / 2) - 2;
		WORD addr = userAddr;

		while (pos >= -2)
		{
			WORD nxaddr = addr - 1;
			WORD instrSz = 1;

			for (WORD off = 8; off > 0; off--)
			{
				instrSz = z80ex_dasm(buf, MAX_INSTR_SIZE, 0, &t, &t2, ReadByteDasm, (WORD)(addr - off), NULL);

				if (instrSz == off)
				{
					nxaddr = addr - off;
					break;
				}
			}

			addr = nxaddr;
			pos--;
		}

		addr += z80ex_dasm(buf, MAX_INSTR_SIZE, 0, &t, &t2, ReadByteDasm, addr, NULL);
		addr += z80ex_dasm(buf, MAX_INSTR_SIZE, 0, &t, &t2, ReadByteDasm, addr, NULL);

		pos = 0;
		userPos = (mx / 2) - 1;

		while (pos < mx)
		{
			WORD instrSz = z80ex_dasm(dispBuf[pos].cmd, MAX_INSTR_SIZE, 0, &t, &t2, ReadByteDasm, addr, NULL);

			if (exactAddr && (addr < userAddr && (addr+instrSz) > userAddr))
			{
				strcpy(dispBuf[pos].cmd, "DB ");
				bool sep = false;

				for (WORD tmpAddr = addr; tmpAddr < userAddr ; tmpAddr++)
				{
					if (sep) strcat(dispBuf[pos].cmd, ",");

					sprintf(buf, "%02X", ReadByteDasm(tmpAddr, NULL));
					strcat(dispBuf[pos].cmd, buf);

					sep = true;
				}

				dispBuf[pos].addr = addr;
				dispBuf[pos].size = userAddr - addr;
				addr = userAddr;

				exactAddr = false;
			}
			else
			{
				dispBuf[pos].addr = addr;
				dispBuf[pos].size = instrSz;
				addr += instrSz;
			}

			pos++;
		}

		do
		{
			for (key = 0; SDL_PollEvent(&event);)
			{
				if (event.type == SDL_QUIT) exit(0);
				if (event.type == SDL_KEYDOWN) key = event.key.keysym.sym;
			}
 
	 		DlgClearScreen();
			Bar(0, scrEnd, WIDTH-1, HEIGHT-1, DRGB(0x80, 0x20, 0x20));

			for (int i = 0; i < mx; i++)
			{
				if ( (dispBuf[i].addr <= curAddr) && ((dispBuf[i].addr+dispBuf[i].size) > curAddr) ) {
					Bar(0, i*h, WIDTH/2, i*h+h-1, DRGB(0x80, 0x20, 0x20));
				}

				if (dispBuf[i].addr == userAddr)
				{
					Bar(0, i*h, WIDTH/2, i*h, DRGB(0xFF, 0xCC, 0x00));
					Bar(0, i*h+h-1, WIDTH/2, i*h+h-1, DRGB(0xFF, 0xCC, 0x00));
					Bar(0, i*h+1, 1, i*h+h-2, DRGB(0xFF, 0xCC, 0x00));
					Bar(WIDTH/2-1, i*h+1, WIDTH/2, i*h+h-2, DRGB(0xFF, 0xCC, 0x00));

					userPos = i;
				}

				if (breakpoints[dispBuf[i].addr])
				{
					Bar(WIDTH/2-8-1, i*h+h/2-2-1, WIDTH/2-8+3+1, i*h+h/2+1+1, DRGB(0x00, 0x00, 0x00));
					Bar(WIDTH/2-8, i*h+h/2-2, WIDTH/2-8+3, i*h+h/2+1, DRGB(0xFF, 0x00, 0x00));
				}

				PrintWord(8, i*h, dispBuf[i].addr);

				if (hexMode)
				{
					int xpos = 50;

					for (int j = 0; j < dispBuf[i].size; j++)
					{
					    PrintByte(xpos, i*h, ReadByteDasm(dispBuf[i].addr + j, NULL));
					    xpos += 20;
					}
				}
				else
				{
					font.PrintString(50, i*h, dispBuf[i].cmd);
				}
			}

			int x = WIDTH/2 + 8;
			int cx = x + 30;
			int y = 0;

			font.PrintString(x, y, "AF");	PrintWord(cx, y, z80ex_get_reg(cpu, regAF));	y += h;
			font.PrintString(x, y, "BC");	PrintWord(cx, y, z80ex_get_reg(cpu, regBC));	y += h;
			font.PrintString(x, y, "DE");	PrintWord(cx, y, z80ex_get_reg(cpu, regDE));	y += h;
			font.PrintString(x, y, "HL");	PrintWord(cx, y, z80ex_get_reg(cpu, regHL));	y += h;
			font.PrintString(x, y, "IX");	PrintWord(cx, y, z80ex_get_reg(cpu, regIX));	y += h;
			font.PrintString(x, y, "IY");	PrintWord(cx, y, z80ex_get_reg(cpu, regIY));	y += h;
			y += h;
			font.PrintString(x, y, "AF'");	PrintWord(cx, y, z80ex_get_reg(cpu, regAF_));	y += h;
			font.PrintString(x, y, "BC'");	PrintWord(cx, y, z80ex_get_reg(cpu, regBC_));	y += h;
			font.PrintString(x, y, "DE'");	PrintWord(cx, y, z80ex_get_reg(cpu, regDE_));	y += h;
			font.PrintString(x, y, "HL'");	PrintWord(cx, y, z80ex_get_reg(cpu, regHL_));	y += h;
			y += h;
			font.PrintString(x, y, "SP");	PrintWord(cx, y, z80ex_get_reg(cpu, regSP));	y += h;
			font.PrintString(x, y, "PC");	PrintWord(cx, y, z80ex_get_reg(cpu, regPC));

			x += 80;
			cx = x + 40;
			y = h * 7;

			font.PrintString(x, y, "I");	PrintByte(cx, y, z80ex_get_reg(cpu, regI));	y += h;
			font.PrintString(x, y, "R");	PrintByte(cx, y, z80ex_get_reg(cpu, regR));	y += h;
			y += h;
			font.PrintString(x, y, "IFF1");	PrintByte(cx, y, z80ex_get_reg(cpu, regIFF1));	y += h;
			font.PrintString(x, y, "IFF2");	PrintByte(cx, y, z80ex_get_reg(cpu, regIFF2));	y += h;
			font.PrintString(x, y, "IM");	PrintByte(cx, y, z80ex_get_reg(cpu, regIM));

			x = WIDTH/2 + 8;
			cx = x + 40;
			y = h * 15;

			font.PrintString(x, y, "7FFD"); PrintByte(cx, y, C_MemoryManager::port7FFD);	y += h;
			font.PrintString(x, y, "EFF7"); PrintByte(cx, y, C_ExtPort::portEFF7);		y += h;
			font.PrintString(x, y, "DOS");  font.PrintString(cx, y, (C_TrDos::trdos ? "ON" : "OFF"));

			x += 80;
			cx = x + 50;
			y = h * 15;

			font.PrintString(x, y, "INT?"); font.PrintString(cx, y, (z80ex_int_possible(cpu) ? "Y" : "N"));

			OutputGimpImage(WIDTH - img_zemuIco.width - 8, 8, (s_GimpImage *) &img_zemuIco);
			ScaleImage();
			UpdateScreen();
			SDL_Delay(10);
	 	} while (key == 0);

		if (key == SDLK_UP)
		{
			userAddr = dispBuf[userPos > 0 ? userPos-1 : 0].addr;
		}
		else
		if (key == SDLK_DOWN)
		{
			userAddr = dispBuf[userPos < mx-1 ? userPos+1 : mx-1].addr;
		}
		else
		if (key == SDLK_PAGEUP)
		{
		        userAddr = dispBuf[0].addr;
		}
		else
		if (key == SDLK_PAGEDOWN)
		{
			userAddr = dispBuf[mx-2].addr;
		}
		else
		if (key == SDLK_F2)
		{
			breakpoints[userAddr] = !breakpoints[userAddr];
		}
		else
		if (key == SDLK_F4)
		{
			hexMode = !hexMode;
		}
		else
		if (key=='g' || key=='G')
		{
			const char* buf = DlgInputString("GoTo address");
			int sz = strlen(buf);

			if (sz>=1 && sz<=4)
			{
			    addr = 0;
			    bool goodAddr = true;

			    for (int i = 0; i < sz; i++)
			    {
				if (buf[i]>='0' && buf[i]<='9') addr = addr*0x10 + (buf[i]-'0');
				else
				if (buf[i]>='a' && buf[i]<='f') addr = addr*0x10 + (buf[i]-'a'+10);
				else
				if (buf[i]>='A' && buf[i]<='F') addr = addr*0x10 + (buf[i]-'A'+10);
				else { goodAddr = false; break; }
			    }

			    if (goodAddr)
			    {
			    	userAddr = addr;
			    	exactAddr = true;
			    }
			}
		}
		else
		if (key == SDLK_F7)
		{
			DebugStep();
			curAddr = z80ex_get_reg(cpu, regPC);
			userAddr = curAddr;
    			correctAddr = true;
		}
		else
		if (key == SDLK_F8)
		{
			addr = z80ex_get_reg(cpu, regPC);
			unsigned int maxCnt = 71680 / 4;

			do
			{
				DebugStep();
				maxCnt--;
			} while (z80ex_get_reg(cpu, regPC)==addr && maxCnt);

			curAddr = z80ex_get_reg(cpu, regPC);
			userAddr = curAddr;
    			correctAddr = true;
		}

	} while (key != SDLK_ESCAPE);

	delete[] dispBuf;

	do
	{
		SDL_Delay(1);
		for (keyx = 0; SDL_PollEvent(&event);)
		{
			if (event.type == SDL_QUIT) exit(0);
			if (event.type == SDL_KEYUP) keyx = event.key.keysym.sym;
		}
	} while (key != keyx);
}

void RunDebugger(void)
{
	disableSound = true;
	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);

	DebugIt();

	SDL_EnableKeyRepeat(0, SDL_DEFAULT_REPEAT_INTERVAL);
	disableSound = false;
}

