#include <string.h>
#include <ctype.h>
#include "zemu.h"
#include "exceptions.h"
#include "bin.h"
#include "lib_wd1793/unreal_wd1793.h"
#include "dirwork.h"
#include "ftbos_font.h"
#include "font.h"
#include "dialog.h"
#include "graphics.h"
#include "sdlwaveplayer.h"
#include "cpu_trace.h"
#include "tape/tape.h"
//[boo_boo]
#include "joystick_manager.h"

#include "renderer/render_speccy.h"
#include "renderer/render_16c.h"
#include "renderer/render_multicolor.h"

#include "devs.h"

#include "snap_z80.h"
#include "snap_sna.h"

const unsigned SND_FQ = 44100;

#include "sound/linux.cpp"
#include "sound/win32.cpp"

SNDSAMPLE mixBuffer[MIX_BUFFER_SIZE];
volatile int audioCnt;

int turboMultiplier = 1;
int turboMultiplierNx = 1;

Z80EX_CONTEXT *cpu;
unsigned long long clk, devClk, lastDevClk, devClkCounter;
s_Params params;
SDL_Surface *screen, *scrSurf[2];
int PITCH;
bool drawFrame;
int frames;
C_Config config;
C_Font font;
bool disableSound = false;

int videoSpec;
int actualWidth;
int actualHeight;

SDLWavePlayer sdlWp(SDLWAVE_CALLBACK_BUFFER_SIZE * 4);

char tempFolderName[MAX_PATH];

//--------------------------------------------------------------------------------------------------------------

bool (* hnd_z80read[MAX_HANDLERS])(Z80EX_WORD, bool, Z80EX_BYTE&);
bool (* hnd_z80write[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE);
bool (* hnd_z80input[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE&);
bool (* hnd_z80output[MAX_HANDLERS])(Z80EX_WORD, Z80EX_BYTE);
void (* hnd_frameStart[MAX_HANDLERS])(void);
void (* hnd_afterFrameRender[MAX_HANDLERS])(void);
int types_sdl[MAX_HANDLERS];
bool (* hnd_sdl[MAX_HANDLERS])(SDL_Event&);
void (* hnd_reset[MAX_HANDLERS])(void);

int cnt_z80read = 0;
int cnt_z80write = 0;
int cnt_z80input = 0;
int cnt_z80output = 0;
int cnt_frameStart = 0;
int cnt_afterFrameRender = 0;
int cnt_sdl = 0;
int cnt_reset = 0;

void AttachZ80ReadHandler(bool (* func)(Z80EX_WORD, bool, Z80EX_BYTE&))
{
	if (cnt_z80read >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80read[cnt_z80read++] = func;
}

void AttachZ80WriteHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE))
{
	if (cnt_z80write >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80write[cnt_z80write++] = func;
}

void AttachZ80InputHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE&))
{
	if (cnt_z80input >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80input[cnt_z80input++] = func;
}

void AttachZ80OutputHandler(bool (* func)(Z80EX_WORD, Z80EX_BYTE))
{
	if (cnt_z80output >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_z80output[cnt_z80output++] = func;
}

void AttachFrameStartHandler(void (* func)(void))
{
	if (cnt_frameStart >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_frameStart[cnt_frameStart++] = func;
}

void AttachAfterFrameRenderHandler(void (* func)(void))
{
	if (cnt_afterFrameRender >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_afterFrameRender[cnt_afterFrameRender++] = func;
}

void AttachSDLHandler(int eventType, bool (* func)(SDL_Event&))
{
	if (cnt_sdl >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	types_sdl[cnt_sdl] = eventType;
	hnd_sdl[cnt_sdl++] = func;
}

void AttachResetHandler(void (* func)(void))
{
	if (cnt_reset >= MAX_HANDLERS) StrikeError("Increase MAX_HANDLERS");
	hnd_reset[cnt_reset++] = func;
}

//--------------------------------------------------------------------------------------------------------------

C_Border dev_border;
C_ExtPort dev_extport;
C_Keyboard dev_keyboard;
C_TrDos dev_trdos;
C_MemoryManager dev_mman;
C_TsFm dev_tsfm;
C_Mouse dev_mouse;
//[boo_boo]
C_KempstonStick dev_kempston;

C_Device* devs[] =
{
	&dev_border,
	&dev_extport,
	&dev_keyboard,
	&dev_trdos,
	&dev_mman,
	&dev_tsfm,
	&dev_mouse,
	&dev_kempston, //[boo_boo]
	NULL
};

int colors_digital[0x10] = {
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 128),
	DRGB(128,   0,   0),
	DRGB(128,   0, 128),
	DRGB(  0, 128,   0),
	DRGB(  0, 128, 128),
	DRGB(128, 128,   0),
	DRGB(128, 128, 128),
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 255),
	DRGB(255,   0,   0),
	DRGB(255,   0, 255),
	DRGB(  0, 255,   0),
	DRGB(  0, 255, 255),
	DRGB(255, 255,   0),
	DRGB(255, 255, 255)
};

int colors_breeze[0x10] = {
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 192),
	DRGB(192,   0,   0),
	DRGB(192,   0, 192),
	DRGB(  0, 192,   0),
	DRGB(  0, 192, 192),
	DRGB(192, 192,   0),
	DRGB(192, 192, 192),
	DRGB(  0,   0,   0),
	DRGB(  0,   0, 255),
	DRGB(255,   0,   0),
	DRGB(255,   0, 255),
	DRGB(  0, 255,   0),
	DRGB(  0, 255, 255),
	DRGB(255, 255,   0),
	DRGB(255, 255, 255)
};

int* colors = colors_breeze;

//--------------------------------------------------------------------------------------------------------------

int messageTimeout = 0;
char message[0x100];

void OutputText(char *str)
{
	int x, y;

	x = (WIDTH - font.StrLenPx(str)) / 2;
	y = HEIGHT - font.Height() - 4;

	font.PrintString(x, y, str);
}

void ShowMessage(void)
{
	if (messageTimeout <= 0) return;
	messageTimeout--;
	OutputText(message);
}

void SetMessage(const char *str)
{
	strcpy(message, str);
	messageTimeout = 50;
}

//--------------------------------------------------------------------------------------------------------------

void ResetSequence(void);

void StrToLower(char *str)
{
	while (*str) *(str++) = tolower(*str);
}

#define MAX_FILES 4096
#define MAX_FNAME 256

bool IsArchive(char *fname)
{
	char plgName[MAX_PATH];
	char ext[MAX_FNAME];

	if (!(*params.arcPluginsDir)) return false;
	strcpy(ext, C_DirWork::ExtractExt(fname));
	StrToLower(ext);

	strcpy(plgName, params.arcPluginsDir);
	strcat(plgName, ext);

	return C_File::FileExists(plgName);
}

void LoadNormalFile(char *fname, int drive)
{
	if (!stricmp(C_DirWork::ExtractExt(fname), "tap")) {
		C_Tape::Insert(fname);
	} else if (!stricmp(C_DirWork::ExtractExt(fname), "z80")) {
		if (!load_z80_snap(fname, cpu, dev_mman, dev_border)) StrikeMessage("Error loading snapshot");
	} else if (!stricmp(C_DirWork::ExtractExt(fname), "sna")) {
		if (!load_sna_snap(fname, cpu, dev_mman, dev_border)) StrikeMessage("Error loading snapshot");
	} else {
		wd1793_load_dimage(fname, drive);
	}
}

void LoadArcFile(char *arcName, int drive)
{
	C_File fl;
	char plgName[MAX_PATH], res[MAX_PATH];
	char tmp[MAX_PATH], str[MAX_FNAME];
	char files[MAX_FILES][MAX_FNAME];
	int filesCount;

	strcpy(tmp, C_DirWork::ExtractExt(arcName));
	StrToLower(tmp);

	strcpy(plgName, params.arcPluginsDir);
	strcat(plgName, tmp);

	sprintf(tmp, "%s list %s %s/files.txt", plgName, arcName, tempFolderName);
	system(tmp);

	sprintf(tmp, "%s/files.txt", tempFolderName);
	if (!C_File::FileExists(tmp)) return;

	fl.Read(tmp);
	for (filesCount = 0; !fl.Eof(); )
	{
		fl.GetS(str, sizeof(str));
		if (strcmp(str, "")) strcpy(files[filesCount++], str);
	}

	unlink(tmp);
	if (!filesCount) return;

	// currently load only first file
	sprintf(tmp, "%s extract %s %s %s", plgName, arcName, files[0], tempFolderName);
	system(tmp);
	strcpy(tmp, C_DirWork::ExtractFileName(files[0]));

	// TODO: check if lresult strlen > sizeof(res)
	sprintf(res, "%s/%s", tempFolderName, tmp);

	LoadNormalFile(res, drive);
	unlink(res);
}

void TryNLoadFile(char *fname, int drive)
{
	if (IsArchive(fname)) LoadArcFile(fname, drive);
	else LoadNormalFile(fname, drive);
}

void TryNLoadFile(char *fname)
{
	TryNLoadFile(fname, 0);
}

//--------------------------------------------------------------------------------------------------------------

void Action_Reset(void) {
	ResetSequence();
}

void Action_ResetTrDos(void)
{
	ResetSequence();
	dev_mman.OnOutputByte(0x7FFD, 0x10);
	dev_trdos.trdos = true;
}

void Action_MaxSpeed(void) {
	params.maxSpeed = !params.maxSpeed;
	SetMessage(params.maxSpeed ? "MaxSpeed ON" : "MaxSpeed OFF");
}

void Action_QuickLoad(void) {
	if (!load_z80_snap("snap.z80", cpu, dev_mman, dev_border)) SetMessage("Error loading snapshot");
	else SetMessage("Snapshot loaded");
}

void Action_QuickSave(void) {
	save_z80_snap("snap.z80", cpu, dev_mman, dev_border);
	SetMessage("Snapshot saved");
}

void Action_AntiFlicker(void)
{
	params.antiFlicker = !params.antiFlicker;
	SetMessage(params.antiFlicker ? "AntiFlicker ON" : "AntiFlicker OFF");
}

void Action_LoadFile(void) {
	FileDialog();
}

void Action_Fullscreen(void)
{
	/* SDL_WM_ToggleFullscreen works only on X11 */

	videoSpec ^= SDL_FULLSCREEN;
	SDL_FreeSurface(screen);
	screen = SDL_SetVideoMode(actualWidth, actualHeight, 32, videoSpec);
	PITCH = screen->pitch / 4;
}

void Action_Debugger(void) {
	RunDebugger();
}

void Action_Turbo(void)
{
	turboMultiplierNx *= 2;

	if (turboMultiplierNx > 4) turboMultiplierNx = 1;

	if (turboMultiplierNx < 2) {
		SetMessage("Turbo OFF");
	} else {
		char buf[0x10];
		sprintf(buf, "Turbo %dx", turboMultiplierNx);
		SetMessage(buf);
	}
}

void Action_AttributesHack(void)
{
	attributesHack = !attributesHack;
	SetMessage(attributesHack ? "AttributesHack ON" : "AttributesHack OFF");
}

s_Action cfgActions[] =
{
	{"reset",		Action_Reset},
	{"reset_trdos",		Action_ResetTrDos},
	{"max_speed",		Action_MaxSpeed},
	{"quick_load",		Action_QuickLoad},
	{"quick_save",		Action_QuickSave},
	{"anti_flicker",	Action_AntiFlicker},
	{"load_file",		Action_LoadFile},
	{"fullscreen",		Action_Fullscreen},
	{"debugger",		Action_Debugger},
	{"turbo",			Action_Turbo},
	{"attrib_hack",		Action_AttributesHack},
	{"",			NULL}
};

//--------------------------------------------------------------------------------------------------------------

bool runDebuggerFlag = false;
Z80EX_WORD lastCommandAddr = 0;
bool breakpoints[0x10000];

Z80EX_BYTE ReadByteDasm(Z80EX_WORD addr, void *userData)
{
	int i;
	Z80EX_BYTE retval;

	for (i = 0; i < cnt_z80read; i++) {
		if (hnd_z80read[i](addr, false, retval)) return retval;
	}

	return 0xFF;
}

Z80EX_BYTE ReadByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, int m1_state, void *userData)
{
	int i;
	Z80EX_BYTE retval;

	if (m1_state) lastCommandAddr = addr;

	for (i = 0; i < cnt_z80read; i++) {
		if (hnd_z80read[i](addr, m1_state, retval)) return retval;
	}

	return 0xFF;
}

void WriteByte(Z80EX_CONTEXT *cpu, Z80EX_WORD addr, Z80EX_BYTE value, void *userData)
{
	int i;
	for (i = 0; i < cnt_z80write; i++) {
		if (hnd_z80write[i](addr, value)) return;
	}
}

Z80EX_BYTE InputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, void *userData)
{
	int i;
	Z80EX_BYTE retval;

	for (i = 0; i < cnt_z80input; i++) {
		if (hnd_z80input[i](port, retval)) return retval;
	}

	return 0xFF;
}

void OutputByte(Z80EX_CONTEXT *cpu, Z80EX_WORD port, Z80EX_BYTE value, void *userData)
{
//	if (port == 0xEFF7) runDebuggerFlag = true;

	int i;
	for (i = 0; i < cnt_z80output; i++) {
		if (hnd_z80output[i](port, value)) return;
	}
}

Z80EX_BYTE ReadIntVec(Z80EX_CONTEXT *cpu, void *userData)
{
	return 0xFF;
}

//--------------------------------------------------------------------------------------------------------------

void InitSurfaces(void)
{
	SDL_PixelFormat *fmt = screen->format;

	scrSurf[0] = SDL_CreateRGBSurface(SDL_SWSURFACE, WIDTH, HEIGHT, fmt->BitsPerPixel, fmt->Rmask, fmt->Gmask, fmt->Bmask, 0);
	if (scrSurf[0] == NULL) StrikeError("Unable to create primary surface: %s\n", SDL_GetError());

	scrSurf[1] = SDL_CreateRGBSurface(SDL_SWSURFACE, WIDTH, HEIGHT, fmt->BitsPerPixel, fmt->Rmask, fmt->Gmask, fmt->Bmask, 0);
	if (scrSurf[1] == NULL) StrikeError("Unable to create secondary surface: %s\n", SDL_GetError());
}

void InitFont(void)
{
	font.Init(ftbos_font_data);
	font.CopySym('-', '_');
	font.SetSymOff('_', 0, 4);
	font.SetSymOff('-', 0, 1);
}

void InitAll(void)
{
	int i;
	for (i = 0; devs[i]; i++) devs[i]->Init();

	InitSurfaces();
	InitFont();
	FileDialogInit();

	for (i = 0; i < 0x10000; i++) breakpoints[i] = false;

	cpu = z80ex_create(
			ReadByte, NULL,
			WriteByte, NULL,
			InputByte, NULL,
			OutputByte, NULL,
			ReadIntVec, NULL
			);
}

#define MAX_FRAME_TACTS 71680
#define INT_LENGTH 32
#define INT_BEGIN 68069

bool attributesHack = false;

// processor_speed    = 3546900
// ay_speed           = 1773400

// left_border        = 36
// horizontal_screen  = 128
// right_border       = 28
// horizontal_retrace = 32

// top_border         = 64
// vertical_screen    = 192
// bottom_border      = 48
// vertical_retrace   = 16

// top_left_pixel     = 17988

void AntiFlicker(void)
{
	int i, j;
	BYTE *s1, *s2, *sr;
	BYTE *s1w, *s2w, *srw;

	if (SDL_MUSTLOCK(screen)) {if (SDL_LockSurface(screen) < 0) return;}
	if (SDL_MUSTLOCK(scrSurf[0])) {if (SDL_LockSurface(scrSurf[0]) < 0) return;}
	if (SDL_MUSTLOCK(scrSurf[1])) {if (SDL_LockSurface(scrSurf[1]) < 0) return;}

	sr = (BYTE *)screen->pixels;
	s1 = (BYTE *)scrSurf[0]->pixels;
	s2 = (BYTE *)scrSurf[1]->pixels;

	for (i = 0; i < HEIGHT; i++)
	{
		srw = sr;
		s1w = s1;
		s2w = s2;

		for (j = 0; j < WIDTH; j++)
		{
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
			*srw = (BYTE)(((unsigned int)(*s1w) + (unsigned int)(*s2w)) >> 1);
			srw++; s1w++; s2w++;
		}

		sr += screen->pitch;
		s1 += scrSurf[0]->pitch;
		s2 += scrSurf[1]->pitch;
	}

	if (SDL_MUSTLOCK(scrSurf[1])) SDL_UnlockSurface(scrSurf[1]);
	if (SDL_MUSTLOCK(scrSurf[0])) SDL_UnlockSurface(scrSurf[0]);
	if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
}

int actDevClkCounter = 0;
int actClk = 0;

void InitActClk(void)
{
	actDevClkCounter = devClkCounter * turboMultiplier;
	actClk = clk * turboMultiplier;
}

inline void CpuCalcTacts(unsigned long cmdClk)
{
	if (turboMultiplier > 1)
	{
		actDevClkCounter += cmdClk;
		actClk += cmdClk;

		devClkCounter = (actDevClkCounter + turboMultiplier-1) / turboMultiplier;
		clk = (actClk + turboMultiplier-1) / turboMultiplier;
	}
	else
	{
		devClkCounter += cmdClk;
		clk += cmdClk;
	}

	devClk = clk;
	C_Tape::Process();

	if (runDebuggerFlag)
	{
		runDebuggerFlag = false;
		RunDebugger();
	}
	else
	if (breakpoints[z80ex_get_reg(cpu, regPC)])
	{
		RunDebugger();
	}
}

inline void DebugCpuCalcTacts(unsigned long cmdClk)
{
	if (turboMultiplier > 1)
	{
		actDevClkCounter += cmdClk;
		actClk += cmdClk;

		devClkCounter = (actDevClkCounter + turboMultiplier-1) / turboMultiplier;
		clk = (actClk + turboMultiplier-1) / turboMultiplier;
	}
	else
	{
		devClkCounter += cmdClk;
		clk += cmdClk;
	}

	devClk = clk;
	C_Tape::Process();
}

int (* DoCpuStep)(Z80EX_CONTEXT *cpu) = z80ex_step;
int (* DoCpuInt)(Z80EX_CONTEXT *cpu) = z80ex_int;

int TraceCpuStep(Z80EX_CONTEXT *cpu)
{
	CpuTrace_Log();
	cpuTrace_dT = z80ex_step(cpu);
	return cpuTrace_dT;
}

int TraceCpuInt(Z80EX_CONTEXT *cpu)
{
	// CpuTrace_Log();
	cpuTrace_dT = z80ex_int(cpu);
	return cpuTrace_dT;
}

inline void CpuStep(void)
{
	CpuCalcTacts(DoCpuStep(cpu));
}

inline void CpuInt(void)
{
	CpuCalcTacts(DoCpuInt(cpu));
}

inline void DebugCpuStep(void)
{
	DebugCpuCalcTacts(DoCpuStep(cpu));
}

inline void DebugCpuInt(void)
{
	DebugCpuCalcTacts(DoCpuInt(cpu));
}

void DebugStep(void)
{
	int cnt = 4;
	unsigned long cmdClk;

	do
	{
		if (clk < MAX_FRAME_TACTS)
		{
			CpuStep();
			if (clk>=INT_BEGIN && clk<(INT_BEGIN+INT_LENGTH)) CpuInt();
		}
		else
		{
			lastDevClk = devClk;
			clk -= MAX_FRAME_TACTS;
			devClk = clk;

			InitActClk();
		}

		cnt--;
	} while (z80ex_last_op_type(cpu) && cnt>0);
}

void Render(void)
{
	SDL_Surface *surf;
	unsigned long lastClk;
	static int sn = 0;
	unsigned long (* render)(SDL_Surface*, int, unsigned long, unsigned long);

	if (params.antiFlicker)
	{
		surf = scrSurf[sn];
		sn = 1 - sn;
	} else surf = screen;

	if (SDL_MUSTLOCK(surf)) {if (SDL_LockSurface(surf) < 0) return;}
	int ptch = surf->pitch / 4;

	if (dev_extport.Is16Colors()) render = Render16c;
	else if (dev_extport.IsMulticolor()) render = RenderMulticolor;
//	else if (dev_extport.Is512x192()) render = Render512x192;
//	else if (dev_extport.Is384x304()) render = Render384x304;
	else render = RenderSpeccy;

	InitActClk();
	lastClk = 0;

	if (drawFrame)
	{
		while (clk < MAX_FRAME_TACTS)
		{
			CpuStep();
			lastClk = render(surf, ptch, lastClk, clk);

			if (clk>=INT_BEGIN && clk<(INT_BEGIN+INT_LENGTH)) CpuInt();
		}
	}
	else
	{
		while (clk < MAX_FRAME_TACTS)
		{
			CpuStep();
			if (clk>=INT_BEGIN && clk<(INT_BEGIN+INT_LENGTH)) CpuInt();
		}
	}

	lastDevClk = devClk;
	clk -= MAX_FRAME_TACTS;
	devClk = clk;

	if (SDL_MUSTLOCK(surf)) SDL_UnlockSurface(surf);
	if (params.antiFlicker && drawFrame) AntiFlicker();
}

#include "images/floppy.h"
#include "images/floppy_read.h"
#include "images/floppy_write.h"
#include "images/turbo_off.h"
#include "images/turbo_on.h"

void DrawIndicators(void)
{
	DRIVE_STATE st = dev_trdos.GetIndicatorState();

	if (st == DS_READ) OutputGimpImage(8, 0, (s_GimpImage *) &img_floppyRead);
	else if (st == DS_WRITE) OutputGimpImage(8, 0, (s_GimpImage *) &img_floppyWrite);
	else if (params.showInactiveIcons) OutputGimpImage(8, 0, (s_GimpImage *) &img_floppy);

	if (params.maxSpeed) OutputGimpImage(32, 0, (s_GimpImage *) &img_turboOn);
	else if (params.showInactiveIcons) OutputGimpImage(32, 0, (s_GimpImage *) &img_turboOff); 

	if (C_Tape::IsActive())
	{
		char buf[0x100];
		sprintf(buf, "%d%%", C_Tape::GetPosPerc());

		int wdt = font.StrLenPx(buf);
		font.PrintString(WIDTH - 4 - wdt, 4, buf);
	}
}

void ResetSequence(void)
{
	int i;
	z80ex_reset(cpu);
	for (i = 0; i < cnt_reset; i++) hnd_reset[i]();
}

void ScaleImage(void)
{
	if (!params.scale2x) return;
	if (SDL_MUSTLOCK(screen)) {if (SDL_LockSurface(screen) < 0) return;}

	if (params.scanlines)
	{
		for (int i = HEIGHT-1; i >= 0; i--)
		{
			int* line = (int*)screen->pixels + i * PITCH + WIDTH-1;
			int* lineA = (int*)screen->pixels + (i*2) * PITCH + (WIDTH*2-1);
			int* lineB = (int*)screen->pixels + (i*2+1) * PITCH + (WIDTH*2-1);

			for (int j = WIDTH; j--;)
			{
				int c = *(line--);

				*(lineA--) = c;
				*(lineA--) = c;
				*(lineB--) = 0;
				*(lineB--) = 0;
			}
		}
	}
	else
	{
		for (int i = HEIGHT-1; i >= 0; i--)
		{
			int* line = (int*)screen->pixels + i * PITCH + WIDTH-1;
			int* lineA = (int*)screen->pixels + (i*2) * PITCH + (WIDTH*2-1);
			int* lineB = (int*)screen->pixels + (i*2+1) * PITCH + (WIDTH*2-1);

			for (int j = WIDTH; j--;)
			{
				int c = *(line--);

				*(lineA--) = c;
				*(lineA--) = c;
				*(lineB--) = c;
				*(lineB--) = c;
			}
		}
	}

	if (SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
}

void Process(void)
{
	int key;
	SDL_Event event;
	int i, j, *s, *p, st;
	unsigned int btick;
	int frameSkip = 0;

	devClkCounter = 0;
	clk = 0;
	devClk = 0;
	lastDevClk = 0;
	frames = 0;
	params.maxSpeed = false;

	btick = SDL_GetTicks() + FRAME_WAIT_MS;

	for (;;)
	{
		for (i = 0; i < cnt_frameStart; i++) hnd_frameStart[i]();

		if (params.maxSpeed)
		{
			if (frameSkip > 0) {
				frameSkip--;
				drawFrame = false;
			} else {
				frameSkip = MAX_SPEED_FRAME_SKIP;
				drawFrame = true;
			}
		} else drawFrame = true;

		bool tapePrevActive = C_Tape::IsActive();

		Render();
		frames++;

		if (drawFrame)
		{
			DrawIndicators();
			ShowMessage();
			ScaleImage();
			UpdateScreen();
		}

		if (!params.maxSpeed)
		{
			SDL_Delay(1);
			if (SDL_GetTicks() < btick) SDL_Delay(btick-SDL_GetTicks());
		}

		btick = SDL_GetTicks() + FRAME_WAIT_MS;

		for (i = 0; i < cnt_afterFrameRender; i++) hnd_afterFrameRender[i]();

		if (params.sound && !params.maxSpeed)
		{
			int samples = min(dev_tsfm.samples, dev_border.samples);

			if (samples)
			{
				SNDSAMPLE *o = dev_tsfm.soundBuffer;
				SNDSAMPLE *p = dev_border.soundBuffer;
				SNDSAMPLE *r = mixBuffer;

				for (i = samples; i--;)
				{
					r->ch.left = (o->ch.left >> 1) + (p->ch.left >> 1);
					r->ch.right = (o->ch.right >> 1) + (p->ch.right >> 1);

					o++; p++; r++;
				}

				if (params.sdl_sound) sdlWp.Write((BYTE*)mixBuffer, samples * sizeof(SNDSAMPLE));
				else wav_play(mixBuffer, samples);
			}

			if (dev_tsfm.samples > samples) memmove(dev_tsfm.soundBuffer, &dev_tsfm.soundBuffer[samples], (dev_tsfm.samples - samples) * sizeof(SNDSAMPLE));
			if (dev_border.samples > samples) memmove(dev_border.soundBuffer, &dev_border.soundBuffer[samples], (dev_border.samples - samples) * sizeof(SNDSAMPLE));

			dev_tsfm.samples -= samples;
			dev_border.samples -= samples;
		}
		else
		{
			dev_tsfm.samples = 0;
			dev_border.samples = 0;
		}

		while (SDL_PollEvent(&event))
		{
			if (event.type == SDL_QUIT) exit(0);

			if (event.type == SDL_KEYUP)
			{
				key = event.key.keysym.sym;
				if (key == SDLK_ESCAPE) return;
			}

			for (i = 0; i < cnt_sdl; i++) {
				if (types_sdl[i] == event.type) {
					if (hnd_sdl[i](event)) break;
				}
			}
		}

		turboMultiplier = turboMultiplierNx;

		if (tapePrevActive && !C_Tape::IsActive())
		{
			if (params.maxSpeed)
			{
				SetMessage("Tape end : MaxSpeed OFF");
				params.maxSpeed = false;
			}
			else SetMessage("Tape end");
		}
	}
}

void InitAudio(void)
{
	if (params.sdl_sound) sdlWp.Init();
	else wav_start(params.soundParam, AUDIO_HW_BUFFER);
}

void FreeAudio(void)
{
	if (params.sdl_sound) SDL_CloseAudio();
	else wav_stop();
}

void UpdateScreen(void)
{
    if (params.useFlipSurface) SDL_Flip(screen);
    else SDL_UpdateRect(screen, 0, 0, 0, 0);
}

void FreeAll(void)
{
	int i;
	for (i = 0; devs[i]; i++) devs[i]->Close();

	C_Tape::Close();

	#ifdef OS_LINUX
		// TODO: do in in more portable way
		char cmd[MAX_PATH];
		strcpy(cmd, "rm -rf ");
		strcat(cmd, tempFolderName);
		system(cmd);
	#endif

	if (videoSpec & SDL_FULLSCREEN)
	{
	    videoSpec ^= SDL_FULLSCREEN;
	    SDL_FreeSurface(screen);
	    screen = SDL_SetVideoMode(actualWidth, actualHeight, 32, videoSpec);
	}

	SDL_FreeSurface(scrSurf[0]);
	SDL_FreeSurface(scrSurf[1]);
	SDL_FreeSurface(screen);

	z80ex_destroy(cpu);
	if (params.sound) FreeAudio();
}

void ParseCmdLine(int argc, char *argv[])
{
	if (argc != 2) return;
	TryNLoadFile(argv[1]);
}

void OutputLogo(void)
{
	printf("                                     \n");
	printf("   $ww,.                             \n");
	printf("    `^$$$ww,.                        \n");
	printf("       `^$$$$$$                      \n");
	printf("         ,$$7'                       \n");
	printf("       _j$$'   __    __  _ _         \n");
	printf("     ,j$$7      /__ (-_ | ) ) (_|    \n");
	printf("    $$$$$$w.                         \n");
	printf("     `^^T$$$w,             rst'o6    \n");
	printf("           `^T$$                     \n");
	printf("                                     \n");
	printf(" restorer [ restorer.fct@gmail.com ] \n");
	printf(" boo_boo  [    boo_boo@inbox.ru    ] \n");
	printf("                                     \n");
	printf(" breeze (gfx)                        \n");
	printf("                                     \n");
	printf(" with help of SMT                    \n");
	printf("                                     \n");
}

void InitCfgName(char* cfg_fname)
{
	char* homePath = getenv("HOME");

	if (homePath)
	{
		#ifdef OS_LINUX
			// TODO: do in in more portable way
			char cmd[MAX_PATH];
			sprintf(cmd, "mkdir -p %s/.zemu", homePath);
			system(cmd);
		#endif

		strcat(cfg_fname, homePath);
		strcat(cfg_fname, "/.zemu/config.xml");

		if (C_File::FileExists(cfg_fname)) return;
	}

	strcpy(cfg_fname, CONFIG_PATH);
	strcat(cfg_fname, "/config.xml");
}

int main(int argc, char *argv[])
{
	int spec;
	char *str;
	bool wd1793delay;

	OutputLogo();

	try
	{
		char cfg_fname[MAX_PATH] = "";
		InitCfgName(cfg_fname);

		if (C_File::FileExists(cfg_fname)) config.Load(cfg_fname);
		else StrikeError("Unable to load config from \"%s\"", cfg_fname);

		if (config.GetString("root/ArcPluginsDir", &str)) strcpy(params.arcPluginsDir, str);
		else *params.arcPluginsDir = 0;

		if (config.GetString("root/CpuTrace/Format", &str)) strcpy(params.cpuTraceFormat, str);
		else *params.cpuTraceFormat = 0;

		if (config.GetString("root/CpuTrace/FileName", &str)) strcpy(params.cpuTraceFileName, str);
		else *params.cpuTraceFileName = 0;

		if (!config.GetBool("root/UseFlipSurface", &params.useFlipSurface)) params.useFlipSurface = false;
		if (!config.GetBool("root/SoundEnable", &params.sound)) params.sound = false;
		if (!config.GetBool("root/FullScreen", &params.fullscreen)) params.fullscreen = false;
		if (!config.GetBool("root/Scale2x", &params.scale2x)) params.scale2x = false;
		if (!config.GetBool("root/Scanlines", &params.scanlines)) params.scanlines = false;
		if (!config.GetBool("root/AntiFlicker", &params.antiFlicker)) params.antiFlicker = false;
		if (!config.GetBool("root/UseSdlSound", &params.sdl_sound)) params.sdl_sound = false;
		if (!config.GetBool("root/ShowInactiveIcons", &params.showInactiveIcons)) params.showInactiveIcons = true;
		if (!config.GetBool("root/CpuTrace/Enabled", &params.cpuTraceEnabled)) params.cpuTraceEnabled = false;

		if (config.GetString("root/Drives/A", &str)) wd1793_load_dimage(str, 0);
		if (config.GetString("root/Drives/B", &str)) wd1793_load_dimage(str, 1);
		if (config.GetString("root/Drives/C", &str)) wd1793_load_dimage(str, 2);
		if (config.GetString("root/Drives/D", &str)) wd1793_load_dimage(str, 3);

		if (config.GetBool("root/Drives/NoDelay", &wd1793delay)) wd1793_set_nodelay(wd1793delay);

		if (config.GetString("root/Drives/AppendBoot", &str)) {
			if (strcmp(str, "")) wd1793_set_appendboot(str);
		}

		if (!config.GetInt("root/MouseDiv", &params.mouseDiv)) params.mouseDiv = 1;
		else {
			if (params.mouseDiv <= 0) params.mouseDiv = 1;
			if (params.mouseDiv > 8) params.mouseDiv = 8;
		}

		#ifdef OS_LINUX
			if (!config.GetInt("root/OssFragNum", &params.soundParam)) params.soundParam = 8;
		#endif
		#ifdef OS_WINDOWS
			if (!config.GetInt("root/WqSize", &params.soundParam)) params.soundParam = 4;
		#endif

		spec = SDL_INIT_VIDEO;
		if (params.sound && params.sdl_sound) spec |= SDL_INIT_AUDIO;
		if (SDL_Init(spec) < 0) StrikeError("Unable to init SDL: %s\n", SDL_GetError());

		atexit(SDL_Quit);

		videoSpec = SDL_SWSURFACE;
		if (params.useFlipSurface) videoSpec |= SDL_DOUBLEBUF;
		if (params.fullscreen) videoSpec |= SDL_FULLSCREEN;

		int actualWidth = WIDTH;
		int actualHeight = HEIGHT;

		if (params.scale2x)
		{
			actualWidth *= 2;
			actualHeight *= 2;
		}

		screen = SDL_SetVideoMode(actualWidth, actualHeight, 32, videoSpec);

		if (screen == NULL) StrikeError("Unable to set requested video mode: %s\n", SDL_GetError());
		PITCH = screen->pitch / 4;

		SDL_WM_SetCaption("ZEmu", "ZEmu");
		SDL_ShowCursor(SDL_DISABLE);

		#ifdef OS_WINDOWS
			strcpy(tempFolderName, "./_temp");
		#endif
		#ifdef OS_LINUX
			strcpy(tempFolderName, "/tmp/zemu-XXXXXX");
			mkdtemp(tempFolderName);
		#endif

		atexit(FreeAll);

		if (params.cpuTraceEnabled)
		{
			DoCpuStep = TraceCpuStep;
			DoCpuInt = TraceCpuInt;
			CpuTrace_Init();
		}

		InitAll();
		ResetSequence();
		if (argc != 1) ParseCmdLine(argc, argv);
		if (params.sound) InitAudio();
		
		//[boo_boo]
		C_JoystickManager *jmanager = C_JoystickManager::Instance();
		//
		if(config.GetBool("root/Joysticks/Kempston/Enable", &params.KempstonEnable) && params.KempstonEnable)
		{
			if(!config.GetInt("root/Joysticks/Kempston/SysJoystickNum", &params.KempstonOnStickNum))
				StrikeError("Joystick system number must be given");
			
			if(!config.GetInt("root/Joysticks/Kempston/AxisTreshold", &params.KempstonAxisTreshold))
				params.KempstonAxisTreshold = DEF_JOY_AXIS_TRESHOLD;
			
			if(jmanager->AddJoystick(params.KempstonOnStickNum, params.KempstonAxisTreshold))
			{
				dev_kempston.SetJoyNum(params.KempstonOnStickNum);

				jmanager->EnableProcessing();
			}
			else StrikeMessage("Error initializing kempston joystick");
		}
		
		Process();

		if (params.cpuTraceEnabled) CpuTrace_Close();
	}
	catch (C_E &e)
	{
		StrikeError("Error %d: %s (%s)", e.exc, e.Descr(), e.param);
	}

	return 0;
}

